package jp.co.sra.jun.opengl.grapher;

import java.awt.Point;
import java.awt.Rectangle;
import java.io.Writer;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Vector;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispNil;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.picking.JunOpenGLPickingObjects;

/**
 * JunOpenGL3dGraph class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/12/07 (by Hirotsugu Kondo)
 *  @updated   2004/11/12 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun630 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dGraph.java,v 8.13 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGL3dGraph extends JunOpenGL3dGraphAbstract implements JunOpenGLPickingObjects {
	protected Vector graphNodes;
	protected Vector graphArcs;
	protected Dictionary arrangeFormat;

	/**
	 * Create a new instance of <code>JunOpenGL3dGraph</code> from a JunLispList.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @deprecated since Jun454, use the constructor
	 * @category Lisp support
	 */
	public static JunOpenGL3dGraphAbstract FromLispList_(JunLispList aList) {
		return new JunOpenGL3dGraph(aList);
	}

	/**
	 * Load a <code>JunOpenGL3dGraph</code> from the object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param anObject java.lang.Object
	 * @category Lisp support
	 */
	public static JunOpenGL3dObject LoadFrom_(Object anObject) {
		JunLispList aList = (JunLispList) JunLispParser.Parse_(anObject);
		return JunOpenGL3dGraph.FromLispList_(aList);
	}

	/**
	 * Create a new instance of <code>JunOpenGL3dGraph</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dGraph() {
		super();
	}

	/**
	 * Create a new instance of <code>JunOpenGL3dGraph</code> and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dGraph(JunLispList aList) {
		super(aList);
	}

	/**
	 * Generate a tree from a lisp list with node block and arc block.
	 * 
	 * @param table java.util.Hashtable
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category generating
	 */
	public void _generateTreeFromLispList_nodeBlock_arcBlock_(Hashtable table, JunLispList aList, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		if (StSymbol.class.isInstance(aList)) {
			return;
		}

		JunOpenGL3dNode fromNode = null;
		if (table.containsKey(((JunLispCons) aList).head())) {
			fromNode = (JunOpenGL3dNode) table.get(((JunLispCons) aList).head());
		} else {
			fromNode = (JunOpenGL3dNode) nodeBlock.value_(((JunLispCons) aList).head());
			this.addNode_(fromNode);
			table.put(((JunLispCons) aList).head(), fromNode);
		}
		if (JunLispNil.class.isInstance(((JunLispCons) aList).tail())) {
			return;
		}

		final JunLispCons tails = (JunLispCons) ((JunLispCons) aList).tail();
		final Hashtable table_ = table;
		final JunOpenGL3dNode fromNode_ = fromNode;
		final StBlockClosure nodeBlock_ = nodeBlock;
		final StBlockClosure arcBlock_ = arcBlock;
		final JunOpenGL3dGraph this_ = this;
		tails.do_(new StBlockClosure() {
			public Object value_(Object object) {
				JunOpenGL3dNode toNode;
				JunLispCons each = (JunLispCons) object;

				if (table_.containsKey(each.head())) {
					toNode = (JunOpenGL3dNode) table_.get(each.head());
				} else {
					toNode = (JunOpenGL3dNode) nodeBlock_.value_(each.head());
					this_.addNode_((JunOpenGL3dNode) toNode);
					table_.put(each.head(), toNode);
				}

				JunOpenGL3dArc anArc_ = (JunOpenGL3dArc) arcBlock_.value_value_(fromNode_, toNode);
				this_.addArc_(anArc_);

				return null;
			}
		});

		final JunLispCons tails2 = (JunLispCons) ((JunLispCons) aList).tail();
		tails2.do_(new StBlockClosure() {
			public Object value_(Object object) {
				JunLispList each = (JunLispList) object;
				this_._generateTreeFromLispList_nodeBlock_arcBlock_(table_, each, nodeBlock_, arcBlock_);
				return null;
			}
		});
	}

	/**
	 * Added an arc object.
	 * 
	 * @param anArc jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc
	 * @category arc accessing
	 */
	public void addArc_(JunOpenGL3dArc anArc) {
		JunOpenGL3dArc[] oldArcs = this.arcs();
		Vector newArcs = new Vector();
		int size = oldArcs.length;

		for (int index = 0; index < size; index++) {
			newArcs.addElement(oldArcs[index]);
		}

		newArcs.addElement(anArc);
		this.arcs_(newArcs);
		this.flushDisplayObject();
	}

	/**
	 * Added a node object.
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category node accessing
	 */
	public void addNode_(JunOpenGL3dNode aNode) {
		JunOpenGL3dNode[] oldNodes = this.nodes();
		Vector newNoodes = new Vector();
		int size = oldNodes.length;

		for (int index = 0; index < size; index++) {
			newNoodes.addElement(oldNodes[index]);
		}

		newNoodes.addElement(aNode);
		this.nodes_(newNoodes);
		this.flushDisplayObject();
	}

	/**
	 * Get the arc objects as array.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc[]
	 * @category arc accessing
	 */
	public JunOpenGL3dArc[] arcs() {
		Vector arcs = graphArcs;
		if (arcs == null) {
			arcs = new Vector();
			this.graphArcs = arcs;
		}

		int size = arcs.size();
		JunOpenGL3dArc[] result = new JunOpenGL3dArc[size];
		for (int index = 0; index < size; index++) {
			result[index] = (JunOpenGL3dArc) arcs.elementAt(index);
		}
		return result;
	}

	/**
	 * Get the arc objects.
	 * 
	 * @param newVector java.util.Vector
	 * @category arc accessing
	 */
	private void arcs_(Vector newVector) {
		graphArcs = newVector;
	}

	/**
	 * Enumerate every arc objects and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category arc accessing
	 */
	public void arcsDo_(StBlockClosure aBlock) {
		JunOpenGL3dArc[] arcs = this.arcs();
		int size = arcs.length;
		for (int i = 0; i < size; i++) {
			aBlock.value_(arcs[i]);
		}
	}

	/**
	 * Get my arcs from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected void arcsFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("arcs")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		final JunOpenGL3dGraph aGraph_ = this;
		((JunLispCons) list.tail()).do_(new StBlockClosure() {
			public Object value_(Object each) {
				try {
					JunOpenGL3dArc arc = new JunOpenGL3dArc((JunLispCons) each);
					aGraph_.addArc_(arc);
				} catch (Exception e) {
					throw new SmalltalkException(e);
				}

				return null;
			}
		});
	}

	/**
	 * Convert the receiver's arcs as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons arcsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("arcs"));

		final JunOpenGL3dGraph this_ = this;
		final JunLispCons list_ = list;
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) each;
				JunLispCons pair = this_.lispCons();
				pair.head_(this_.generateIdString());
				pair.tail_(arc.toLispListWithoutFromNodeAndToNode());
				list_.add_(pair);

				return null;
			}
		});

		return list;
	}

	/**
	 * Get my arc table from the lisp list.
	 * 
	 * @return java.util.Dictionary
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected Dictionary arcTableFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("arcs")));
			}
		}, new StBlockClosure());

		if (list == null) {
			return null;
		}

		final JunLispCons list_ = (JunLispCons) list.tail();
		Hashtable table = new Hashtable(list_.size());
		final Hashtable table_ = table;
		final StValueHolder index = StValueHolder.With_(new Integer(1));
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) anObject;
				String id = (String) ((JunLispCons) list_.nth_(((Integer) index.value()).intValue())).head();
				table_.put(id, arc);
				index.value_(((Integer) index.value()).intValue() + 1);
				return null;
			}
		});
		return table;
	}

	/**
	 * Arranging this receiver.
	 * 
	 * @category arranging
	 */
	public void arrange() {
		this.arrange_(null);
	}

	/**
	 * Set the arrange.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category arranging
	 */
	public void arrange_(StBlockClosure aBlock) {
		this.arrange1_(aBlock);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aDictionary java.util.Dictionary
	 * @param aBlock jp.co.sra.smalltalk.StBlock
	 * @category arranging
	 */
	public Jun3dPoint arrange_xyz_visited_interrim_(JunOpenGL3dNode aNode, Jun3dPoint aPoint, Dictionary aDictionary, StBlockClosure aBlock) {
		aNode.location_(aPoint);

		if (aBlock != null) {
			int numArgs = aBlock.numArgs();
			if (numArgs == 0) {
				aBlock.value();
			}
			if (numArgs == 1) {
				aBlock.value_(aNode);
			}
		}

		aDictionary.put(aNode, $("visited"));
		JunOpenGL3dNode[] connectedNodes = this.connectedNodes_(aNode);
		int size = connectedNodes.length;
		if (size == 0) {
			double width = aPoint.x() + aNode.width();
			double height = aPoint.y() + aNode.height();
			return new Jun3dPoint(width, height, aPoint.z());
		}

		double width = aPoint.x() + aNode.width();
		double height = aPoint.y();
		double x = width + this.arrangeIntervalX();
		double y = height;
		double min = height;
		for (int index = 0; index < size; index++) {
			JunOpenGL3dNode node = connectedNodes[index];
			Object value = aDictionary.get(node);
			if (value == null) {
				value = $("unvisited");
			}
			if (value == $("unvisited")) {
				Jun3dPoint extent = this.arrange_xyz_visited_interrim_(node, new Jun3dPoint(x, y, aPoint.z()), aDictionary, aBlock);
				y = Math.max(extent.y(), y + aNode.height());
				width = Math.max(extent.x(), width);
				height = Math.max(y, height);
				y = y + this.arrangeIntervalY();
			}
			if (this.arrangeBalance()) {
				this.arrangeBalance_xyz_minHeight_interrim_(aNode, new Jun3dPoint(x, y, aPoint.z()), min, aBlock);
				height = Math.max(height, aNode.location().y() + aNode.height());
			}
		}

		return new Jun3dPoint(width, height, aPoint.z());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @category arranging
	 */
	public void arrange1() {
		this.arrange1_(null);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category arranging
	 */
	public void arrange1_(StBlockClosure aBlock) {
		Jun3dPoint origin = new Jun3dPoint(0, 0, 0);
		Vector roots = this.arrangeRoots();

		for (int index = 0; index < roots.size(); index++) {
			JunOpenGL3dNode root = (JunOpenGL3dNode) roots.elementAt(index);
			this.arrange_xyz_visited_interrim_(root, origin, new Hashtable(), aBlock);

			if (this.arrangeBalance()) {
				Jun3dPoint delta = new Jun3dPoint(0, -1 * (root.location().y()), 0);
				final Jun3dTransformation transformation = Jun3dTransformation.Translate_(delta);
				final StBlockClosure block = aBlock;
				this.breadthFirstFrom_nodesDo_(root, new StBlockClosure() {
					public Object value_(Object each) {
						JunOpenGL3dNode eachNode = (JunOpenGL3dNode) each;
						eachNode.transform_(transformation);
						if (block != null) {
							int numArgs = block.numArgs();
							if (numArgs == 0) {
								block.value();
							}
							if (numArgs == 1) {
								block.value_(eachNode);
							}
						}
						return null;
					}
				});
			}
			origin = (Jun3dPoint) origin.plus_(new Jun3dPoint(0, 0, this.arrangeIntervalZ()));
		}
		this.flushDisplayObject();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @category arranging
	 */
	public void arrange2() {
		this.arrange2_(null);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category arranging
	 */
	public void arrange2_(StBlockClosure aBlock) {
		Jun3dPoint origin = new Jun3dPoint(0, 0, 0);
		Jun3dPoint corner = origin;
		Vector roots = this.arrangeRoots();

		for (int index = 0; index < roots.size(); index++) {
			JunOpenGL3dNode root = (JunOpenGL3dNode) roots.elementAt(index);
			Jun3dPoint point = new Jun3dPoint(0, 0, origin.z());
			JunOpenGL3dNode[] connectedNodes = this.connectedNodes_(root);
			for (int i = 0; i < connectedNodes.length; i++) {
				JunOpenGL3dNode node = connectedNodes[i];
				Jun3dPoint xyz = this.arrange_xyz_visited_interrim_(node, origin, new Hashtable(), aBlock);
				if (this.arrangeBalance()) {
					Jun3dPoint delta = new Jun3dPoint(0, -1 * (node.location().y()), 0);
					final Jun3dTransformation transformation = Jun3dTransformation.Translate_(delta);
					final StBlockClosure block = aBlock;
					this.breadthFirstFrom_nodesDo_(node, new StBlockClosure() {
						public Object value_(Object each) {
							JunOpenGL3dNode eachNode = (JunOpenGL3dNode) each;
							eachNode.transform_(transformation);
							if (block != null) {
								int numArgs = block.numArgs();
								if (numArgs == 0) {
									block.value();
								}
								if (numArgs == 1) {
									block.value_(eachNode);
								}
							}
							return null;
						}
					});
				}
				corner = new Jun3dPoint(Math.max(corner.x(), xyz.x()), Math.max(corner.y(), xyz.y()), Math.max(corner.z(), xyz.z()));
				origin = (Jun3dPoint) origin.plus_(new Jun3dPoint(0, 0, this.arrangeIntervalZ()));
			}

			point = (Jun3dPoint) point.plus_((new Jun3dPoint(0, 0, corner.z()).minus_(point)).dividedBy_(2));
			root.location_((Jun3dPoint) point.minus_(new Jun3dPoint((this.arrangeIntervalX() + root.width()), 0, 0)));
			if (aBlock != null) {
				int numArgs = aBlock.numArgs();
				if (numArgs == 0) {
					aBlock.value();
				}
				if (numArgs == 1) {
					aBlock.value_(root);
				}
				Jun3dPoint delta = new Jun3dPoint(-1 * root.location().x(), 0, 0);
				final Jun3dTransformation transformation = Jun3dTransformation.Translate_(delta);
				final StBlockClosure block = aBlock;
				this.depthFirstFrom_nodeDo_(root, new StBlockClosure() {
					public Object value_(Object each) {
						JunOpenGL3dNode eachNode = (JunOpenGL3dNode) each;
						eachNode.transform_(transformation);
						if (block != null) {
							int numArgs2 = block.numArgs();
							if (numArgs2 == 0) {
								block.value();
							}
							if (numArgs2 == 1) {
								block.value_(eachNode);
							}
						}
						return null;
					}
				});
			}
		}
		this.flushDisplayObject();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 * @category arranging
	 */
	public boolean arrangeBalance() {
		return ((Boolean) this.arrangeFormat().get($("balance"))).booleanValue();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBoolean boolean
	 * @category arranging
	 */
	public void arrangeBalance_(boolean aBoolean) {
		this.arrangeFormat().put($("balance"), Boolean.valueOf(aBoolean));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param min double
	 * @param aBlock jp.co.sra.smalltalk.StBlocClosure
	 * @category arranging
	 */
	public void arrangeBalance_xyz_minHeight_interrim_(JunOpenGL3dNode aNode, Jun3dPoint aPoint, double min, StBlockClosure aBlock) {
		double y = aPoint.y() - this.arrangeIntervalY();
		if (y > (aNode.location().y() + aNode.height())) {
			y = Math.max(min, min + ((y - min - aNode.height()) / 2));
		} else {
			y = min;
		}
		aNode.location_(new Jun3dPoint(aNode.location().x(), y, aPoint.z()));
		if (aBlock != null) {
			int numArgs = aBlock.numArgs();

			if (numArgs == 0) {
				aBlock.value();
			}
			if (numArgs == 1) {
				aBlock.value_(aNode);
			}
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return java.util.Dictionary
	 * @category arranging
	 */
	public Dictionary arrangeFormat() {
		Dictionary format = this.arrangeFormat;
		if (format == null) {
			format = new Hashtable();
			format.put($("interval"), new Jun3dPoint(10, 2, 4));
			format.put($("balance"), Boolean.TRUE);
			this.arrangeFormat = format;
		}
		return format;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arranging
	 */
	public Jun3dPoint arrangeInterval() {
		return (Jun3dPoint) this.arrangeFormat().get($("interval"));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.Jun3dPoint
	 * @category arranging
	 */
	public void arrangeInterval_(Jun3dPoint aPoint) {
		this.arrangeFormat().put($("interval"), aPoint);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 * @category arranging
	 */
	public double arrangeIntervalX() {
		return ((Jun3dPoint) this.arrangeFormat().get($("interval"))).x();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 * @category arranging
	 */
	public double arrangeIntervalY() {
		return ((Jun3dPoint) this.arrangeFormat().get($("interval"))).y();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 * @category arranging
	 */
	public double arrangeIntervalZ() {
		return ((Jun3dPoint) this.arrangeFormat().get($("interval"))).z();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return java.util.Vector
	 * @category arranging
	 */
	public Vector arrangeRoots() {
		if (this.nodes().length == 0) {
			return new Vector();
		}
		final Vector exceptions = new Vector();
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object arc) {
				exceptions.addElement(((JunOpenGL3dArc) arc).to());

				return null;
			}
		});
		final Vector roots = new Vector();
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object node) {
				JunOpenGL3dNode aNode = (JunOpenGL3dNode) node;
				if (!exceptions.contains(aNode)) {
					roots.addElement(aNode);
				}
				return null;
			}
		});
		if (roots.isEmpty()) {
			roots.addElement(this.atNode_(1));
		}
		return roots;
	}

	/**
	 * Get the node at an index.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category node accessing
	 */
	public JunOpenGL3dNode atNode_(int anInteger) {
		return this.nodes()[anInteger - 1];
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	public void breadthFirstFrom_nodesDo_(JunOpenGL3dNode aNode, StBlockClosure nodeBlock) {
		this.breadthFirstFrom_nodesDo_arcDo_(aNode, nodeBlock, new StBlockClosure());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	public void breadthFirstFrom_nodesDo_arcDo_(JunOpenGL3dNode aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		this.breadthFirstFrom_nodesDo_arcDo_visited_queue_(aNode, nodeBlock, arcBlock, new Hashtable(), new Vector());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aDictionary java.util.Dictionary
	 * @param aQueue java.util.Vector
	 * @category visiting
	 */
	public void breadthFirstFrom_nodesDo_arcDo_visited_queue_(JunOpenGL3dNode aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock, Dictionary aDictionary, Vector aQueue) {
		aDictionary.put(aNode, $("entered"));
		aQueue.addElement(aNode);

		while (!aQueue.isEmpty()) {
			JunOpenGL3dNode node = (JunOpenGL3dNode) aQueue.elementAt(0);
			aQueue.removeElementAt(0);

			Object value = aDictionary.get(node);
			if (value == null) {
				value = $("unvisited");
			}
			if (value != $("visited")) {
				nodeBlock.value_(node);
			}
			aDictionary.put(node, $("visited"));

			JunOpenGL3dArc[] arcs = this.connectedArcs_(node);
			for (int index = 0; index < arcs.length; index++) {
				JunOpenGL3dArc arc = arcs[index];
				JunOpenGL3dNode toNode = arc.to();
				arcBlock.value_(arc);
				value = aDictionary.get(toNode);
				if (value == null) {
					value = $("unvisited");
					if (value == $("unvisited")) {
						aQueue.addElement(toNode);
						aDictionary.put(toNode, $("entered"));
					}
				}
			}
		}
	}

	/**
	 * Answer the receiver's component objects.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @see jp.co.sra.jun.opengl.picking.JunOpenGLPickingObjects#components()
	 * @category accessing
	 */
	public JunOpenGL3dObject[] components() {
		return ((JunOpenGL3dCompoundObject) this.displayObject()).components();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc[]
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category arc accessing
	 */
	public JunOpenGL3dArc[] connectedArcs_(JunOpenGL3dNode aNode) {
		JunOpenGL3dArc[] arcs = this.arcs();
		Vector connectedArcs = new Vector();
		int size = arcs.length;
		for (int index = 0; index < size; index++) {
			JunOpenGL3dArc arc = (JunOpenGL3dArc) arcs[index];
			if (arc.from() == aNode) {
				if (connectedArcs.indexOf(arc) == -1) {
					connectedArcs.addElement(arc);
				}
			}
		}
		JunOpenGL3dArc[] result = new JunOpenGL3dArc[connectedArcs.size()];
		connectedArcs.copyInto(result);
		return result;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode[]
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category node accessing
	 */
	public JunOpenGL3dNode[] connectedNodes_(JunOpenGL3dNode aNode) {
		JunOpenGL3dArc[] arcs = this.connectedArcs_(aNode);
		int size = arcs.length;
		JunOpenGL3dNode[] result = new JunOpenGL3dNode[size];
		for (int index = 0; index < size; index++) {
			result[index] = arcs[index].to();
		}
		return result;
	}

	/**
	 * Get my connections from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param anArray java.util.Dictionary[]
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected void connectionsFromLispList_tables_(JunLispList aList, Dictionary[] anArray) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("connections")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		final Dictionary table1 = anArray[0];
		final Dictionary table2 = anArray[1];
		((JunLispCons) list.tail()).do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLispCons each = (JunLispCons) anObject;
				JunOpenGL3dNode from = (JunOpenGL3dNode) table1.get(each.head());
				JunOpenGL3dNode to = (JunOpenGL3dNode) table1.get(((JunLispCons) each.tail()).head());
				JunOpenGL3dArc arc = (JunOpenGL3dArc) table2.get(((JunLispCons) each.tail()).tail());
				if (arc != null) {
					arc.from_to_(from, to);
				}
				return null;
			}
		});
	}

	/**
	 * Convert the receiver's connections as a JunLispList.
	 * 
	 * @param anArray jp.co.sra.jun.goodies.lisp.JunLispCons[]
	 * @category lisp support
	 */
	protected JunLispCons connectionsToLispList_(JunLispCons[] anArray) {
		JunLispCons list = this.lispCons();
		list.head_($("connections"));

		final JunLispCons list_ = list;
		final JunLispCons list1 = anArray[0];
		final Hashtable table1 = new Hashtable();
		final StValueHolder index1 = new StValueHolder(1);
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dNode node = (JunOpenGL3dNode) each;
				int indexValue = ((Integer) index1.value()).intValue();
				String id = (String) ((JunLispCons) list1.nth_(indexValue)).head();
				table1.put(node, id);
				index1.value_(indexValue + 1);

				return null;
			}
		});

		final JunLispCons list2 = anArray[1];
		final Hashtable table2 = new Hashtable();
		final StValueHolder index2 = new StValueHolder(1);
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) each;
				int indexValue = ((Integer) index2.value()).intValue();
				String id = (String) ((JunLispCons) list2.nth_(indexValue)).head();
				table2.put(arc, id);
				index2.value_(indexValue + 1);

				return null;
			}
		});
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) each;
				JunOpenGL3dNode from = arc.from();
				JunOpenGL3dNode to = arc.to();
				Object car = table1.get(from);
				Object cdar = table1.get(to);
				Object cddr = table2.get(arc);
				JunLispList connection = (JunLispList) JunLispCons.Head_tail_(cdar, cddr);
				connection = JunLispCons.Head_tail_(car, connection);
				list_.add_(connection);

				return null;
			}
		});

		return list;
	}

	/**
	 * Create receiver's display object and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract#createDisplayObject()
	 * @category displaying
	 */
	protected JunOpenGL3dObject createDisplayObject() {
		final JunOpenGL3dCompoundObject object = new JunOpenGL3dCompoundObject();
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dNode node = (JunOpenGL3dNode) each;
				node.flushDisplayObject();
				object.add_(node.displayObject());

				return null;
			}
		});
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) each;
				arc.flushDisplayObject();
				object.add_(arc.displayObject());

				return null;
			}
		});

		return (JunOpenGL3dObject) object;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	public void depthFirstFrom_nodeDo_(JunOpenGL3dNode aNode, StBlockClosure nodeBlock) {
		this.depthFirstFrom_nodeDo_arcDo_(aNode, nodeBlock, new StBlockClosure());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	public void depthFirstFrom_nodeDo_arcDo_(JunOpenGL3dNode aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		this.depthFirstFrom_nodeDo_arcDo_visited_(aNode, nodeBlock, arcBlock, new Hashtable());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aDictionary java.util.Dictionary
	 * @category visiting
	 */
	public void depthFirstFrom_nodeDo_arcDo_visited_(JunOpenGL3dNode aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock, Dictionary aDictionary) {
		Object value = aDictionary.get(aNode);
		if (value == null) {
			value = $("unvisited");
		}
		if (value != $("visited")) {
			nodeBlock.value_(aNode);
		}

		aDictionary.put(aNode, $("visited"));
		JunOpenGL3dArc[] arcs = this.connectedArcs_(aNode);
		for (int index = 0; index < arcs.length; index++) {
			JunOpenGL3dArc arc = arcs[index];
			arcBlock.value_(arc);
			JunOpenGL3dNode toNode = arc.to();
			value = aDictionary.get(toNode);
			if (value == null) {
				value = $("unvisited");
				if (value == $("unvisited")) {
					this.depthFirstFrom_nodeDo_arcDo_visited_(aNode, nodeBlock, arcBlock, aDictionary);
				}
			}
		}
	}

	/**
	 * Flush the display object.
	 * 
	 * @see jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract#flushDisplayObject()
	 * @category displaying
	 */
	protected void flushDisplayObject() {
		super.flushDisplayObject();
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dNode node = (JunOpenGL3dNode) each;
				node.flushDisplayObject();
				return null;
			}
		});
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) each;
				arc.flushDisplayObject();
				return null;
			}
		});
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);

		this.nodesFromLispList_(aList);
		Dictionary[] tables = new Dictionary[2];
		tables[0] = this.nodeTableFromLispList_(aList);
		this.arcsFromLispList_(aList);
		tables[1] = this.arcTableFromLispList_(aList);
		this.connectionsFromLispList_tables_(aList, tables);

	}

	/**
	 * Generate a tree from a lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category generating
	 */
	public void generateTreeFromLispList_(JunLispList aList) {
		this.generateTreeFromLispList_nodeBlock_arcBlock_(aList, new StBlockClosure() {
			public Object value_(Object each) {
				String nodeName = each.toString();
				Point extent = SystemResourceSupport._getStringExtentFor_(nodeName, null);
				Rectangle aRectangle = new Rectangle(0, 0, extent.x, extent.y);
				double aWidth;
				if (aRectangle.height <= 0) {
					aWidth = 1;
				} else {
					aWidth = aRectangle.width / aRectangle.height;
				}
				JunOpenGL3dNode aNode = JunOpenGL3dNode.Extent_color_entity_(new Jun3dPoint(aWidth, 1, 1), JunOpenGL3dObject.SampleColor(), nodeName);
				return aNode;
			}
		}, new StBlockClosure() {
			public Object value_value_(Object each1, Object each2) {
				JunOpenGL3dArc anArc = JunOpenGL3dArc.From_to_((JunOpenGL3dNode) each1, (JunOpenGL3dNode) each2);
				return anArc;
			}
		});
	}

	/**
	 * Generate a tree from a lisp list with node block and arc block.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockclosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockclosure
	 * @category generating
	 */
	public void generateTreeFromLispList_nodeBlock_arcBlock_(JunLispList aList, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		this._generateTreeFromLispList_nodeBlock_arcBlock_(new Hashtable(), aList, nodeBlock, arcBlock);
	}

	/**
	 * Answer true if the receiver includes an arc object, otherwise false.
	 * 
	 * @return boolean
	 * @param anArc jp.co.sra.jun.opengl.grapher.JunOpenGL3dArc
	 * @category testing
	 */
	public boolean includesArc_(JunOpenGL3dArc anArc) {
		JunOpenGL3dArc[] arcs = this.arcs();
		int size = arcs.length;
		for (int index = 0; index < size; index++) {
			if (arcs[index] == anArc) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Answer true if the receiver includes a node object, otherwise false.
	 * 
	 * @return boolean
	 * @param aNode jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode
	 * @category testing
	 */
	public boolean includesNode_(JunOpenGL3dNode aNode) {
		JunOpenGL3dNode[] nodes = this.nodes();
		int size = nodes.length;
		for (int index = 0; index < size; index++) {
			if (nodes[index] == aNode) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		graphNodes = null;
		graphArcs = null;
		arrangeFormat = null;
	}

	/**
	 * Answer true if the receiver is a compound object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#isCompound()
	 * @category testing
	 */
	public boolean isCompound() {
		return true;
	}

	/**
	 * Answer true if the receiver is a graph object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract#isGraph()
	 * @category testing
	 */
	public boolean isGraph() {
		return true;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Graph");
	}

	/**
	 * Answer the name.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#name()
	 * @category accessing
	 */
	public String name() {
		return JunOpenGL3dObject.UniqueId();
	}

	/**
	 * Answer the receiver's node objects as array.
	 * 
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dNode[]
	 * @category node accessing
	 */
	public JunOpenGL3dNode[] nodes() {
		Vector nodes = this.graphNodes;

		if (nodes == null) {
			nodes = new Vector();
			this.graphNodes = nodes;
		}

		int size = nodes.size();
		JunOpenGL3dNode[] result = new JunOpenGL3dNode[size];

		for (int index = 0; index < size; index++) {
			result[index] = (JunOpenGL3dNode) nodes.elementAt(index);
		}

		return result;
	}

	/**
	 * Set the collection of a node objects.
	 * 
	 * @param newVector java.util.Vector
	 * @category node accessing
	 */
	private void nodes_(Vector newVector) {
		this.graphNodes = newVector;
	}

	/**
	 * Enumerate every node objects and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category node accessing
	 */
	public void nodesDo_(StBlockClosure aBlock) {
		JunOpenGL3dNode[] nodes = this.nodes();
		int size = nodes.length;

		for (int i = 0; i < size; i++) {
			aBlock.value_(nodes[i]);
		}
	}

	/**
	 * Get my nodes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected void nodesFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("nodes")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		final JunOpenGL3dGraph aGraph_ = this;
		((JunLispCons) list.tail()).do_(new StBlockClosure() {
			public Object value_(Object each) {
				try {
					JunOpenGL3dNode node = new JunOpenGL3dNode((JunLispCons) each);
					aGraph_.addNode_(node);
				} catch (Exception e) {
					throw new SmalltalkException(e);
				}
				return null;
			}
		});
	}

	/**
	 * Convert the receiver's nodes as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons nodesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("nodes"));

		final JunOpenGL3dGraph this_ = this;
		final JunLispCons list_ = list;
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dNode node = (JunOpenGL3dNode) each;
				JunLispCons pair = this_.lispCons();
				pair.head_(this_.generateIdString());
				pair.tail_(node.toLispList());
				list_.add_(pair);

				return null;
			}
		});

		return list;
	}

	/**
	 * Get my node table from the lisp list.
	 * 
	 * @return java.util.Dictionary
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category lisp support
	 */
	protected Dictionary nodeTableFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("nodes")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return null;
		}

		final JunLispCons list_ = (JunLispCons) list.tail();
		Hashtable table = new Hashtable(list_.size());
		final Hashtable table_ = table;
		final StValueHolder index = StValueHolder.With_(new Integer(1));
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dNode node = (JunOpenGL3dNode) anObject;
				String id = (String) ((JunLispCons) list_.nth_(((Integer) index.value()).intValue())).head();
				table_.put(id, node);
				index.value_(((Integer) index.value()).intValue() + 1);
				return null;
			}
		});
		return table;
	}

	/**
	 * Show the receiver with a JunOpenGL3dGrapher.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#show()
	 * @category utilities
	 */
	public JunOpenGLDisplayModel show() {
		return JunOpenGL3dGrapher.Show_(this);
	}

	/**
	 * Convert the receiver as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}

		JunLispCons list1 = this.nodesToLispList();
		JunLispCons list2 = this.arcsToLispList();
		list.add_(list1);
		list.add_(list2);

		JunLispCons[] array = new JunLispCons[2];
		array[0] = (JunLispCons) list1.tail();
		array[1] = (JunLispCons) list2.tail();
		list.add_(this.connectionsToLispList_(array));

		return list;
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		final Jun3dTransformation transformation = aTransformation;
		this.nodesDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dNode node = (JunOpenGL3dNode) each;
				node.transform_(transformation);

				return null;
			}
		});
		this.arcsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				JunOpenGL3dArc arc = (JunOpenGL3dArc) each;
				arc.transform_(transformation);

				return null;
			}
		});
		this.flushDisplayObject();

		return this;
	}

	/**
	 * Write the VRML string of the receiver on the writer.
	 * 
	 * @param writer java.io.Writer
	 * @see jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract#vrmlOn_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrmlOn_(Writer writer) {
		JunOpenGL3dObject displayObject = this.displayObject();
		if (displayObject == null) {
			return;
		}
		displayObject.vrmlOn_(writer);
	}
}
