package jp.co.sra.jun.opengl.objects;

import java.io.PrintWriter;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunOpenGL3dCylinder class
 * 
 *  @author    nisinaka
 *  @created   2004/06/08 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dCylinder.java,v 8.12 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dCylinder extends JunOpenGL3dCompoundObject {

	protected static final Jun3dPoint DefaultTopCenter = new Jun3dPoint(0, 0, 2);
	protected static final Jun3dPoint DefaultBottomCenter = new Jun3dPoint(0, 0, 0);
	protected static final double DefaultRadius = 1;
	protected static final int DefaultSlices = 20;

	protected Jun3dPoint topCenter;
	protected Jun3dPoint bottomCenter;
	protected double radius;
	protected int slices;

	/**
	 * Create a new instance of JunOpenGL3dCylinder.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dCylinder() {
		super();
		this.flushComponents();
	}

	/**
	 * Create a new instance of JunOpenGL3dCylinder and initialize it.
	 * Mainly used for conversion from VRML.
	 * 
	 * @param radius double
	 * @param height double
	 * @category Instance creation
	 */
	public JunOpenGL3dCylinder(double radius, double height) {
		this(new Jun3dPoint(0, height / 2, 0), new Jun3dPoint(0, -height / 2, 0), radius, DefaultSlices);
	}

	/**
	 * Create a new instance of JunOpenGL3dCylinder and initialize it.
	 * 
	 * @param topCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param bottomCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radius double
	 * @param slices int
	 * @category Instance creation
	 */
	public JunOpenGL3dCylinder(Jun3dPoint topCenter, Jun3dPoint bottomCenter, double radius, int slices) {
		super();
		this.topCenter_(topCenter);
		this.bottomCenter_(bottomCenter);
		this.radius_(radius);
		this.slices_(slices);
	}

	/**
	 * Create a new instance of JunOpenGL3dCone and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dCylinder(JunLispList aList) {
		super();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		topCenter = null;
		bottomCenter = null;
		radius = Double.NaN;
		slices = DefaultSlices;
	}

	/**
	 * Answer my current topCenter point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint topCenter() {
		if (topCenter == null) {
			topCenter = DefaultTopCenter;
		}
		return topCenter;
	}

	/**
	 * Set my new topCenter point.
	 * 
	 * @param newTopCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void topCenter_(Jun3dPoint newTopCenter) {
		topCenter = newTopCenter;
		this.flushComponents();
	}

	/**
	 * Answer my current bottomCenter point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint bottomCenter() {
		if (bottomCenter == null) {
			bottomCenter = DefaultBottomCenter;
		}
		return bottomCenter;
	}

	/**
	 * Set my new bottomCenter point.
	 * 
	 * @param newBottomCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void bottomCenter_(Jun3dPoint newBottomCenter) {
		bottomCenter = newBottomCenter;
		this.flushComponents();
	}

	/**
	 * Answer my current radius.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radius() {
		if (Double.isNaN(radius)) {
			radius = DefaultRadius;
		}
		return radius;
	}

	/**
	 * Set my new radius.
	 * 
	 * @param newRadius double
	 * @category accessing
	 */
	public void radius_(double newRadius) {
		radius = Math.max(newRadius, 0.001);
		this.flushComponents();
	}

	/**
	 * Answer my current number of slices.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int slices() {
		return slices;
	}

	/**
	 * Set my new number of slices.
	 * 
	 * @param newSlices int
	 * @category accessing
	 */
	public void slices_(int newSlices) {
		slices = Math.max(newSlices, 4);
		this.flushComponents();
	}

	/**
	 * Answer my current height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#height()
	 * @category accessing
	 */
	public double height() {
		return this.topCenter().distance_(this.bottomCenter());
	}

	/**
	 * Answer my current components as ArrayList.
	 * If not exist, create one.
	 * 
	 * @return java.util.ArrayList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#_components()
	 * @category accessing
	 */
	protected ArrayList _components() {
		if (components == null) {
			components = new ArrayList();
			this.flushBounds();

			Jun3dTransformation basicTransformation = this.basicTransformation();

			int slices = this.slices();
			Jun3dPoint[] bottomPoints = new Jun3dPoint[slices];
			Jun3dPoint[] topPoints = new Jun3dPoint[slices];
			Jun3dPoint bottom2top = this.topCenter().minus_(this.bottomCenter());
			double dPhi = Math.PI * 2 / this.slices();
			for (int i = 0; i < slices; i++) {
				double phi = dPhi * i;
				double x = this.radius() * Math.sin(phi);
				if (Math.abs(x) < JunGeometry.ACCURACY) {
					x = 0.0d;
				}
				double y = this.radius() * Math.cos(phi);
				if (Math.abs(y) < JunGeometry.ACCURACY) {
					y = 0.0d;
				}
				bottomPoints[i] = basicTransformation.applyTo_(new Jun3dPoint(x, y, 0));
				topPoints[i] = bottomPoints[i].plus_(bottom2top);
			}

			JunOpenGL3dPolygon[] polygons = new JunOpenGL3dPolygon[slices];
			for (int i = 0, prev = slices - 1; i < slices; prev = i, i++) {
				polygons[i] = new JunOpenGL3dPolygon(new Jun3dPoint[] { topPoints[prev], topPoints[i], bottomPoints[i], bottomPoints[prev] });
				polygons[i].paint_(null);
				components.add(polygons[i]);
			}

			JunOpenGL3dPolygon bottom = new JunOpenGL3dPolygon(bottomPoints);
			bottom.paint_(null);
			bottom.normalVectors(); // to set its preferred normal vector.
			components.add(bottom);

			JunOpenGL3dPolygon top = (JunOpenGL3dPolygon) (new JunOpenGL3dPolygon(topPoints)).reversed();
			top.paint_(null);
			components.add(top);
		}
		return components;
	}

	/**
	 * Create a new JunOpenGL3dCylinder transformed with the transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming 
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dCylinder aCylinder = (JunOpenGL3dCylinder) this.copy();

		Jun3dPoint oldPoint = this.basicTransformation().applyTo_(new Jun3dPoint(this.radius(), 0, 0));
		Jun3dPoint newPoint = oldPoint.transform_(aTransformation);
		Jun3dPoint newBottomCenter = this.bottomCenter().transform_(aTransformation);
		double newRadius = newBottomCenter.distance_(newPoint);

		aCylinder.topCenter_(this.topCenter().transform_(aTransformation));
		aCylinder.bottomCenter_(newBottomCenter);
		aCylinder.radius_(newRadius);
		return aCylinder;
	}

	/**
	 * Answer the basic transformation which translate the basic cone to the receiver.
	 * Basic cone means a cone which bottom radius and height are the same but on the origin of X-Y plane.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	protected Jun3dTransformation basicTransformation() {
		Jun3dTransformation aTransformation = Jun3dTransformation.AlignVector_to_(new Jun3dPoint(0, 0, this.height()), this.topCenter().minus_(this.bottomCenter()));
		aTransformation = aTransformation.product_(Jun3dTransformation.Translate_(this.bottomCenter()));
		return aTransformation;
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();

		if (topCenter != null) {
			topCenter = new Jun3dPoint(topCenter.x(), topCenter.y(), topCenter.z());
		}

		if (bottomCenter != null) {
			bottomCenter = new Jun3dPoint(bottomCenter.x(), bottomCenter.y(), bottomCenter.z());
		}

		return this;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Cylinder");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}

		list.add_(this.topCenterToLispList());
		list.add_(this.bottomCenterToLispList());
		list.add_(this.radiusToLispList());
		list.add_(this.slicesToLispList());

		return list;
	}

	/**
	 * Convert the receiver's top center as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList topCenterToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("topCenter"));
		list.tail_(this.topCenter());
		return list;
	}

	/**
	 * Convert the receiver's bottom center as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList bottomCenterToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("bottomCenter"));
		list.tail_(this.bottomCenter());
		return list;
	}

	/**
	 * Convert the receiver's bottom radius as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList radiusToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("radius"));
		list.tail_(new Double(this.radius()));
		return list;
	}

	/**
	 * Convert the receiver's number of slices as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList slicesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("slices"));
		list.tail_(new Integer(this.slices()));
		return list;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.apexFromLispList(aList);
		this.bottomCenterFromLispList(aList);
		this.bottomRadiusFromLispList(aList);
		this.slicesFromLispList(aList);
	}

	/**
	 * Get my top center from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void apexFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("topCenter")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.topCenter_((Jun3dPoint) list.tail());
	}

	/**
	 * Get my bottom center from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void bottomCenterFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("bottomCenter")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.bottomCenter_((Jun3dPoint) list.tail());
	}

	/**
	 * Get my radius from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void bottomRadiusFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("radius")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.radius_(((Number) list.tail()).doubleValue());
	}

	/**
	 * Get my slices from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void slicesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("slices")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.slices_(((Number) list.tail()).intValue());
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml20On_(PrintWriter pw, String leader) {
		Jun3dLine l = new Jun3dLine(this.topCenter(), this.bottomCenter());
		Jun3dPoint t = l.center();
		l = l.translatedBy_(t.negated());
		Jun3dLine y_axes = new Jun3dLine(new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 0, 0));
		Jun3dPoint r = (new JunPlane(y_axes.to(), y_axes.from(), l.from())).normalVector();
		JunAngle a = y_axes.angleWithLine_(l);

		pw.println(leader + "Transform {");
		pw.println(leader + INDENT + "rotation " + r.x() + ' ' + r.y() + ' ' + r.z() + ' ' + a.rad());
		pw.println(leader + INDENT + "translation " + t.x() + ' ' + t.y() + ' ' + t.z());
		pw.println(leader + INDENT + "children [");
		this.vrml20ShapeOn_(pw, leader + INDENT + INDENT);
		pw.println(leader + INDENT + "] # children");
		pw.println(leader + "} #Transform");
		pw.flush();
	}

	/**
	 * Write my geometry as VRML2.0 to the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20GeometryOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		pw.println(leader + "geometry Cylinder {");
		pw.println(leader + INDENT + "height " + this.height());
		pw.println(leader + INDENT + "radius " + this.radius());
		pw.println(leader + "} #Cylinder");
	}

}
