package jp.co.sra.jun.opengl.objects;

import java.io.Writer;
import java.util.Vector;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.collections.sequences.JunDoubleMatrix;
import jp.co.sra.jun.collections.sequences.JunMatrix;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.surfaces.JunNurbsSurface;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.opengl.texture.JunOpenGLStipple;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;

/**
 * JunOpenGL3dNurbsSurface class
 * 
 *  @author    nisinaka
 *  @created   1998/10/29 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2001/11/20 (by nisinaka)
 *  @updated   2004/05/14 (by nisinaka)
 *  @updated   2004/11/15 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun500 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dNurbsSurface.java,v 8.11 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dNurbsSurface extends JunOpenGL3dPrimitiveObject {

	/** The stipple of the receiver. */
	protected JunOpenGLStipple stipple;

	/** The texture of the receiver. */
	protected JunOpenGLTexture texture;

	/** The matrix of the control points. */
	protected JunMatrix controlPoints;

	/** The matrix of the weight number. */
	protected JunDoubleMatrix weights;

	/** The knot vector of u-parameter. */
	protected double[] uKnotVector;

	/** The knot vector of v-parameter. */
	protected double[] vKnotVector;

	/**
	 * Create an instance of JunOpenGL3dNurbsSurface and initialize it with the arguments.
	 * 
	 * @param aMatrix1 jp.co.sra.jun.collections.sequences.JunMatrix as control points
	 * @param aMatrix2 jp.co.sra.jun.collections.sequences.JunMatrix as weights
	 * @param anArray1 double[] as uKnotVector
	 * @param anArray2 double[] as vKnotVector
	 * @category Instance creation
	 */
	public JunOpenGL3dNurbsSurface(JunMatrix aMatrix1, JunDoubleMatrix aMatrix2, double[] anArray1, double[] anArray2) {
		super();
		this.controlPoints_(aMatrix1);
		this.weights_(aMatrix2);
		this.uKnotVector_(anArray1);
		this.vKnotVector_(anArray2);
	}

	/**
	 * Create a new instance of JunOpenGL3dNurbsSurface and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dNurbsSurface(JunLispList aList) {
		super();
		this.fromLispList(aList);
	}

	/**
	 * Create a new instance of JunOpenGL3dNurbsSurface and initialize it with the bezier control points.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsSurface
	 * @category Instance creation
	 */
	public static JunOpenGL3dNurbsSurface BezierControlPoints_(JunMatrix aMatrix) {
		int columnSize = aMatrix.columnSize();
		int rowSize = aMatrix.rowSize();
		double[] uKnotVector = new double[rowSize * 2];
		for (int index = 0; index < rowSize; index++) {
			uKnotVector[index] = 0.0;
			uKnotVector[index + rowSize] = 1.0;
		}
		double[] vKnotVector = new double[columnSize * 2];
		for (int index = 0; index < columnSize; index++) {
			vKnotVector[index] = 0.0;
			vKnotVector[index + columnSize] = 1.0;
		}
		return BSplineControlPoints_uKnotVector_vKnotVector_(aMatrix, uKnotVector, vKnotVector);
	}

	/**
	 * Create an instance of JunOpenGL3dNurbsSurface and initialize it with the arguments.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @param anArray1 double[]
	 * @param anArray2 double[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsSurface
	 * @category Instance creation
	 */
	public static JunOpenGL3dNurbsSurface BSplineControlPoints_uKnotVector_vKnotVector_(JunMatrix aMatrix, double[] anArray1, double[] anArray2) {
		int rowSize = aMatrix.rowSize();
		int columnSize = aMatrix.columnSize();
		double[] weights = new double[rowSize * columnSize];
		for (int i = 0; i < weights.length; i++) {
			weights[i] = 1;
		}
		return new JunOpenGL3dNurbsSurface(aMatrix, new JunDoubleMatrix(rowSize, columnSize, weights), anArray1, anArray2);
	}

	/**
	 * Create an instance of JunOpenGL3dNurbsSurface and initialize it with the arguments.
	 * 
	 * @param aMatrix1 jp.co.sra.jun.collections.sequences.JunMatrix
	 * @param aMatrix2 jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @param anArray1 double[]
	 * @param anArray2 double[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsSurface
	 * @deprecated since Jun454, use the constructor
	 * @category Instance creation
	 */
	public static JunOpenGL3dNurbsSurface ControlPoints_weights_uKnotVector_vKnotVector_(JunMatrix aMatrix1, JunDoubleMatrix aMatrix2, double[] anArray1, double[] anArray2) {
		return new JunOpenGL3dNurbsSurface(aMatrix1, aMatrix2, anArray1, anArray2);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		stipple = null;
		texture = null;
		controlPoints = null;
		weights = null;
		uKnotVector = null;
		vKnotVector = null;
	}

	/**
	 * Set the specified halftone scale.
	 * 
	 * @param halftoneScale double
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#halftone_(double)
	 * @category accessing
	 */
	public void halftone_(double halftoneScale) {
		this.stipple_(JunOpenGLStipple.Halftone_(halftoneScale));
	}

	/**
	 * Answer the stipple of the receiver. Does not have a stipple as a default.
	 * 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLStipple
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#stipple()
	 * @category accessing
	 */
	public JunOpenGLStipple stipple() {
		return stipple;
	}

	/**
	 * Set the stipple of the receiver. Do nothing as a default.
	 * 
	 * @param aStipple jp.co.sra.jun.opengl.texture.JunOpenGLStipple
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#stipple_(jp.co.sra.jun.opengl.texture.JunOpenGLStipple)
	 * @category accessing
	 */
	public void stipple_(JunOpenGLStipple aStipple) {
		stipple = aStipple;
	}

	/**
	 * Answer my current texture.
	 * 
	 * @return jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#texture()
	 * @category accessing
	 */
	public JunOpenGLTexture texture() {
		return texture;
	}

	/**
	 * Set the texture of the receiver.
	 * Do some initialization for subclasses.
	 * 
	 * @param aTexture jp.co.sra.jun.opengl.texture.JunOpenGLTexture
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#texture_(jp.co.sra.jun.opengl.texture.JunOpenGLTexture)
	 * @category accessing
	 */
	public void texture_(JunOpenGLTexture aTexture) {
		texture = aTexture;
		super.texture_(aTexture);
	}

	/**
	 * Answer my current matrix of control points.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category accessing
	 */
	public JunMatrix controlPoints() {
		return controlPoints;
	}

	/**
	 * Set my new matrix of control points.
	 * 
	 * @param aJunMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category accessing
	 */
	public void controlPoints_(JunMatrix aJunMatrix) {
		controlPoints = aJunMatrix;
	}

	/**
	 * Answer my current matrix of the weight numbers.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix weights() {
		return weights;
	}

	/**
	 * Set my new matrix of the weight numbers.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public void weights_(JunDoubleMatrix aMatrix) {
		weights = aMatrix;
	}

	/**
	 * Answer my current knot vector of u-parameter.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] uKnotVector() {
		return uKnotVector;
	}

	/**
	 * Set my new knot vector of u-parameter.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void uKnotVector_(double[] anArray) {
		uKnotVector = anArray;
	}

	/**
	 * Answer my current knot vector of v-parameter.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] vKnotVector() {
		return vKnotVector;
	}

	/**
	 * Set my new knot vector of v-parameter.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void vKnotVector_(double[] anArray) {
		vKnotVector = anArray;
	}

	/**
	 * Answer true if this JunOpenGL3dObject has a stipple, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasStipple()
	 * @category testing
	 */
	public boolean hasStipple() {
		return stipple != null;
	}

	/**
	 * Answer true if this JunOpenGL3dObject has a texture, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasTexture()
	 * @category testing
	 */
	public boolean hasTexture() {
		return texture != null;
	}

	/**
	 * Render the OpenGL object on a rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		int oldTextureCoordinateIndex = Integer.MIN_VALUE;

		if (this.hasPaint()) {
			aRenderingContext.paint_(this.paint());
		}
		if (this.hasAlpha()) {
			aRenderingContext.alpha_(this.alpha());
		}
		if (this.hasStipple()) {
			this.stipple().enableStippleOn_(aRenderingContext);
		}
		if (this.hasTexture()) {
			oldTextureCoordinateIndex = aRenderingContext.textureCoordinateIndex();
			aRenderingContext.textureCoordinateIndex_(0);
			this.texture().enableTextureOn_(aRenderingContext);
		}
		if (this.hasTransparency()) {
			aRenderingContext.enableCullFace();
			aRenderingContext.disableDepthMask();
			aRenderingContext.enableBlending();
		}

		aRenderingContext.displayNurbsSurface_(this._asJunNurbsSurface());

		if (this.hasTransparency()) {
			aRenderingContext.disableCullFace();
			aRenderingContext.enableDepthMask();
			aRenderingContext.disableBlending();
		}
		if (this.hasTexture()) {
			this.texture().disableTextureOn_(aRenderingContext);
			aRenderingContext.textureCoordinateIndex_(oldTextureCoordinateIndex);
		}
		if (this.hasStipple()) {
			this.stipple().disableStippleOn_(aRenderingContext);
		}
	}

	/**
	 * Enumerate every geometries and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object geometriesDo_(StBlockClosure aBlock) {
		return aBlock.value_(JunNurbsSurface.ControlPoints_weights_uKnotVector_vKnotVector_(controlPoints, weights, uKnotVector, vKnotVector));
	}

	/**
	 * Enumerate every points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		controlPoints.do_(aBlock);
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(final Jun3dTransformation aTransformation) {
		JunOpenGL3dNurbsSurface theNurbsSurface = (JunOpenGL3dNurbsSurface) this.copy();
		final JunMatrix cps = theNurbsSurface.controlPoints();
		cps.doIJ_(new StBlockClosure() {
			public Object value_value_value_(Object o1, Object o2, Object o3) {
				Jun3dPoint point = ((Jun3dPoint) o1).transform_(aTransformation);
				int i = ((Number) o2).intValue();
				int j = ((Number) o3).intValue();
				cps.row_column_put_(i, j, point);
				return null;
			}
		});
		return theNurbsSurface;
	}

	/**
	 * Convert the receiver as a JunNurbsSurface.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunNurbsSurface
	 * @category converting
	 */
	public JunNurbsSurface _asJunNurbsSurface() {
		return JunNurbsSurface.ControlPoints_weights_uKnotVector_vKnotVector_(controlPoints, weights, uKnotVector, vKnotVector);
	}

	/**
	 * Anser the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		throw new SmalltalkException("not implemented yet");
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();

		controlPoints = (JunMatrix) controlPoints.copy();
		weights = (JunDoubleMatrix) weights.copy();

		double[] newUKnotVector = new double[uKnotVector.length];
		System.arraycopy(uKnotVector, 0, newUKnotVector, 0, uKnotVector.length);
		uKnotVector = newUKnotVector;

		double[] newVKnotVector = new double[vKnotVector.length];
		System.arraycopy(vKnotVector, 0, newVKnotVector, 0, vKnotVector.length);
		vKnotVector = newVKnotVector;

		return this;
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @category utilities
	 */
	public int numberOfPolygons() {
		return 1;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("NurbsSurface");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();

		if (this.hasStipple()) {
			list.add_(this.stippleToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.controlsToLispList());
		list.add_(this.weightsToLispList());
		list.add_(this.uKnotsToLispList());
		list.add_(this.vKnotsToLispList());

		return list;
	}

	/**
	 * Convert the receiver's stipple as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList stippleToLispList() {
		if (!this.hasStipple()) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("stipple"));
		list.tail_(this.stipple().image());
		return list;
	}

	/**
	 * Convert the receiver's texture as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList textureToLispList() {
		if (!this.hasTexture()) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("texture"));
		list.tail_(this.texture().toLispList());
		return list;
	}

	/**
	 * Convert the receiver's control points as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList controlsToLispList() {
		final Vector collection = new Vector(controlPoints.size() + 3);
		collection.addElement($("controls"));
		collection.addElement(new Integer(controlPoints.rowSize()));
		collection.addElement(new Integer(controlPoints.columnSize()));
		controlPoints.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				collection.addElement(anObject);
				return null;
			}
		});
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's weight numbers as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList weightsToLispList() {
		final Vector collection = new Vector(weights.size() + 3);
		collection.addElement($("weights"));
		collection.addElement(new Integer(weights.rowSize()));
		collection.addElement(new Integer(weights.columnSize()));
		weights.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				collection.addElement(anObject);
				return null;
			}
		});
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's knot vector of u-parameter as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList uKnotsToLispList() {
		Object[] collection = new Object[uKnotVector.length + 1];
		collection[0] = $("uKnots");
		for (int i = 0; i < uKnotVector.length; i++) {
			collection[i + 1] = new Double(uKnotVector[i]);
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Convert the receiver's knot vector of v-parameter as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList vKnotsToLispList() {
		Object[] collection = new Object[vKnotVector.length + 1];
		collection[0] = $("vKnots");
		for (int i = 0; i < vKnotVector.length; i++) {
			collection[i + 1] = new Double(vKnotVector[i]);
		}
		return JunLispCons.List_(collection);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);

		this.stippleFromLispList(aList);
		this.textureFromLispList(aList);
		this.controlsFromLispList(aList);
		this.weightsFromLispList(aList);
		this.uKnotsFromLispList(aList);
		this.vKnotsFromLispList(aList);
		this.transformationFromLispList(aList);
		this.trimsFromLispList(aList);
	}

	/**
	 * Get my controlsPoints from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void controlsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("controls")));
			}
		}, new StBlockClosure());

		if (list == null) {
			this.controlPoints_(new JunMatrix());
		} else {
			list = (JunLispCons) list.tail();
			int rows = ((Number) list.head()).intValue();
			list = (JunLispCons) list.tail();
			int columns = ((Number) list.head()).intValue();
			list = (JunLispCons) list.tail();
			Object[] collection = list.asArray();
			JunMatrix matrix = new JunMatrix(rows, columns, collection);
			this.controlPoints_(matrix);
		}
	}

	/**
	 * Get my stipple from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void stippleFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("stipple")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.stipple_(new JunOpenGLStipple((StImage) list.tail()));
	}

	/**
	 * Get my texture from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void textureFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("texture")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.texture_(new JunOpenGLTexture((JunLispList) list.tail()));
	}

	/**
	 * Get my trimCurves from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void trimsFromLispList(JunLispList aList) {
		// TODO: JunOpenGL2dTrimCurve is not imprement.
	}

	/**
	 * Get my knot vector of u-parameter from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void uKnotsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("uKnots")));
			}
		}, new StBlockClosure());

		Object[] numbers = (list == null) ? ((JunLispList) aList.tail()).asArray() : ((JunLispList) list.tail()).asArray();
		double[] doubles = new double[numbers.length];
		for (int i = 0; i < doubles.length; i++) {
			doubles[i] = ((Number) numbers[i]).doubleValue();
		}
		this.uKnotVector_(doubles);
	}

	/**
	 * Get my knot vector of v-parameter from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void vKnotsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("vKnots")));
			}
		}, new StBlockClosure());

		Object[] numbers = (list == null) ? ((JunLispList) aList.tail()).asArray() : ((JunLispList) list.tail()).asArray();
		double[] doubles = new double[numbers.length];
		for (int i = 0; i < doubles.length; i++) {
			doubles[i] = ((Number) numbers[i]).doubleValue();
		}
		this.vKnotVector_(doubles);
	}

	/**
	 * Get my wights from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	protected void weightsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("weights")));
			}
		}, new StBlockClosure());

		if (list == null) {
			this.controlPoints_(new JunMatrix());
		} else {
			list = (JunLispCons) list.tail();
			int rows = ((Number) list.head()).intValue();
			list = (JunLispCons) list.tail();
			int columns = ((Number) list.head()).intValue();
			list = (JunLispCons) list.tail();
			Object[] collection = list.asArray();
			double[] doubles = new double[collection.length];
			for (int i = 0; i < doubles.length; i++) {
				doubles[i] = ((Number) collection[i]).doubleValue();
			}
			JunDoubleMatrix matrix = new JunDoubleMatrix(rows, columns, doubles);
			this.weights_(matrix);
		}
	}

	/**
	 * Write the VRML1.0 string of the receiver on the writer.
	 * 
	 * @param writer java.io.Writer
	 * @category vrml support
	 */
	public void vrml10On_(Writer writer) {
		throw new SmalltalkException("not implemented yet");
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param writer java.io.Writer
	 * @category vrml support
	 */
	public void vrml20On_(Writer writer) {
		throw new SmalltalkException("not implemented yet");
	}

}
