package jp.co.sra.jun.opengl.objects;

import java.awt.Color;
import java.util.Collection;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunOpenGL3dPolyline class
 * 
 *  @author    nisinaka
 *  @created   1998/10/21 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2001/11/20 (by nisinaka)
 *  @updated   2004/05/14 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun691 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dPolyline.java,v 8.12 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dPolyline extends JunOpenGL3dVertexesObject {

	/** The line width. */
	protected float lineWidth;

	/** The stipple factor. */
	protected int stippleFactor = 1;

	/** The stipple pattern. */
	protected short stipplePattern;

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline() {
		super();
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with a collection of 3d points.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Collection aCollectionOf3dPoints) {
		super(aCollectionOf3dPoints);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with a collection of 3d points.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @param aColor java.awt.Color
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Collection aCollectionOf3dPoints, Color aColor) {
		super(aCollectionOf3dPoints, aColor);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with a collection of 3d points and alpha value.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Collection aCollectionOf3dPoints, Color aColor, float aNumber) {
		super(aCollectionOf3dPoints, aColor, aNumber);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints) {
		super(anArrayOf3dPoints);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points and a Color.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aColor java.awt.Color
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Color aColor) {
		super(anArrayOf3dPoints, aColor);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points, a Color and alpha value.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Color aColor, float aNumber) {
		super(anArrayOf3dPoints, aColor, aNumber);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points and an array of colors.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Color[] anArrayOfColors) {
		super(anArrayOf3dPoints, anArrayOfColors);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points, an array of colors and alpha value.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Color[] anArrayOfColors, float aNumber) {
		super(anArrayOf3dPoints, anArrayOfColors, aNumber);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points, and an array of 3d vectors.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOf3dVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Jun3dPoint[] anArrayOf3dVectors) {
		super(anArrayOf3dPoints, anArrayOf3dVectors);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points, an array of 3d vectors and an array of colors.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOf3dVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Jun3dPoint[] anArrayOf3dVectors, Color[] anArrayOfColors) {
		super(anArrayOf3dPoints, anArrayOf3dVectors, anArrayOfColors);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points, an array of 3d vectors, an array of colors and alpha value.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOf3dVectors jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param anArrayOfColors java.awt.Color[]
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(Jun3dPoint[] anArrayOf3dPoints, Jun3dPoint[] anArrayOf3dVectors, Color[] anArrayOfColors, float aNumber) {
		super(anArrayOf3dPoints, anArrayOf3dVectors, anArrayOfColors, aNumber);
	}

	/**
	 * Create a new instance of JunOpenGL3dPolyline and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dPolyline(JunLispList aList) {
		super(aList);
	}

	/**
	 * Create a JunOpenGL3dPolyline from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor
	 * @category Instance creation
	 */
	public static JunOpenGL3dObject FromLispList_(JunLispCons aList) {
		return new JunOpenGL3dPolyline(aList);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor
	 * @category Instance creation
	 */
	public static JunOpenGL3dObject Vertexes_(Jun3dPoint[] anArrayOf3dPoints) {
		return new JunOpenGL3dPolyline(anArrayOf3dPoints);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with a collection of 3d points.
	 * 
	 * @param aCollectionOf3dPoints java.util.Collection
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor
	 * @category Instance creation
	 */
	public static JunOpenGL3dObject Vertexes_(Collection aCollectionOf3dPoints) {
		return new JunOpenGL3dPolyline(aCollectionOf3dPoints);
	}

	/**
	 * Create a new JunOpenGL3dPolyline and initialize it with an array of 3d points and a Color.
	 * 
	 * @param anArrayOf3dPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @deprecated since Jun454, use the constructor
	 * @category Instance creation
	 */
	public static JunOpenGL3dObject Vertexes_paint_(Jun3dPoint[] anArrayOf3dPoints, Color aColor) {
		return new JunOpenGL3dPolyline(anArrayOf3dPoints, aColor);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		lineWidth = this.defaultLineWidth();
		stippleFactor = 1;
		stipplePattern = Short.MIN_VALUE;
	}

	/**
	 * Answer my current line width.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float lineWidth() {
		return lineWidth;
	}

	/**
	 * Set my new line width.
	 * 
	 * @param aNumber float
	 * @category accessing
	 */
	public void lineWidth_(float aNumber) {
		lineWidth = aNumber;
	}

	/**
	 * Answer my current stipple factor.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int stippleFactor() {
		return stippleFactor;
	}

	/**
	 * Set my new stipple factor.
	 * 
	 * @param aNumber int
	 * @category accessing
	 */
	public void stippleFactor_(int aNumber) {
		stippleFactor = aNumber;
	}

	/**
	 * Answer my current stipple pattern
	 * 
	 * @return short
	 * @category accessing
	 */
	public short stipplePattern() {
		return stipplePattern;
	}

	/**
	 * Set my new stipple pattern.
	 * 
	 * @param aNumber short
	 * @category accessing
	 */
	public void stipplePattern_(short aNumber) {
		stipplePattern = aNumber;
	}

	/**
	 * Set the halftone.
	 * 
	 * @param halftoneScale double
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#halftone_(double)
	 * @category accessing
	 */
	public void halftone_(double halftoneScale) {
		int patternNumber = JunImageProcessor.Stipple_(Math.max(0, Math.min(halftoneScale, 1)));
		stipplePattern = (short) patternNumber;
	}

	/**
	 * Answer true if this JunOpenGL3dObject has a stipple, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#hasStipple()
	 * @category testing
	 */
	public boolean hasStipple() {
		return stipplePattern != Short.MIN_VALUE;
	}

	/**
	 * Answer true if the receiver is looped line segments.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isLoop() {
		return false;
	}

	/**
	 * Answer true if the receiver is a kind of JunOpenGL3dPolyline, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#isPolyline()
	 * @category testing
	 */
	public boolean isPolyline() {
		return true;
	}

	/**
	 * Render the OpenGL object on a rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		if (this.hasPaint()) {
			aRenderingContext.paint_(this.paint());
		}

		if (this.hasAlpha()) {
			aRenderingContext.alpha_(this.alpha());
		}

		aRenderingContext.lineWidth_(this.lineWidth());

		if (this.hasStipple()) {
			aRenderingContext.lineStippleFactor_pattern_(this.stippleFactor(), this.stipplePattern());
			aRenderingContext.enableLineStipple();
		}

		aRenderingContext.displayLines_(this.vertexes());

		if (this.hasStipple()) {
			aRenderingContext.disableLineStipple();
		}
	}

	/**
	 * Enumerate every geometries and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#geometriesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object geometriesDo_(StBlockClosure aBlock) {
		for (int i = 0; i < (vertexes.length - 1); i++) {
			Object result = aBlock.value_(new Jun3dLine(vertexes[i], vertexes[i + 1]));

			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * Enumerate every polylines and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#polylinesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void polylinesDo_(StBlockClosure aBlock) {
		aBlock.value_(this);
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPolygons()
	 * @category utilities
	 */
	public int numberOfPolygons() {
		return 0;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Polyline");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (!this.hasName() && !this.hasColor() && !this.hasStipple() && (this.lineWidth <= 1)) {
			for (int i = 0; i < vertexes.length; i++) {
				list.add_(vertexes[i]);
			}
			return list;
		}

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.lineWidth() > 1) {
			list.add_(this.lineWidthToLispList());
		}
		if (this.hasStipple()) {
			list.add_(this.stippleToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.vertexesToLispList());

		return list;
	}

	/**
	 * Convert the receiver's stipple as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList lineWidthToLispList() {
		if (this.lineWidth <= 1) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("width"));
		list.tail_(new Float(this.lineWidth()));
		return list;
	}

	/**
	 * Convert the receiver's stipple as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#stippleToLispList()
	 * @category lisp support
	 */
	protected JunLispList stippleToLispList() {
		if (!this.hasStipple()) {
			return this.lispNil();
		}

		JunLispCons list = this.lispCons();
		list.head_($("stipple"));
		list.add_(new Integer(this.stippleFactor()));
		list.add_(new Short(this.stipplePattern()));
		return list;
	}

	/**
	 * Convert the receiver's texture as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#textureToLispList()
	 * @category lisp support
	 */
	protected JunLispList textureToLispList() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.lineWidthFromLispList(aList);
	}

	/**
	 * Get my lineWidth from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void lineWidthFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("width")));
			}
		}, new StBlockClosure());

		if (list == null) {
			return;
		}

		this.lineWidth_(((Number) list.tail()).intValue());
	}

	/**
	 * Get my stipple from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#stippleFromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	protected void stippleFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("stipple")));
			}
		}, new StBlockClosure());

		if (list == null) {
			return;
		}

		this.stippleFactor_(((Number) ((JunLispCons) list.tail()).nth_(1)).intValue());
		this.stipplePattern_(((Number) ((JunLispCons) list.tail()).nth_(2)).shortValue());
	}

	/**
	 * Get my texture from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#textureFromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	protected void textureFromLispList(JunLispList aList) {
		// Just ignore it.
	}

	/**
	 * Answer the name of my indexed set node.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#vrml10IndexedSetName()
	 * @category vrml support
	 */
	protected String vrml10IndexedSetName() {
		return "IndexedLineSet";
	}

	/**
	 * Answer the name of my indexed set node.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject#vrml20IndexedSetName()
	 * @category vrml support
	 */
	protected String vrml20IndexedSetName() {
		return "IndexedLineSet";
	}

}
