package jp.co.sra.jun.opengl.projection;

import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StValued;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.opengl.lights.JunOpenGLLight;
import jp.co.sra.jun.opengl.lights.JunOpenGLSpotLight;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunOpenGLProjector class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/10/29 (by MATSUDA Ryouichi)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   2003/03/07 (by nisinaka)
 *  @updated   2003/05/13 (by nisinaka)
 *  @updated   2006/10/13 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLProjector.java,v 8.14 2008/02/20 06:32:49 nisinaka Exp $
 */
public class JunOpenGLProjector extends JunAbstractObject implements StBlockValued {
	protected JunOpenGLProjection projection;
	protected StSymbol shading;
	protected StSymbol presentation;
	protected ArrayList lights; // JunOpenGLLight or StBlockValue
	protected boolean lineSmoothBoolean;
	protected boolean polygonSmoothBoolean;

	/**
	 * Create a new instance of <code>JunOpenGLProjector</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLProjector() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		shading = this.defaultShading();
		presentation = this.defaultPresentation();
		projection = this.defaultProjection();
		lights = new ArrayList();
	}

	/**
	 * Answer my current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public JunOpenGLProjection projection() {
		return (JunOpenGLProjection) projection.copy();
	}

	/**
	 * Set my new projection.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category accessing
	 */
	public void projection_(JunOpenGLProjection aProjection) {
		projection = (JunOpenGLProjection) aProjection.copy();
	}

	/**
	 * Answer my current eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint eyePoint() {
		return projection.eyePoint();
	}

	/**
	 * Set my new eye point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void eyePoint_(Jun3dPoint aPoint) {
		if (projection.eyePoint() == null || this.sightPoint().equal_(aPoint) == false) {
			projection.eyePoint_(aPoint);
		}
	}

	/**
	 * Answer my current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint sightPoint() {
		return projection.sightPoint();
	}

	/**
	 * Set my new sight point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void sightPoint_(Jun3dPoint aPoint) {
		if (projection.sightPoint() == null || this.eyePoint().equal_(aPoint) == false) {
			projection.sightPoint_(aPoint);
		}
	}

	/**
	 * Set the eye point and the sight point at the same time.
	 * 
	 * @param pointOfEye jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointOfSight jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void eyePoint_sightPoint_(Jun3dPoint pointOfEye, Jun3dPoint pointOfSight) {
		projection.eyePoint_sightPoint_(pointOfEye, pointOfSight);
	}

	/**
	 * Answer my current up vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint upVector() {
		return projection.upVector();
	}

	/**
	 * Answer my current unit up vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitUpVector() {
		return projection.unitUpVector();
	}

	/**
	 * Answer my current unit sight vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitSightVector() {
		return projection.unitSightVector();
	}

	/**
	 * Answer my current unit right vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitRightVector() {
		return projection.unitRightVector();
	}

	/**
	 * Set my new up vector.
	 * 
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void upVector_(Jun3dPoint a3dPoint) {
		projection.upVector_(a3dPoint);
	}

	/**
	 * Answer my current view factor.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double viewFactor() {
		return projection.viewFactor();
	}

	/**
	 * Set my new view factor.
	 * 
	 * @param factor double
	 * @category accessing
	 */
	public void viewFactor_(double factor) {
		factor = (factor < 2) ? 2 : factor;
		factor = (1000 < factor) ? 1000 : factor;
		projection.viewFactor_(factor);
	}

	/**
	 * Answer my current near value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double near() {
		return projection.near();
	}

	/**
	 * Set my new near value.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void near_(double aNumber) {
		projection.near_(aNumber);
	}

	/**
	 * Answer my current far value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double far() {
		return projection.far();
	}

	/**
	 * Set my new far value.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void far_(double aNumber) {
		projection.far_(aNumber);
	}

	/**
	 * Answer my current zoom height.
	 *
	 * @return double
	 * @category accessing
	 */
	public double zoomHeight() {
		return projection.zoomHeight();
	}

	/**
	 * Set my new zoom height.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void zoomHeight_(double aNumber) {
		projection.zoomHeight_(aNumber);
	}

	/**
	 * Answer my current distance.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double distance() {
		return projection.distance();
	}

	/**
	 * Pan with the default factor.
	 * 
	 * @category accessing
	 */
	public void pan() {
		projection.pan();
	}

	/**
	 * Pan with the factor.
	 * 
	 * @param factor double
	 * @category accessing
	 */
	public void pan_(double factor) {
		projection.pan_(factor);
	}

	/**
	 * Zoom with the default factor.
	 * 
	 * @category accessing
	 */
	public void zoom() {
		projection.zoom();
	}

	/**
	 * Zoom with the factor.
	 * 
	 * @param factor double
	 * @category accessing
	 */
	public void zoom_(double factor) {
		projection.zoom_(factor);
	}

	/**
	 * Answer the value.
	 *
	 * @return java.lang.Object
	 * @see jp.co.sra.smalltalk.StValued#value()
	 * @category accessing
	 */
	public Object value() {
		return this;
	}

	/**
	 * Answer a BlockValue that computes aBlock with the receiver's value  as
	 * the argument. aBlock will become a dependent of the receiver, and will
	 * be sent the message value: when the receiver is sent the message
	 * value:.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Do something after a shallow copy.
	 * Do not copy the dependents list.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();
		this.breakDependents();
		return this;
	}

	/**
	 * Answer a default projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category defaults
	 */
	protected JunOpenGLProjection defaultProjection() {
		return JunOpenGLProjection.Default();
	}

	/**
	 * Answer a symbol of my default presentation.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	protected StSymbol defaultPresentation() {
		return $("solidPresentation");
	}

	/**
	 * Answer a symbol of my default shading.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	protected StSymbol defaultShading() {
		return $("flatShading");
	}

	/**
	 * Project on the rendering context.
	 * 
	 * @param a3dObjectOrModel jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category displaying
	 */
	public void project_on_(JunOpenGL3dObject a3dObjectOrModel, JunOpenGLRenderingContext aRenderingContext) {
		this.project_withLights_on_(a3dObjectOrModel, null, aRenderingContext);
	}

	/**
	 * Project the 3D object on the rendering context with lights.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aCollectionOfJunOpenGLLight jp.co.sra.jun.opengl.lights.JunOpenGLLight[]
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category displaying
	 */
	public void project_withLights_on_(JunOpenGL3dObject a3dObject, JunOpenGLLight[] aCollectionOfJunOpenGLLight, JunOpenGLRenderingContext aRenderingContext) {
		this.project_withLights_on_withDisplayList_(a3dObject, aCollectionOfJunOpenGLLight, aRenderingContext, null);
	}

	/**
	 * Project the 3D object on the rendering context with lights and a display list.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aCollectionOfJunOpenGLLight jp.co.sra.jun.opengl.lights.JunOpenGLLight[]
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @param displayListHolder jp.co.sra.smalltalk.StValueHolder
	 * @category displaying
	 */
	public void project_withLights_on_withDisplayList_(JunOpenGL3dObject a3dObject, JunOpenGLLight[] aCollectionOfJunOpenGLLight, JunOpenGLRenderingContext aRenderingContext, StValueHolder displayListHolder) {
		this.projectOn_(aRenderingContext);

		if (aCollectionOfJunOpenGLLight != null) {
			for (int index = 0; index < aCollectionOfJunOpenGLLight.length; index++) {
				aCollectionOfJunOpenGLLight[index].lightingOn_(aRenderingContext);
			}
		}

		try {
			if (a3dObject != null) {
				this.render_on_withDisplayList_(a3dObject, this.renderingContextFrom_(aRenderingContext), displayListHolder);
			}
		} finally {
			aRenderingContext.flush();
		}
	}

	/**
	 * Project on a RenderingContext.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category displaying
	 */
	public void projectOn_(final JunOpenGLRenderingContext aRenderingContext) {
		try {
			projection.projectOn_(aRenderingContext);
			this.lightsDo_(new StBlockClosure() {
				public Object value_(Object light) {
					((JunOpenGLLight) light).lightingOn_(aRenderingContext);
					return null;
				}
			});
		} finally {
			aRenderingContext.flush();
		}
	}

	/**
	 * Arrange to receive a message with aSymbol when the value aspect changes on anObject.
	 * 
	 * @param aSymbol java.lang.String
	 * @param anObject jp.co.sra.smalltalk.StObject
	 * @category interests
	 */
	public void onChangeProjectionSend_to_(String aSymbol, StObject anObject) {
		this.expressInterestIn_for_sendBack_($("projection"), anObject, aSymbol);
	}

	/**
	 * Arrange to receive a message with aSymbol when the value aspect changes on anObject.
	 * 
	 * @param aSymbol java.lang.String
	 * @param anObject jp.co.sra.smalltalk.StObject
	 * @category interests
	 */
	public void onChangeSend_to_(String aSymbol, StObject anObject) {
		this.expressInterestIn_for_sendBack_($("value"), anObject, aSymbol);
	}

	/**
	 * Undo a send of onChangeSend:to:.
	 * 
	 * @param anObject jp.co.sra.smalltalk.StObject
	 * @category interests
	 */
	public void retractInterestsFor_(StObject anObject) {
		this.retractInterestIn_for_($("value"), anObject);
	}

	/**
	 * Undo a send of onChangeSend:to:.
	 * 
	 * @param anObject jp.co.sra.smalltalk.StObject
	 * @category interests
	 */
	public void retractProjectionInterestsFor_(StObject anObject) {
		this.retractInterestIn_for_($("projection"), anObject);
	}

	/**
	 * Add all lights.
	 * 
	 * @param aCollectionOfJunOpenGLLight java.util.Collection
	 * @category lights-adding
	 */
	public void addAllLights_(Collection aCollectionOfJunOpenGLLight) {
		lights.addAll(aCollectionOfJunOpenGLLight);
	}

	/**
	 * Add a head light.
	 * 
	 * @category lights-adding
	 */
	public void addHeadLight() {
		final JunOpenGLSpotLight headLight = JunOpenGLSpotLight.At3dPoint_(this.eyePoint());
		this.addLight_(new StBlockValue(new StBlockClosure() {
			public Object value_(Object projector) {
				headLight.position_(((JunOpenGLProjector) projector).eyePoint());
				return headLight;
			}
		}, this));
	}

	/**
	 * Add a light.
	 * 
	 * @param aJunOpenGLLight jp.co.sra.smalltalk.StBlockValue
	 * @category lights-adding
	 */
	public void addLight_(JunOpenGLLight aJunOpenGLLight) {
		lights.add(aJunOpenGLLight);
	}

	/**
	 * Add a light.
	 * 
	 * @param aJunOpenGLLight jp.co.sra.smalltalk.StBlockValue
	 * @category lights-adding
	 */
	public void addLight_(StBlockValue aJunOpenGLLight) {
		lights.add(aJunOpenGLLight);
	}

	/**
	 * Enumerate every lights and evaluate a Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category lights-enumerating
	 */
	public void lightsDo_(StBlockClosure aBlock) {
		StValued[] values = (StValued[]) lights.toArray(new StValued[lights.size()]);
		for (int i = 0; i < values.length; i++) {
			aBlock.value_(values[i].value());
		}
	}

	/**
	 * Remove all lights.
	 * 
	 * @param aCollectionOfJunOpenGLLight java.util.Collection
	 * @category lights-removing
	 */
	public void removeAllLight_(Collection aCollectionOfJunOpenGLLight) {
		lights.removeAll(aCollectionOfJunOpenGLLight);
	}

	/**
	 * Remove a light.
	 * 
	 * @param aJunOpenGLLight jp.co.sra.jun.opengl.lights.JunOpenGLLight
	 * @category lights-removing
	 */
	public void removeLight_(JunOpenGLLight aJunOpenGLLight) {
		lights.remove(aJunOpenGLLight);
	}

	/**
	 * Answer my current presentation.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category presentation
	 */
	public StSymbol presentation() {
		return presentation;
	}

	/**
	 * Set my new presentation
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category presentation
	 */
	public void presentation_(StSymbol aSymbol) {
		presentation = aSymbol;
	}

	/**
	 * Set the presentation mode as solid.
	 * 
	 * @category presentation
	 */
	public void solidPresentation() {
		this.presentation_($("solidPresentation"));
	}

	/**
	 * Set the presentation mode as wireframe.
	 * 
	 * @category presentation
	 */
	public void wireframePresentation() {
		this.presentation_($("wireframePresentation"));
	}

	/**
	 * Set the presentation mode as hidden-line.
	 * 
	 * @category presentation
	 */
	public void hiddenlinePresentation() {
		this.presentation_($("hiddenlinePresentation"));
	}

	/**
	 * Set the parallel projection.
	 * 
	 * @category projection
	 */
	public void parallelProjection() {
		this.projection_(projection.asParallelProjection());
	}

	/**
	 * Set the perspective projection.
	 * 
	 * @category projection
	 */
	public void perspectiveProjection() {
		this.projection_(projection.asPerspectiveProjection());
	}

	/**
	 * Answer my current shading.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category shading
	 */
	public StSymbol shading() {
		return shading;
	}

	/**
	 * Set my new shading.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category shading
	 */
	public void shading_(StSymbol aSymbol) {
		shading = aSymbol;
	}

	/**
	 * Set the flat shading.
	 * 
	 * @category shading
	 */
	public void flatShading() {
		this.shading_($("flatShading"));
	}

	/**
	 * Set the smooth shading.
	 * 
	 * @category shading
	 */
	public void smoothShading() {
		this.shading_($("smoothShading"));
	}

	/**
	 * Answer whether do a line smoothing effect or not.
	 * 
	 * @return boolean
	 * @category smoothing
	 */
	public boolean lineSmooth() {
		return lineSmoothBoolean;
	}

	/**
	 * Set whether do a line smoothing effect or not.
	 * 
	 * @param aBoolean boolean
	 * @category smoothing
	 */
	public void lineSmooth_(boolean aBoolean) {
		lineSmoothBoolean = aBoolean;
	}

	/**
	 * Answer whether do a polygon smoothing effect or not.
	 * 
	 * @return boolean
	 * @category smoothing
	 */
	public boolean polygonSmooth() {
		return polygonSmoothBoolean;
	}

	/**
	 * Set whether do a polygon smoothing effect or not.
	 * 
	 * @param aBoolean boolean
	 * @category smoothing
	 */
	public void polygonSmooth_(boolean aBoolean) {
		polygonSmoothBoolean = aBoolean;
	}

	/**
	 * Render the 3D object on the rendering context with the specified display list.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @param displayListHolder jp.co.sra.smalltalk.StValueHolder
	 * @category private
	 */
	protected void render_on_withDisplayList_(final JunOpenGL3dObject a3dObject, final JunOpenGLRenderingContext aRenderingContext, StValueHolder displayListHolder) {
		if (displayListHolder == null) {
			a3dObject.renderOn_(aRenderingContext);
		} else if (displayListHolder.value() == null) {
			int id = aRenderingContext.createDisplayList_displayImmediately_(new StBlockClosure() {
				public Object value() {
					a3dObject.renderOn_(aRenderingContext);
					return null;
				}
			}, true);
			displayListHolder.value_(id);
		} else {
			aRenderingContext.callDisplayList_(((Number) displayListHolder.value()).intValue());
		}
	}

	/**
	 * Set projector mode to a the rendering context and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category private
	 */
	protected JunOpenGLRenderingContext renderingContextFrom_(JunOpenGLRenderingContext aRenderingContext) {
		if (presentation == $("solidPresentation")) {
			aRenderingContext.solidPresentation();
		} else if (presentation == $("wireframePresentation")) {
			aRenderingContext.wireframePresentation();
		} else if (presentation == $("hiddenlinePresentation")) {
			aRenderingContext.hiddenlinePresentation();
		}

		if (shading == $("smoothShading")) {
			aRenderingContext.smoothShading();
		} else if (shading == $("flatShading")) {
			aRenderingContext.flatShading();
		}

		if (this.lineSmooth()) {
			aRenderingContext.lineSmoothNicest();
			aRenderingContext.enableLineSmooth();
		} else {
			aRenderingContext.disableLineSmooth();
		}

		if (this.polygonSmooth()) {
			aRenderingContext.polygonSmoothFastest();
			aRenderingContext.enablePolygonSmooth();
		} else {
			aRenderingContext.disablePolygonSmooth();
		}

		return aRenderingContext;
	}

	/**
	 * Set a projection.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category private
	 */
	protected void setProjection_(JunOpenGLProjection aProjection) {
		projection = aProjection;
	}

	/**
	 * Set the eye point in private.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	public void setEyePoint_(Jun3dPoint aPoint) {
		projection.setEyePoint_(aPoint);
	}

	/**
	 * Set the sight point in private.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	public void setSightPoint_(Jun3dPoint aPoint) {
		projection.setSightPoint_(aPoint);
	}

	/**
	 * Set the up vector in private.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	public void setUpVector_(Jun3dPoint aPoint) {
		projection.setUpVector_(aPoint);
	}

	/**
	 * Set the view factor in private.
	 * 
	 * @param factor double
	 * @category private
	 */
	public void setViewFactor_(double factor) {
		projection.setViewFactor_(factor);
	}

	/**
	 * Set the zoom height in private.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	public void setZoomHeight_(double aNumber) {
		projection.setZoomHeight_(aNumber);
	}

	/**
	 * Set the projection in private.
	 * 
	 * @category private
	 */
	public void setProjection() {
		if (this.eyePoint().equal_(this.sightPoint())) {
			this.setEyePoint_(this.sightPoint().plus_(this.projection().DefaultEyePoint));
		}

		projection.setProjection();
	}
}
