package jp.co.sra.jun.system.framework;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Window;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer;
import jp.co.sra.jun.goodies.tables.JunAttributeTable;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunAbstractModel class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2008/01/28 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunAbstractModel.java,v 8.2 2008/02/20 06:32:50 nisinaka Exp $
 */
public abstract class JunAbstractModel extends JunApplicationModel {
	protected JunAttributeTable attributeTable;

	/**
	 * Initialize the receiver when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		attributeTable = new JunAttributeTable();
	}

	/**
	 * Answer the receiver's attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category accessing
	 */
	public JunAttributeTable attributeTable() {
		return attributeTable;
	}

	/**
	 * Convert the receiver to an image as <code>StImage</code>.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		StImage anImage = null;
		StView aView = this.getView();
		Dimension aSize = (aView == null) ? new Dimension(100, 100) : aView.toComponent().getBounds().getSize();
		JunCursors aCursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			aCursor._show();
			anImage = new StImage(aSize);
			Graphics aGraphicsContext = anImage.image().getGraphics();
			try {
				Window aWindow = this.getWindow();
				if (aWindow == null) {
					aGraphicsContext.setColor(Color.white);
				} else {
					aGraphicsContext.setColor(aWindow.getBackground());
				}
				aGraphicsContext.drawRect(0, 0, anImage.width(), anImage.height());
				if (aView != null) {
					aView.toComponent().paint(aGraphicsContext);
				}
			} finally {
				if (aGraphicsContext != null) {
					aGraphicsContext.dispose();
				}
			}
		} finally {
			aCursor._restore();
		}
		return anImage;
	}

	/**
	 * Answer the receiver's display block.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category displaying
	 */
	public StBlockClosure displayBlock() {
		return (StBlockClosure) this.attributeTable().at_ifAbsentPut_($("displayBlock"), new StBlockClosure() {
			public Object value() {
				return null;
			}
		});
	}

	/**
	 * Set the receiver's display block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category displaying
	 */
	public void displayBlock_(final StBlockClosure aBlock) {
		final JunAbstractModel self = this;
		this.attributeTable().at_put_($("displayBlock"), new StBlockClosure() {
			public Object value_(Object anObject) {
				Graphics graphicsContext = (Graphics) anObject;
				int numberOfArguments = aBlock.numArgs();
				if (numberOfArguments == 0) {
					aBlock.value();
				}
				if (numberOfArguments == 1) {
					aBlock.value_(graphicsContext);
				}
				if (numberOfArguments == 2) {
					aBlock.value_value_(graphicsContext, self.getView());
				}
				if (numberOfArguments == 3) {
					aBlock.value_value_value_(graphicsContext, self.getView(), ((self.getView() == null) ? null : self.getView().controller()));
				}
				if (numberOfArguments == 4) {
					aBlock.valueWithArguments_(new Object[] { graphicsContext, self.getView(), ((self.getView() == null) ? null : self.getView().controller()), self.getViews() });
				}
				return null;
			}
		});
		this.changed();
	}

	/**
	 * Paint the receiver.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StViewJPanel#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		if (this.displayBlock() != null) {
			this.displayBlock().value_(graphicsContext);
		}
	}

	/**
	 * Update the menu indication.
	 * 
	 * @category menu accessing
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();
		this.updateFileMenuIndication();
		this.updateMiscMenuIndication();
		this.updateYellowButtonMenuIndication();
	}

	/**
	 * Update the file menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateFileMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("fileMenu"));
		if (aMenu == null) {
			return;
		}

		StMenu menuItem = (StMenu) aMenu.atNameKey_($("newModel"));
		if (menuItem != null) {
			menuItem.enable();
		}
		menuItem = (StMenu) aMenu.atNameKey_($("quitModel"));
		if (menuItem != null) {
			menuItem.enable();
		}
	}

	/**
	 * Update the misc menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateMiscMenuIndication() {
		if (this._menuBar() == null) {
			return;
		}

		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("miscMenu"));
		if (aMenu == null) {
			return;
		}

		StMenu menuItem = (StMenu) aMenu.atNameKey_($("inspectModel"));
		if (menuItem != null) {
			menuItem.enable();
		}
	}

	/**
	 * Update the yellow button menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateYellowButtonMenuIndication() {
		if (this._popupMenu() == null) {
			return;
		}

		StMenu menuItem = (StMenu) this._popupMenu().atNameKey_($("yourself"));
		if (menuItem == null) {
			menuItem.enable();
		}
	}

	/**
	 * Open an image file.
	 * 
	 * @category menu messages
	 */
	public void openImage() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Gazou file", "image files"), JunSystem.DefaultImageExtensionPatterns()), };
		File aFile = JunFileRequesterDialog.RequestFile($String("Select a gazou file.", "Select a image file."), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return;
		}
		this.openImageFrom_(aFile);
	}

	/**
	 * Open an image from the specified file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.smalltalk.StImage
	 * @category menu messages
	 */
	public StImage openImageFrom_(File aFile) {
		if (aFile.exists() == false) {
			return null;
		}

		StImage anImage = null;
		JunMoviePlayer aMoviePlayer = null;
		JunCursors aCursor = new JunCursors(JunCursors.ReadCursor());
		try {
			aCursor._show();
			aMoviePlayer = new JunMoviePlayer(aFile);
			anImage = aMoviePlayer.asImage();
		} catch (Exception e) {
			anImage = JunImageProcessor.ImageFromFile_(aFile);
		} finally {
			if (aMoviePlayer != null) {
				aMoviePlayer.release();
				aMoviePlayer = null;
			}
			aCursor._restore();
		}

		return anImage;
	}

	/**
	 * Save an image file.
	 * 
	 * @return java.io.File
	 * @category menu messages
	 */
	public File saveImage() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Gazou file", "image files"), JunSystem.DefaultImageExtensionPatterns()), };
		File aFile = JunFileRequesterDialog.RequestNewFile($String("Input a gazou file.", "Input a image file."), new File(this.defaultBaseName() + "." + JunSystem.DefaultImageExtension()), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return null;
		}
		return this.saveImageTo_(aFile);
	}

	/**
	 * Quit the receiver.
	 * 
	 * @category menu messages
	 */
	public void quitModel() {
		this.closeRequest();
	}

	/**
	 * Save an image to the specified file.
	 * 
	 * @param aFile java.io.File
	 * @return java.io.File
	 * @category menu messages
	 */
	public File saveImageTo_(File aFile) {
		StImage anImage = this.asImage();
		if (anImage == null) {
			return null;
		}

		JunImageStream aStream = null;
		JunCursors aCursor = new JunCursors(JunCursors.WriteCursor());
		try {
			aCursor._show();
			aStream = JunJpegImageStream.On_(new BufferedOutputStream(new FileOutputStream(aFile)));
			((JunJpegImageStream) aStream).quality_(0.95f);
			aStream.nextPutImage_(anImage);
		} catch (IOException e) {
			System.out.println(e.getMessage());
			e.printStackTrace();
		} finally {
			if (aStream != null) {
				try {
					aStream.flush();
					aStream.close();
				} catch (IOException e) {
					System.out.println(e.getMessage());
					e.printStackTrace();
				}
			}
			aCursor._restore();
		}
		return aFile;
	}

	/**
	 * Answer the receiver's menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		return (StMenuBar) this.attributeTable().at_ifAbsentPut_($("menuBar"), new StBlockClosure() {
			public Object value() {
				StMenuBar menuBar = new StMenuBar();

				StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));
				fileMenu.add(new StMenuItem($String("New"), $("New"), new MenuPerformer(JunAbstractModel.this, "newModel")));
				fileMenu.add(new StMenuItem($String("Quit"), $("Quit"), new MenuPerformer(JunAbstractModel.this, "quitModel")));
				menuBar.add(fileMenu);

				StMenu miscMenu = new StMenu($String("Misc"), $("miscMenu"));
				miscMenu.add(new StMenuItem($String("Inspect"), $("Inspect"), new MenuPerformer(JunAbstractModel.this, "inspectModel")));
				menuBar.add(miscMenu);

				return menuBar;
			}
		});
	}

	/**
	 * Answer the receiver's popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		return (StPopupMenu) this.attributeTable().at_ifAbsentPut_($("_popupMenu"), new StBlockClosure() {
			public Object value() {
				return null;
			}
		});
	}
}
