package jp.co.sra.jun.system.support;

import java.awt.*;
import java.awt.image.*;
import java.util.Vector;
import jp.co.sra.smalltalk.*;
import jp.co.sra.jun.geometry.basic.*;

/**
 * JunSmallCompiler class
 * 
 *  @author    ASTI Beijing
 *  @created   1998/12/09 (by ASTI Beijing)
 *  @updated   1999/06/21 (by nisinaka)
 *  @updated   2003/04/25 (by nisinaka)
 *  @updated   2003/05/07 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSmallCompiler.java,v 8.11 2008/02/20 06:32:59 nisinaka Exp $
 */
public class JunSmallCompiler extends StObject {
	protected StReadStream source;
	protected int mark;
	protected int prevEnd;
	protected char hereChar;
	protected Object token;
	protected StSymbol tokenType;
	protected boolean saveComments;
	protected Vector currentComment;
	protected StringBuffer buffer;
	protected JunSmallTypeTable typeTable;
	protected Object parseNode;

	/**
	 * Create a new instance of JunLispSmallCompiler and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunSmallCompiler() {
		super();
		this.initScanner();
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @category Evaluating
	 */
	public static Object Evaluate_(String aString) {
		return Evaluate_for_logged_(aString, null, false);
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @param anObject java.lang.Object
	 * @param logFlag boolean
	 * @category Evaluating
	 */
	public static Object Evaluate_for_logged_(String aString, Object anObject, boolean logFlag) {
		return Evaluate_for_notifying_logged_(aString, anObject, null, logFlag);
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @param anObject java.lang.Object
	 * @param aController java.lang.Object
	 * @param logFlag boolean
	 * @category Evaluating
	 */
	public static Object Evaluate_for_notifying_logged_(String aString, Object anObject, Object aController, boolean logFlag) {
		return (new JunSmallCompiler()).evaluate_in_receiver_notifying_ifFail_(aString, null, anObject, aController, new StBlockClosure());
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @param logFlag boolean
	 * @category Evaluating
	 */
	public static Object Evaluate_logged_(String aString, boolean logFlag) {
		return Evaluate_for_logged_(aString, null, logFlag);
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @param aController java.lang.Object
	 * @param logFlag boolean
	 * @category Evaluating
	 */
	public static Object Evaluate_notifying_logged_(String aString, Object aController, boolean logFlag) {
		return Evaluate_for_notifying_logged_(aString, null, aController, logFlag);
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @param aContext java.lang.Object
	 * @param receiver java.lang.Object
	 * @param aRequestor java.lang.Object
	 * @param failBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category evaluating
	 */
	public Object evaluate_in_receiver_notifying_ifFail_(String aString, Object aContext, Object receiver, Object aRequestor, StBlockClosure failBlock) {
		return this.compile_(aString);
	}

	/**
	 * Evaluate the string as a Smalltalk expression.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @param aContext java.lang.Object
	 * @param receiver java.lang.Object
	 * @param aRequestor java.lang.Object
	 * @param failBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category evaluating
	 */
	public Object evaluate_in_to_notifying_ifFail_(String aString, Object aContext, Object receiver, Object aRequestor, StBlockClosure failBlock) {
		Object value = null;
		value = this.evaluate_in_receiver_notifying_ifFail_(aString, aContext, receiver, aRequestor, failBlock);
		return value;
	}

	/**
	 * Set the source stream.
	 * 
	 * @param inputStream jp.co.sra.smalltalk.StReadStream
	 * @category scanning
	 */
	public void on_(StReadStream inputStream) {
		source = inputStream;
		hereChar = source.next();
		prevEnd = 0;
	}

	/**
	 * Scan the source stream.
	 * 
	 * @param inputStream jp.co.sra.smalltalk.StReadStream
	 * @category scanning
	 */
	public void scan_(StReadStream inputStream) {
		this.on_(inputStream);
		this.scanToken();
	}

	/**
	 * Scan the field names.
	 * 
	 * @return java.lang.Object[]
	 * @param aString java.lang.String
	 * @category scanning
	 */
	public Object[] scanFieldNames_(String aString) {
		if (aString.length() == 0) {
			return new Object[0];
		}

		this.scan_(new StReadStream(aString));
		Vector collection = new Vector(10);
		while (tokenType != $("doIt")) {
			if (tokenType == $("word")) {
				collection.addElement(token);
			}
			this.scanToken();
		}

		Object[] array = new Object[collection.size()];
		collection.copyInto(array);
		return array;
	}

	/**
	 * Scan the positions of the Symbol in the String.
	 * 
	 * @return int[]
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aString java.lang.String
	 * @category scanning
	 */
	public int[] scanPositionsFor_inString_(StSymbol aSymbol, String aString) {
		this.scan_(new StReadStream(aString));
		String string = aSymbol.toString();
		Vector positions = new Vector(16);
		while (tokenType != $("doIt")) {
			if ((tokenType == $("word")) && string.equals(token)) {
				positions.addElement(new Integer(mark));
			}
			this.scanToken();
		}

		int[] array = new int[positions.size()];
		for (int i = 0; i < positions.size(); i++) {
			array[i] = ((Integer) positions.elementAt(i)).intValue();
		}
		return array;
	}

	/**
	 * Scan the string and get an array of parsed nodes.
	 * 
	 * @return java.lang.Object[]
	 * @param aString java.lang.String
	 * @category scanning
	 */
	public Object[] scanTokens_(String aString) {
		this.scan_(new StReadStream(aString));
		Vector collection = new Vector(16);
		while (tokenType != $("doIt")) {
			this.arrayLiteralElement();
			collection.addElement(parseNode);
		}

		Object[] array = new Object[collection.size()];
		collection.copyInto(array);
		return array;
	}

	/**
	 * Process for a binary token.
	 * 
	 * @category scanning
	 */
	public void xBinary() {
		tokenType = $("binary");
		char ch = hereChar;
		hereChar = source.next();
		StringBuffer buffer = new StringBuffer();
		buffer.append(ch);
		if ((hereChar != 0) && (typeTable.at_(hereChar) == $("xBinary")) && (hereChar != '-')) {
			buffer.append(this.step());
		}
		token = $(buffer.toString());
	}

	/**
	 * Process for a default token.
	 * 
	 * @category scanning
	 */
	public void xDefault() {
		this.notify_("unknown character");
	}

	/**
	 * Process for a delimiter token.
	 * 
	 * @category scanning
	 */
	public void xDelimiter() {
		this.scanToken();
	}

	/**
	 * Process for a digit token.
	 * 
	 * @category scanning
	 */
	public void xDigit() {
		tokenType = $("number");
		if (hereChar != 0) {
			source.skip_(-1);
		}
		token = StNumber.ReadFrom_(source);
		hereChar = source.next();
	}

	/**
	 * Process for a dollar token.
	 * 
	 * @category scanning
	 */
	public void xDollar() {
		token = new Character(source.next());
		hereChar = source.next();
		tokenType = $("character");
	}

	/**
	 * Process a double quote token.
	 * 
	 * @category scanning
	 */
	public void xDoubleQuote() {
		char c;
		int savedEnd = prevEnd;
		if (!saveComments) {
			while ((c = source.next()) != '\"') {
				if (c == 0) {
					this.offEnd_("unmatched comment quote");
				}
			}
		} else {
			buffer.setLength(0);
			while ((c = source.next()) != '\"') {
				if (c == 0) {
					this.offEnd_("unmatched comment quote");
				}
				buffer.append(c);
			}
			if (currentComment == null) {
				currentComment = new Vector();
			}
			currentComment.addElement(buffer.toString());
			if (buffer.length() > 200) {
				buffer = new StringBuffer(40);
			}
		}

		hereChar = source.next();
		this.scanToken();
		prevEnd = savedEnd;
	}

	/**
	 * Process a letter token.
	 * 
	 * @category scanning
	 */
	public void xLetter() {
		buffer.setLength(0);
		buffer.append(hereChar);
		char c = 0;
		StSymbol type = null;
		while (((c = source.next()) != 0) && ((type = typeTable.at_(c)) == $("xLetter") || type == $("xDigit"))) {
			buffer.append(c);
		}
		if (type == $("colon")) {
			buffer.append(c);
			c = source.next();
			tokenType = $("keyword");
		} else {
			tokenType = $("word");
		}
		hereChar = c;
		token = buffer.toString();
	}

	/**
	 * Process a lit quote token
	 * 
	 * @category scanning
	 */
	public void xLitQuote() {
		this.step();
		this.scanToken();
		if (tokenType == $("leftParenthesis")) {
			int start = mark;
			this.scanToken();
			this.scanLitVec();
			if (tokenType == $("doIt")) {
				mark = start;
				this.offEnd_("unmatched parenthesis");
			}
		} else {
			if (tokenType == $("word")) {
				token = $(token.toString());
			} else {
				if (tokenType == $("keyword")) {
					this.scanLitKeywords();
				}
			}
		}

		tokenType = $("literal");
	}

	/**
	 * Process a single quote token
	 * 
	 * @category scanning
	 */
	public void xSingleQuote() {
		char c = source.next();
		if (c == 0) {
			this.offEnd_("unmatched string quote");
			return;
		}

		buffer = new StringBuffer(32);
		while (!(c == '\'' && (c = source.next()) != '\'')) {
			buffer.append(c);
			c = source.next();
		}

		hereChar = c;
		token = buffer.toString();
		tokenType = $("string");
	}

	/**
	 * Parse an array literal.
	 * 
	 * @category scanning
	 */
	protected void arrayLiteral() {
		Vector collection = new Vector(16);
		while (tokenType != $("rightParenthesis")) {
			if (tokenType == $("doIt")) {
				this.notify_("array element or right  parenthesis");
				return;
			}

			this.arrayLiteralElement();
			collection.addElement(parseNode);
		}

		Object[] array = new Object[collection.size()];
		collection.copyInto(array);
		parseNode = array;
		this.scanToken();
	}

	/**
	 * Parse an array literal element.
	 * 
	 * @category scanning
	 */
	protected void arrayLiteralElement() {
		if (this.constant()) {
			return;
		}

		if (tokenType == $("leftParenthesis")) {
			this.scanToken();
			this.arrayLiteral();
			return;
		}

		if (tokenType == $("leftBracket")) {
			this.scanToken();
			this.byteArrayLiteral();
			return;
		}

		this.scanLitToken();
		parseNode = token;
		this.scanToken();
	}

	/**
	 * Parse a byte array literal.
	 * 
	 * @category scanning
	 */
	protected void byteArrayLiteral() {
		StringBuffer buffer = new StringBuffer(16);
		int size = 0;
		while (tokenType != $("rightBracket")) {
			int number;
			if (!((tokenType == $("number")) && token instanceof Integer && ((number = ((Integer) token).intValue()) >= 0) && number <= 255)) {
				this.notify_("8-bit integer or right bracket");
				return;
			}

			buffer.append((char) number);
			size++;
			this.scanToken();
		}

		byte[] byteArray = new byte[size];
		for (int i = 0; i < size; i++) {
			byteArray[i] = (byte) buffer.charAt(i);
		}

		parseNode = byteArray;
		this.scanToken();
	}

	/**
	 * Compile the string.
	 * 
	 * @return java.lang.Object
	 * @param aString java.lang.String
	 * @category public access
	 */
	protected Object compile_(String aString) {
		Object[] tokenArray = this.scanTokens_(aString);
		return this.translate_(tokenArray);
	}

	/**
	 * Parse a constance value.
	 * 
	 * @return boolean
	 * @category constants
	 */
	protected boolean constant() {
		if (tokenType == $("word")) {
			String tokenString = (String) token;
			int size = tokenString.length();
			if (size == 5) {
				if (tokenString.equals("false")) {
					this.scanToken();
					parseNode = Boolean.FALSE;
					return true;
				}
			} else if (size == 4) {
				if (tokenString.equals("true")) {
					this.scanToken();
					parseNode = Boolean.TRUE;
					return true;
				}
			} else if (size == 3) {
				if (tokenString.equals("nil")) {
					scanToken();
					parseNode = null;
					return true;
				}
			}
			return false;
		}

		if ((tokenType == $("string")) || (tokenType == $("number")) || (tokenType == $("character"))) {
			parseNode = token;
			this.scanToken();
			return true;
		}

		if ((token == $("-")) && (hereChar != 0) && Character.isDigit(hereChar)) {
			this.scanToken();
			parseNode = StNumber._Negate((Number) token);
			this.scanToken();
			return true;
		}

		if (tokenType != $("literalQuote")) {
			return false;
		}

		this.scanToken();
		if ((tokenType == $("word")) || (tokenType == $("binary")) || (tokenType == $("verticalBar")) || (tokenType == $("string"))) {
			parseNode = $(token.toString());
			this.scanToken();
			return true;
		}

		if (tokenType == $("keyword")) {
			this.scanLitKeywords();
			parseNode = $(token.toString());
			this.scanToken();
			return true;
		}

		if (tokenType == $("leftParenthesis")) {
			this.scanToken();
			this.arrayLiteral();
			return true;
		}

		if (tokenType == $("leftBracket")) {
			this.scanToken();
			this.byteArrayLiteral();
			return true;
		}

		this.notify_("word,binary,keyword,(,or [");
		return false;
	}

	/**
	 * Initialize the scanner.
	 * 
	 * @category initialize-release
	 */
	protected void initScanner() {
		buffer = new StringBuffer(32);
		saveComments = true;
		typeTable = JunSmallTypeTable.TypeTable();
	}

	/**
	 * Throw a SmalltalkException to nofity an error or something.
	 * 
	 * @param message java.lang.String
	 * @category scanning
	 */
	protected void notify_(String message) {
		throw SmalltalkException.Error(message);
	}

	/**
	 * Notify the unmatched grouping.
	 * 
	 * @param aString java.lang.String
	 * @category scanning
	 */
	protected void offEnd_(String aString) {
		this.notify_(aString);
	}

	/**
	 * Scan a literal keyword.
	 * 
	 * @category scanning
	 */
	protected void scanLitKeywords() {
		boolean inKeywords = true;
		while (inKeywords && (hereChar != 0) && (typeTable.at_(hereChar) == $("xLetter"))) {
			Object t = token;
			int lastP = source.position();
			char lastCh = hereChar;
			this.xLetter();
			if (tokenType == $("keyword")) {
				token = t.toString() + token;
			} else {
				inKeywords = false;
				token = t;
				hereChar = lastCh;
				source.position_(lastP);
				tokenType = $("keyword");
			}
		}

		token = $(token.toString());
	}

	/**
	 * Scan a literal token.
	 * 
	 * @category scanning
	 */
	protected void scanLitToken() {
		if (tokenType == $("leftParenthesis")) {
			this.scanToken();
			this.scanLitVec();
		} else {
			if (tokenType == $("word")) {
				token = $(token.toString());
			} else {
				if (tokenType == $("keyword")) {
					this.scanLitKeywords();
				} else {
					if ((token == $("-")) && (hereChar != 0) && (typeTable.at_(hereChar) == $("xDigit"))) {
						this.scanToken();
						token = StNumber._Negate((Number) token);
					} else {
						if (token instanceof Character && (tokenType != $("character"))) {
							token = $(token.toString());
						}
					}
				}
			}
		}
	}

	/**
	 * Scan a literal vector.
	 * 
	 * @category scanning
	 */
	protected void scanLitVec() {
		Vector collection = new Vector(16);
		while ((tokenType != $("rightParenthesis")) && (tokenType != $("doIt"))) {
			this.scanLitToken();
			collection.addElement(token);
			this.scanToken();
		}

		Object[] array = new Object[collection.size()];
		collection.copyInto(array);
		token = array;
	}

	/**
	 * Scan a token.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category scanning
	 */
	protected void scanToken() {
		if (hereChar == 0) {
			prevEnd = source.position();
			mark = prevEnd + 1;
			tokenType = $("doIt");
			token = null;
			return;
		}

		int skipped = 0;
		while ((tokenType = typeTable.at_(hereChar)) == $("xDelimiter")) {
			skipped++;
			hereChar = source.next();
			if (hereChar == 0) {
				prevEnd = source.position();
				mark = prevEnd + 1;
				tokenType = $("doIt");
				token = null;
				return;
			}
		}

		mark = source.position();
		prevEnd = mark - skipped - 1;
		if (tokenType == $("xLetter")) {
			this.xLetter();
		} else {
			String selector = tokenType.toString();
			if (selector.charAt(0) == 'x') {
				try {
					this.perform_(selector);
				} catch (Exception e) {
					throw new SmalltalkException(e);
				}
			} else {
				token = new Character(hereChar);
				hereChar = source.next();
			}
		}
	}

	/**
	 * Answer a string which consists separator characters.
	 * 
	 * @return jp.co.sra.smalltalk.StString
	 * @category string utilities
	 */
	protected String separators() {
		StringBuffer buffer = new StringBuffer(4);
		buffer.append(' ');
		buffer.append('\t');
		buffer.append('\r');
		buffer.append('\n');
		return buffer.toString();
	}

	/**
	 * Step into the next character.
	 * 
	 * @return char
	 * @category scanning
	 */
	protected char step() {
		char c = hereChar;
		hereChar = source.next();
		return c;
	}

	/**
	 * Answer true if the string matches the pattern.
	 * 
	 * @return boolean
	 * @param string java.lang.String
	 * @param pattern java.lang.String
	 * @category string utilities
	 */
	protected boolean stringMatch_and_(String string, String pattern) {
		StringBuffer s1 = new StringBuffer(20);
		s1.append(string);
		s1.append('\0');
		StringBuffer s2 = new StringBuffer(20);
		s2.append(string);
		s2.append('\0');
		return stringMatch_sIndex_and_pIndex_(s1.toString(), 0, s2.toString(), 0);
	}

	/**
	 * Answer true if the string matches the pattern.
	 * 
	 * @return boolean
	 * @param string java.lang.String
	 * @param sindex int
	 * @param pattern java.lang.String
	 * @param pindex int
	 * @category string utilities
	 */
	protected boolean stringMatch_sIndex_and_pIndex_(String string, int sindex, String pattern, int pindex) {
		int si = sindex;
		int pi = pindex;
		int slength = string.length();
		int plength = pattern.length();
		char scc = (si <= slength) ? string.charAt(si) : 0;
		si++;
		char c = (pi <= plength) ? pattern.charAt(pi) : 0;
		pi++;
		if (c == '[') {
			boolean ok = false;
			char lc = 255;
			while (pi <= plength) {
				c = pattern.charAt(pi);
				pi++;
				if (c == 0) {
					break;
				}

				if (c == ']') {
					if (ok) {
						return this.stringMatch_sIndex_and_pIndex_(string, si, pattern, pi);
					} else {
						return false;
					}
				} else {
					if (c == '-') {
						if ((lc <= scc) && (scc <= pattern.charAt(pi))) {
							ok = true;
						}
						pi = pi + 1;
					} else {
						lc = c;
						if (scc == c) {
							ok = true;
						}
					}
				}
			}

			c = 0;
		}

		if (c == '?') {
			if (scc != 0) {
				return this.stringMatch_sIndex_and_pIndex_(string, si, pattern, pi);
			} else {
				return false;
			}
		}

		if (c == '*') {
			if (pattern.charAt(pi) == 0) {
				return true;
			}

			si = si - 1;
			while (string.charAt(si) != 0) {
				if (this.stringMatch_sIndex_and_pIndex_(string, si, pattern, pi)) {
					return true;
				}
				si = si + 1;
			}

			return false;
		}

		if (c == 0) {
			if (scc == 0) {
				return true;
			} else {
				return false;
			}
		}

		if (scc != c) {
			return false;
		}

		if (scc != 0) {
			return this.stringMatch_sIndex_and_pIndex_(string, si, pattern, pi);
		} else {
			return false;
		}
	}

	/**
	 * Translate the token array and generate an object as a evaluation result.
	 * 
	 * @return java.lang.Object
	 * @param tokenArray java.lang.Object[]
	 * @category transforming
	 */
	protected Object translate_(Object[] tokenArray) {
		if (tokenArray.length == 0) {
			return null;
		}

		if ((tokenArray.length == 1) && tokenArray[0] instanceof Object[]) {
			return this.translate_((Object[]) tokenArray[0]);
		}

		if (tokenArray.length == 1) {
			return tokenArray[0];
		}

		Object receiver = tokenArray[0];
		Object selector = tokenArray[1];
		if (receiver instanceof Number && (selector == $("@") || selector == $(","))) {
			return this.translateToPoint_(tokenArray);
		}

		if ((receiver == $("Text")) && (selector == $("string:"))) {
			return this.translateToText_(tokenArray);
		}

		if ((selector == $("x:")) && ((receiver == $("Point")) || (receiver == $("Jun3dPoint")) || receiver == $("Jun2dPoint"))) {
			return this.translateToPoint_(tokenArray);
		}

		if ((receiver == $("ByteArray")) && (selector == $("fromPackedString:"))) {
			return this.translateToByteArray_(tokenArray);
		}

		if ((receiver == $("Image")) && (selector == $("extent:"))) {
			return this.translateToImage_(tokenArray);
		}

		if (receiver == $("ColorValue")) {
			return this.translateToColor_(tokenArray);
		}

		return null;
	}

	/**
	 * Translate the token array to an array of byte.
	 * 
	 * @return byte[]
	 * @param anObject java.lang.Object
	 * @category transforming
	 */
	protected byte[] translateToByteArray_(Object anObject) {
		if (anObject instanceof byte[]) {
			return (byte[]) anObject;
		}

		Object[] tokenArray = (Object[]) anObject;
		Object aSymbol = tokenArray[0];
		// Object aSelector = tokenArray[1];
		if ((aSymbol != $("ByteArray")) && (aSymbol != $("fromPackedString:"))) {
			throw SmalltalkException.Error("unexpected error");
		}

		return StByteArray.FromPackedString_((String) tokenArray[2])._asBytes();
	}

	/**
	 * Translate the token array to a color.
	 * 
	 * @return java.lang.Object
	 * @param tokenArray java.lang.Object[]
	 * @category transforming
	 */
	protected Object translateToColor_(Object[] tokenArray) {
		Object receiver = tokenArray[0];
		if (receiver != $("ColorValue") && tokenArray.length >= 2) {
			this.notify_("unexpected error");
			return null;
		}

		Object selector = tokenArray[1];
		if (StColorValue.ConstantNames().contains(selector)) {
			return StColorValue._GetColorByName_(selector.toString());
		}
		if (selector == $("hue:")) {
			return new Color(Color.HSBtoRGB(((Number) tokenArray[2]).floatValue(), ((Number) tokenArray[4]).floatValue(), ((Number) tokenArray[6]).floatValue()));
		}
		if (selector == $("red:")) {
			return new Color(((Number) tokenArray[2]).floatValue(), ((Number) tokenArray[4]).floatValue(), ((Number) tokenArray[6]).floatValue());
		}
		if (selector == $("scaledRed:")) {
			return StColorValue.ScaledRed_ScaledGreen_ScaledBlue_(((Number) tokenArray[2]).intValue(), ((Number) tokenArray[4]).intValue(), ((Number) tokenArray[6]).intValue());
		}
		if (selector == $("brightness:")) {
			return StColorValue.Brightness_(((Number) tokenArray[2]).doubleValue());
		}

		this.notify_("unexpected error");
		return null;
	}

	/**
	 * Translate the token array to an image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @param tokenArray java.lang.Object[]
	 * @category transforming
	 */
	protected StImage translateToImage_(Object[] tokenArray) {
		Object aSymbol = tokenArray[0];
		if ((aSymbol != $("Image")) && (aSymbol != $("OpaqueImage")) && (aSymbol != $("CachedImage"))) {
			throw SmalltalkException.Error("unexpected error");
		}

		StImage anImage = null;
		if (aSymbol == $("Image")) {
			int width = ((Number) tokenArray[2]).intValue();
			int height = ((Number) tokenArray[4]).intValue();
			// int colorDepth = ((Number) tokenArray[6]).intValue();
			int bitsPerPixel = ((Number) tokenArray[8]).intValue();
			Object anObject = tokenArray[10];
			ColorModel colorModel = null;
			byte[] bytes = null;
			if (anObject instanceof Object[]) {
				colorModel = this.translateToPalette_((Object[]) tokenArray[10], bitsPerPixel);
				bytes = this.translateToByteArray_(tokenArray[12]);
			} else {
				colorModel = this.translateToPalette_(new Object[] { tokenArray[10], tokenArray[11] }, bitsPerPixel);
				bytes = this.translateToByteArray_(tokenArray[13]);
			}
			anImage = new StImage(width, height, bytes, colorModel);
		} else {
			throw SmalltalkException.Error("The image [" + aSymbol + "] is not supported yet.");
		}

		return anImage;
	}

	/**
	 * Translate the token array to a ColorModel.
	 * 
	 * @return java.awt.image.ColorModel
	 * @param tokenArray java.lang.Object[]
	 * @param bitsPerPixel int
	 * @category transforming
	 */
	protected ColorModel translateToPalette_(Object[] tokenArray, int bitsPerPixel) {
		int offset = 0;
		if ((tokenArray[0] == $("Graphics")) && (tokenArray[1] == $("."))) {
			offset += 2;
		}
		Object aSymbol = tokenArray[offset];
		if ((aSymbol != $("CoveragePalette")) && (aSymbol != $("FixedPalette")) && (aSymbol != $("MappedPalette")) && (aSymbol != $("MonoMappedPalette"))) {
			throw SmalltalkException.Error("unepxected error");
		}

		ColorModel colorModel = null;
		if (aSymbol == $("FixedPalette")) {
			int redShift = 0;
			int redMask = 0;
			int greenShift = 0;
			int greenMask = 0;
			int blueShift = 0;
			int blueMask = 0;
			for (int i = offset + 1; i < tokenArray.length; i += 2) {
				Object key = tokenArray[i];
				int value = ((Number) tokenArray[i + 1]).intValue();
				if (key == $("redShift:")) {
					redShift = value;
				} else if (key == $("redMask:")) {
					redMask = value;
				} else if (key == $("greenShift:")) {
					greenShift = value;
				} else if (key == $("greenMask:")) {
					greenMask = value;
				} else if (key == $("blueShift:")) {
					blueShift = value;
				} else if (key == $("blueMask:")) {
					blueMask = value;
				}
			}

			if (bitsPerPixel == 32) {
				// Workaround for a bug of java.awt.image.DirectColorModel.isCompatibleRaster(Raster) in JDK1.2.2.
				colorModel = new DirectColorModel(bitsPerPixel, redMask << redShift, greenMask << greenShift, blueMask << blueShift, 0xff000000);
			} else {
				colorModel = new DirectColorModel(bitsPerPixel, redMask << redShift, greenMask << greenShift, blueMask << blueShift);
			}
		} else if ((aSymbol == $("MappedPalette")) || (aSymbol == $("MonoMappedPalette"))) {
			int size = 0;
			byte[] r = null;
			byte[] g = null;
			byte[] b = null;
			byte[] a = null;
			StSymbol selector = (StSymbol) tokenArray[1];
			if (selector == $("blackWhite")) {
				size = 2;
				r = g = b = new byte[] {(byte) 0, (byte) 255 };
			} else if (selector == $("whiteBlack")) {
				size = 2;
				r = g = b = new byte[] {(byte) 255, (byte) 0 };
			} else {
				throw SmalltalkException.Error("The palette [" + aSymbol + "] in general form is not supported yet.");
			}
			colorModel = new IndexColorModel(bitsPerPixel, size, r, g, b, a);
		} else {
			throw SmalltalkException.Error("The palette [" + aSymbol + "] is not supported yet.");
		}

		return colorModel;
	}

	/**
	 * Translate the token array to a point.
	 * 
	 * @return java.lang.Object
	 * @param tokenArray java.lang.Object[]
	 * @category transforming
	 */
	protected Object translateToPoint_(Object[] tokenArray) {
		Object receiver = tokenArray[0];
		// Object selector = tokenArray[1];
		if (receiver instanceof Number) {
			if (tokenArray.length == 3) {
				return new Jun2dPoint(((Number) tokenArray[0]).doubleValue(), ((Number) tokenArray[2]).doubleValue());
			} else if (tokenArray.length == 5) {
				return new Jun3dPoint(((Number) tokenArray[0]).doubleValue(), ((Number) tokenArray[2]).doubleValue(), ((Number) tokenArray[4]).doubleValue());
			} else if (tokenArray.length == 7) {
				return new Rectangle(((Number) tokenArray[0]).intValue(), ((Number) tokenArray[2]).intValue(), ((Number) tokenArray[4]).intValue(), ((Number) tokenArray[6]).intValue());
			}

			this.notify_("unexpected error");
			return null;
		}

		if ((receiver == $("Point")) || (receiver == $("Jun2dPoint"))) {
			return new Jun2dPoint(((Number) tokenArray[2]).doubleValue(), ((Number) tokenArray[4]).doubleValue());
		}

		if (receiver == $("Jun3dPoint")) {
			return new Jun3dPoint(((Number) tokenArray[2]).doubleValue(), ((Number) tokenArray[4]).doubleValue(), ((Number) tokenArray[6]).doubleValue());
		}

		this.notify_("unexpected error");
		return null;
	}

	/**
	 * Translate the token array to a point.
	 * 
	 * @return java.lang.Object (java.lang.String)
	 * @param tokenArray java.lang.Object[]
	 * @category transforming
	 */
	protected Object translateToText_(Object[] tokenArray) {
		Object reveiver = tokenArray[0];
		Object selector1 = tokenArray[1];
		Object selector2 = tokenArray[3];
		if (reveiver != $("Text") || selector1 != $("string:") || selector2 != $("runs:") || tokenArray.length != 5) {
			throw SmalltalkException.Error("unexpected error");
		}

		return (String) tokenArray[2];
	}
}
