package jp.co.sra.jun.topology.elements;

import java.io.*;
import java.util.*;
import jp.co.sra.jun.geometry.abstracts.*;
import jp.co.sra.jun.geometry.basic.*;
import jp.co.sra.jun.geometry.boundaries.*;
import jp.co.sra.jun.geometry.boundaries.Jun2dPolygonalBoundary;
import jp.co.sra.jun.geometry.curves.*;
import jp.co.sra.jun.geometry.surfaces.*;
import jp.co.sra.jun.goodies.lisp.*;
import jp.co.sra.jun.opengl.objects.*;
import jp.co.sra.jun.opengl.support.*;
import jp.co.sra.jun.topology.abstracts.*;
import jp.co.sra.smalltalk.*;

/**
 * JunLoop class
 * 
 *  @author    ASTI Shanghai
 *  @created   UNKNOWN
 *  @updated   1999/08/04 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunLoop.java,v 8.11 2008/02/20 06:33:01 nisinaka Exp $
 */
public class JunLoop extends JunTopologicalElement {
	/** The edge contained in the loop. */
	protected JunEdge edge;

	/** The parent loop. */
	protected JunLoop parentLoop;

	/** The child loop. */
	protected JunLoop childLoop;

	/** An surface which corresponds to the loop. */
	protected JunSurface surface;

	/**
	 * Create a new instance of <code>JunLoop</code> and initialize it with the JunEdge.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category Instance creation
	 */
	public static final JunLoop Edge_(JunEdge anEdge) {
		JunLoop aLoop = new JunLoop();
		aLoop.edge_(anEdge);
		return aLoop;
	}

	/**
	 * Add a loop as a child loop of the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category adding
	 */
	public JunLoop addChildLoop_(JunLoop aLoop) {
		if (this.isChildLoop() == true) {
			// return this.parentLoop().addChildLoop_(aLoop);
			throw SmalltalkException.Error("should not be a child loop.");
		}

		aLoop.parentLoop_(this);
		aLoop.childLoop_(this.childLoop());
		childLoop = aLoop;
		return aLoop;
	}

	/**
	 * Convert the surface of the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunSurface surface = this.surface();

		if (surface == null) {
			return null;
		}
		if (surface instanceof Jun3dPolygon == false) {
			// So far, it's only support JunPolygon as a surface.
			return null;
		}

		final Jun3dPolygon polygon = (Jun3dPolygon) surface;
		if (this.hasChildLoop() == false) {
			return polygon.asJunOpenGL3dObject();
		}

		Jun2dPolygon parameterPolygon = polygon.parameterPolygon();
		final Jun2dPolygonalBoundary boundary = new Jun2dPolygonalBoundary();
		boundary.addPolygon_(parameterPolygon);
		this.childLoopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop childPolygon = (JunLoop) anObject;
				Vector parameterPoints = new Vector();
				Jun3dPoint[] points = childPolygon.points();

				for (int i = 0; i < points.length; i++) {
					parameterPoints.addElement(polygon.parameterPointAtPoint_(points[i]));
				}

				boundary.addHole_(new Jun2dPolygon(parameterPoints));

				return null;
			}
		});

		Jun2dPolygon[] newParameterPolygons = boundary.asArrayOfPolygons();
		if (newParameterPolygons.length == 1) {
			Jun3dPolygon p = Jun3dPolygon.Origin_uVector_vVector_parameterPolygon_(polygon.origin(), polygon.uVector(), polygon.vVector(), newParameterPolygons[0]);
			return p.asJunOpenGL3dObject();
		} else {
			JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
			for (int i = 0; i < newParameterPolygons.length; i++) {
				Jun3dPolygon p = Jun3dPolygon.Origin_uVector_vVector_parameterPolygon_(polygon.origin(), polygon.uVector(), polygon.vVector(), newParameterPolygons[i]);
				compoundObject.add_(p.asJunOpenGL3dObject());
			}
			return compoundObject;
		}
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aJunBody jp.co.sra.jun.topology.elements.JunBody
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @category converting
	 */
	public JunLoopProxy asProxyIn_(JunBody aJunBody) {
		return (JunLoopProxy) this.asProxyIn_advise_(aJunBody, null);
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * @param aTopologicalElementProxy jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElementOrProxy#asProxyIn_advise_(jp.co.sra.jun.topology.elements.JunBody, jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy)
	 * @category converting
	 */
	public JunTopologicalElementProxy asProxyIn_advise_(JunBody aBody, JunTopologicalElementProxy aTopologicalElementProxy) {
		JunLoopProxy proxy = (JunLoopProxy) aBody.proxyForLoop_(this);
		if (proxy != null) {
			return proxy;
		} else {
			return aBody.addLoop_as_(this, (JunLoopProxy) aTopologicalElementProxy);
		}
	}

	/**
	 * Convert the corresponding geometry as a wireframed JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#asWireframedOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asWireframedOpenGL3dObject() {
		final JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object o) {
				JunEdge e = (JunEdge) o;
				JunCurve _curve = e.curve();
				if (_curve instanceof Jun3dLine) {
					compoundObject.add_(((Jun3dLine) _curve).asJunOpenGL3dObject());
				} else if (_curve instanceof JunNurbsCurve) {
					compoundObject.add_(((JunNurbsCurve) _curve).asJunOpenGL3dObject());
				}
				return null;
			}
		});

		return compoundObject;
	}

	/**
	 * Answer the surface of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface
	 * @category accessing
	 */
	public final JunSurface basicSurface() {
		return surface;
	}

	/**
	 * Answer the bounding box of the surface of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return ((Jun3dPolygon) this.surface()).boundingBox();
	}

	/**
	 * Answer the child loop of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public final JunLoop childLoop() {
		return childLoop;
	}

	/**
	 * Set a child loop of the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public final void childLoop_(JunLoop aLoop) {
		childLoop = aLoop;
	}

	/**
	 * Answer the array of my child loops.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop[]
	 * @category accessing
	 */
	public JunLoop[] childLoops() {
		final Vector loops = new Vector();
		this.childLoopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				loops.addElement(anObject);

				return null;
			}
		});
		JunLoop[] anArrayOfLoops = new JunLoop[loops.size()];
		loops.copyInto(anArrayOfLoops);
		return anArrayOfLoops;
	}

	/**
	 * Enumerate the child loops and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object childLoopsDo_(StBlockClosure aBlock) {
		JunLoop currentLoop;
		JunLoop nextLoop;
		if (this.isChildLoop() == true) {
			// this.parentLoop().childLoopsDo_(aBlock);
			throw SmalltalkException.Error("should not be a child loop.");
		} else {
			currentLoop = this.childLoop();
			while (currentLoop != null) {
				nextLoop = currentLoop.childLoop();
				Object result = aBlock.value_(currentLoop);
				if (result != null) {
					return result;
				}
				currentLoop = nextLoop;
			}
		}

		return null;
	}

	/**
	 * Detect an edge which matches the condition aBlock. If none found,
	 * evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category enumerating
	 */
	public JunEdge detectEdge_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object e) {
				if (((Boolean) aBlock.value_(e)).booleanValue() == true) {
					return e;
				}
				return null;
			}
		});

		return (JunEdge) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Detect a vertex which matches the condition aBlock. If none found,
	 * evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category enumerating
	 */
	public JunVertex detectVertex_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				if (((Boolean) aBlock.value_(v1)).booleanValue() == true) {
					return v1;
				}
				return null;
			}
		});

		return (JunVertex) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Answer the edge of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public final JunEdge edge() {
		return edge;
	}

	/**
	 * Set an edge of the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public final void edge_(JunEdge anEdge) {
		edge = anEdge;
	}

	/**
	 * Answer the edge specified with two vertexes.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge edgeForVertex_and_(final JunVertex aVertex1, final JunVertex aVertex2) {
		return (JunEdge) this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				if ((v1 == aVertex1 && v2 == aVertex2) || (v1 == aVertex2 && v2 == aVertex1)) {
					return e;
				}
				return null;
			}
		});
	}

	/**
	 * Answer the edge which starts from the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge edgeFromVertex_(final JunVertex aVertex) {
		return (JunEdge) this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				if (v1 == aVertex) {
					return e;
				}
				return null;
			}
		});
	}

	/**
	 * Answer the edges contained in the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge[]
	 * @category accessing
	 */
	public JunEdge[] edges() {
		final Vector edges = new Vector(this.numberOfEdges());
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				edges.addElement(anObject);
				return null;
			}
		});

		JunEdge[] array = new JunEdge[edges.size()];
		edges.copyInto(array);
		return array;
	}

	/**
	 * Enumerate the edges and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object edgesDo_(final StBlockClosure aBlock) {
		return this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				return aBlock.value_(e);
			}
		});
	}

	/**
	 * Enumerate the edges and evaluate aBlock while aConditionBlock returns true.
	 * 
	 * @param aConditionBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object edgesWhileTrue_do_(final StBlockClosure aConditionBlock, final StBlockClosure aBlock) {
		final JunLoop self = this;
		return this.edgesDo_(new StBlockClosure() {
			public Object value_(Object e) {
				if (((Boolean) aConditionBlock.value_(e)).booleanValue() == false) {
					return self;
				}
				aBlock.value_(e);
				return null;
			}
		});
	}

	/**
	 * Answer the edge which ends with the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunEdge edgeToVertex_(final JunVertex aVertex) {
		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				if (aVertex == v2) {
					return e;
				}
				return null;
			}
		};
		return (JunEdge) this.edge().onLoop_vertexEdgeVertexDo_(this, aBlock);
	}

	/**
	 * Answer the first edge of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category private
	 */
	public JunEdge firstEdge() {
		return this.edge();
	}

	/**
	 * Answer the first vertex of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category private
	 */
	public JunVertex firstVertex() {
		JunEdge firstEdge = this.firstEdge();
		return firstEdge.oppositeVertexOfVertex_(firstEdge.directionVertexOnLoop_(this));
	}

	/**
	 * Forget every instance variables.
	 * 
	 * @category private
	 */
	public void forget() {
		if (this.isChildLoop() == true) {
			throw SmalltalkException.Error("should not be a child loop.");
		}

		this.detouchFromParentChildReference();
		edge = null;
		parentLoop = null;
		childLoop = null;
		surface = null;
	}

	/**
	 * Initialize the receiver with the lisp list.
	 * 
	 * @param aLispList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @category lisp support
	 */
	public void fromLispList_forLoops_edges_vertexes_(JunLispCons aLispList, final JunLoop[] loopArray, final JunEdge[] edgeArray, JunVertex[] vertexArray) {
		final JunLoop self = this;
		((JunLispCons) aLispList.tail()).do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLispCons tuple = (JunLispCons) anObject;
				if (tuple.head() == $("edge")) {
					edge = edgeArray[((Integer) tuple.tail()).intValue() - 1];
				}
				if (tuple.head() == $("parentLoop")) {
					if (tuple.tail() == JunLispList.NullList()) {
						parentLoop = null;
					} else {
						parentLoop = loopArray[((Integer) tuple.tail()).intValue() - 1];
					}
				}
				if (tuple.head() == $("childLoop")) {
					if (tuple.tail() == JunLispList.NullList()) {
						childLoop = null;
					} else {
						childLoop = loopArray[((Integer) tuple.tail()).intValue() - 1];
					}
				}
				if (tuple.head() == $("surface")) {
					if (tuple.tail() == JunLispList.NullList()) {
						surface = null;
					} else {
						surface = self.surfaceFromList_((JunLispList) tuple.tail());
					}
				}
				return null;
			}
		});
	}

	/**
	 * Answer true if I have a child loop.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasChildLoop() {
		return this.isParentLoop() && (childLoop != null);
	}

	/**
	 * Answer true if the receiver contains the edge, otherwise false.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @return boolean
	 * @category testing
	 */
	public boolean includesEdge_(final JunEdge anEdge) {
		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object e) {
				if (e == anEdge) {
					return Boolean.TRUE;
				}
				return null;
			}
		});

		return (result != null);
	}

	/**
	 * Answer true if the receiver contains a path from aVertex1 to aVertex2, otherwise false.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @return boolean
	 * @category testing
	 */
	public boolean includesPathFrom_to_(final JunVertex aVertex1, final JunVertex aVertex2) {
		Object result = this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				if ((v1 == aVertex1) && (v2 == aVertex2)) {
					return Boolean.TRUE;
				}
				return null;
			}
		});

		return (result != null);
	}

	/**
	 * Answer true if the receiver contains aVertex, otherwise false.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return boolean
	 * @category testing
	 */
	public boolean includesVertex_(final JunVertex aVertex) {
		Object result = this.vertexesDo_(new StBlockClosure() {
			public Object value_(Object v) {
				if (v == aVertex) {
					return Boolean.TRUE;
				}
				return null;
			}
		});

		return (result != null);
	}

	/**
	 * Answer true if the receiver is a child loop, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isChildLoop() {
		if (this.isParentLoop() == true) {
			return false;
		}

		return true;
	}

	/**
	 * Answer true if the receiver is killed, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isKilled() {
		return (edge == null);
	}

	/**
	 * Answer true if the receiver is a parent loop, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isParentLoop() {
		if (this.parentLoop() == null) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Enumerate the neighbor loops and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void neighborLoopsDo_(final StBlockClosure aBlock) {
		final JunLoop self = this;
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunEdge myEdge = (JunEdge) anObject;
				JunLoop loop = myEdge.oppositeLoopOfLoop_(self);
				if (self != loop) {
					aBlock.value_(loop);
				}
				return null;
			}
		});
	}

	/**
	 * Answer the neighbor loop specified with the vertex.
	 * 
	 * @param aJunVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunLoop neighborLoopWithVertex_(JunVertex aJunVertex) {
		final JunLoop self = this;
		return (JunLoop) aJunVertex.loopsDo_(new StBlockClosure() {
			public Object value_(Object hisLoop) {
				if (hisLoop != self) {
					return hisLoop;
				}
				return null;
			}
		});
	}

	/**
	 * Answer the vertex which is next to aVertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunVertex nextVertexOfVertex_(final JunVertex aVertex) {
		return (JunVertex) this.edge().onLoop_vertexEdgeVertexDo_(this, new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				if (aVertex == v1) {
					return v2;
				}
				return null;
			}
		});
	}

	/**
	 * Answer the number of my child loops.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfChildLoops() {
		final StValueHolder num = new StValueHolder(0);
		if (this.isChildLoop() == false) {
			this.childLoopsDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					num.value_(num._intValue() + 1);
					return null;
				}
			});
		}
		return num._intValue();
	}

	/**
	 * Answer the number of edges contained in the receiver.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfEdges() {
		final StValueHolder numberOfEdges = new StValueHolder(0);
		this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				numberOfEdges.value_(numberOfEdges._intValue() + 1);
				return null;
			}
		});
		return numberOfEdges._intValue();
	}

	/**
	 * Answer the number of vertexes contained in the receiver.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfVertexes() {
		final StValueHolder numberOfVertexes = new StValueHolder(0);
		this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				numberOfVertexes.value_(numberOfVertexes._intValue() + 1);
				return null;
			}
		});
		return numberOfVertexes._intValue();
	}

	/**
	 * Answer the parent loop of the receiver
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public final JunLoop parentLoop() {
		return parentLoop;
	}

	/**
	 * Set a loop as the receiver's parent loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @category private
	 */
	public final void parentLoop_(JunLoop aLoop) {
		parentLoop = aLoop;
	}

	/**
	 * Answer the array of Jun3dPoint which corresponds to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] points() {
		final Vector points = new Vector();
		this.pointsDo_(new StBlockClosure() {
			public Object value_(Object p) {
				points.addElement(p);
				return null;
			}
		});
		Jun3dPoint[] anArrayOfPoints = new Jun3dPoint[points.size()];
		points.copyInto(anArrayOfPoints);
		return anArrayOfPoints;
	}

	/**
	 * Enumerate the points and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void pointsDo_(final StBlockClosure aBlock) {
		StBlockClosure b = new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				aBlock.value_(((JunVertex) v1).point());
				return null;
			}
		};
		this.vertexEdgeVertexDo_(b);
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		final BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
		try {
			bw.write("Loop (");
			if (this.edge() != null) {
				this.vertexEdgeVertexDo_(new StBlockClosure() {
					public Object value_value_value_(Object v1, Object e, Object v2) {
						try {
							bw.newLine();
							((JunVertex) v1).printOn_(bw);
							bw.write("->");
							((JunVertex) v2).printOn_(bw);
						} catch (IOException ex) {
						}
						return null;
					}
				});
			} else {
				bw.write("removed");
			}
			bw.write(')');
		} finally {
			bw.flush();
		}
	}

	/**
	 * Remove the loop from the child loops of the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @category removing
	 */
	public void removeChildLoop_(final JunLoop aLoop) {
		if (this.isChildLoop() == true) {
			throw SmalltalkException.Error("should not be a child loop.");
		}
		aLoop.parentLoop_(null);
		if (this.childLoop() == aLoop) {
			this.childLoop_(aLoop.childLoop());
			aLoop.childLoop_(null);
			return;
		}
		Object result = this.childLoopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;
				if (loop.childLoop() == aLoop) {
					loop.childLoop_(aLoop.childLoop());
					aLoop.childLoop_(null);
					return aLoop;
				}
				return null;
			}
		});
		if (result == null) {
			throw SmalltalkException.Error("internal error");
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aRenderingContext DOCUMENT ME!
	 * @category displaying
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		((JunNurbsSurface) this.surface()).renderOn_(aRenderingContext);
	}

	/**
	 * Repatch the receiver.
	 * 
	 * @category accessing
	 */
	public void repatch() {
		if (this.basicSurface() == null) {
			return;
		}
		this.surface_(null);
	}

	/**
	 * Set a surface of the receiver.
	 * 
	 * @param aSurface jp.co.sra.jun.geometry.abstracts.JunSurface
	 * @category accessing
	 */
	public final void setSurface_(JunSurface aSurface) {
		surface = aSurface;
	}

	/**
	 * Answer the surface of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface
	 * @category accessing
	 */
	public JunSurface surface() {
		JunSurface s = this.basicSurface();
		if (s != null) {
			return s;
		}
		return this.defaultSurface();
	}

	/**
	 * Set a surface of the receiver.
	 * 
	 * @param aSurface jp.co.sra.jun.geometry.abstracts.JunSurface
	 * @category accessing
	 */
	public final void surface_(JunSurface aSurface) {
		surface = aSurface;
	}

	/**
	 * Answer the surface created from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface
	 * @category lisp support
	 */
	public JunSurface surfaceFromList_(JunLispList aList) {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Enumerate the surrounding edges and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void surroundingEdgesDo_(final StBlockClosure aBlock) {
		this.edgesDo_(aBlock);
		this.childLoopsDo_(new StBlockClosure() {
			public Object value_(Object child) {
				((JunLoop) child).edgesDo_(aBlock);
				return null;
			}
		});
	}

	/**
	 * Answer the lisp list which represents the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#toLispListForLoops_edges_vertexes_(jp.co.sra.jun.topology.elements.JunLoop[], jp.co.sra.jun.topology.elements.JunEdge[], jp.co.sra.jun.topology.elements.JunVertex[])
	 * @category lisp support
	 */
	public JunLispList toLispListForLoops_edges_vertexes_(final JunLoop[] loopArray, final JunEdge[] edgeArray, final JunVertex[] vertexArray) {
		int indexOfEdge;
		int indexOfParentLoop = -1;
		int indexOfChildLoop = -1;
		for (indexOfEdge = 0; indexOfEdge < edgeArray.length; indexOfEdge++) {
			if (edgeArray[indexOfEdge] == edge) {
				break;
			}
		}
		for (int i = 0; i < loopArray.length; i++) {
			if (loopArray[i] == parentLoop) {
				indexOfParentLoop = i;
			}
			if (loopArray[i] == childLoop) {
				indexOfChildLoop = i;
			}
		}
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		list.add_(JunLispCons.Head_tail_($("edge"), new Integer(indexOfEdge + 1)));
		list.add_(JunLispCons.Head_tail_($("parentLoop"), (parentLoop == null) ? (Object) this.lispNil() : (Object) new Integer(indexOfParentLoop + 1)));
		list.add_(JunLispCons.Head_tail_($("childLoop"), (childLoop == null) ? (Object) this.lispNil() : (Object) new Integer(indexOfChildLoop + 1)));
		list.add_(JunLispCons.Head_tail_($("surface"), this.lispNil()));
		return list;
	}

	/**
	 * Answer the representative vertex of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunVertex vertex() {
		return this.edge().startVertex();
	}

	/**
	 * Enumerate the Vertex-Edge-Vertex trios and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object vertexEdgeVertexDo_(StBlockClosure aBlock) {
		JunEdge firstEdge = this.firstEdge();
		JunEdge focusedEdge = firstEdge;
		JunVertex firstVertex = this.firstVertex();
		JunVertex focusedVertex = firstVertex;
		while (focusedEdge != null) {
			JunVertex oppositeVertex = focusedEdge.oppositeVertexOfVertex_(focusedVertex);
			Object result = aBlock.value_value_value_(focusedVertex, focusedEdge, oppositeVertex);
			if (result != null) {
				return result;
			}
			focusedVertex = oppositeVertex;
			focusedEdge = focusedEdge.nextEdgeWithVertex_(focusedVertex);
			if ((firstEdge == focusedEdge) && (focusedVertex == firstVertex)) {
				return null;
			}
		}

		return null;
	}

	/**
	 * Answer the vertexes contained in the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex[]
	 * @category accessing
	 */
	public JunVertex[] vertexes() {
		final Vector vertexes = new Vector(this.numberOfVertexes());
		this.vertexesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				vertexes.addElement(anObject);
				return null;
			}
		});

		JunVertex[] array = new JunVertex[vertexes.size()];
		vertexes.copyInto(array);
		return array;
	}

	/**
	 * Enumerate the vertexes and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object vertexesDo_(final StBlockClosure aBlock) {
		return this.vertexEdgeVertexDo_(new StBlockClosure() {
			public Object value_value_value_(Object v1, Object e, Object v2) {
				return aBlock.value_(v1);
			}
		});
	}

	/**
	 * Answer the default surface of JunLoop.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunSurface
	 * @category private
	 */
	protected JunSurface defaultSurface() {
		JunVertex[] vertexes = this.vertexes();
		Jun3dPoint[] points = new Jun3dPoint[vertexes.length];
		for (int index = 0; index < vertexes.length; index++) {
			points[index] = vertexes[index].point();
		}
		return Jun3dPolygon.Vertexes_(points);
	}

	/**
	 * Detouch the receiver from the parent loop.
	 * 
	 * @category private
	 */
	private void detouchFromParentChildReference() {
		if (this.isChildLoop()) {
			this.parentLoop().removeChildLoop_(this);
		}
	}

	/**
	 * Answer my volume.
	 * 
	 * @return double
	 * @category private
	 */
	protected double volume() {
		JunVertex[] vertexes = this.vertexes();
		Jun3dPoint[] points = new Jun3dPoint[vertexes.length];
		for (int i = 0; i < vertexes.length; i++) {
			points[i] = vertexes[i].point();
		}
		double area = Jun2dPolygon.SignedAreaForPoints_(points);
		double height = 0;
		for (int i = 0; i < points.length; i++) {
			height += points[i].z();
		}
		return (area * height) / points.length;
	}
}
