package jp.co.sra.jun.topology.elements;

import java.awt.*;
import java.io.*;
import java.util.Vector;
import jp.co.sra.jun.geometry.basic.*;
import jp.co.sra.jun.goodies.lisp.*;
import jp.co.sra.jun.opengl.objects.*;
import jp.co.sra.jun.opengl.support.*;
import jp.co.sra.jun.topology.abstracts.*;
import jp.co.sra.smalltalk.*;

/**
 * JunVertex class
 * 
 *  @author    ASTI Shanghai
 *  @created   UNKNOWN
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVertex.java,v 8.10 2008/02/20 06:33:01 nisinaka Exp $
 */
public class JunVertex extends JunTopologicalElement {
	/** The edge in which the JunVertex is contained. */
	protected JunEdge edge = null;

	/** The Jun3dPoint which corresponds to the JunVertex. */
	protected Jun3dPoint point = null;

	/**
	 * Create a new JunVertex and initialize it with the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 */
	public static final JunVertex Point_(Jun3dPoint aPoint) {
		JunVertex aVertex = new JunVertex();
		aVertex.point_(aPoint);

		return aVertex;
	}

	/**
	 * Convert the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		return JunOpenGL3dVertex.Point_(this.point());
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * 
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 */
	public JunTopologicalElementProxy asProxyIn_(JunBody aBody) {
		return this.asProxyIn_advise_(aBody, null);
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * @param aTopologicalElementProxy
	 *        jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * 
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 */
	public JunTopologicalElementProxy asProxyIn_advise_(JunBody aBody, JunTopologicalElementProxy aTopologicalElementProxy) {
		JunVertexProxy proxy;
		proxy = (JunVertexProxy) aBody.proxyForVertex_(this);

		if (proxy != null) {
			return proxy;
		} else {
			return aBody.addVertex_as_(this, (JunVertexProxy) aTopologicalElementProxy);
		}
	}

	/**
	 * Convert the corresponding geometry as a wireframed JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 */
	public JunOpenGL3dObject asWireframedOpenGL3dObject() {
		return JunOpenGL3dVertex.Point_(this.point());
	}

	/**
	 * Answer the common loop of the receiver specified with the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 */
	public JunLoop commonLoopWithVertex_(final JunVertex aVertex) {
		return (JunLoop) this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				final JunLoop myLoop = (JunLoop) anObject;

				return aVertex.loopsDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						JunLoop hisLoop = (JunLoop) anObject;

						if (myLoop == hisLoop) {
							return myLoop;
						}

						return null;
					}
				});
			}
		});
	}

	/**
	 * Detect an edge which meets the condition specified with aBlock. If not
	 * found, evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 */
	public JunEdge detectEdge_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object myEdge) {
				if (((Boolean) aBlock.value_(myEdge)).booleanValue() == true) {
					return myEdge;
				}

				return null;
			}
		});

		return (JunEdge) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Detect a loop which meets the condition specified with aBlock. If not
	 * found, evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 */
	public JunLoop detectLoop_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.loopsDo_(new StBlockClosure() {
			public Object value_(Object myLoop) {
				if (((Boolean) aBlock.value_(myLoop)).booleanValue() == true) {
					return myLoop;
				}

				return null;
			}
		});

		return (JunLoop) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Answer the edge of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 */
	public JunEdge edge() {
		return edge;
	}

	/**
	 * Set an edge of the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 */
	public void edge_(JunEdge anEdge) {
		edge = anEdge;
	}

	/**
	 * Enumerate the edge and loop pairs and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object edgeAndLoopsDo_(StBlockClosure aBlock) {
		JunEdge currentEdge = this.edge();
		JunLoop currentLoop = currentEdge.loopToVertex_(this);
		Object result = aBlock.value_value_(currentEdge, currentLoop);

		if (result != null) {
			return result;
		}

		JunEdge nextEdge = currentEdge.nextEdgeWithVertex_(this);

		while (nextEdge != this.edge()) {
			currentEdge = nextEdge;
			currentLoop = currentEdge.loopToVertex_(this);

			if ((result = aBlock.value_value_(currentEdge, currentLoop)) != null) {
				return result;
			}

			nextEdge = currentEdge.nextEdgeWithVertex_(this);
		}

		return null;
	}

	/**
	 * Answer the edge on the loop.  If none found, evaluate the error block.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 */
	public JunEdge edgeOnLoop_ifAbsent_(final JunLoop aLoop, StBlockClosure errorBlock) {
		StBlockClosure b = new StBlockClosure() {
			public Object value_(Object myEdge) {
				return new Boolean(((JunVertex) myEdge).hasLoop_(aLoop));
			}
		};

		return this.detectEdge_ifNone_(b, errorBlock);
	}

	/**
	 * Answer the edges.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge[]
	 */
	public JunEdge[] edges() {
		final Vector edges = new Vector(this.numberOfEdges());
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				edges.addElement(anObject);

				return null;
			}
		});

		JunEdge[] array = new JunEdge[edges.size()];
		edges.copyInto(array);

		return array;
	}

	/**
	 * Enumerate the edges and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object edgesDo_(final StBlockClosure aBlock) {
		StBlockClosure b = new StBlockClosure() {
			public Object value_value_(Object e, Object l) {
				return aBlock.value_(e);
			}
		};

		return this.edgeAndLoopsDo_(b);
	}

	/**
	 * Answer the edge from the receiver to aVertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 */
	public JunEdge edgeToVertex_(final JunVertex aVertex) {
		if (this.isKilled()) {
			return null;
		}

		final JunVertex self = this;
		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunEdge myEdge = (JunEdge) anObject;

				if (myEdge.oppositeVertexOfVertex_(self) == aVertex) {
					return myEdge;
				}

				return null;
			}
		};

		return (JunEdge) this.edgesDo_(aBlock);
	}

	/**
	 * Forget every instance variables.
	 */
	public void forget() {
		edge = null;
		point = null;
	}

	/**
	 * Initialize the receiver with the lisp list.
	 * 
	 * @param aLispList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 */
	public void fromLispList_forLoops_edges_vertexes_(JunLispCons aLispList, JunLoop[] loopArray, final JunEdge[] edgeArray, JunVertex[] vertexArray) {
		((JunLispCons) aLispList.tail()).do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLispCons tuple = (JunLispCons) anObject;

				if (tuple.head() == $("edge")) {
					edge = (JunEdge) edgeArray[((Integer) tuple.tail()).intValue() - 1];
				}

				if (tuple.head() == $("point")) {
					point = (Jun3dPoint) tuple.tail();
				}

				return null;
			}
		});
	}

	/**
	 * Answer true if there exists an edge from the receiver to aVertex,
	 * otherwise false.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return boolean
	 */
	public boolean hasEdgeToVertex_(JunVertex aVertex) {
		return (this.edgeToVertex_(aVertex) != null);
	}

	/**
	 * Answer true if there exists a loop which contains the receiver,
	 * otherwise false.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * 
	 * @return boolean
	 */
	public boolean hasLoop_(final JunLoop aLoop) {
		if (this.isKilled()) {
			return false;
		}

		Object result = this.loopsDo_(new StBlockClosure() {
			public Object value_(Object myLoop) {
				if (myLoop == aLoop) {
					return Boolean.TRUE;
				}

				return null;
			}
		});

		return (result != null);
	}

	/**
	 * Answer true if there exists the edge which contains the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * 
	 * @return boolean
	 */
	public boolean includesEdge_(final JunEdge anEdge) {
		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object myEdge) {
				if (myEdge == anEdge) {
					return Boolean.TRUE;
				}

				return null;
			}
		});

		return (result != null);
	}

	/**
	 * Answer true if the receiver is killed, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean isKilled() {
		return (edge == null);
	}

	/**
	 * Answer the loop.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 */
	public JunLoop loop() {
		return this.edge().loop();
	}

	/**
	 * Answer the loops related to the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop[]
	 */
	public JunLoop[] loops() {
		final Vector loops = new Vector(this.numberOfLoops());
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				loops.addElement(anObject);

				return null;
			}
		});

		JunLoop[] array = new JunLoop[loops.size()];
		loops.copyInto(array);

		return array;
	}

	/**
	 * Enumerate the loops and evaluate aBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object loopsDo_(final StBlockClosure aBlock) {
		return this.edgeAndLoopsDo_(new StBlockClosure() {
			public Object value_value_(Object e, Object l) {
				return aBlock.value_(l);
			}
		});
	}

	/**
	 * Answer the loop from me to the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 */
	public JunLoop loopToVertex_(JunVertex aVertex) {
		JunEdge theEdge = this.edgeToVertex_(aVertex);

		if (theEdge == null) {
			return null;
		}

		return theEdge.loopToVertex_(aVertex);
	}

	/**
	 * Answer the number of the edges related to the receiver.
	 * 
	 * @return int
	 */
	public int numberOfEdges() {
		final StValueHolder numberOfEdges = new StValueHolder(0);
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object e) {
				numberOfEdges.value_(numberOfEdges._intValue() + 1);

				return null;
			}
		});

		return numberOfEdges._intValue();
	}

	/**
	 * Answer the number of the loops related to the receiver.
	 * 
	 * @return int
	 */
	public int numberOfLoops() {
		final StValueHolder numberOfLoops = new StValueHolder(0);
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				numberOfLoops.value_(numberOfLoops._intValue() + 1);

				return null;
			}
		});

		return numberOfLoops._intValue();
	}

	/**
	 * Answer the Jun3dPoint which corresponds to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint point() {
		return point;
	}

	/**
	 * Set the Jun3dPoint as a corresponding geometry.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public void point_(Jun3dPoint aPoint) {
		point = aPoint;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * 
	 * @throws IOException DOCUMENT ME!
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Vertex (");
		this.point.printOn_(aWriter);
		aWriter.write(")");
	}

	/**
	 * Render the point of the receiver on the rendering context.
	 * 
	 * @param aRenderingContext
	 *        jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		JunOpenGL3dVertex vertex;
		vertex = JunOpenGL3dVertex.Point_paint_(this.point(), Color.black);
		vertex.size_(5);
		vertex.renderOn_(aRenderingContext);
	}

	/**
	 * Answer the lisp list which represents the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 */
	public JunLispList toLispListForLoops_edges_vertexes_(JunLoop[] loopArray, JunEdge[] edgeArray, JunVertex[] vertexArray) {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		int index = -1;

		for (int i = 0; i < edgeArray.length; i++) {
			if (edgeArray[i] == edge) {
				index = i;

				break;
			}
		}

		list.add_(JunLispCons.Head_tail_($("edge"), new Integer(index + 1)));
		list.add_(JunLispCons.Head_tail_($("point"), point));

		return list;
	}
}
