package jp.co.sra.jun.topology.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.geom.Arc2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeSet;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.display.JunDisplayModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.tables.JunAdjacencyMatrix;
import jp.co.sra.jun.goodies.tables.JunAttributeTable;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.topology.grapher.JunGrapher;

/**
 * JunElementalGraph class
 * 
 *  @author    nisinaka
 *  @created   2006/04/05 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunElementalGraph.java,v 8.12 2008/02/20 06:33:02 nisinaka Exp $
 */
public class JunElementalGraph extends JunElementalStuff {

	protected static JunAttributeTable DefaultAttributes = new JunAttributeTable();
	protected static StSymbol[] AttributeSymbolsToReset = new StSymbol[] {
			$("labelString"),
			$("selectionBorderColor"),
			$("selectionBorderWidth"),
			$("arrangeFormat"),
			$("arrangeForestInterval"),
			$("arrangeForestMargin"),
			$("arrangeConcentricRadius"),
			$("arrangeConcentricSpiral"),
			$("arrangeConcentricGradation"),
			$("arrangeAttributeSymbol") };

	protected ArrayList elementalNodes;
	protected JunAdjacencyMatrix adjacencyMatrix;
	protected ArrayList selectedNodes;
	protected JunElementalNode[] cachedNodes;
	protected HashMap cachedNodeTable;
	protected JunElementalArc[] cachedArcs;
	protected HashMap cachedArcTable;
	protected JunElementalNode[] cachedRoots;
	protected HashMap cachedNodeToArcs;
	protected StRectangle boundingBox;

	/**
	 * Create a new instance of the default graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Defaults
	 */
	public static JunElementalGraph DefaultNewGraph() {
		return new JunElementalGraph();
	}

	/**
	 * Create a new instance of the default node.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category Defaults
	 */
	public static JunElementalNode DefaultNewNode(String labelString) {
		return new JunElementalNode(labelString);
	}

	/**
	 * Create a new instance of the default node.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Defaults
	 */
	public static JunElementalNode DefaultNewNode(JunLispList aList) {
		return new JunElementalNode(aList);
	}

	/**
	 * Create a new simple graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Examples
	 */
	public static JunElementalGraph ExampleSimple() {
		JunElementalGraph aGraph = JunElementalGraph.DefaultNewGraph();
		JunElementalNode nodeABC = JunElementalGraph.DefaultNewNode("abc");
		JunElementalNode nodeDEF = JunElementalGraph.DefaultNewNode("def");
		JunElementalNode nodeGHI = JunElementalGraph.DefaultNewNode("ghi");
		JunElementalNode nodeJKL = JunElementalGraph.DefaultNewNode("jkl");
		JunElementalNode nodeMNO = JunElementalGraph.DefaultNewNode("mno");
		JunElementalNode nodePQR = JunElementalGraph.DefaultNewNode("pqr");
		JunElementalNode nodeSTU = JunElementalGraph.DefaultNewNode("stu");
		JunElementalNode nodeVWX = JunElementalGraph.DefaultNewNode("vwx");
		JunElementalNode nodeYZ0 = JunElementalGraph.DefaultNewNode("yz0");
		JunElementalNode node123 = JunElementalGraph.DefaultNewNode("123");
		JunElementalNode node456 = JunElementalGraph.DefaultNewNode("456");
		JunElementalNode node789 = JunElementalGraph.DefaultNewNode("789");

		aGraph.connect_with_(nodeABC, nodeDEF);
		aGraph.connect_with_cost_(nodeABC, nodeGHI, 3);
		aGraph.connect_with_(nodeABC, nodeJKL);
		aGraph.connect_with_(nodeDEF, nodeMNO);
		aGraph.connect_with_(nodeGHI, nodePQR);
		aGraph.connect_with_(nodeGHI, nodeSTU);
		aGraph.connect_with_(nodeJKL, nodeVWX);
		aGraph.connect_with_cost_(nodeJKL, nodeYZ0, 2);
		aGraph.connect_with_cost_(nodeSTU, node123, 4);
		aGraph.connect_with_(nodeSTU, node456);
		aGraph.add_(node789);

		return aGraph;
	}

	/**
	 * Create a new cycle graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Examples
	 */
	public static JunElementalGraph ExampleCycle() {
		JunElementalGraph aGraph = JunElementalGraph.DefaultNewGraph();
		JunElementalNode nodeABC = JunElementalGraph.DefaultNewNode("abc");
		JunElementalNode nodeDEF = JunElementalGraph.DefaultNewNode("def");
		JunElementalNode nodeGHI = JunElementalGraph.DefaultNewNode("ghi");
		JunElementalNode nodeJKL = JunElementalGraph.DefaultNewNode("jkl");
		JunElementalNode nodeMNO = JunElementalGraph.DefaultNewNode("mno");
		JunElementalNode nodePQR = JunElementalGraph.DefaultNewNode("pqr");
		JunElementalNode nodeSTU = JunElementalGraph.DefaultNewNode("stu");
		JunElementalNode nodeVWX = JunElementalGraph.DefaultNewNode("vwx");
		JunElementalNode nodeYZ0 = JunElementalGraph.DefaultNewNode("yz0");
		JunElementalNode node123 = JunElementalGraph.DefaultNewNode("123");
		JunElementalNode node456 = JunElementalGraph.DefaultNewNode("456");
		JunElementalNode node789 = JunElementalGraph.DefaultNewNode("789");

		aGraph.connect_with_(nodeABC, nodeDEF);
		aGraph.connect_with_cost_(nodeABC, nodeGHI, 3);
		aGraph.connect_with_(nodeABC, nodeJKL);
		aGraph.connect_with_(nodeDEF, nodeMNO);
		aGraph.connect_with_(nodeGHI, nodePQR);
		aGraph.connect_with_(nodeGHI, nodeSTU);
		aGraph.connect_with_(nodeJKL, nodeVWX);
		aGraph.connect_with_cost_(nodeJKL, nodeYZ0, 2);
		aGraph.connect_with_cost_(nodeSTU, node123, 4);
		aGraph.connect_with_(nodeSTU, node456);
		aGraph.connect_with_(node789, nodeABC);
		aGraph.connect_with_(node123, node789);

		return aGraph;
	}

	/**
	 * Create a new tree elemental graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Examples
	 */
	public static JunElementalGraph ExampleTree() {
		final HashMap aMap = new HashMap();
		final StBlockClosure aClassNodeBlock = new StBlockClosure() {
			public Object value_(Object aClass) {
				JunElementalNode aNode = (JunElementalNode) aMap.get(aClass);
				if (aNode == null) {
					aNode = JunElementalGraph.DefaultNewNode(((Class) aClass).getName());
					aMap.put(aClass, aNode);
				}
				return aNode;
			}
		};

		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_value_(Object o1, Object o2) {
				JunElementalGraph aGraph = (JunElementalGraph) o1;
				Class aClass = (Class) o2;
				Class superClass = aClass.getSuperclass();
				if (superClass != null) {
					aGraph.connect_with_((JunElementalNode) aClassNodeBlock.value_(superClass), (JunElementalNode) aClassNodeBlock.value_(aClass));
					this.value_value_(aGraph, superClass);
				}
				return null;
			}
		};

		JunElementalGraph aGraph = JunElementalGraph.DefaultNewGraph();
		aBlock.value_value_(aGraph, java.util.ArrayList.class);
		aBlock.value_value_(aGraph, java.util.HashMap.class);
		aBlock.value_value_(aGraph, java.util.HashSet.class);
		aBlock.value_value_(aGraph, java.util.Hashtable.class);
		aBlock.value_value_(aGraph, java.util.LinkedHashSet.class);
		aBlock.value_value_(aGraph, java.util.LinkedHashMap.class);
		aBlock.value_value_(aGraph, java.util.LinkedList.class);
		aBlock.value_value_(aGraph, java.util.Stack.class);
		aBlock.value_value_(aGraph, java.util.TreeMap.class);
		aBlock.value_value_(aGraph, java.util.TreeSet.class);
		aBlock.value_value_(aGraph, java.util.Vector.class);

		return aGraph;
	}

	/**
	 * Create a new forest elemental graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Examples
	 */
	public static JunElementalGraph ExampleForest() {
		final HashMap aMap = new HashMap();
		final StBlockClosure aClassNodeBlock = new StBlockClosure() {
			public Object value_(Object aClass) {
				JunElementalNode aNode = (JunElementalNode) aMap.get(aClass);
				if (aNode == null) {
					aNode = JunElementalGraph.DefaultNewNode(((Class) aClass).getName());
					aMap.put(aClass, aNode);
				}
				return aNode;
			}
		};

		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_value_(Object o1, Object o2) {
				JunElementalGraph aGraph = (JunElementalGraph) o1;
				Class aClass = (Class) o2;
				Class superClass = aClass.getSuperclass();
				if (superClass != java.lang.Object.class) {
					aGraph.connect_with_((JunElementalNode) aClassNodeBlock.value_(superClass), (JunElementalNode) aClassNodeBlock.value_(aClass));
					this.value_value_(aGraph, superClass);
				}
				return null;
			}
		};

		JunElementalGraph aGraph = JunElementalGraph.DefaultNewGraph();
		aBlock.value_value_(aGraph, java.util.ArrayList.class);
		aBlock.value_value_(aGraph, java.util.HashMap.class);
		aBlock.value_value_(aGraph, java.util.HashSet.class);
		aBlock.value_value_(aGraph, java.util.Hashtable.class);
		aBlock.value_value_(aGraph, java.util.LinkedHashSet.class);
		aBlock.value_value_(aGraph, java.util.LinkedHashMap.class);
		aBlock.value_value_(aGraph, java.util.LinkedList.class);
		aBlock.value_value_(aGraph, java.util.Stack.class);
		aBlock.value_value_(aGraph, java.util.TreeMap.class);
		aBlock.value_value_(aGraph, java.util.TreeSet.class);
		aBlock.value_value_(aGraph, java.util.Vector.class);

		aBlock.value_value_(aGraph, java.io.BufferedInputStream.class);
		aBlock.value_value_(aGraph, java.io.ByteArrayInputStream.class);
		aBlock.value_value_(aGraph, java.io.DataInputStream.class);
		aBlock.value_value_(aGraph, java.io.FileInputStream.class);
		aBlock.value_value_(aGraph, java.io.FilterInputStream.class);
		aBlock.value_value_(aGraph, java.io.ObjectInputStream.class);
		aBlock.value_value_(aGraph, java.io.PipedInputStream.class);
		aBlock.value_value_(aGraph, java.io.PushbackInputStream.class);
		aBlock.value_value_(aGraph, java.io.SequenceInputStream.class);

		aBlock.value_value_(aGraph, java.io.BufferedOutputStream.class);
		aBlock.value_value_(aGraph, java.io.ByteArrayOutputStream.class);
		aBlock.value_value_(aGraph, java.io.DataOutputStream.class);
		aBlock.value_value_(aGraph, java.io.FileOutputStream.class);
		aBlock.value_value_(aGraph, java.io.FilterOutputStream.class);
		aBlock.value_value_(aGraph, java.io.ObjectOutputStream.class);
		aBlock.value_value_(aGraph, java.io.PipedOutputStream.class);

		aBlock.value_value_(aGraph, java.io.BufferedReader.class);
		aBlock.value_value_(aGraph, java.io.CharArrayReader.class);
		aBlock.value_value_(aGraph, java.io.FileReader.class);
		aBlock.value_value_(aGraph, java.io.InputStreamReader.class);
		aBlock.value_value_(aGraph, java.io.LineNumberReader.class);
		aBlock.value_value_(aGraph, java.io.PipedReader.class);
		aBlock.value_value_(aGraph, java.io.PushbackReader.class);
		aBlock.value_value_(aGraph, java.io.StringReader.class);

		aBlock.value_value_(aGraph, java.io.BufferedWriter.class);
		aBlock.value_value_(aGraph, java.io.CharArrayWriter.class);
		aBlock.value_value_(aGraph, java.io.FileWriter.class);
		aBlock.value_value_(aGraph, java.io.OutputStreamWriter.class);
		aBlock.value_value_(aGraph, java.io.PipedWriter.class);
		aBlock.value_value_(aGraph, java.io.PrintWriter.class);
		aBlock.value_value_(aGraph, java.io.StringWriter.class);

		return aGraph;
	}

	/**
	 * Create a new directory elemental graph.
	 * 
	 * @param diveLevel int
	 * @param withFiles boolean
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Examples
	 */
	public static JunElementalGraph ExampleDirectory_withFiles_(final int diveLevel, final boolean withFiles) {
		File theDirectory = JunFileRequesterDialog.RequestDirectory($String("Select a directory."));
		if (theDirectory == null) {
			return null;
		}
		if (theDirectory.isDirectory() == false) {
			return null;
		}

		final HashMap aMap = new HashMap();
		final StBlockClosure aFileNodeBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				File aFile = (File) anObject;
				JunElementalNode aNode = (JunElementalNode) aMap.get(aFile);
				if (aNode == null) {
					String aString = aFile.getName();
					if (aFile.isDirectory()) {
						aString += '/';
					}
					aNode = JunElementalGraph.DefaultNewNode(aString);
					aMap.put(aFile, aNode);
				}
				return aNode;
			}
		};

		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_value_value_(Object o1, Object o2, Object o3) {
				JunElementalGraph aMatrix = (JunElementalGraph) o1;
				File aDirectory = (File) o2;
				int level = ((Number) o3).intValue();
				if (level < diveLevel) {
					File[] files = aDirectory.listFiles();
					for (int i = 0; i < files.length; i++) {
						if (files[i].isDirectory()) {
							aMatrix.connect_with_((JunElementalNode) aFileNodeBlock.value_(aDirectory), (JunElementalNode) aFileNodeBlock.value_(files[i]));
							this.value_value_value_(aMatrix, files[i], new Integer(level + 1));
						} else if (withFiles) {
							aMatrix.connect_with_((JunElementalNode) aFileNodeBlock.value_(aDirectory), (JunElementalNode) aFileNodeBlock.value_(files[i]));
						}
					}
				}
				return null;
			}
		};

		JunElementalGraph aGraph = JunElementalGraph.DefaultNewGraph();
		aBlock.value_value_value_(aGraph, theDirectory, new Integer(1));

		return aGraph;
	}

	/**
	 * Create a new tokyo elemental graph.
	 * 
	 * 
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Examples
	 */
	public static JunElementalGraph ExampleTokyo() {
		JunElementalGraph aGraph = JunElementalGraph.DefaultNewGraph();

		/*
		 self nodesDo: [:node | 
		 Transcript cr.
		 Transcript show: 'JunElementalNode node', node labelString, ' =  JunElementalGraph.DefaultNewNode($String("', node labelString, '"));'.
		 Transcript cr.
		 Transcript show: 'node', node labelString, '.addAttribute($("longitude"), "' , (node attributeAt: #longitude), '");'.
		 Transcript cr.
		 Transcript show: 'node', node labelString, '.addAttribute($("latitude"), "' , (node attributeAt: #latitude), '");'
		 ]
		 */
		JunElementalNode nodeAkihabara = JunElementalGraph.DefaultNewNode($String("Akihabara"));
		nodeAkihabara.addAttribute_($("longitude"), "E/139/46/34.5");
		nodeAkihabara.addAttribute_($("latitude"), "N/35/41/42.3");
		JunElementalNode nodeAsagaya = JunElementalGraph.DefaultNewNode($String("Asagaya"));
		nodeAsagaya.addAttribute_($("longitude"), "E/139/38/20.1");
		nodeAsagaya.addAttribute_($("latitude"), "N/35/42/5.9");
		JunElementalNode nodeAsakusa = JunElementalGraph.DefaultNewNode($String("Asakusa"));
		nodeAsakusa.addAttribute_($("longitude"), "E/139/48/2.4");
		nodeAsakusa.addAttribute_($("latitude"), "N/35/42/26.5");
		JunElementalNode nodeAsakusabashi = JunElementalGraph.DefaultNewNode($String("Asakusabashi"));
		nodeAsakusabashi.addAttribute_($("longitude"), "E/139/47/16.1");
		nodeAsakusabashi.addAttribute_($("latitude"), "N/35/41/38.9");
		JunElementalNode nodeDaiba = JunElementalGraph.DefaultNewNode($String("Daiba"));
		nodeDaiba.addAttribute_($("longitude"), "E/139/46/29.0");
		nodeDaiba.addAttribute_($("latitude"), "N/35/37/21.7");
		JunElementalNode nodeEbisu = JunElementalGraph.DefaultNewNode($String("Ebisu"));
		nodeEbisu.addAttribute_($("longitude"), "E/139/42/47.6");
		nodeEbisu.addAttribute_($("latitude"), "N/35/38/36.5");
		JunElementalNode nodeGotanda = JunElementalGraph.DefaultNewNode($String("Gotanda"));
		nodeGotanda.addAttribute_($("longitude"), "E/139/43/36.1");
		nodeGotanda.addAttribute_($("latitude"), "N/35/37/23.7");
		JunElementalNode nodeHamamatsucho = JunElementalGraph.DefaultNewNode($String("Hamamatsucho"));
		nodeHamamatsucho.addAttribute_($("longitude"), "E/139/45/37.6");
		nodeHamamatsucho.addAttribute_($("latitude"), "N/35/39/8.3");
		JunElementalNode nodeHarajuku = JunElementalGraph.DefaultNewNode($String("Harajuku"));
		nodeHarajuku.addAttribute_($("longitude"), "E/139/42/21.6");
		nodeHarajuku.addAttribute_($("latitude"), "N/35/40/5.3");
		JunElementalNode nodeHigashiNakano = JunElementalGraph.DefaultNewNode($String("HigashiNakano"));
		nodeHigashiNakano.addAttribute_($("longitude"), "E/139/41/17.5");
		nodeHigashiNakano.addAttribute_($("latitude"), "N/35/42/10.9");
		JunElementalNode nodeHirai = JunElementalGraph.DefaultNewNode($String("Hirai"));
		nodeHirai.addAttribute_($("longitude"), "E/139/50/44.2");
		nodeHirai.addAttribute_($("latitude"), "N/35/42/11.7");
		JunElementalNode nodeIchigaya = JunElementalGraph.DefaultNewNode($String("Ichigaya"));
		nodeIchigaya.addAttribute_($("longitude"), "E/139/44/19.6");
		nodeIchigaya.addAttribute_($("latitude"), "N/35/41/17.1");
		JunElementalNode nodeIidabashi = JunElementalGraph.DefaultNewNode($String("Iidabashi"));
		nodeIidabashi.addAttribute_($("longitude"), "E/139/44/53.1");
		nodeIidabashi.addAttribute_($("latitude"), "N/35/41/54.4");
		JunElementalNode nodeIkebukuro = JunElementalGraph.DefaultNewNode($String("Ikebukuro"));
		nodeIkebukuro.addAttribute_($("longitude"), "E/139/42/52.0");
		nodeIkebukuro.addAttribute_($("latitude"), "N/35/43/37.6");
		JunElementalNode nodeKameido = JunElementalGraph.DefaultNewNode($String("Kameido"));
		nodeKameido.addAttribute_($("longitude"), "E/139/49/46.6");
		nodeKameido.addAttribute_($("latitude"), "N/35/41/38.7");
		JunElementalNode nodeKanda = JunElementalGraph.DefaultNewNode($String("Kanda"));
		nodeKanda.addAttribute_($("longitude"), "E/139/46/26.6");
		nodeKanda.addAttribute_($("latitude"), "N/35/41/18.6");
		JunElementalNode nodeKinshicho = JunElementalGraph.DefaultNewNode($String("Kinshicho"));
		nodeKinshicho.addAttribute_($("longitude"), "E/139/49/2.9");
		nodeKinshicho.addAttribute_($("latitude"), "N/35/41/36.6");
		JunElementalNode nodeKoenji = JunElementalGraph.DefaultNewNode($String("Koenji"));
		nodeKoenji.addAttribute_($("longitude"), "E/139/39/10.0");
		nodeKoenji.addAttribute_($("latitude"), "N/35/42/7.3");
		JunElementalNode nodeKoiwa = JunElementalGraph.DefaultNewNode($String("Koiwa"));
		nodeKoiwa.addAttribute_($("longitude"), "E/139/53/6.3");
		nodeKoiwa.addAttribute_($("latitude"), "N/35/43/48.4");
		JunElementalNode nodeKokkaiGijidouMae = JunElementalGraph.DefaultNewNode($String("KokkaiGijidouMae"));
		nodeKokkaiGijidouMae.addAttribute_($("longitude"), "E/139/44/57.1");
		nodeKokkaiGijidouMae.addAttribute_($("latitude"), "N/35/40/18.4");
		JunElementalNode nodeKomagome = JunElementalGraph.DefaultNewNode($String("Komagome"));
		nodeKomagome.addAttribute_($("longitude"), "E/139/45/4.8");
		nodeKomagome.addAttribute_($("latitude"), "N/35/44/1.1");
		JunElementalNode nodeKourakuen = JunElementalGraph.DefaultNewNode($String("Kourakuen"));
		nodeKourakuen.addAttribute_($("longitude"), "E/139/45/15.5");
		nodeKourakuen.addAttribute_($("latitude"), "N/35/42/14.8");
		JunElementalNode nodeMeguro = JunElementalGraph.DefaultNewNode($String("Meguro"));
		nodeMeguro.addAttribute_($("longitude"), "E/139/43/8.2");
		nodeMeguro.addAttribute_($("latitude"), "N/35/37/49.9");
		JunElementalNode nodeMejiro = JunElementalGraph.DefaultNewNode($String("Mejiro"));
		nodeMejiro.addAttribute_($("longitude"), "E/139/42/34.2");
		nodeMejiro.addAttribute_($("latitude"), "N/35/43/1.8");
		JunElementalNode nodeNakano = JunElementalGraph.DefaultNewNode($String("Nakano"));
		nodeNakano.addAttribute_($("longitude"), "E/139/40/8.4");
		nodeNakano.addAttribute_($("latitude"), "N/35/42/9.0");
		JunElementalNode nodeNippori = JunElementalGraph.DefaultNewNode($String("Nippori"));
		nodeNippori.addAttribute_($("longitude"), "E/139/46/27.8");
		nodeNippori.addAttribute_($("latitude"), "N/35/43/26.7");
		JunElementalNode nodeNishiNippori = JunElementalGraph.DefaultNewNode($String("NishiNippori"));
		nodeNishiNippori.addAttribute_($("longitude"), "E/139/46/12.1");
		nodeNishiNippori.addAttribute_($("latitude"), "N/35/43/43.6");
		JunElementalNode nodeNishiOgikubo = JunElementalGraph.DefaultNewNode($String("NishiOgikubo"));
		nodeNishiOgikubo.addAttribute_($("longitude"), "E/139/36/9.3");
		nodeNishiOgikubo.addAttribute_($("latitude"), "N/35/42/1.9");
		JunElementalNode nodeOchanomizu = JunElementalGraph.DefaultNewNode($String("Ochanomizu"));
		nodeOchanomizu.addAttribute_($("longitude"), "E/139/46/5.6");
		nodeOchanomizu.addAttribute_($("latitude"), "N/35/41/46.9");
		JunElementalNode nodeOgikubo = JunElementalGraph.DefaultNewNode($String("Ogikubo"));
		nodeOgikubo.addAttribute_($("longitude"), "E/139/37/24.5");
		nodeOgikubo.addAttribute_($("latitude"), "N/35/42/4.3");
		JunElementalNode nodeOkachimachi = JunElementalGraph.DefaultNewNode($String("Okachimachi"));
		nodeOkachimachi.addAttribute_($("longitude"), "E/139/46/40.2");
		nodeOkachimachi.addAttribute_($("latitude"), "N/35/42/13.5");
		JunElementalNode nodeOkubo = JunElementalGraph.DefaultNewNode($String("Okubo"));
		nodeOkubo.addAttribute_($("longitude"), "E/139/42/2.7");
		nodeOkubo.addAttribute_($("latitude"), "N/35/41/50.0");
		JunElementalNode nodeOsaki = JunElementalGraph.DefaultNewNode($String("Osaki"));
		nodeOsaki.addAttribute_($("longitude"), "E/139/43/53.0");
		nodeOsaki.addAttribute_($("latitude"), "N/35/37/0.2");
		JunElementalNode nodeOtsuka = JunElementalGraph.DefaultNewNode($String("Otsuka"));
		nodeOtsuka.addAttribute_($("longitude"), "E/139/43/52.4");
		nodeOtsuka.addAttribute_($("latitude"), "N/35/43/42.7");
		JunElementalNode nodeRyogoku = JunElementalGraph.DefaultNewNode($String("Ryogoku"));
		nodeRyogoku.addAttribute_($("longitude"), "E/139/47/47.9");
		nodeRyogoku.addAttribute_($("latitude"), "N/35/41/32.9");
		JunElementalNode nodeSendagaya = JunElementalGraph.DefaultNewNode($String("Sendagaya"));
		nodeSendagaya.addAttribute_($("longitude"), "E/139/42/53.4");
		nodeSendagaya.addAttribute_($("latitude"), "N/35/40/40.8");
		JunElementalNode nodeShibuya = JunElementalGraph.DefaultNewNode($String("Shibuya"));
		nodeShibuya.addAttribute_($("longitude"), "E/139/42/16.8");
		nodeShibuya.addAttribute_($("latitude"), "N/35/39/20.4");
		JunElementalNode nodeShimokitazawa = JunElementalGraph.DefaultNewNode($String("Shimokitazawa"));
		nodeShimokitazawa.addAttribute_($("longitude"), "E/139/40/11.0");
		nodeShimokitazawa.addAttribute_($("latitude"), "N/35/39/30.2");
		JunElementalNode nodeShinagawa = JunElementalGraph.DefaultNewNode($String("Shinagawa"));
		nodeShinagawa.addAttribute_($("longitude"), "E/139/44/31.7");
		nodeShinagawa.addAttribute_($("latitude"), "N/35/37/33.1");
		JunElementalNode nodeShinanomachi = JunElementalGraph.DefaultNewNode($String("Shinanomachi"));
		nodeShinanomachi.addAttribute_($("longitude"), "E/139/43/27.8");
		nodeShinanomachi.addAttribute_($("latitude"), "N/35/40/36.1");
		JunElementalNode nodeShinbashi = JunElementalGraph.DefaultNewNode($String("Shinbashi"));
		nodeShinbashi.addAttribute_($("longitude"), "E/139/45/40.9");
		nodeShinbashi.addAttribute_($("latitude"), "N/35/39/47.2");
		JunElementalNode nodeShinjuku = JunElementalGraph.DefaultNewNode($String("Shinjuku"));
		nodeShinjuku.addAttribute_($("longitude"), "E/139/42/14.1");
		nodeShinjuku.addAttribute_($("latitude"), "N/35/41/11.2");
		JunElementalNode nodeShinKoiwa = JunElementalGraph.DefaultNewNode($String("ShinKoiwa"));
		nodeShinKoiwa.addAttribute_($("longitude"), "E/139/51/39.8");
		nodeShinKoiwa.addAttribute_($("latitude"), "N/35/42/48.9");
		JunElementalNode nodeShinOkubo = JunElementalGraph.DefaultNewNode($String("ShinOkubo"));
		nodeShinOkubo.addAttribute_($("longitude"), "E/139/42/12.5");
		nodeShinOkubo.addAttribute_($("latitude"), "N/35/41/51.9");
		JunElementalNode nodeSugamo = JunElementalGraph.DefaultNewNode($String("Sugamo"));
		nodeSugamo.addAttribute_($("longitude"), "E/139/44/37.0");
		nodeSugamo.addAttribute_($("latitude"), "N/35/43/49.8");
		JunElementalNode nodeSuidoubashi = JunElementalGraph.DefaultNewNode($String("Suidoubashi"));
		nodeSuidoubashi.addAttribute_($("longitude"), "E/139/45/24.7");
		nodeSuidoubashi.addAttribute_($("latitude"), "N/35/41/55.7");
		JunElementalNode nodeTabata = JunElementalGraph.DefaultNewNode($String("Tabata"));
		nodeTabata.addAttribute_($("longitude"), "E/139/45/53.5");
		nodeTabata.addAttribute_($("latitude"), "N/35/44/3.2");
		JunElementalNode nodeTakadanobaba = JunElementalGraph.DefaultNewNode($String("Takadanobaba"));
		nodeTakadanobaba.addAttribute_($("longitude"), "E/139/42/25.1");
		nodeTakadanobaba.addAttribute_($("latitude"), "N/35/42/34.1");
		JunElementalNode nodeTamachi = JunElementalGraph.DefaultNewNode($String("Tamachi"));
		nodeTamachi.addAttribute_($("longitude"), "E/139/45/3.3");
		nodeTamachi.addAttribute_($("latitude"), "N/35/38/33.0");
		JunElementalNode nodeTokyo = JunElementalGraph.DefaultNewNode($String("Tokyo"));
		nodeTokyo.addAttribute_($("longitude"), "E/139/46/13.9");
		nodeTokyo.addAttribute_($("latitude"), "N/35/40/39.8");
		JunElementalNode nodeToshimaen = JunElementalGraph.DefaultNewNode($String("Toshimaen"));
		nodeToshimaen.addAttribute_($("longitude"), "E/139/39/4.3");
		nodeToshimaen.addAttribute_($("latitude"), "N/35/44/20.0");
		JunElementalNode nodeUeno = JunElementalGraph.DefaultNewNode($String("Ueno"));
		nodeUeno.addAttribute_($("longitude"), "E/139/46/48.0");
		nodeUeno.addAttribute_($("latitude"), "N/35/42/37.4");
		JunElementalNode nodeUguisudani = JunElementalGraph.DefaultNewNode($String("Uguisudani"));
		nodeUguisudani.addAttribute_($("longitude"), "E/139/46/52.8");
		nodeUguisudani.addAttribute_($("latitude"), "N/35/43/5.8");
		JunElementalNode nodeYotsuya = JunElementalGraph.DefaultNewNode($String("Yotsuya"));
		nodeYotsuya.addAttribute_($("longitude"), "E/139/44/1.7");
		nodeYotsuya.addAttribute_($("latitude"), "N/35/40/57.4");
		JunElementalNode nodeYotsuyaSanchome = JunElementalGraph.DefaultNewNode($String("YotsuyaSanchome"));
		nodeYotsuyaSanchome.addAttribute_($("longitude"), "E/139/43/25.4");
		nodeYotsuyaSanchome.addAttribute_($("latitude"), "N/35/41/4.9");
		JunElementalNode nodeYoyogi = JunElementalGraph.DefaultNewNode($String("Yoyogi"));
		nodeYoyogi.addAttribute_($("longitude"), "E/139/42/19.4");
		nodeYoyogi.addAttribute_($("latitude"), "N/35/40/50.0");
		JunElementalNode nodeYurakucho = JunElementalGraph.DefaultNewNode($String("Yurakucho"));
		nodeYurakucho.addAttribute_($("longitude"), "E/139/45/58.7");
		nodeYurakucho.addAttribute_($("latitude"), "N/35/40/18.0");

		/*
		 self arcsDo: [:arc |
		 Transcript cr.
		 Transcript show: 'aGraph.connect_with_key_value_(', arc firstNode labelString, ', ', arc lastNode labelString, ', $("distance"), new Float(', (arc arcAttributes at: #distance) printString, '));'.
		 ]
		 */
		aGraph.connect_with_key_value_(nodeTokyo, nodeKanda, $("distance"), new Float(1.3));
		aGraph.connect_with_key_value_(nodeKanda, nodeAkihabara, $("distance"), new Float(0.7));
		aGraph.connect_with_key_value_(nodeAkihabara, nodeOkachimachi, $("distance"), new Float(1.0));
		aGraph.connect_with_key_value_(nodeOkachimachi, nodeUeno, $("distance"), new Float(0.6));
		aGraph.connect_with_key_value_(nodeUeno, nodeUguisudani, $("distance"), new Float(1.1));
		aGraph.connect_with_key_value_(nodeUguisudani, nodeNippori, $("distance"), new Float(1.1));
		aGraph.connect_with_key_value_(nodeNippori, nodeNishiNippori, $("distance"), new Float(0.5));
		aGraph.connect_with_key_value_(nodeNishiNippori, nodeTabata, $("distance"), new Float(0.8));
		aGraph.connect_with_key_value_(nodeTabata, nodeKomagome, $("distance"), new Float(1.6));
		aGraph.connect_with_key_value_(nodeKomagome, nodeSugamo, $("distance"), new Float(0.7));
		aGraph.connect_with_key_value_(nodeSugamo, nodeOtsuka, $("distance"), new Float(1.1));
		aGraph.connect_with_key_value_(nodeOtsuka, nodeIkebukuro, $("distance"), new Float(1.8));
		aGraph.connect_with_key_value_(nodeIkebukuro, nodeMejiro, $("distance"), new Float(1.2));
		aGraph.connect_with_key_value_(nodeMejiro, nodeTakadanobaba, $("distance"), new Float(0.9));
		aGraph.connect_with_key_value_(nodeTakadanobaba, nodeShinOkubo, $("distance"), new Float(1.4));
		aGraph.connect_with_key_value_(nodeShinOkubo, nodeShinjuku, $("distance"), new Float(1.3));
		aGraph.connect_with_key_value_(nodeShinjuku, nodeYoyogi, $("distance"), new Float(0.7));
		aGraph.connect_with_key_value_(nodeYoyogi, nodeHarajuku, $("distance"), new Float(1.5));
		aGraph.connect_with_key_value_(nodeHarajuku, nodeShibuya, $("distance"), new Float(1.2));
		aGraph.connect_with_key_value_(nodeShibuya, nodeEbisu, $("distance"), new Float(1.6));
		aGraph.connect_with_key_value_(nodeEbisu, nodeMeguro, $("distance"), new Float(1.5));
		aGraph.connect_with_key_value_(nodeMeguro, nodeGotanda, $("distance"), new Float(1.2));
		aGraph.connect_with_key_value_(nodeGotanda, nodeOsaki, $("distance"), new Float(0.9));
		aGraph.connect_with_key_value_(nodeOsaki, nodeShinagawa, $("distance"), new Float(2.0));
		aGraph.connect_with_key_value_(nodeShinagawa, nodeTamachi, $("distance"), new Float(2.2));
		aGraph.connect_with_key_value_(nodeTamachi, nodeHamamatsucho, $("distance"), new Float(1.5));
		aGraph.connect_with_key_value_(nodeHamamatsucho, nodeShinbashi, $("distance"), new Float(1.2));
		aGraph.connect_with_key_value_(nodeShinbashi, nodeYurakucho, $("distance"), new Float(1.1));
		aGraph.connect_with_key_value_(nodeYurakucho, nodeTokyo, $("distance"), new Float(0.8));

		aGraph.connect_with_key_value_(nodeNishiOgikubo, nodeOgikubo, $("distance"), new Float(1.9));
		aGraph.connect_with_key_value_(nodeOgikubo, nodeAsagaya, $("distance"), new Float(1.4));
		aGraph.connect_with_key_value_(nodeAsagaya, nodeKoenji, $("distance"), new Float(1.2));
		aGraph.connect_with_key_value_(nodeKoenji, nodeNakano, $("distance"), new Float(1.4));
		aGraph.connect_with_key_value_(nodeNakano, nodeHigashiNakano, $("distance"), new Float(1.9));
		aGraph.connect_with_key_value_(nodeHigashiNakano, nodeOkubo, $("distance"), new Float(1.1));
		aGraph.connect_with_key_value_(nodeOkubo, nodeShinjuku, $("distance"), new Float(1.4));
		aGraph.connect_with_key_value_(nodeYoyogi, nodeSendagaya, $("distance"), new Float(1.0));
		aGraph.connect_with_key_value_(nodeSendagaya, nodeShinanomachi, $("distance"), new Float(0.7));
		aGraph.connect_with_key_value_(nodeShinanomachi, nodeYotsuya, $("distance"), new Float(1.3));
		aGraph.connect_with_key_value_(nodeYotsuya, nodeIchigaya, $("distance"), new Float(0.8));
		aGraph.connect_with_key_value_(nodeIchigaya, nodeIidabashi, $("distance"), new Float(1.5));
		aGraph.connect_with_key_value_(nodeIidabashi, nodeSuidoubashi, $("distance"), new Float(0.9));
		aGraph.connect_with_key_value_(nodeSuidoubashi, nodeOchanomizu, $("distance"), new Float(0.8));
		aGraph.connect_with_key_value_(nodeOchanomizu, nodeAkihabara, $("distance"), new Float(0.9));
		aGraph.connect_with_key_value_(nodeAkihabara, nodeAsakusabashi, $("distance"), new Float(1.1));
		aGraph.connect_with_key_value_(nodeAsakusabashi, nodeRyogoku, $("distance"), new Float(0.8));
		aGraph.connect_with_key_value_(nodeRyogoku, nodeKinshicho, $("distance"), new Float(1.5));
		aGraph.connect_with_key_value_(nodeKinshicho, nodeKameido, $("distance"), new Float(1.5));
		aGraph.connect_with_key_value_(nodeKameido, nodeHirai, $("distance"), new Float(1.9));
		aGraph.connect_with_key_value_(nodeHirai, nodeShinKoiwa, $("distance"), new Float(1.8));
		aGraph.connect_with_key_value_(nodeShinKoiwa, nodeKoiwa, $("distance"), new Float(2.8));

		aGraph.connect_with_key_value_(nodeOgikubo, nodeNakano, $("distance"), new Float(4.0));
		aGraph.connect_with_key_value_(nodeNakano, nodeShinjuku, $("distance"), new Float(4.4));
		aGraph.connect_with_key_value_(nodeShinjuku, nodeYotsuya, $("distance"), new Float(3.7));
		aGraph.connect_with_key_value_(nodeYotsuya, nodeOchanomizu, $("distance"), new Float(4.0));
		aGraph.connect_with_key_value_(nodeOchanomizu, nodeKanda, $("distance"), new Float(1.3));

		/*
		 self arcsDo: [:arc |
		 Transcript cr.
		 (arc arcAttributes includesKey: #lineColor) ifTrue: [
		 Transcript show: 'attributes = aGraph.between_and_(node' , arc firstNode labelString , ', node' ,  arc lastNode labelString , ');'.
		 Transcript cr.
		 Transcript show: 'attributes.at_put_($("lineColor"), Color.' , (ColorValue constantNameFor: (arc arcAttributes at: #lineColor)) asString , ');'.
		 Transcript cr.
		 Transcript show: 'attributes.at_put_($("neededTime"), new Integer(' , (arc arcAttributes at: #neededTime) printString , '));'.
		 ]
		 ]
		 */
		JunAttributeTable attributes;
		attributes = aGraph.between_and_(nodeTokyo, nodeKanda);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeKanda, nodeAkihabara);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(1));
		attributes = aGraph.between_and_(nodeAkihabara, nodeOkachimachi);
		attributes.at_put_($("lineColor"), Color.green);
		attributes = aGraph.between_and_(nodeOkachimachi, nodeUeno);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeUeno, nodeUguisudani);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeUguisudani, nodeNippori);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeNippori, nodeNishiNippori);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(1));
		attributes = aGraph.between_and_(nodeNishiNippori, nodeTabata);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeTabata, nodeKomagome);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeKomagome, nodeSugamo);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeSugamo, nodeOtsuka);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(1));
		attributes = aGraph.between_and_(nodeOtsuka, nodeIkebukuro);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(4));
		attributes = aGraph.between_and_(nodeIkebukuro, nodeMejiro);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeMejiro, nodeTakadanobaba);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeTakadanobaba, nodeShinOkubo);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeShinOkubo, nodeShinjuku);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeShinjuku, nodeYoyogi);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeYoyogi, nodeHarajuku);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeHarajuku, nodeShibuya);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(3));
		attributes = aGraph.between_and_(nodeShibuya, nodeEbisu);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeEbisu, nodeMeguro);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(3));
		attributes = aGraph.between_and_(nodeMeguro, nodeGotanda);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeGotanda, nodeOsaki);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(1));
		attributes = aGraph.between_and_(nodeOsaki, nodeShinagawa);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(3));
		attributes = aGraph.between_and_(nodeShinagawa, nodeTamachi);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(3));
		attributes = aGraph.between_and_(nodeTamachi, nodeHamamatsucho);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeHamamatsucho, nodeShinbashi);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeShinbashi, nodeYurakucho);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeYurakucho, nodeTokyo);
		attributes.at_put_($("lineColor"), Color.green);
		attributes.at_put_($("neededTime"), new Integer(2));

		attributes = aGraph.between_and_(nodeNishiOgikubo, nodeOgikubo);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeOgikubo, nodeAsagaya);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeAsagaya, nodeKoenji);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeKoenji, nodeNakano);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeNakano, nodeHigashiNakano);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeHigashiNakano, nodeOkubo);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeOkubo, nodeShinjuku);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeYoyogi, nodeSendagaya);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeSendagaya, nodeShinanomachi);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeShinanomachi, nodeYotsuya);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeYotsuya, nodeIchigaya);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeIchigaya, nodeIidabashi);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeIidabashi, nodeSuidoubashi);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeSuidoubashi, nodeOchanomizu);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeOchanomizu, nodeAkihabara);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeAkihabara, nodeAsakusabashi);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeAsakusabashi, nodeRyogoku);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeRyogoku, nodeKinshicho);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeKinshicho, nodeKameido);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeKameido, nodeHirai);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeHirai, nodeShinKoiwa);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(2));
		attributes = aGraph.between_and_(nodeShinKoiwa, nodeKoiwa);
		attributes.at_put_($("lineColor"), Color.yellow);
		attributes.at_put_($("neededTime"), new Integer(3));

		attributes = aGraph.between_and_(nodeOgikubo, nodeNakano);
		attributes.at_put_($("lineColor"), Color.orange);
		attributes.at_put_($("neededTime"), new Integer(4));
		attributes = aGraph.between_and_(nodeNakano, nodeShinjuku);
		attributes.at_put_($("lineColor"), Color.orange);
		attributes.at_put_($("neededTime"), new Integer(4));
		attributes = aGraph.between_and_(nodeShinjuku, nodeYotsuya);
		attributes.at_put_($("lineColor"), Color.orange);
		attributes.at_put_($("neededTime"), new Integer(5));
		attributes = aGraph.between_and_(nodeYotsuya, nodeOchanomizu);
		attributes.at_put_($("lineColor"), Color.orange);
		attributes.at_put_($("neededTime"), new Integer(5));
		attributes = aGraph.between_and_(nodeOchanomizu, nodeKanda);
		attributes.at_put_($("lineColor"), Color.orange);
		attributes.at_put_($("neededTime"), new Integer(2));

		JunElementalArc[] arcs = aGraph.arcs();
		for (int i = 0; i < arcs.length; i++) {
			arcs[i].firstPointAlignment_($("center"));
			arcs[i].lastPointAlignment_($("center"));
			arcs[i].lineWidth_(3);
		}

		aGraph.add_(nodeYotsuyaSanchome);
		aGraph.add_(nodeKourakuen);
		aGraph.add_(nodeKokkaiGijidouMae);
		aGraph.add_(nodeToshimaen);
		aGraph.add_(nodeAsakusa);
		aGraph.add_(nodeShimokitazawa);
		aGraph.add_(nodeDaiba);

		aGraph.beUndirectedGraph();

		return aGraph;
	}

	/**
	 * Create a new instance of the default arc.
	 * 
	 * @param firstNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param lastNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param arcAttribute jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @return jp.co.sra.jun.topology.graph.JunElementalArc
	 * @category Defaults
	 */
	public static JunElementalArc DefaultNewArc(JunElementalNode firstNode, JunElementalNode lastNode, JunAttributeTable arcAttribute) {
		return new JunElementalArc(firstNode, lastNode, arcAttribute);
	}

	/**
	 * Answer the default arrange format.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Defaults
	 */
	protected static StSymbol DefaultArrangeFormat() {
		return $("none");
	}

	/**
	 * Answer the default arrange attribute symbol.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Defaults
	 */
	protected static StSymbol DefaultArrangeAttributeSymbol() {
		return $("distance");
	}

	/**
	 * Answer the default arrange concentric radius.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultArrangeConcentricRadius() {
		return 350;
	}

	/**
	 * Answer the default arrange concentric spiral.
	 * 
	 * @return float
	 * @category Defaults
	 */
	protected static float DefaultArrangeConcentricSpiral() {
		return 1.0f;
	}

	/**
	 * Answer the default arrange concentric gradation.
	 * 
	 * @return java.lang.Boolean
	 * @category Defaults
	 */
	protected static Boolean DefaultArrangeConcentricGradation() {
		return Boolean.TRUE;
	}

	/**
	 * Answer the default arrange forest margin.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultArrangeForestMargin() {
		return 2;
	}

	/**
	 * Answer the default arrange forest interval.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultArrangeForestInterval() {
		return 50;
	}

	/**
	 * Create a new instance of JunElementalGraph and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunElementalGraph() {
		super();
	}

	/**
	 * Create a new instance of JunElementalGraph and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunElementalGraph(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		elementalNodes = new ArrayList();
		adjacencyMatrix = new JunAdjacencyMatrix();
		selectedNodes = new ArrayList();
		this.flushBoundingBox();
		this.flushCachedAll();
	}

	/**
	 * Set the receiver to be a directed graph.
	 * 
	 * @category accessing
	 */
	public void beDirectedGraph() {
		adjacencyMatrix.beDirectedGraph();
		this.flushCachedArcs();
	}

	/**
	 * Set the receiver to be an undirected graph.
	 * 
	 * @category accessing
	 */
	public void beUndirectedGraph() {
		adjacencyMatrix.beUndirectedGraph();
		this.flushCachedArcs();
	}

	/**
	 * Answer the number of nodes.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfNodes() {
		return this.nodes().length;
	}

	/**
	 * Answer the number of arcs.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfArcs() {
		return this.arcs().length;
	}

	/**
	 * Answer the number of roots.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfRoots() {
		return this.roots().length;
	}

	/**
	 * Answer my size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return elementalNodes.size();
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#boundingBox()
	 * @category accessing
	 */
	public StRectangle boundingBox() {
		if (boundingBox == null) {
			StRectangle aBox = new StRectangle(0, 0, 0, 0);
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				aBox = aBox.merge_(arcs[i].boundingBox());
			}
			JunElementalNode[] nodes = this.nodes();
			for (int i = 0; i < nodes.length; i++) {
				aBox = aBox.merge_(nodes[i].boundingBox());
			}

			boundingBox = aBox;
		}
		return boundingBox;
	}

	/**
	 * Answer my current nodes.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] nodes() {
		if (cachedNodes == null) {
			JunElementalNode[] nodes = (JunElementalNode[]) elementalNodes.toArray(new JunElementalNode[elementalNodes.size()]);
			Arrays.sort(nodes, new Comparator() {
				public int compare(Object o1, Object o2) {
					return ((JunElementalNode) o1).labelString().compareTo(((JunElementalNode) o2).labelString());
				}
			});
			cachedNodes = nodes;
		}
		return cachedNodes;
	}

	/**
	 * Answer the nodes in partial order.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] nodesPartialOrder() {
		return this.nodesPartialOrder_(null);
	}

	/**
	 * Answer the nodes in partial order.
	 * 
	 * @param startNode java.lang.Object
	 * @return java.lang.Object[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] nodesPartialOrder_(JunElementalNode startNode) {
		final ArrayList aList = new ArrayList(this.numberOfNodes());
		this.nodesPartialOrderDo_startNode_(new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				aList.add(node);
				return null;
			}
		}, startNode);
		return (JunElementalNode[]) aList.toArray(new JunElementalNode[aList.size()]);
	}

	/**
	 * Answer the nodes in topological sort.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] nodesTopologicalSort() {
		return this.nodesPartialOrder();
	}

	/**
	 * Answer the nodes in topological sort.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] nodesTopologicalSort_(JunElementalNode startNode) {
		return this.nodesPartialOrder_(startNode);
	}

	/**
	 * Answer the reachable nodes from the specified node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] reachableNodesFrom_(final JunElementalNode startNode) {
		final ArrayList aList = new ArrayList(this.numberOfNodes());
		this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				if (node.equals(startNode) == false) {
					aList.add(node);
				}
				return null;
			}
		}, new StBlockClosure());
		return (JunElementalNode[]) aList.toArray(new JunElementalNode[aList.size()]);
	}

	/**
	 * Answer my current roots.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category accessing nodes
	 */
	public JunElementalNode[] roots() {
		if (cachedRoots == null) {
			if (this.isEmpty()) {
				cachedRoots = new JunElementalNode[0];
			} else {
				TreeSet aTreeSet = new TreeSet(new Comparator() {
					public int compare(Object o1, Object o2) {
						return ((JunElementalNode) o1).labelString().compareTo(((JunElementalNode) o2).labelString());
					}
				});

				Object[] roots = adjacencyMatrix.roots();
				for (int i = 0; i < roots.length; i++) {
					aTreeSet.add(this.uniqueNumberAt_((StSymbol) roots[i]));
				}
				cachedRoots = (JunElementalNode[]) aTreeSet.toArray(new JunElementalNode[aTreeSet.size()]);
			}
		}
		return cachedRoots;
	}

	/**
	 * Answer my current node table.
	 * 
	 * @return java.util.Map
	 * @category accessing nodes
	 */
	protected Map nodeTable() {
		if (cachedNodeTable == null) {
			JunElementalNode[] nodes = this.nodes();
			HashMap table = new HashMap(nodes.length);
			for (int i = 0; i < nodes.length; i++) {
				table.put(nodes[i].uniqueNumber(), nodes[i]);
			}
			cachedNodeTable = table;
		}
		return cachedNodeTable;
	}

	/**
	 * Answer the node with the specified unique number.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category accessing nodes
	 */
	protected JunElementalNode uniqueNumberAt_(StSymbol aSymbol) {
		return (JunElementalNode) this.nodeTable().get(aSymbol);
	}

	/**
	 * Answer my current arcs.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalArc[]
	 * @category accessing arcs
	 */
	public JunElementalArc[] arcs() {
		if (cachedArcs == null) {
			Object[] collection = adjacencyMatrix.arcs();
			JunElementalArc[] arcs = new JunElementalArc[collection.length];
			HashMap table = new HashMap(collection.length);
			for (int i = 0; i < collection.length; i++) {
				Object[] array = (Object[]) collection[i];
				StSymbol first = (StSymbol) array[0];
				StSymbol last = (StSymbol) array[2];
				JunElementalArc arc = DefaultNewArc(this.uniqueNumberAt_(first), this.uniqueNumberAt_(last), (JunAttributeTable) array[1]);
				arcs[i] = arc;
				table.put(this.arcTableKey(first, last), arc);
			}
			Arrays.sort(arcs, new Comparator() {
				public int compare(Object o1, Object o2) {
					JunElementalArc a1 = (JunElementalArc) o1;
					JunElementalArc a2 = (JunElementalArc) o2;
					if (a1.firstNode().labelString().equals(a2.firstNode().labelString())) {
						return a1.lastNode().labelString().compareTo(a2.lastNode().labelString());
					} else {
						return a1.firstNode().labelString().compareTo(a2.firstNode().labelString());
					}
				}
			});

			cachedArcs = arcs;
			cachedArcTable = table;
		}
		return cachedArcs;
	}

	/**
	 * Answer the arcs of the specified node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalArc[]
	 * @category accessing arcs
	 */
	public JunElementalArc[] arcsOf_(JunElementalNode aNode) {
		if (cachedNodeToArcs == null) {
			cachedNodeToArcs = new HashMap();
		}
		JunElementalArc[] nodeToArcs = (JunElementalArc[]) cachedNodeToArcs.get(aNode);
		if (nodeToArcs == null) {
			TreeSet aTreeSet = new TreeSet(new Comparator() {
				public int compare(Object o1, Object o2) {
					JunElementalArc a1 = (JunElementalArc) o1;
					JunElementalArc a2 = (JunElementalArc) o2;
					return a1.lastNode().labelString().compareTo(a2.lastNode().labelString());
				}
			});
			Object[] arcs = adjacencyMatrix.arcsOf_(aNode.uniqueNumber());
			for (int i = 0; i < arcs.length; i++) {
				Object[] arc = (Object[]) arcs[i];
				aTreeSet.add(this.uniqueNumberAt_with_((StSymbol) arc[0], (StSymbol) arc[2]));
			}

			nodeToArcs = (JunElementalArc[]) aTreeSet.toArray(new JunElementalArc[aTreeSet.size()]);
			cachedNodeToArcs.put(aNode, nodeToArcs);
		}
		return nodeToArcs;
	}

	/**
	 * Answer the attributes of the arc between the nodes.
	 * 
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param toNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category accessing arcs
	 */
	public JunAttributeTable between_and_(JunElementalNode fromNode, JunElementalNode toNode) {
		return (JunAttributeTable) adjacencyMatrix.between_and_(fromNode.uniqueNumber(), toNode.uniqueNumber());
	}

	/**
	 * Answer the down arcs of the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalArc[]
	 * @category accessing arcs
	 */
	public JunElementalArc[] downArcsOfNode_(JunElementalNode aNode) {
		TreeSet aTreeSet = new TreeSet(new Comparator() {
			public int compare(Object o1, Object o2) {
				JunElementalArc a1 = (JunElementalArc) o1;
				JunElementalArc a2 = (JunElementalArc) o2;
				return a1.lastNode().labelString().compareTo(a2.lastNode().labelString());
			}
		});

		Object[] arcs = adjacencyMatrix.downArcsOfNode_(aNode.uniqueNumber());
		for (int i = 0; i < arcs.length; i++) {
			Object[] arc = (Object[]) arcs[i];
			aTreeSet.add(this.uniqueNumberAt_with_((StSymbol) arc[0], (StSymbol) arc[2]));
		}
		return (JunElementalArc[]) aTreeSet.toArray(new JunElementalArc[aTreeSet.size()]);
	}

	/**
	 * Answer the up arcs of the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalArc[]
	 * @category accessing arcs
	 */
	public JunElementalArc[] upArcsOfNode_(JunElementalNode aNode) {
		TreeSet aTreeSet = new TreeSet(new Comparator() {
			public int compare(Object o1, Object o2) {
				JunElementalArc a1 = (JunElementalArc) o1;
				JunElementalArc a2 = (JunElementalArc) o2;
				return a1.firstNode().labelString().compareTo(a2.firstNode().labelString());
			}
		});

		Object[] arcs = adjacencyMatrix.upArcsOfNode_(aNode.uniqueNumber());
		for (int i = 0; i < arcs.length; i++) {
			Object[] arc = (Object[]) arcs[i];
			aTreeSet.add(this.uniqueNumberAt_with_((StSymbol) arc[0], (StSymbol) arc[2]));
		}
		return (JunElementalArc[]) aTreeSet.toArray(new JunElementalArc[aTreeSet.size()]);
	}

	/**
	 * Answer my current arc table.
	 * 
	 * @return java.util.Map
	 * @category accessing arcs
	 */
	protected Map arcTable() {
		if (cachedArcTable == null) {
			this.arcs();
		}
		return cachedArcTable;
	}

	/**
	 * Answer the arc table key with two nodes.
	 * 
	 * @param from jp.co.sra.smalltalk.StSymbol
	 * @param to jp.co.sra.smalltalk.StSymbol
	 * @return java.lang.String
	 * @category accessing arcs
	 */
	private String arcTableKey(StSymbol from, StSymbol to) {
		return from.toString() + '+' + to.toString();
	}

	/**
	 * Answer the arc with the specified unique number.
	 * 
	 * @param fromUniqueNumber jp.co.sra.smalltalk.StSymbol
	 * @param toUniqueNumber jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.jun.topology.graph.JunElementalArc
	 * @category accessing arcs
	 */
	protected JunElementalArc uniqueNumberAt_with_(StSymbol fromUniqueNumber, StSymbol toUniqueNumber) {
		return (JunElementalArc) this.arcTable().get(this.arcTableKey(fromUniqueNumber, toUniqueNumber));
	}

	/**
	 * Answer the distances.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] distances() {
		return this.distancesAttributeSymbol_($("cost"));
	}

	/**
	 * Answer the distances.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk..StSymbol
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] distancesAttributeSymbol_(StSymbol aSymbol) {
		if (this.isEmpty()) {
			return new Object[] { new Float(0), new ArrayList(0) };
		}

		JunElementalNode startNode = (JunElementalNode) elementalNodes.get(0);
		if (this.hasSelections()) {
			startNode = this.selection();
		}
		return this.distancesFrom_attributeSymbol_(startNode, aSymbol);

	}

	/**
	 * Answer the distances from the node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] distancesFrom_(JunElementalNode startNode) {
		return this.distancesFrom_attributeSymbol_(startNode, $("cost"));
	}

	/**
	 * Answer the distances from the node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param aSymbol jp.co.sra.smalltalk..StSymbol
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] distancesFrom_attributeSymbol_(JunElementalNode startNode, StSymbol aSymbol) {
		if (this.isEmpty()) {
			return new Object[] { new Float(0), new ArrayList(0) };
		}

		Map nodeTable = this.shortestPathFrom_attributeSymbol_(startNode, aSymbol);

		float maxDistance = 0;
		Iterator iterator = nodeTable.entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			Number distance = (Number) entry.getValue();
			if (distance != null) {
				maxDistance = Math.max(maxDistance, distance.floatValue());
			}
		}
		if (maxDistance <= 0) {
			maxDistance = 1;
		}

		HashMap distanceTable = new HashMap();
		iterator = nodeTable.entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			JunElementalNode node = (JunElementalNode) entry.getKey();
			Number distance = (Number) entry.getValue();
			if (distance == null) {
				distance = new Float(maxDistance * 2);
			}

			Collection aCollection = (Collection) distanceTable.get(distance);
			if (aCollection == null) {
				aCollection = new ArrayList();
				distanceTable.put(distance, aCollection);
			}
			aCollection.add(node);

		}

		TreeSet aValue = new TreeSet(new Comparator() {
			public int compare(Object o1, Object o2) {
				float d1 = ((Number) ((Map.Entry) o1).getKey()).floatValue();
				float d2 = ((Number) ((Map.Entry) o2).getKey()).floatValue();
				return (d1 == d2) ? 0 : (d1 < d2) ? -1 : 1;
			}
		});
		aValue.addAll(distanceTable.entrySet());

		return new Object[] { new Float(maxDistance), aValue };
	}

	/**
	 * Answer the normalized distances.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] normalizedDistances() {
		return this.normalizedDistancesAttributeSymbol_($("cost"));
	}

	/**
	 * Answer the normalized distances.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] normalizedDistancesAttributeSymbol_(StSymbol aSymbol) {
		if (this.isEmpty()) {
			return new Object[] { new Float(0), new ArrayList(0) };
		}

		JunElementalNode startNode = (JunElementalNode) elementalNodes.get(0);
		if (this.hasSelections()) {
			startNode = this.selection();
		}
		return this.normalizedDistancesFrom_attributeSymbol_(startNode, aSymbol);
	}

	/**
	 * Answer the normalized distances from the node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] normalizedDistancesFrom_(JunElementalNode startNode) {
		return this.normalizedDistancesFrom_attributeSymbol_(startNode, $("cost"));
	}

	/**
	 * Answer the normalized distances from the node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param aSymbol jp.co.sra.smalltalk..StSymbol
	 * @return java.lang.Object[]
	 * @category accessing distances
	 */
	public Object[] normalizedDistancesFrom_attributeSymbol_(JunElementalNode startNode, StSymbol aSymbol) {
		Object[] anAssociation = this.distancesFrom_attributeSymbol_(startNode, aSymbol);
		Number aKey = (Number) anAssociation[0];

		ArrayList aValue = new ArrayList();
		Iterator iterator = ((Collection) anAssociation[1]).iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			float key = ((Number) entry.getKey()).floatValue() / aKey.floatValue();
			aValue.add(new Object[] { new Float(key), entry.getValue() });
		}

		return new Object[] { aKey, aValue };
	}

	/**
	 * Answer the shortest paths of all nodes.
	 * 
	 * @return java.util.Map
	 * @category accessing distances
	 */
	public Map shortestPaths() {
		return this.shortestPathsAttributeSymbol_($("cost"));
	}

	/**
	 * Answer the shortest paths of all nodes.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return java.util.Map
	 * @category accessing distances
	 */
	public Map shortestPathsAttributeSymbol_(StSymbol aSymbol) {
		HashMap aMap = new HashMap(this.numberOfNodes());
		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			aMap.put(nodes[i], this.shortestPathFrom_attributeSymbol_(nodes[i], aSymbol));
		}
		return aMap;
	}

	/**
	 * Answer the shortest path from the specified node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return java.util.Map
	 * @category accessing distances
	 */
	public Map shortestPathFrom_(JunElementalNode startNode) {
		return this.shortestPathFrom_attributeSymbol_(startNode, $("cost"));
	}

	/**
	 * Answer the shortest path from the specified node.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return java.util.Map
	 * @category accessing distances
	 */
	public Map shortestPathFrom_attributeSymbol_(JunElementalNode startNode, StSymbol aSymbol) {
		return (Map) this.visitDijkstra_attributeSymbol_(startNode, aSymbol).at_($("distanceTable"));
	}

	/**
	 * Answer my arrange format.
	 * 
	 * @return jp.co.sra.smalltalk..StSymbol
	 * @category attributes
	 */
	public StSymbol arrangeFormat() {
		StSymbol aSymbol = (StSymbol) myAttributes.at_($("arrangeFormat"));
		if (aSymbol == null) {
			aSymbol = this.defaultArrangeFormat();
			myAttributes.at_put_($("arrangeFormat"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Set my arrange format.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk..StSymbol
	 * @category attributes
	 */
	public void arrangeFormat_(StSymbol aSymbol) {
		this.updateModificationDate();
		if (aSymbol == null) {
			myAttributes.removeKey_($("arrangeFormat"));
			return;
		}

		if (this.isNoneArrangement()) {
			JunAttributeTable aTable = this.noneArrangementTable();
			Object[] keys = aTable.keys().toArray();
			for (int i = 0; i < keys.length; i++) {
				aTable.removeKey_(keys[i]);
			}
			JunElementalNode[] nodes = this.nodes();
			for (int i = 0; i < nodes.length; i++) {
				aTable.at_put_(nodes[i].uniqueNumber(), nodes[i].locationPoint());
			}
		}

		myAttributes.at_put_($("arrangeFormat"), aSymbol);
	}

	/**
	 * Answer my arrange format.
	 * 
	 * @return jp.co.sra.smalltalk..StSymbol
	 * @category attributes
	 */
	public StSymbol arrangeAttributeSymbol() {
		StSymbol aSymbol = (StSymbol) myAttributes.at_($("arrangeAttributeSymbol"));
		if (aSymbol == null) {
			aSymbol = this.defaultArrangeAttributeSymbol();
			myAttributes.at_put_($("arrangeAttributeSymbol"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Set my arrange format.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk..StSymbol
	 * @category attributes
	 */
	public void arrangeAttributeSymbol_(StSymbol aSymbol) {
		this.updateModificationDate();
		if (aSymbol == null) {
			myAttributes.removeKey_($("arrangeAttributeSymbol"));
			return;
		}

		myAttributes.at_put_($("arrangeAttributeSymbol"), aSymbol);
	}

	/**
	 * Answer my arrange concentric radius.
	 * 
	 * @return int
	 * @category attributes
	 */
	public int arrangeConcentricRadius() {
		Number aNumber = (Number) myAttributes.at_($("arrangeConcentricRadius"));
		if (aNumber == null) {
			aNumber = new Integer(this.defaultArrangeConcentricRadius());
			myAttributes.at_put_($("arrangeConcentricRadius"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set my arrange concentric radius.
	 * 
	 * @param anInteger int
	 * @category attributes
	 */
	public void arrangeConcentricRadius_(int anInteger) {
		this.updateModificationDate();
		if (anInteger < 0) {
			myAttributes.removeKey_($("arrangeConcentricRadius"));
			return;
		}

		myAttributes.at_put_($("arrangeConcentricRadius"), new Integer(anInteger));
	}

	/**
	 * Answer my arrange concentric spiral.
	 * 
	 * @return float
	 * @category attributes
	 */
	public float arrangeConcentricSpiral() {
		Number aNumber = (Number) myAttributes.at_($("arrangeConcentricSpiral"));
		if (aNumber == null) {
			aNumber = new Float(this.defaultArrangeConcentricSpiral());
			myAttributes.at_put_($("arrangeConcentricSpiral"), aNumber);
		}
		return aNumber.floatValue();
	}

	/**
	 * Set my arrange concenric spiral
	 * 
	 * @param aFloat float
	 * @category attributes
	 */
	public void arrangeConcentricSpiral_(float aFloat) {
		this.updateModificationDate();
		if (aFloat < 0) {
			myAttributes.removeKey_($("arrangeConcentricSpiral"));
			return;
		}

		myAttributes.at_put_($("arrangeConcentricSpiral"), new Float(aFloat));
	}

	/**
	 * Answer my arrange concentric gradation.
	 * 
	 * @return java.lang.Boolean
	 * @category attributes
	 */
	public Boolean arrangeConcentricGradation() {
		Boolean aBoolean = (Boolean) myAttributes.at_($("arrangeConcentricGradation"));
		if (aBoolean == null) {
			aBoolean = this.defaultArrangeConcentricGradation();
			myAttributes.at_put_($("arrangeConcentricGradation"), aBoolean);
		}
		return aBoolean;
	}

	/**
	 * Set my arrange forest margin.
	 * 
	 * @param aBoolean
	 * @category attributes
	 */
	public void arrangeConcentricGradation_(Boolean aBoolean) {
		this.updateModificationDate();
		if (aBoolean == null) {
			myAttributes.removeKey_($("arrangeConcentricGradation"));
			return;
		}

		myAttributes.at_put_($("arrangeConcentricGradation"), aBoolean);
	}

	/**
	 * Answer my arrange concentric rectangles.
	 * 
	 * @return java.util.ArrayList
	 * @category attributes
	 */
	public ArrayList arrangeConcentricRectangles() {
		ArrayList aList = (ArrayList) myAttributes.at_($("arrangeConcentricRectangles"));
		if (aList == null) {
			aList = new ArrayList();
			myAttributes.at_put_($("arrangeConcentricRectangles"), aList);
		}
		return aList;
	}

	/**
	 * Set my arrange concentric rectangles.
	 * 
	 * @param aList java.util.ArrayList
	 * @category attributes
	 */
	public void arrangeConcentricRectangles_(ArrayList aList) {
		if (aList == null) {
			myAttributes.removeKey_($("arrangeConcentricRectangles"));
			return;
		}
		myAttributes.at_put_($("arrangeConcentricRectangles"), aList);
	}

	/**
	 * Answer my arrange forest margin.
	 * 
	 * @return int
	 * @category attributes
	 */
	public int arrangeForestMargin() {
		Number aNumber = (Number) myAttributes.at_($("arrangeForestMargin"));
		if (aNumber == null) {
			aNumber = new Integer(this.defaultArrangeForestMargin());
			myAttributes.at_put_($("arrangeForestMargin"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set my arrange forest margin.
	 * 
	 * @param anInteger int
	 * @category attributes
	 */
	public void arrangeForestMargin_(int anInteger) {
		this.updateModificationDate();
		if (anInteger < 0) {
			myAttributes.removeKey_($("arrangeForestMargin"));
			return;
		}

		myAttributes.at_put_($("arrangeForestMargin"), new Integer(anInteger));
	}

	/**
	 * Answer my arrange forest interval.
	 * 
	 * @return int
	 * @category attributes
	 */
	public int arrangeForestInterval() {
		Number aNumber = (Number) myAttributes.at_($("arrangeForestInterval"));
		if (aNumber == null) {
			aNumber = new Integer(this.defaultArrangeForestInterval());
			myAttributes.at_put_($("arrangeForestInterval"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Set my arrange forest interval.
	 * 
	 * @param anInteger int
	 * @category attributes
	 */
	public void arrangeForestInterval_(int anInteger) {
		this.updateModificationDate();
		if (anInteger < 0) {
			myAttributes.removeKey_($("arrangeForestInterval"));
			return;
		}

		myAttributes.at_put_($("arrangeForestInterval"), new Integer(anInteger));
	}

	/**
	 * Answer my none arrangement table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category attributes
	 */
	public JunAttributeTable noneArrangementTable() {
		JunAttributeTable aTable = (JunAttributeTable) myAttributes.at_($("noneArrangementTable"));
		if (aTable == null) {
			aTable = new JunAttributeTable();
			myAttributes.at_put_($("noneArrangementTable"), aTable);
		}
		return aTable;
	}

	/**
	 * Answer the attribute symbols to reset.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol[]
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#attributeSymbolsToReset()
	 * @category attributes
	 */
	protected StSymbol[] attributeSymbolsToReset() {
		return AttributeSymbolsToReset;
	}

	/**
	 * Add the node to the receiver.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category adding
	 */
	public JunElementalNode add_(JunElementalNode aNode) {
		if (adjacencyMatrix.add_(aNode.uniqueNumber) == null) {
			return null;
		}

		elementalNodes.add(aNode);
		this.flushBoundingBox();
		this.flushCachedNodes();
		return aNode;
	}

	/**
	 * Remove the node from the receiver.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category removing
	 */
	public JunElementalNode remove_(JunElementalNode aNode) {
		if (elementalNodes.contains(aNode) == false) {
			return null;
		}

		elementalNodes.remove(aNode);
		adjacencyMatrix.remove_(aNode.uniqueNumber());
		selectedNodes.remove(aNode);
		this.flushBoundingBox();
		this.flushCachedNodes();
		return aNode;
	}

	/**
	 * Connect the from node with the to node. 
	 * 
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param toNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean connect_with_(JunElementalNode fromNode, JunElementalNode toNode) {
		this.add_(fromNode);
		this.add_(toNode);
		Boolean triplet = adjacencyMatrix.connect_with_(fromNode.uniqueNumber(), toNode.uniqueNumber());
		if (triplet != null && triplet.booleanValue()) {
			this.flushCachedArcs();
			JunAttributeTable attributes = this.between_and_(fromNode, toNode);
			if (this.isConcentricArrangement()) {
				attributes.at_put_($("firstPointAlignment"), $("center"));
				attributes.at_put_($("lastPointAlignment"), $("center"));
			} else if (this.isHierarchicalArrangement()) {
				attributes.at_put_($("firstPointAlignment"), $("rightCenter"));
				attributes.at_put_($("lastPointAlignment"), $("leftCenter"));
			}
		}

		return triplet;
	}

	/**
	 * Connect the from node with the to node, and set the specified attribute to the arc..
	 * 
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param toNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param key java.lang.Object
	 * @param value java.lang.Object
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean connect_with_key_value_(JunElementalNode fromNode, JunElementalNode toNode, Object key, Object value) {
		this.add_(fromNode);
		this.add_(toNode);
		Boolean triplet = adjacencyMatrix.connect_with_key_value_(fromNode.uniqueNumber(), toNode.uniqueNumber(), key, value);
		if (triplet != null && triplet.booleanValue()) {
			this.flushCachedArcs();
		}
		return triplet;
	}

	/**
	 * Connect the from node with the to node, and specifies its cost. 
	 * 
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param toNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param value int
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean connect_with_cost_(JunElementalNode fromNode, JunElementalNode toNode, int value) {
		this.add_(fromNode);
		this.add_(toNode);
		Boolean triplet = adjacencyMatrix.connect_with_cost_(fromNode.uniqueNumber(), toNode.uniqueNumber(), value);
		if (triplet != null && triplet.booleanValue()) {
			this.flushCachedArcs();
		}

		return triplet;
	}

	/**
	 * Disconnect the from node with the to node. 
	 * 
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param toNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return java.lang.Boolean
	 * @category connecting
	 */
	public Boolean disconnect_with_(JunElementalNode fromNode, JunElementalNode toNode) {
		Boolean triplet = adjacencyMatrix.disconnect_with_(fromNode.uniqueNumber(), toNode.uniqueNumber());
		if (triplet != null && triplet.booleanValue()) {
			this.flushCachedArcs();
		}

		return triplet;
	}

	/**
	 * Visit the nodes in the breadth first order.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitBreadthFirst_nodeDo_arcDo_(JunElementalNode aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		JunAttributeTable visitingAttributes = new JunAttributeTable();
		visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("nodesBreadthFirst"), new ArrayList(this.numberOfNodes()));
		visitingAttributes.at_put_($("arcsBreadthFirst"), new ArrayList(this.numberOfArcs()));
		visitingAttributes.at_put_($("closedPath"), new ArrayList());
		HashMap visitingTable = new HashMap();
		for (int i = 0; i < this.nodes().length; i++) {
			visitingTable.put(this.nodes()[i], $("unvisited"));
		}
		visitingAttributes.at_put_($("visitingTable"), visitingTable);

		LinkedList graphQueue = new LinkedList();

		if (aNode == null) {
			for (int i = 0; i < this.roots().length; i++) {
				JunElementalNode fromNode = this.roots()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(visitingAttributes, fromNode, graphQueue, nodeBlock, arcBlock);
				}
			}
			for (int i = 0; i < this.nodes().length; i++) {
				JunElementalNode fromNode = this.nodes()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(visitingAttributes, fromNode, graphQueue, nodeBlock, arcBlock);
				}
			}
		} else {
			if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(aNode) == $("unvisited")) {
				this.visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(visitingAttributes, aNode, graphQueue, nodeBlock, arcBlock);
			}
		}

		return visitingAttributes;
	}

	/**
	 * Visit the nodes in the breadth first order.
	 * 
	 * @param visitingAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param graphQueue java.util.LinkedList
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	protected void visitBreadthFirst_fromNode_graphQueue_nodeDo_arcDo_(JunAttributeTable visitingAttributes, JunElementalNode fromNode, LinkedList graphQueue, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		ArrayList nodesBreadthFirst = (ArrayList) visitingAttributes.at_($("nodesBreadthFirst"));
		ArrayList arcsBreadthFirst = (ArrayList) visitingAttributes.at_($("arcsBreadthFirst"));
		ArrayList closedPath = (ArrayList) visitingAttributes.at_($("closedPath"));
		HashMap visitingTable = (HashMap) visitingAttributes.at_($("visitingTable"));
		visitingTable.put(fromNode, $("entered"));
		graphQueue.add(new Object[] { fromNode, new Integer(0) });
		while (graphQueue.isEmpty() == false) {
			Object[] anArray = (Object[]) graphQueue.removeFirst();
			JunElementalNode theNode = (JunElementalNode) anArray[0];
			int indentLevel = ((Number) anArray[1]).intValue();
			visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("nodeSequenceNumber"))).intValue() + 1));
			Object[] argumentArray = new Object[] { theNode, new Integer(indentLevel), visitingAttributes.at_($("nodeSequenceNumber")) };
			nodesBreadthFirst.add(argumentArray);
			nodeBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
			visitingTable.put(theNode, $("visited"));
			JunElementalArc[] arcs = this.arcsOf_(theNode);
			for (int i = 0; i < arcs.length; i++) {
				JunElementalNode toNode = arcs[i].lastNode();
				visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("arcSequenceNumber"))).intValue() + 1));
				argumentArray = new Object[] { arcs[i], new Integer(indentLevel), visitingAttributes.at_($("arcSequenceNumber")) };
				arcsBreadthFirst.add(argumentArray);
				arcBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
				if (visitingTable.get(toNode) == $("unvisited")) {
					graphQueue.add(new Object[] { toNode, new Integer(indentLevel + 1) });
					visitingTable.put(toNode, $("entered"));
				} else {
					closedPath.add(toNode);
				}
			}
		}
	}

	/**
	 * Visit the nodes in the depth first order.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitDepthFirst_nodeDo_arcDo_(JunElementalNode aNode, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		JunAttributeTable visitingAttributes = new JunAttributeTable();
		visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(0));
		visitingAttributes.at_put_($("nodesDepthFirst"), new ArrayList(this.numberOfNodes()));
		visitingAttributes.at_put_($("arcsDepthFirst"), new ArrayList(this.numberOfArcs()));
		visitingAttributes.at_put_($("topologicalSort"), new ArrayList(this.numberOfNodes()));
		visitingAttributes.at_put_($("closedPath"), new ArrayList());
		HashMap visitingTable = new HashMap();
		for (int i = 0; i < this.nodes().length; i++) {
			visitingTable.put(this.nodes()[i], $("unvisited"));
		}
		visitingAttributes.at_put_($("visitingTable"), visitingTable);

		if (aNode == null) {
			for (int i = 0; i < this.roots().length; i++) {
				JunElementalNode fromNode = this.roots()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, fromNode, 0, nodeBlock, arcBlock);
				}
			}
			for (int i = 0; i < this.nodes().length; i++) {
				JunElementalNode fromNode = this.nodes()[i];
				if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(fromNode) == $("unvisited")) {
					this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, fromNode, 0, nodeBlock, arcBlock);
				}
			}
		} else {
			if (((HashMap) visitingAttributes.at_($("visitingTable"))).get(aNode) == $("unvisited")) {
				this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, aNode, 0, nodeBlock, arcBlock);
			}
		}

		return visitingAttributes;
	}

	/**
	 * Visit the nodes in the depth first order.
	 * 
	 * @param visitingAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param indentLevel int
	 * @param nodeBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param arcBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category visiting
	 */
	protected void visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(JunAttributeTable visitingAttributes, JunElementalNode fromNode, int indentLevel, StBlockClosure nodeBlock, StBlockClosure arcBlock) {
		ArrayList nodesDepthFirst = (ArrayList) visitingAttributes.at_($("nodesDepthFirst"));
		ArrayList arcsDepthFirst = (ArrayList) visitingAttributes.at_($("arcsDepthFirst"));
		ArrayList topologicalSort = (ArrayList) visitingAttributes.at_($("topologicalSort"));
		ArrayList closedPath = (ArrayList) visitingAttributes.at_($("closedPath"));
		HashMap visitingTable = (HashMap) visitingAttributes.at_($("visitingTable"));
		visitingAttributes.at_put_($("nodeSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("nodeSequenceNumber"))).intValue() + 1));
		Object[] argumentArray = new Object[] { fromNode, new Integer(indentLevel), visitingAttributes.at_($("nodeSequenceNumber")) };
		nodesDepthFirst.add(argumentArray);
		nodeBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		visitingTable.put(fromNode, $("visited"));
		JunElementalArc[] arcs = this.arcsOf_(fromNode);
		for (int i = 0; i < arcs.length; i++) {
			JunElementalNode toNode = arcs[i].lastNode();
			visitingAttributes.at_put_($("arcSequenceNumber"), new Integer(((Number) visitingAttributes.at_($("arcSequenceNumber"))).intValue() + 1));
			argumentArray = new Object[] { arcs[i], new Integer(indentLevel), visitingAttributes.at_($("arcSequenceNumber")) };
			arcsDepthFirst.add(argumentArray);
			arcBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
			if (visitingTable.get(toNode) == $("unvisited")) {
				this.visitDepthFirst_fromNode_indentLevel_nodeDo_arcDo_(visitingAttributes, toNode, indentLevel + 1, nodeBlock, arcBlock);
			} else {
				closedPath.add(toNode);
			}
		}
		topologicalSort.add(fromNode);
	}

	/**
	 * Visit the nodes in Dijkstra's algorithm.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitDijkstra_(JunElementalNode aNode) {
		return this.visitDijkstra_attributeSymbol_(aNode, $("cost"));
	}

	/**
	 * Visit the nodes in Dijkstra's algorithm.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category visiting
	 */
	public JunAttributeTable visitDijkstra_attributeSymbol_(JunElementalNode aNode, StSymbol aSymbol) {
		HashMap distanceTable = new HashMap();
		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			distanceTable.put(nodes[i], null);
		}

		JunAttributeTable visitingAttributes = new JunAttributeTable();
		visitingAttributes.at_put_($("distanceTable"), distanceTable);
		visitingAttributes.at_put_($("visitedNodes"), new HashSet());
		visitingAttributes.at_put_($("unvisitedNodes"), new HashSet());

		if (aNode != null) {
			this.visitDijkstra_fromNode_attributeSymbol_(visitingAttributes, aNode, aSymbol);
		}

		return visitingAttributes;
	}

	/**
	 * Visit the nodes in Dijkstra's algorithm.
	 * 
	 * @param visitingAttributes jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param fromNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category visiting
	 */
	protected void visitDijkstra_fromNode_attributeSymbol_(JunAttributeTable visitingAttributes, JunElementalNode fromNode, StSymbol aSymbol) {
		Collection visitedNodes = (Collection) visitingAttributes.at_($("visitedNodes"));
		Collection unvisitedNodes = (Collection) visitingAttributes.at_($("unvisitedNodes"));
		Map distanceTable = (Map) visitingAttributes.at_($("distanceTable"));
		distanceTable.put(fromNode, new Float(0));
		visitedNodes.add(fromNode);
		JunElementalNode[] reachableNodes = this.reachableNodesFrom_(fromNode);
		for (int i = 0; i < reachableNodes.length; i++) {
			distanceTable.put(reachableNodes[i], null);
			unvisitedNodes.add(reachableNodes[i]);
		}
		JunElementalArc[] arcs = this.arcsOf_(fromNode);
		for (int i = 0; i < arcs.length; i++) {
			JunElementalNode toNode = arcs[i].lastNode();
			JunAttributeTable attributes = arcs[i].arcAttributes();
			float length = 1;
			if (attributes.at_(aSymbol) != null) {
				length = ((Number) attributes.at_(aSymbol)).floatValue();
				length = Math.max(length, 0);
			}
			distanceTable.put(toNode, new Float(length));
		}
		while (unvisitedNodes.isEmpty() == false) {
			JunElementalNode aNode = null;
			Number minDistance = null;
			Iterator iterator = unvisitedNodes.iterator();
			while (iterator.hasNext()) {
				JunElementalNode node = (JunElementalNode) iterator.next();
				Number theDistance = (Number) distanceTable.get(node);
				if (theDistance != null) {
					if (minDistance == null) {
						minDistance = theDistance;
						aNode = node;
					} else {
						if (theDistance.floatValue() < minDistance.floatValue()) {
							minDistance = theDistance;
							aNode = node;
						}
					}
				}
			}
			if (aNode == null) {
				aNode = (JunElementalNode) unvisitedNodes.iterator().next();
			}
			unvisitedNodes.remove(aNode);
			visitedNodes.add(aNode);

			arcs = this.arcsOf_(aNode);
			for (int i = 0; i < arcs.length; i++) {
				JunElementalNode toNode = arcs[i].lastNode();
				if (unvisitedNodes.contains(toNode)) {
					JunAttributeTable attributes = (JunAttributeTable) arcs[i].arcAttributes();
					float length = 1;
					if (attributes.at_(aSymbol) != null) {
						length = ((Number) attributes.at_(aSymbol)).floatValue();
						length = Math.max(length, 0);
					}
					Number toDistance = (Number) distanceTable.get(toNode);
					if (toDistance == null) {
						Number aDistance = (Number) distanceTable.get(aNode);
						if (aDistance == null) {
							//
						} else {
							length += aDistance.floatValue();
						}
					} else {
						Number aDistance = (Number) distanceTable.get(aNode);
						if (aDistance == null) {
							length = Math.min(toDistance.floatValue(), length);
						} else {
							length = Math.min(toDistance.floatValue(), length + aDistance.floatValue());
						}
					}
					distanceTable.put(toNode, new Float(length));
				}
			}
		}
	}

	/**
	 * Enumerate nodes in breadth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesBreadthFirstDo_(StBlockClosure aBlock) {
		this.nodesBreadthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate nodes in breadth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category enumerating nodes
	 */
	public void nodesBreadthFirstDo_startNode_(StBlockClosure aBlock, JunElementalNode startNode) {
		JunAttributeTable attributes = this.visitBreadthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("nodesBreadthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Enumerate nodes in depth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesDepthFirstDo_(StBlockClosure aBlock) {
		this.nodesDepthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate nodes in depth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category enumerating nodes
	 */
	public void nodesDepthFirstDo_startNode_(StBlockClosure aBlock, JunElementalNode startNode) {
		JunAttributeTable attributes = this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("nodesDepthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Enumerate nodes in partial order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesPartialOrderDo_(StBlockClosure aBlock) {
		this.nodesPartialOrderDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate nodes in partial order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category enumerating nodes
	 */
	public void nodesPartialOrderDo_startNode_(StBlockClosure aBlock, JunElementalNode startNode) {
		final HashMap visitingDictionary = new HashMap(this.size());
		JunAttributeTable visitingAttributes = this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				visitingDictionary.put(node, new Object[] { indent, sequence });
				return null;
			}
		}, new StBlockClosure());

		ArrayList aList = (ArrayList) visitingAttributes.at_($("topologicalSort"));
		for (int i = 0; i < aList.size(); i++) {
			Object node = aList.get(i);
			Object[] anArray = (Object[]) visitingDictionary.get(node);
			Object indent = anArray[0];
			Object sequence = anArray[1];
			aBlock.value_value_value_(node, indent, sequence);
		}
	}

	/**
	 * Enumerate nodes in topological sort order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating nodes
	 */
	public void nodesTopologicalSortDo_(StBlockClosure aBlock) {
		this.nodesPartialOrderDo_(aBlock);
	}

	/**
	 * Enumerate nodes in topological sort order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category enumerating nodes
	 */
	public void nodesTopologicalSortDo_startNode_(StBlockClosure aBlock, JunElementalNode startNode) {
		this.nodesPartialOrderDo_startNode_(aBlock, startNode);
	}

	/**
	 * Enumerate arcs in the breadth first order and evaluate the block with the arc.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating arcs
	 */
	public void arcsBreadthFirstDo_(StBlockClosure aBlock) {
		this.arcsBreadthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate arcs in the breadth first order and evaluate the block with the arc.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category enumerating arcs
	 */
	public void arcsBreadthFirstDo_startNode_(StBlockClosure aBlock, JunElementalNode startNode) {
		JunAttributeTable attributes = this.visitBreadthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("arcsBreadthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Enumerate arcs in depth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating arcs
	 */
	public void arcsDepthFirstDo_(StBlockClosure aBlock) {
		this.arcsDepthFirstDo_startNode_(aBlock, null);
	}

	/**
	 * Enumerate arcs in depth first order and evaluate the block with the node.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category enumerating arcs
	 */
	public void arcsDepthFirstDo_startNode_(StBlockClosure aBlock, JunElementalNode startNode) {
		JunAttributeTable attributes = this.visitDepthFirst_nodeDo_arcDo_(startNode, new StBlockClosure(), new StBlockClosure());
		ArrayList aList = (ArrayList) attributes.at_($("arcsDepthFirst"));
		for (int i = 0; i < aList.size(); i++) {
			Object[] argumentArray = (Object[]) aList.get(i);
			aBlock.value_value_value_(argumentArray[0], argumentArray[1], argumentArray[2]);
		}
	}

	/**
	 * Answer the selected nodes.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode[]
	 * @category selecting
	 */
	public JunElementalNode[] selectedNodes() {
		return (JunElementalNode[]) selectedNodes.toArray(new JunElementalNode[selectedNodes.size()]);
	}

	/**
	 * Answer the collection of the selected nodes.
	 * 
	 * @return java.util.Collection
	 * @category selecting
	 */
	public Collection selections() {
		return selectedNodes;
	}

	/**
	 * Select the nodes.
	 * 
	 * @param aCollection java.util.Collection
	 * @category selecting
	 */
	public void selections_(Collection aCollection) {
		selectedNodes.clear();

		if (aCollection == null || aCollection.isEmpty()) {
			return;
		}

		elementalNodes.removeAll(aCollection);
		elementalNodes.addAll(aCollection);

		selectedNodes.addAll(aCollection);
	}

	/**
	 * Answer a selected node if exists.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category selecting
	 */
	public JunElementalNode selection() {
		if (this.hasSelections() == false) {
			return null;
		}
		return (JunElementalNode) selectedNodes.get(0);
	}

	/**
	 * Set a selected node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category selecting
	 */
	public void selection_(JunElementalNode aNode) {
		ArrayList aList = new ArrayList(1);
		aList.add(aNode);
		this.selections_(aList);
	}

	/**
	 * Answer a node which is placed at the specified point.
	 * If none, return null.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category selecting
	 */
	public JunElementalNode whichAt_(Point aPoint) {
		JunElementalNode[] nodes = this.nodes();
		for (int i = nodes.length - 1; i >= 0; i--) {
			if (nodes[i].boundingBox().containsPoint_(aPoint)) {
				return nodes[i];
			}
		}
		return null;
	}

	/**
	 * Arrange the graph.
	 * 
	 * @category arranging
	 */
	public void arrange() {
		this.arrangeConcentricRectangles_(null);

		if (this.isConcentricArrangement()) {
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				arcs[i].firstPointAlignment_($("center"));
				arcs[i].lastPointAlignment_($("center"));
			}
			JunAttributeTable attributeTable = (JunAttributeTable) this.arrangeConcentricCirclesAttributeSymbol_(this.arrangeAttributeSymbol());
			Map nodeCenterTable = (Map) attributeTable.at_($("nodeCenterTable"));
			Iterator i = nodeCenterTable.entrySet().iterator();
			while (i.hasNext()) {
				Map.Entry entry = (Map.Entry) i.next();
				((JunElementalNode) entry.getKey()).center_((Point) entry.getValue());
			}
		} else if (this.isHierarchicalArrangement()) {
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				arcs[i].firstPointAlignment_($("rightCenter"));
				arcs[i].lastPointAlignment_($("leftCenter"));
			}
			JunAttributeTable attributeTable = (JunAttributeTable) this.arrangeForest();
			Map nodeOriginTable = (Map) attributeTable.at_($("nodeOriginTable"));
			Iterator i = nodeOriginTable.entrySet().iterator();
			while (i.hasNext()) {
				Map.Entry entry = (Map.Entry) i.next();
				((JunElementalNode) entry.getKey()).origin_((Point) entry.getValue());
			}
		} else if (this.isNoneArrangement()) {
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				arcs[i].firstPointAlignment_($("center"));
				arcs[i].lastPointAlignment_($("center"));
			}
			JunAttributeTable aTable = this.noneArrangementTable();
			JunElementalNode[] nodes = this.nodes();
			for (int i = 0; i < nodes.length; i++) {
				if (aTable.includesKey_(nodes[i].uniqueNumber())) {
					nodes[i].locationPoint_((Point) aTable.at_(nodes[i].uniqueNumber()));
				}
			}
		}

		this.flushBoundingBox();
	}

	/**
	 * Arrange the graph as concentric circle.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category arranging
	 */
	public JunAttributeTable arrangeConcentricCircles() {
		return this.arrangeConcentricCirclesAttributeSymbol_($("cost"));
	}

	/**
	 * Arrange the graph as concentric circle.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category arranging
	 */
	public JunAttributeTable arrangeConcentricCirclesAttributeSymbol_(StSymbol aSymbol) {
		JunAttributeTable attributeTable = new JunAttributeTable();
		HashMap nodeCenterTable = new HashMap();
		attributeTable.at_put_($("nodeCenterTable"), nodeCenterTable);

		Object[] anAssociation = this.normalizedDistancesAttributeSymbol_(aSymbol);
		// Number aScalar = (Number) anAssociation[0];
		Object[] anArray = ((Collection) anAssociation[1]).toArray();
		if (anArray.length == 0) {
			this.arrangeConcentricRectangles_(null);
			return attributeTable;
		}

		ArrayList concentricRectangles = new ArrayList(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			int degrees = (int) (360 * this.arrangeConcentricSpiral() * i / Math.max(anArray.length - 1, 1));

			Object[] association = (Object[]) anArray[i];
			double distance = ((Number) association[0]).doubleValue() * this.arrangeConcentricRadius();

			Object[] nodes = ((Collection) association[1]).toArray();
			for (int n = 0; n < nodes.length; n++) {
				double radians = Math.toRadians(degrees + (360 * n / nodes.length));
				int x = (int) Math.round(Math.cos(radians) * distance);
				int y = (int) Math.round(Math.sin(radians) * distance);
				nodeCenterTable.put(nodes[n], new Point(x, y));
			}
			StRectangle rectangle = new StRectangle(0, 0, 0, 0).expandedBy_((int) distance);
			concentricRectangles.add(rectangle);
		}

		this.arrangeConcentricRectangles_(concentricRectangles);
		return attributeTable;
	}

	/**
	 * Arrange the graph as tree.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category arranging
	 */
	public JunAttributeTable arrangeTree() {
		return this.arrangeForest();
	}

	/**
	 * Arrange the graph as forest.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @category arranging
	 */
	public JunAttributeTable arrangeForest() {
		HashMap visitingTable = new HashMap();
		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			visitingTable.put(nodes[i], $("unvisited"));
		}

		JunAttributeTable attributeTable = new JunAttributeTable();
		attributeTable.at_put_($("nodeOriginTable"), new HashMap());
		attributeTable.at_put_($("visitingTable"), visitingTable);

		Point origin = new Point(0, 0);
		JunElementalNode[] roots = this.roots();
		for (int i = 0; i < roots.length; i++) {
			if (visitingTable.get(roots[i]) == $("unvisited")) {
				Dimension extent = this.arrangeForest_startNode_originPoint_(attributeTable, roots[i], origin);
				origin = new Point(origin.x, extent.height + this.arrangeForestMargin());
			}
		}
		for (int i = 0; i < nodes.length; i++) {
			if (visitingTable.get(nodes[i]) == $("unvisited")) {
				Dimension extent = this.arrangeForest_startNode_originPoint_(attributeTable, nodes[i], origin);
				origin = new Point(origin.x, extent.height + this.arrangeForestMargin());
			}
		}

		return attributeTable;
	}

	/**
	 * Arrange the subgraph as forest.
	 * 
	 * @param attributeTable jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param aPoint java.awt.Point
	 * @return java.awt.Dimension
	 * @category arranging
	 */
	protected Dimension arrangeForest_startNode_originPoint_(JunAttributeTable attributeTable, JunElementalNode aNode, Point aPoint) {
		Map visitingTable = (Map) attributeTable.at_($("visitingTable"));
		Map nodeOriginTable = (Map) attributeTable.at_($("nodeOriginTable"));

		visitingTable.put(aNode, $("visited"));
		nodeOriginTable.put(aNode, aPoint);

		JunElementalArc[] arcs = this.arcsOf_(aNode);
		if (arcs.length == 0) {
			return new Dimension(aPoint.x + aNode.width(), aPoint.y + aNode.height());
		}

		int width = aPoint.x + aNode.width();
		int height = aPoint.y;
		int x = width + this.arrangeForestInterval();
		int y = height;
		int min = y;
		for (int i = 0; i < arcs.length; i++) {
			JunElementalNode node = arcs[i].lastNode();
			if (visitingTable.get(node) == $("unvisited")) {
				Dimension extent = this.arrangeForest_startNode_originPoint_(attributeTable, node, new Point(x, y));
				y = Math.max(extent.height, y + node.height());
				width = Math.max(extent.width, width);
				height = Math.max(y, height);
				y += this.arrangeForestMargin();
			}
		}
		y -= this.arrangeForestMargin();
		if (y > aPoint.y + aNode.height()) {
			y = Math.max(min + (y - min - aNode.height()) / 2, min);
		} else {
			y = min;
		}
		nodeOriginTable.put(aNode, new Point(aPoint.x, y));
		height = Math.max(height, y + aNode.height());
		return new Dimension(width, height);
	}

	/**
	 * Arrange the graph with animation.
	 * 
	 * @param aGrapher jp.co.sra.jun.topology.graph.JunGrapher
	 * @category arranging
	 */
	public void arrangeWithAnimation_(JunGrapher aGrapher) {
		this.arrangeConcentricRectangles_(null);
		JunElementalNode aNode = aGrapher.selection();
		if (this.isConcentricArrangement()) {
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				arcs[i].firstPointAlignment_($("center"));
				arcs[i].lastPointAlignment_($("center"));
			}
			JunAttributeTable attributeTable = this.arrangeConcentricCirclesAttributeSymbol_(this.arrangeAttributeSymbol());
			if (aNode == null || aNode.center().equals(new Point(0, 0)) == false) {
				Point oldPoint = aGrapher.scrollAmount();
				Point newPoint = aGrapher.viewBounds().center();
				int howMany = 25;
				HashMap aMap = new HashMap(attributeTable.size());
				Iterator iterator = ((Map) attributeTable.at_($("nodeCenterTable"))).entrySet().iterator();
				while (iterator.hasNext()) {
					Map.Entry entry = (Map.Entry) iterator.next();
					JunElementalNode node = (JunElementalNode) entry.getKey();
					Point center = (Point) entry.getValue();
					ArrayList aList = new ArrayList(howMany + 1);
					for (int i = 0; i <= howMany; i++) {
						int x = node.center().x + (center.x - node.center().x) * i / howMany;
						int y = node.center().y + (center.y - node.center().y) * i / howMany;
						aList.add(new Point(x, y));
					}
					aMap.put(node, aList);
				}
				JunElementalNode[] nodes = this.nodes();
				for (int n = 0; n <= howMany; n++) {
					for (int i = 0; i < nodes.length; i++) {
						nodes[i].center_((Point) ((ArrayList) aMap.get(nodes[i])).get(n));
					}
					int x = oldPoint.x + (newPoint.x - oldPoint.x) * n / howMany;
					int y = oldPoint.y + (newPoint.y - oldPoint.y) * n / howMany;
					aGrapher.scrollTo_(new Point(x, y));
					// aGrapher.redisplay();

					try {
						Thread.sleep(40);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
				aGrapher.flushBoundingBox();
			}
			Iterator iterator = ((Map) attributeTable.at_($("nodeCenterTable"))).entrySet().iterator();
			while (iterator.hasNext()) {
				Map.Entry entry = (Map.Entry) iterator.next();
				JunElementalNode node = (JunElementalNode) entry.getKey();
				Point center = (Point) entry.getValue();
				node.center_(center);
			}
		} else if (this.isHierarchicalArrangement()) {
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				arcs[i].firstPointAlignment_($("rightCenter"));
				arcs[i].lastPointAlignment_($("leftCenter"));
			}
			JunAttributeTable attributeTable = this.arrangeForest();
			Iterator iterator = ((Map) attributeTable.at_($("nodeOriginTable"))).entrySet().iterator();
			while (iterator.hasNext()) {
				Map.Entry entry = (Map.Entry) iterator.next();
				JunElementalNode node = (JunElementalNode) entry.getKey();
				Point origin = (Point) entry.getValue();
				node.origin_(origin);
			}
		} else if (this.isNoneArrangement()) {
			JunElementalArc[] arcs = this.arcs();
			for (int i = 0; i < arcs.length; i++) {
				arcs[i].firstPointAlignment_($("center"));
				arcs[i].lastPointAlignment_($("center"));
			}
			JunAttributeTable attributeTable = this.noneArrangementTable();
			JunElementalNode[] nodes = this.nodes();
			for (int i = 0; i < nodes.length; i++) {
				if (attributeTable.includesKey_(nodes[i].uniqueNumber())) {
					nodes[i].locationPoint_((Point) attributeTable.at_(nodes[i].uniqueNumber()));
				}
			}
		}

		if (aNode != null) {
			Point oldPoint = aGrapher.scrollAmount();
			Point newPoint = aGrapher.viewBounds().center();
			newPoint.translate(-aNode.center().x, -aNode.center().y);
			if (oldPoint.equals(newPoint) == false) {
				int howMany = 25;
				for (int n = 0; n <= howMany; n++) {
					int x = oldPoint.x + (newPoint.x - oldPoint.x) * n / howMany;
					int y = oldPoint.y + (newPoint.y - oldPoint.y) * n / howMany;
					aGrapher.scrollTo_(new Point(x, y));

					try {
						Thread.sleep(40);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
			}
		}

		this.flushBoundingBox();
	}

	/**
	 * Flush the cached information.
	 * 
	 * @category flushing
	 */
	public void flushCachedAll() {
		this.flushCachedNodes();
		this.flushCachedArcs();
	}

	/**
	 * Flush the cached nodes.
	 * 
	 * @category flushing
	 */
	protected void flushCachedNodes() {
		cachedNodes = null;
		cachedNodeTable = null;
		cachedRoots = null;
	}

	/**
	 * Flush the cached arcs.
	 * 
	 * @category flushing
	 */
	protected void flushCachedArcs() {
		cachedArcs = null;
		cachedArcTable = null;
		cachedNodeToArcs = null;
	}

	/**
	 * Flush the bounding box.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#flushBoundingBox()
	 * @category flushing
	 */
	public void flushBoundingBox() {
		boundingBox = null;
	}

	/**
	 * Answer the default attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#defaultAttributes()
	 * @category defaults
	 */
	protected JunAttributeTable defaultAttributes() {
		return DefaultAttributes;
	}

	/**
	 * Answer the default arrange format.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	protected StSymbol defaultArrangeFormat() {
		StSymbol aSymbol = (StSymbol) this.defaultAttributes().at_($("arrangeFormat"));
		if (aSymbol == null) {
			aSymbol = DefaultArrangeFormat();
			this.defaultAttributes().at_put_($("arrangeFormat"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Answer the default arrange attribute symbol.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	protected StSymbol defaultArrangeAttributeSymbol() {
		StSymbol aSymbol = (StSymbol) this.defaultAttributes().at_($("arrangeAttributeSymbol"));
		if (aSymbol == null) {
			aSymbol = DefaultArrangeAttributeSymbol();
			this.defaultAttributes().at_put_($("arrangeAttributeSymbol"), aSymbol);
		}
		return aSymbol;
	}

	/**
	 * Answer the default arrange concentric radius.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultArrangeConcentricRadius() {
		Number aNumber = (Number) this.defaultAttributes().at_($("arrangeConcentricRadius"));
		if (aNumber == null) {
			aNumber = new Integer(DefaultArrangeConcentricRadius());
			this.defaultAttributes().at_put_($("arrangeConcentricRadius"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Answer the default arrange concentric spiral.
	 * 
	 * @return float
	 * @category defaults
	 */
	protected float defaultArrangeConcentricSpiral() {
		Number aNumber = (Number) this.defaultAttributes().at_($("arrangeConcentricSpiral"));
		if (aNumber == null) {
			aNumber = new Float(DefaultArrangeConcentricSpiral());
			this.defaultAttributes().at_put_($("arrangeConcentricSpiral"), aNumber);
		}
		return aNumber.floatValue();
	}

	/**
	 * Answer the default arrange concentric gradation.
	 * 
	 * @return Boolean
	 * @category defaults
	 */
	protected Boolean defaultArrangeConcentricGradation() {
		Boolean aBoolean = (Boolean) this.defaultAttributes().at_($("arrangeConcentricGradation"));
		if (aBoolean == null) {
			aBoolean = DefaultArrangeConcentricGradation();
			this.defaultAttributes().at_put_($("arrangeConcentricGradation"), aBoolean);
		}
		return aBoolean;
	}

	/**
	 * Answer the default arrange forest margin.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultArrangeForestMargin() {
		Number aNumber = (Number) this.defaultAttributes().at_($("arrangeForestMargin"));
		if (aNumber == null) {
			aNumber = new Integer(DefaultArrangeForestMargin());
			this.defaultAttributes().at_put_($("arrangeForestMargin"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Answer the default arrange forest interval.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultArrangeForestInterval() {
		Number aNumber = (Number) this.defaultAttributes().at_($("arrangeForestInterval"));
		if (aNumber == null) {
			aNumber = new Integer(DefaultArrangeForestInterval());
			this.defaultAttributes().at_put_($("arrangeForestInterval"), aNumber);
		}
		return aNumber.intValue();
	}

	/**
	 * Answer true if the receiver's current arrange format is none, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNoneArrangement() {
		return this.arrangeFormat() == $("none");
	}

	/**
	 * Answer true if the receiver's current arrange format is concentric, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConcentricArrangement() {
		return this.arrangeFormat() == $("concentric");
	}

	/**
	 * Answer true if the receiver's current arrange format is forest, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isForestArrangement() {
		return this.arrangeFormat() == $("forest");
	}

	/**
	 * Answer true if the receiver's current arrange format is forest, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTreeArrangement() {
		return this.arrangeFormat() == $("tree");
	}

	/**
	 * Answer true if the receiver's current arrange format is hierarchical, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isHierarchicalArrangement() {
		return this.isForestArrangement() || this.isTreeArrangement();
	}

	/**
	 * Answer true if the receiver has no entities, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return adjacencyMatrix.isEmpty();
	}

	/**
	 * Answer true if the receiver is isomorphic to the specified graph, otherwise false.
	 * 
	 * @param anotherGraph jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @return boolean
	 * @category testing
	 */
	public boolean isIsomorphicTo_(JunElementalGraph anotherGraph) {
		if (anotherGraph == null) {
			return false;
		}
		if (this.isEmpty() && anotherGraph.isEmpty()) {
			return true;
		}
		if (this.numberOfNodes() != anotherGraph.numberOfNodes()) {
			return false;
		}
		if (this.numberOfArcs() != anotherGraph.numberOfArcs()) {
			return false;
		}
		if (this.numberOfRoots() != anotherGraph.numberOfRoots()) {
			return false;
		}
		if (this.numberOfArcs() == 0 && anotherGraph.numberOfArcs() == 0) {
			return true;
		}

		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			JunElementalNode[] anotherNodes = anotherGraph.nodes();
			for (int j = 0; j < anotherNodes.length; j++) {
				if (this.downArcsOfNode_(nodes[i]).length == anotherGraph.downArcsOfNode_(anotherNodes[j]).length && this.upArcsOfNode_(nodes[i]).length == anotherGraph.upArcsOfNode_(anotherNodes[j]).length) {
					if (this.privateIsomorphicTo_with_and_(anotherGraph, nodes[i], anotherNodes[j])) {
						return true;
					}
				}
			}
		}

		return false;
	}

	/**
	 * Answer true if the receiver has one or more selections, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasSelections() {
		return selectedNodes.size() > 0;
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		Graphics2D aGraphics2D = (Graphics2D) aGraphics;
		aGraphics2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		Rectangle clipBounds = aGraphics2D.getClipBounds();
		if (clipBounds != null) {
			aGraphics2D.clearRect(clipBounds.x, clipBounds.y, clipBounds.width, clipBounds.height);
		}

		if (this.isConcentricArrangement()) {
			ArrayList boxCollection = this.arrangeConcentricRectangles();
			StRectangle[] boxes = new StRectangle[boxCollection.size()];
			for (int i = 0; i < boxes.length; i++) {
				boxes[boxes.length - i - 1] = ((StRectangle) boxCollection.get(i)).translatedBy_(aPoint.x, aPoint.y);
			}
			for (int i = 0; i < boxes.length; i++) {
				if (boxes[i].area() > 0) {
					Color color = StColorValue.Brightness_(1 - (i + 1.0) / boxes.length);
					if (this.arrangeConcentricGradation().booleanValue()) {
						aGraphics2D.setColor(StColorValue.Blend(color, Color.white));
						aGraphics2D.fill(new Arc2D.Double(boxes[i].x(), boxes[i].y(), boxes[i].width(), boxes[i].height(), 0, 360, Arc2D.CHORD));
					}
					aGraphics2D.setColor(StColorValue.Blend(color, Color.gray));
					aGraphics2D.setStroke(new BasicStroke(1));
					aGraphics2D.draw(new Ellipse2D.Double(boxes[i].x(), boxes[i].y(), boxes[i].width(), boxes[i].height()));
				}
			}
		}

		JunElementalArc[] arcs = this.arcs();
		for (int i = 0; i < arcs.length; i++) {
			arcs[i].displayOn_at_(aGraphics, aPoint);
		}

		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			nodes[i].displayOn_at_(aGraphics, aPoint);
		}

		aGraphics2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
		JunElementalNode[] selectedNodes = this.selectedNodes();
		for (int i = 0; i < selectedNodes.length; i++) {
			StRectangle displayBox = selectedNodes[i].boundingBox().translatedBy_(aPoint.x, aPoint.y);
			if (clipBounds != null && clipBounds.intersects(displayBox.toRectangle()) == false) {
				continue;
			}

			displayBox = displayBox.insetBy_(this.selectionBorderWidth() - 1);
			aGraphics2D.setColor(this.selectionBorderColor());
			aGraphics2D.setStroke(new BasicStroke(this.selectionBorderWidth()));
			aGraphics2D.draw(new Rectangle2D.Double(displayBox.x(), displayBox.y(), displayBox.width(), displayBox.height()));
		}
	}

	/**
	 * Display the receiver on the graphics at the specified point with the specified scale factor.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param scaleFactor jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category displaying
	 */
	public void displayOn_at_scaledBy_(Graphics aGraphics, Point aPoint, Jun2dPoint scaleFactor) {
		Graphics2D aGraphics2D = (Graphics2D) aGraphics;
		aGraphics2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		Rectangle clipBounds = aGraphics2D.getClipBounds();
		if (clipBounds != null) {
			aGraphics2D.clearRect(clipBounds.x, clipBounds.y, clipBounds.width, clipBounds.height);
		}

		if (this.isConcentricArrangement()) {
			ArrayList boxCollection = this.arrangeConcentricRectangles();
			StRectangle[] boxes = new StRectangle[boxCollection.size()];
			for (int i = 0; i < boxes.length; i++) {
				boxes[boxes.length - i - 1] = ((StRectangle) boxCollection.get(i)).translatedBy_(aPoint.x, aPoint.y).scaledBy_(scaleFactor.x(), scaleFactor.y());
			}
			for (int i = 0; i < boxes.length; i++) {
				if (boxes[i].area() > 0) {
					Color color = StColorValue.Brightness_(1 - (i + 1.0) / boxes.length);
					if (this.arrangeConcentricGradation().booleanValue()) {
						aGraphics2D.setColor(StColorValue.Blend(color, Color.white));
						aGraphics2D.fill(new Arc2D.Double(boxes[i].x(), boxes[i].y(), boxes[i].width(), boxes[i].height(), 0, 360, Arc2D.CHORD));
					}
					aGraphics2D.setColor(StColorValue.Blend(color, Color.gray));
					aGraphics2D.setStroke(new BasicStroke(1));
					aGraphics2D.draw(new Ellipse2D.Double(boxes[i].x(), boxes[i].y(), boxes[i].width(), boxes[i].height()));
				}
			}
		}

		JunElementalArc[] arcs = this.arcs();
		for (int i = 0; i < arcs.length; i++) {
			arcs[i].displayOn_at_scaledBy_(aGraphics, aPoint, scaleFactor);
		}

		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			nodes[i].displayOn_at_scaledBy_(aGraphics, aPoint, scaleFactor);
		}

		aGraphics2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
		JunElementalNode[] selectedNodes = this.selectedNodes();
		for (int i = 0; i < selectedNodes.length; i++) {
			StRectangle displayBox = selectedNodes[i].boundingBox().translatedBy_(aPoint.x, aPoint.y);
			displayBox = displayBox.scaledBy_(scaleFactor.x(), scaleFactor.y());
			if (clipBounds != null && clipBounds.intersects(displayBox.toRectangle()) == false) {
				continue;
			}

			displayBox = displayBox.insetBy_(this.selectionBorderWidth() - 1);
			aGraphics2D.setColor(this.selectionBorderColor());
			aGraphics2D.setStroke(new BasicStroke(this.selectionBorderWidth()));
			aGraphics2D.draw(new Rectangle2D.Double(displayBox.x(), displayBox.y(), displayBox.width(), displayBox.height()));
		}
	}

	/**
	 * Show the graph.
	 * 
	 * @return jp.co.sra.jun.goodies.display.JunDisplayModel
	 * @category viewing
	 */
	public JunDisplayModel show() {
		JunDisplayModel displayModel = new JunDisplayModel(this) {
			public Frame openView_(StView aView) {
				Frame aFrame = this.allButOpenView_(aView);

				Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
				Rectangle bounds = aFrame.getBounds();
				int width = Math.min(bounds.width, screen.width - 100);
				int height = Math.min(bounds.height, screen.height - 100);
				aFrame.setSize(width, height);

				_ShowAtCenterPoint(aFrame);
				return aFrame;
			}

			protected String windowTitle() {
				return JunElementalGraph.this._className() + " " + boundingBox().width() + "x" + boundingBox().height();
			}
		};
		displayModel.open();
		return displayModel;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("graph(");
		aWriter.write("nodes(");
		JunElementalNode[] nodes = this.nodes();
		int size = nodes.length;
		if (size == 0) {
			// no nodes to print.
		} else if (size <= this.printMax()) {
			nodes[0].printOn_(aWriter);
			for (int i = 1; i < size; i++) {
				aWriter.write(',');
				nodes[i].printOn_(aWriter);
			}
		} else {
			int halfsize = this.printMax() / 2;
			nodes[0].printOn_(aWriter);
			for (int i = 1; i < halfsize; i++) {
				aWriter.write(',');
				nodes[i].printOn_(aWriter);
			}
			aWriter.write(" ...");
			nodes[size - halfsize].printOn_(aWriter);
			for (int i = size - halfsize + 1; i < size; i++) {
				aWriter.write(',');
				nodes[i].printOn_(aWriter);
			}
		}
		aWriter.write(')');
		aWriter.write(',');
		aWriter.write("arcs(");
		JunElementalArc[] arcs = this.arcs();
		size = arcs.length;
		if (size == 0) {
			// no nodes to print.
		} else if (size <= this.printMax()) {
			arcs[0].printOn_(aWriter);
			for (int i = 1; i < size; i++) {
				aWriter.write(',');
				arcs[i].printOn_(aWriter);
			}
		} else {
			int halfsize = this.printMax() / 2;
			arcs[0].printOn_(aWriter);
			for (int i = 1; i < halfsize; i++) {
				aWriter.write(',');
				arcs[i].printOn_(aWriter);
			}
			aWriter.write(" ...");
			arcs[size - halfsize].printOn_(aWriter);
			for (int i = size - halfsize + 1; i < size; i++) {
				aWriter.write(',');
				arcs[i].printOn_(aWriter);
			}
		}
		aWriter.write(')');
		aWriter.write(')');
	}

	/**
	 * Answer the maximum number for printing elements.
	 * 
	 * @return int
	 * @category printing
	 */
	protected int printMax() {
		return 100;
	}

	/**
	 * Do something after a shallow copy.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		StObject copiedObject = super.postCopy();

		HashMap uniqueNumberTable = new HashMap();
		for (int i = 0; i < elementalNodes.size(); i++) {
			JunElementalNode node = (JunElementalNode) elementalNodes.get(i);
			StSymbol oldUniqueNumber = node.uniqueNumber();
			StSymbol newUniqueNumber = UniqueNumber();
			uniqueNumberTable.put(oldUniqueNumber, newUniqueNumber);
			node.setUniqueNumber_(newUniqueNumber);
		}
		adjacencyMatrix._updateUniqueNumbers(uniqueNumberTable);
		this.flushCachedAll();

		return copiedObject;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return this._className();
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);

		this.elementalNodesFromLispList(aList);
		this.adjacencyMatrixFromLispList(aList);
		this.selectedNodesFromLispList(aList);
	}

	/**
	 * Convert my attributes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons aList = super.toLispList();
		aList.add_(this.elementalNodesToLispList());
		aList.add_(this.adjacencyMatrixToLispList());
		aList.add_(this.selectedNodesToLispList());
		return aList;
	}

	/**
	 * Get my elemental nodes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void elementalNodesFromLispList(JunLispList aList) {
		JunLispList elementalNodesList = aList._findSublistWithHead($("elementalNodes"));
		if (elementalNodesList == null) {
			return;
		}

		Object[] anArray = ((JunLispList) elementalNodesList.tail()).asArray();
		ArrayList anArrayList = new ArrayList(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			anArrayList.add(DefaultNewNode((JunLispList) anArray[i]));
		}
		this.privateElementalNodes_(anArrayList);
	}

	/**
	 * Convert my elemental nodes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons elementalNodesToLispList() {
		JunLispCons elementalNodesList = new JunLispCons($("elementalNodes"));
		JunElementalNode[] nodes = this.nodes();
		for (int i = 0; i < nodes.length; i++) {
			elementalNodesList.add_(nodes[i].toLispList());
		}
		return elementalNodesList;
	}

	/**
	 * Get my elemental nodes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void adjacencyMatrixFromLispList(JunLispList aList) {
		JunLispList adjacencyMatrixList = aList._findSublistWithHead($("adjacencyMatrix"));
		if (adjacencyMatrixList == null) {
			return;
		}

		this.privateAdjacencyMatrix_(new JunAdjacencyMatrix((JunLispList) adjacencyMatrixList.tail()));
	}

	/**
	 * Convert my elemental nodes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons adjacencyMatrixToLispList() {
		return new JunLispCons($("adjacencyMatrix"), adjacencyMatrix.toLispList());
	}

	/**
	 * Get my selected nodes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void selectedNodesFromLispList(JunLispList aList) {
		JunLispList selectedNodesList = aList._findSublistWithHead($("selectedNodes"));
		if (selectedNodesList == null) {
			return;
		}

		Object[] anArray = ((JunLispList) selectedNodesList.tail()).asArray();
		ArrayList anArrayList = new ArrayList(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			anArrayList.add(this.uniqueNumberAt_((StSymbol) anArray[i]));
		}
		this.selections_(anArrayList);
	}

	/**
	 * Convert my selected nodes to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons selectedNodesToLispList() {
		JunLispCons selectedNodesList = new JunLispCons($("selectedNodes"));

		JunElementalNode[] selectedNodes = this.selectedNodes();
		for (int i = 0; i < selectedNodes.length; i++) {
			selectedNodesList.add_(selectedNodes[i].uniqueNumber());
		}
		return selectedNodesList;
	}

	/**
	 * Set the elemental nodes.
	 * 
	 * @param privateElementalNodes java.util.ArrayList
	 * @category private
	 */
	protected void privateElementalNodes_(ArrayList privateElementalNodes) {
		elementalNodes = privateElementalNodes;
		this.flushCachedAll();
	}

	/**
	 * Set the adjacency matrix.
	 * 
	 * @param privateAdjacencyMatrix jp.co.sra.jun.goodies.tables.JunAdjacencyMatrix
	 * @category private
	 */
	protected void privateAdjacencyMatrix_(JunAdjacencyMatrix privateAdjacencyMatrix) {
		adjacencyMatrix = privateAdjacencyMatrix;
		this.flushCachedAll();
	}

	/**
	 * Private method to check the two matrixes are isomorphic.
	 * 
	 * @param anotherGraph jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param anotherNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return boolean
	 * @category private
	 */
	protected boolean privateIsomorphicTo_with_and_(JunElementalGraph anotherGraph, JunElementalNode aNode, JunElementalNode anotherNode) {
		final Collection aCollection = new ArrayList(this.numberOfNodes());
		JunAttributeTable aTable = this.visitDepthFirst_nodeDo_arcDo_(aNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				aCollection.add(indent);
				return null;
			}
		}, new StBlockClosure());
		final Collection anotherCollection = new ArrayList(anotherGraph.numberOfNodes());
		JunAttributeTable anotherTable = anotherGraph.visitDepthFirst_nodeDo_arcDo_(anotherNode, new StBlockClosure() {
			public Object value_value_value_(Object node, Object indent, Object sequence) {
				anotherCollection.add(indent);
				return null;
			}
		}, new StBlockClosure());
		if (aCollection.equals(anotherCollection) == false) {
			return false;
		}

		aCollection.clear();
		Iterator iterator = ((Map) aTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("unvisited")) {
				aCollection.add(entry);
			}
		}
		anotherCollection.clear();
		iterator = ((Map) anotherTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("unvisited")) {
				anotherCollection.add(entry);
			}
		}
		if (aCollection.size() != anotherCollection.size()) {
			return false;
		}

		aCollection.clear();
		iterator = ((Map) aTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("visited")) {
				aCollection.add(entry);
			}
		}
		anotherCollection.clear();
		iterator = ((Map) anotherTable.at_($("visitingTable"))).entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			if (entry.getValue() == $("visited")) {
				anotherCollection.add(entry);
			}
		}

		JunElementalGraph aCopiedGraph = (JunElementalGraph) this.copy();
		HashMap aMap = new HashMap();
		for (int i = 0; i < elementalNodes.size(); i++) {
			aMap.put(elementalNodes.get(i), aCopiedGraph.elementalNodes.get(i));
		}
		JunElementalGraph anotherCopiedGraph = (JunElementalGraph) anotherGraph.copy();
		HashMap anotherMap = new HashMap();
		for (int i = 0; i < anotherGraph.elementalNodes.size(); i++) {
			anotherMap.put(anotherGraph.elementalNodes.get(i), anotherCopiedGraph.elementalNodes.get(i));
		}
		iterator = aCollection.iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			aCopiedGraph.remove_((JunElementalNode) aMap.get(entry.getKey()));
		}
		iterator = anotherCollection.iterator();
		while (iterator.hasNext()) {
			Map.Entry entry = (Map.Entry) iterator.next();
			anotherCopiedGraph.remove_((JunElementalNode) anotherMap.get(entry.getKey()));
		}
		return aCopiedGraph.isIsomorphicTo_(anotherCopiedGraph);
	}

}
