package jp.co.sra.jun.topology.grapher;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.topology.graph.JunElementalArc;
import jp.co.sra.jun.topology.graph.JunElementalArcSettings;
import jp.co.sra.jun.topology.graph.JunElementalGraph;
import jp.co.sra.jun.topology.graph.JunElementalGraphSettings;
import jp.co.sra.jun.topology.graph.JunElementalNode;
import jp.co.sra.jun.topology.graph.JunElementalNodeSettings;

/**
 * JunGrapher class
 * 
 *  @author    nisinaka
 *  @created   2006/04/14 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGrapher.java,v 8.12 2008/02/20 06:33:13 nisinaka Exp $
 */
public class JunGrapher extends JunApplicationModel {

	protected JunElementalGraph elementalGraph;
	protected StRectangle focusBoundingBox;
	protected StValueHolder selectedNodesHolder;
	protected StPopupMenu graphMenu;
	protected StPopupMenu nodeMenu;
	protected JunElementalGraphSettings graphSettings;
	protected JunElementalNodeSettings nodeSettings;
	protected JunElementalArcSettings arcSettings;
	private Point _popupMenuPoint;

	/**
	 * Create a new instance of the default grapher.
	 * 
	 * @param aGraph jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @return jp.co.sra.jun.topology.graph.JunElementalGrapher
	 * @category Defaults
	 */
	public static JunGrapher DefaultNewGrapher(JunElementalGraph aGraph) {
		return new JunGrapher(aGraph);
	}

	/**
	 * Create a new instance of JunGrapher and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunGrapher() {
		super();
	}

	/**
	 * Create a new instance of JunGrapher and initialize it.
	 *
	 * @param aGraph jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category Instance creation
	 */
	public JunGrapher(JunElementalGraph aGraph) {
		this.graph_(aGraph);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		elementalGraph = null;
		focusBoundingBox = null;
		selectedNodesHolder = null;
		graphMenu = null;
		nodeMenu = null;
		graphSettings = null;
		nodeSettings = null;
		arcSettings = null;
	}

	/**
	 * Answer my current graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category accessing
	 */
	public JunElementalGraph graph() {
		if (elementalGraph == null) {
			elementalGraph = this.defaultGraph();
		}
		return elementalGraph;
	}

	/**
	 * Set my new graph.
	 * 
	 * @param aGraph jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category accessing
	 */
	public void graph_(JunElementalGraph aGraph) {
		elementalGraph = aGraph;

		this.selectedNodesHolder().value_(aGraph.selections());
		this.changed_($("graph"));

		this.graphSettings().closeRequest();
		this.nodeSettings().closeRequest();
		this.arcSettings().closeRequest();
		graphSettings = null;
		nodeSettings = null;
		arcSettings = null;
	}

	/**
	 * Answer my current focus bounding box.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle focusBoundingBox() {
		if (focusBoundingBox == null) {
			JunGrapherView aView = (JunGrapherView) this.getView();
			if (aView == null) {
				return this.boundingBox();
			}

			StRectangle aBox = new StRectangle(aView.toComponent().getBounds());
			focusBoundingBox = StRectangle.Origin_corner_(aView.convertViewPointToModelPoint_(aBox.origin()), aView.convertViewPointToModelPoint_(aBox.corner()));
		}
		return focusBoundingBox;
	}

	/**
	 * Set my new focus bounding box.
	 * 
	 * @param aRectangle jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public void focusBoundingBox_(StRectangle aRectangle) {
		focusBoundingBox = aRectangle;
	}

	/**
	 * Answer the bounding box of the graph.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle boundingBox() {
		return this.graph().boundingBox();
	}

	/**
	 * Answer the graph settings.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraphSettings
	 * @category aspects
	 */
	protected JunElementalGraphSettings graphSettings() {
		if (graphSettings == null) {
			graphSettings = this.defaultGraphSettings();
			graphSettings.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					flushAll();
					arrangeGrapher();
					updateGrapher();
					return null;
				}
			});
		}
		return graphSettings;
	}

	/**
	 * Answer the graph settings for the specified graph.
	 * 
	 * @param aGraph jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraphSettings
	 * @category aspects
	 */
	protected JunElementalGraphSettings graphSettingsFor_(JunElementalGraph aGraph) {
		if (aGraph == null) {
			return this.defaultGraphSettings();
		}

		JunElementalGraphSettings graphSettings = this.graphSettings();
		graphSettings.stuff_(aGraph);
		return graphSettings;
	}

	/**
	 * Answer the node settings.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNodeSettings
	 * @category aspects
	 */
	protected JunElementalNodeSettings nodeSettings() {
		if (nodeSettings == null) {
			nodeSettings = this.defaultNodeSettings();
			nodeSettings.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					updateGrapher();
					return null;
				}
			});
		}
		return nodeSettings;
	}

	/**
	 * Answer the node settings for the specified node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.jun.topology.graph.JunElementalNodeSettings
	 * @category aspects
	 */
	protected JunElementalNodeSettings nodeSettingsFor_(JunElementalNode aNode) {
		if (aNode == null) {
			return this.defaultNodeSettings();
		}

		JunElementalNodeSettings nodeSettings = this.nodeSettings();
		nodeSettings.stuff_(aNode);
		return nodeSettings;
	}

	/**
	 * Answer the arc settings.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalArcSettings
	 * @category aspects
	 */
	protected JunElementalArcSettings arcSettings() {
		if (arcSettings == null) {
			arcSettings = this.defaultArcSettings();
			arcSettings.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					updateGrapher();
					return null;
				}
			});
		}
		return arcSettings;
	}

	/**
	 * Answer the arc settings for the specified arc.
	 * 
	 * @param anArc jp.co.sra.jun.topology.graph.JunElementalArc
	 * @return jp.co.sra.jun.topology.graph.JunElementalArcSettings
	 * @category aspects
	 */
	protected JunElementalArcSettings arcSettingsFor_(JunElementalArc anArc) {
		if (anArc == null) {
			return this.defaultArcSettings();
		}

		JunElementalArcSettings arcSettings = this.arcSettings();
		arcSettings.stuff_(anArc);
		return arcSettings;
	}

	/**
	 * Flush all.
	 * 
	 * @category flushing
	 */
	public void flushAll() {
		this.graph().flushBoundingBox();
		this.graph().flushCachedAll();
		JunElementalNode[] nodes = this.graph().nodes();
		for (int i = 0; i < nodes.length; i++) {
			nodes[i].flushVisualObject();
		}
	}

	/**
	 * Flush the bounding box.
	 * 
	 * @category flushing 
	 */
	public void flushBoundingBox() {
		this.graph().flushBoundingBox();
	}

	/**
	 * Create an instance of my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunGrapherViewAwt(this);
		} else {
			return new JunGrapherViewSwing(this);
		}
	}

	/**
	 * Create an instance of my default graph.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraph
	 * @category defaults
	 */
	protected JunElementalGraph defaultGraph() {
		return new JunElementalGraph();
	}

	/**
	 * Create an intsance of my defualt graph settings.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalGraphSettings
	 * @category defaults
	 */
	protected JunElementalGraphSettings defaultGraphSettings() {
		return new JunElementalGraphSettings();
	}

	/**
	 * Create an instance of my default node.
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category defaults
	 */
	protected JunElementalNode defaultNode(String aString) {
		return new JunElementalNode(aString);
	}

	/**
	 * Create an intsance of my defualt node settings.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNodeSettings
	 * @category defaults
	 */
	protected JunElementalNodeSettings defaultNodeSettings() {
		return new JunElementalNodeSettings();
	}

	/**
	 * Create an intsance of my defualt arc settings.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalArcSettings
	 * @category defaults
	 */
	protected JunElementalArcSettings defaultArcSettings() {
		return new JunElementalArcSettings();
	}

	/**
	 * Answer true if the receiver's current arrange format is none, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNoneArrangement() {
		return this.graph().isNoneArrangement();
	}

	/**
	 * Answer true if the receiver's current arrange format is concentric, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConcentricArrangement() {
		return this.graph().isConcentricArrangement();
	}

	/**
	 * Answer true if the receiver's current arrange format is forest, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isForestArrangement() {
		return this.graph().isForestArrangement();
	}

	/**
	 * Answer true if the receiver's current arrange format is forest, otherwise fasle. 
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTreeArrangement() {
		return this.graph().isTreeArrangement();
	}

	/**
	 * Answer true if the receiver's current arrange format is hierarchical, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isHierarchicalArrangement() {
		return this.isForestArrangement() || this.isTreeArrangement();
	}

	/**
	 * Answer true if the node has one or more connections with the nodes in the collection.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param nodeCollection java.util.Collection
	 * @return boolean
	 * @category testing
	 */
	public boolean hasConnections_with_(JunElementalNode aNode, Collection nodeCollection) {
		JunElementalArc[] arcs = this.graph().arcsOf_(aNode);
		for (int i = 0; i < arcs.length; i++) {
			if (nodeCollection.contains(arcs[i].lastNode())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Answer true if the node has connections with the all nodes in the collection.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param nodeCollection java.util.Collection
	 * @return boolean
	 * @category testing
	 */
	public boolean hasAllConnections_with_(JunElementalNode aNode, Collection nodeCollection) {
		JunElementalArc[] arcs = this.graph().arcsOf_(aNode);
		if (arcs.length == 0) {
			return false;
		}

		for (int i = 0; i < arcs.length; i++) {
			if (nodeCollection.contains(arcs[i].lastNode()) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Fit the viewpoint when a view is opened.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StUIBulider#windowOpened(WindowEvent)
	 * @see jp.co.sra.smalltalk.StApplicationModel#postOpenWith_(jp.co.sra.smalltalk.StView)
	 * @category interface opening
	 */
	public void postOpenWith_(StView aView) {
		super.postOpenWith_(aView);
		this.fitViewpoint();
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Grapher");
	}

	/**
	 * Answer whether the view is placed on a bordered panel or not.
	 * 
	 * @return boolean
	 * @see jp.co.sra.smalltalk.StApplicationModel#_viewOnBorderedPanel()
	 * @category interface opening
	 */
	protected boolean _viewOnBorderedPanel() {
		return true;
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param evt java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent evt) {
		if (graphSettings != null) {
			graphSettings.closeRequest();
		}
		if (nodeSettings != null) {
			nodeSettings.closeRequest();
		}
		if (arcSettings != null) {
			arcSettings.closeRequest();
		}

		super.noticeOfWindowClose(evt);
	}

	/**
	 * Answer the popup menu at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	public StPopupMenu _popupMenuAt(Point aPoint) {
		StPopupMenu aPopupMenu = null;

		_popupMenuPoint = aPoint;
		JunElementalNode aNode = this.whichAt_(aPoint);
		if (aNode == null) {
			aPopupMenu = this.graphMenu();
		} else {
			aPopupMenu = this.nodeMenuFor_(aNode);
		}

		return aPopupMenu;
	}

	/**
	 * Answer my popup menu for the graph.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu graphMenu() {
		if (graphMenu == null) {
			graphMenu = new StPopupMenu();
			graphMenu.add(new StMenuItem($String("New node"), new MenuPerformer(this, "newNode")));
			graphMenu.add(new StMenuItem($String("Search node..."), new MenuPerformer(this, "searchNode")));
			graphMenu.add(new StMenuItem($String("Select all nodes"), new MenuPerformer(this, "selectAllNodes")));
			graphMenu.add(new StMenuItem($String("Remove selected nodes"), $("RemoveNodes"), new MenuPerformer(this, "removeNodes")));
			graphMenu.addSeparator();
			graphMenu.add(new StMenuItem($String("Graph settings"), new MenuPerformer(this, "inspectGraphAttributes")));
			graphMenu.add(new StMenuItem($String("Arrange"), new MenuPerformer(this, "arrangeGrapher")));
			graphMenu.add(new StMenuItem($String("Update"), new MenuPerformer(this, "updateGrapher")));
		}
		this.updateGraphMenuIndication();
		return graphMenu;
	}

	/**
	 * Answer my popup menu for the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	protected StPopupMenu nodeMenuFor_(JunElementalNode aNode) {
		if (nodeMenu == null) {
			nodeMenu = new StPopupMenu();
			nodeMenu.add(new StMenuItem($String("Node settings"), new MenuPerformer(this, "inspectNodeAttributes")));
			nodeMenu.add(new StMenuItem($String("Centering"), new MenuPerformer(this, "centeringNode")));
			nodeMenu.add(new StMenu($String("Neighbor nodes"), $("NeighborNodes")));
			nodeMenu.add(new StMenuItem($String("Remove"), new MenuPerformer(this, "removeNode")));
			nodeMenu.addSeparator();
			nodeMenu.add(new StMenu($String("Arc settings"), $("ArcSettings")));
			nodeMenu.add(new StMenuItem($String("Connect with selections"), $("ConnectWithSelections"), new MenuPerformer(this, "connectWithSelections")));
			nodeMenu.add(new StMenuItem($String("Disconnect with selections"), $("DisconnectWithSelections"), new MenuPerformer(this, "disconnectWithSelections")));
		}
		this.updateNodeMenuIndicationFor_(aNode);
		return nodeMenu;
	}

	/**
	 * Update the menu indication.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();
		this.updateGraphMenuIndication();
		this.updateNodeMenuIndication();
	}

	/**
	 * Update the graph menu indication.
	 * 
	 * @category menu accessing
	 */
	protected void updateGraphMenuIndication() {
		if (graphMenu == null) {
			return;
		}

		StMenuItem menuItem = graphMenu.atNameKey_($("RemoveNodes"));
		if (menuItem != null) {
			if (this.selections().isEmpty()) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}
	}

	/**
	 * Update the node menu indication.
	 * 
	 * @category menu accessing 
	 */
	protected void updateNodeMenuIndication() {
		this.updateNodeMenuIndicationFor_(null);
	}

	/**
	 * Update the node menu indication for the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category menu accessing
	 */
	protected void updateNodeMenuIndicationFor_(JunElementalNode aNode) {
		if (nodeMenu == null) {
			return;
		}

		JunElementalArc[] downArcs = this.graph().downArcsOfNode_(aNode);
		JunElementalNode[] downNodes = new JunElementalNode[downArcs.length];
		for (int i = 0; i < downNodes.length; i++) {
			downNodes[i] = downArcs[i].lastNode();
		}
		JunElementalArc[] upArcs = this.graph().upArcsOfNode_(aNode);
		JunElementalNode[] upNodes = new JunElementalNode[upArcs.length];
		for (int i = 0; i < upNodes.length; i++) {
			upNodes[i] = upArcs[i].firstNode();
		}

		StMenu menu = (StMenu) nodeMenu.atNameKey_($("NeighborNodes"));
		if (menu != null) {
			menu.removeAll();
			if (aNode == null || downNodes.length + upNodes.length == 0) {
				menu.disable();
			} else {
				menu.enable();

				for (int i = 0; i < upNodes.length; i++) {
					menu.add(new StMenuItem(upNodes[i].labelString(), new MenuPerformer(this, "selectAndScrollFor_", upNodes[i])));
				}
				menu.addSeparator();
				for (int i = 0; i < downNodes.length; i++) {
					menu.add(new StMenuItem(downNodes[i].labelString(), new MenuPerformer(this, "selectAndScrollFor_", downNodes[i])));
				}
			}
		}

		menu = (StMenu) nodeMenu.atNameKey_($("ArcSettings"));
		if (menu != null) {
			menu.removeAll();
			if (aNode == null || downArcs.length + upArcs.length == 0) {
				menu.disable();
			} else {
				menu.enable();

				for (int i = 0; i < upArcs.length; i++) {
					menu.add(new StMenuItem("<- " + upArcs[i].firstNode().labelString(), new MenuPerformer(this, "inspectArcAttributes_", upArcs[i])));
				}
				menu.addSeparator();
				for (int i = 0; i < downArcs.length; i++) {
					menu.add(new StMenuItem("-> " + downArcs[i].lastNode().labelString(), new MenuPerformer(this, "inspectArcAttributes_", downArcs[i])));
				}
			}
		}

		StMenuItem menuItem = nodeMenu.atNameKey_($("ConnectWithSelections"));
		if (menuItem != null) {
			if (this.selections().isEmpty()) {
				menuItem.disable();
			} else if (this.selections().size() == 1 && this.selection() == aNode) {
				menuItem.disable();
			} else if (this.hasAllConnections_with_(aNode, this.selections())) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}

		menuItem = nodeMenu.atNameKey_($("DisconnectWithSelections"));
		if (menuItem != null) {
			if (this.selections().isEmpty()) {
				menuItem.disable();
			} else if (this.selections().size() == 1 && this.selection() == aNode) {
				menuItem.disable();
			} else if (this.hasConnections_with_(aNode, this.selections()) == false) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}
	}

	/**
	 * Arrange the grapher.
	 * 
	 * @category menu messages
	 */
	public void arrangeGrapher() {
		this.graph().arrange();
		this.fitViewpoint();
		this.nodeSettingsFor_((JunElementalNode) this.nodeSettings().stuff());
	}

	/**
	 * Centering the node.
	 * 
	 * @category menu messages
	 */
	public void centeringNode() {
		JunElementalNode aNode = this.whichAt_(_popupMenuPoint);
		if (aNode == null) {
			return;
		}

		this.selectAndArrangeAndScrollFor_(aNode);
	}

	/**
	 * Connect the current node with the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void connectWithSelections() {
		JunElementalNode aNode = this.whichAt_(_popupMenuPoint);
		if (aNode == null) {
			return;
		}

		ArrayList selectedNodes = new ArrayList(this.selections());
		if (selectedNodes.contains(aNode)) {
			selectedNodes.remove(aNode);
		}
		if (selectedNodes.isEmpty()) {
			return;
		}

		for (int i = 0; i < selectedNodes.size(); i++) {
			JunElementalNode eachNode = (JunElementalNode) selectedNodes.get(i);
			this.graph().connect_with_(aNode, eachNode);
		}
		this.updateGrapher();
	}

	/**
	 * Disconnect the current node with the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void disconnectWithSelections() {
		JunElementalNode aNode = this.whichAt_(_popupMenuPoint);
		if (aNode == null) {
			return;
		}

		ArrayList selectedNodes = new ArrayList(this.selections());
		if (selectedNodes.contains(aNode)) {
			selectedNodes.remove(aNode);
		}
		if (selectedNodes.isEmpty()) {
			return;
		}

		for (int i = 0; i < selectedNodes.size(); i++) {
			JunElementalNode eachNode = (JunElementalNode) selectedNodes.get(i);
			this.graph().disconnect_with_(aNode, eachNode);
		}
		this.updateGrapher();
	}

	/**
	 * Connect two nodes specified with the freehand drawing.
	 * 
	 * @param startNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @param endNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category menu messages
	 */
	public void freehandConnect_with_(JunElementalNode startNode, JunElementalNode endNode) {
		if (startNode != null && endNode != null) {
			this.graph().connect_with_(startNode, endNode);
			this.updateGrapher();
		} else {
			this.changed();
		}
	}

	/**
	 * Handle points specified with the freehand drawing.
	 * 
	 * @param points java.awt.Point[]
	 * @param e java.awt.event.MouseEvent
	 * @category menu messages
	 */
	public void freehandPoints_(Point[] points, MouseEvent e) {
		if (points == null || points.length == 0) {
			this.changed();
			return;
		}

		Jun2dBoundingBoxes boundingBoxes = Jun2dBoundingBoxes.FromPolyline_bitSize_(points, new Dimension(32, 32));
		Rectangle[] rectangles = boundingBoxes.asRectangles();
		if (e.isControlDown()) {
			JunGrapherView aView = (JunGrapherView) this.getView();
			Point scrollAmount = aView.scrollAmount();

			Graphics2D aGraphics = null;
			try {
				aGraphics = (Graphics2D) aView.toComponent().getGraphics();
				aGraphics.setColor(Color.orange);
				aGraphics.setStroke(new BasicStroke(1));
				for (int i = 0; i < rectangles.length; i++) {
					int x = rectangles[i].x + scrollAmount.x;
					int y = rectangles[i].y + scrollAmount.y;
					int width = rectangles[i].width;
					int height = rectangles[i].height;
					aGraphics.drawRect(x, y, width, height);
				}

				Thread.sleep(3000);
			} catch (InterruptedException ex) {
				ex.printStackTrace();
			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
				}
			}
		}

		HashSet nodeSet = new HashSet();
		JunElementalNode[] nodes = this.graph().nodes();
		for (int i = 0; i < nodes.length; i++) {
			for (int j = 0; j < rectangles.length; j++) {
				if (rectangles[j].intersects(nodes[i].bounds())) {
					nodeSet.add(nodes[i]);
					break;
				}
			}
		}

		Jun2dBoundingBox boundingBox = boundingBoxes.boundingBox();
		if (nodeSet.isEmpty() && boundingBox.area() >= 64) {
			this.newNodeAt_(boundingBox.center()._toPoint());
		} else {
			this.changed();
			if (e.isShiftDown()) {
				nodeSet.addAll(this.selections());
			}
			this.selections_(nodeSet);
		}
	}

	/**
	 * Open an arc settings.
	 * 
	 * @param anArc jp.co.sra.jun.topology.graph.JunElementalArc
	 * @category menu messages
	 */
	public void inspectArcAttributes_(JunElementalArc anArc) {
		if (anArc == null) {
			return;
		}

		JunElementalArcSettings aModel = this.arcSettingsFor_(anArc);
		aModel._show();
		aModel.updateWindowTitle();
	}

	/**
	 * Open a graph settings.
	 * 
	 * @category menu messages
	 */
	public void inspectGraphAttributes() {
		JunElementalGraph aGraph = this.graph();
		if (aGraph == null) {
			return;
		}

		JunElementalGraphSettings aModel = this.graphSettingsFor_(aGraph);
		aModel._show();
		aModel.updateWindowTitle();
	}

	/**
	 * Open a node settings.
	 * 
	 * @category menu messages
	 */
	public void inspectNodeAttributes() {
		JunElementalNode aNode = this.whichAt_(_popupMenuPoint);
		if (aNode == null) {
			return;
		}

		JunElementalNodeSettings aModel = this.nodeSettingsFor_(aNode);
		aModel._show();
		aModel.updateWindowTitle();
	}

	/**
	 * Create a new node and place it at the popup menu point.
	 * 
	 * @category menu messages
	 */
	public void newNode() {
		Point aPoint = _popupMenuPoint;
		if (aPoint == null) {
			aPoint = new Point(0, 0);
		}

		this.newNodeAt_(aPoint);
	}

	/**
	 * Create a new node at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category menu messages
	 */
	public void newNodeAt_(Point aPoint) {
		String aString = $String("Untitled");
		JunElementalNode aNode = this.defaultNode(aString);
		aNode.center_(aPoint);
		this.graph().add_(aNode);
		this.changed();
	}

	/**
	 * Remove the current node.
	 * 
	 * @category menu messages
	 */
	public void removeNode() {
		JunElementalNode aNode = this.whichAt_(_popupMenuPoint);
		if (aNode == null) {
			return;
		}
		if (JunDialog.Confirm_($String("Really remove '<1p>'?", null, aNode.labelString())) == false) {
			return;
		}

		this.graph().remove_(aNode);
		this.graph().flushCachedAll();
		this.changed_($("graph"));
	}

	/**
	 * Remove the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void removeNodes() {
		if (this.selections().isEmpty()) {
			return;
		}
		if (JunDialog.Confirm_($String("Really remove?")) == false) {
			return;
		}

		JunElementalNode[] selectedNodes = (JunElementalNode[]) this.selections().toArray(new JunElementalNode[this.selections().size()]);
		for (int i = 0; i < selectedNodes.length; i++) {
			this.graph().remove_(selectedNodes[i]);
		}
		this.graph().flushCachedAll();
		this.changed_($("graph"));
	}

	/**
	 * Search for a node.
	 * 
	 * @category menu messages
	 */
	public void searchNode() {
		String aString = JunDialog.Request_($String("Input wild card to search."), "*");
		if (aString == null || aString.length() == 0) {
			return;
		}

		ArrayList aList = new ArrayList();
		JunElementalNode[] nodes = this.graph().nodes();
		for (int i = 0; i < nodes.length; i++) {
			if (JunStringUtility.StringMatch_and_(nodes[i].labelString(), aString)) {
				aList.add(nodes[i]);
			}
		}
		if (aList.isEmpty()) {
			return;
		}

		JunElementalNode[] values = (JunElementalNode[]) aList.toArray(new JunElementalNode[aList.size()]);
		String[] labels = new String[values.length];
		for (int i = 0; i < labels.length; i++) {
			labels[i] = values[i].labelString();
		}

		JunElementalNode aNode = (JunElementalNode) JunDialog.Choose_($String("Choose one."), labels, values, 10, new StBlockClosure());
		if (aNode == null) {
			return;
		}

		this.selectAndScrollFor_(aNode);
	}

	/**
	 * Select the all ndoes.
	 * 
	 * @category menu messages
	 */
	public void selectAllNodes() {
		this.selections_(Arrays.asList(this.graph().nodes()));
		this.changed();
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		this.graph().displayOn_at_(aGraphics, aPoint);
	}

	/**
	 * Display the receiver on the graphics at the specified point with the specified scale factor.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param scaleFactor jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category displaying
	 */
	public void displayOn_at_scaledBy_(Graphics aGraphics, Point aPoint, Jun2dPoint scaleFactor) {
		this.graph().displayOn_at_scaledBy_(aGraphics, aPoint, scaleFactor);
	}

	/**
	 * Redisplay the view.
	 * 
	 * @category displaying
	 */
	public void redisplay() {
		JunGrapherView aView = (JunGrapherView) this.getView();
		if (aView != null) {
			aView.redisplay();
		}
	}

	/**
	 * Answer the view bounds.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category displaying
	 */
	public StRectangle viewBounds() {
		StView aView = this.getView();
		if (aView == null) {
			return new StRectangle(0, 0, 0, 0);
		}

		return new StRectangle(aView.toComponent().getBounds());
	}

	/**
	 * Fit the viewpoint.
	 * 
	 * @category scrolling
	 */
	protected void fitViewpoint() {
		if (this.selection() == null) {
			StRectangle viewBox = this.viewBounds();
			if (this.isConcentricArrangement()) {
				this.scrollTo_(viewBox.center());
			} else if (this.isHierarchicalArrangement()) {
				Point viewBoxPoint = viewBox.leftCenter();
				Point boundingBoxPoint = this.boundingBox().leftCenter();
				this.scrollTo_(new Point(viewBoxPoint.x - boundingBoxPoint.x, viewBoxPoint.y - boundingBoxPoint.y));
			} else {
				Point viewBoxPoint = viewBox.center();
				Point boundingBoxPoint = this.boundingBox().center();
				this.scrollTo_(new Point(viewBoxPoint.x - boundingBoxPoint.x, viewBoxPoint.y - boundingBoxPoint.y));
			}
		} else {
			this.scrollFor_(this.selection());
		}
	}

	/**
	 * Answer the scroll amount.
	 * 
	 * @return java.awt.Point
	 * @category scrolling
	 */
	public Point scrollAmount() {
		JunGrapherView aView = (JunGrapherView) this.getView();
		if (aView == null) {
			return new Point(0, 0);
		}

		return aView.scrollAmount();
	}

	/**
	 * Scroll to the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category scrolling
	 */
	public void scrollTo_(Point aPoint) {
		JunGrapherView aView = (JunGrapherView) this.getView();
		if (aView == null) {
			return;
		}

		aView.scrollTo_(aPoint);

		this.focusBoundingBox_(null);
		this.focusBoundingBox();
	}

	/**
	 * Scroll for the specified node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category scrolling
	 */
	public void scrollFor_(JunElementalNode aNode) {
		JunGrapherView aView = (JunGrapherView) this.getView();
		if (aView == null) {
			return;
		}

		aView.scrollFor_(aNode);

		this.focusBoundingBox_(null);
		this.focusBoundingBox();
	}

	/**
	 * Select, arrange, and scroll for the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category scrolling
	 */
	public void selectAndArrangeAndScrollFor_(JunElementalNode aNode) {
		this.selection_(aNode);
		this.arrangeAndScrollFor_(aNode);
	}

	/**
	 * Select and scroll for the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category scrolling
	 */
	public void selectAndScrollFor_(JunElementalNode aNode) {
		this.selection_(aNode);
		this.scrollFor_(aNode);
	}

	/**
	 * Arrange and scroll for the node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category scrolling
	 */
	public void arrangeAndScrollFor_(JunElementalNode aNode) {
		if (aNode == null) {
			this.arrangeGrapher();
		} else {
			this.graph().arrangeWithAnimation_(this);
			this.scrollFor_(aNode);
			this.nodeSettingsFor_((JunElementalNode) this.nodeSettings().stuff());
		}
	}

	/**
	 * Answer my current selections.
	 * 
	 * @return java.util.Collection
	 * @category selecting
	 */
	public Collection selections() {
		return this.graph().selections();
	}

	/**
	 * Set my new selections.
	 * 
	 * @param aCollection java.util.Collection
	 * @category selecting
	 */
	public void selections_(Collection aCollection) {
		ArrayList previous = new ArrayList(this.selections());
		this.graph().selections_(aCollection);
		this.changed_with_($("selections"), previous);
		this.selectedNodesHolder().value_(aCollection);
	}

	/**
	 * Answer a selected node if exists.
	 * 
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category selecting
	 */
	public JunElementalNode selection() {
		return this.graph().selection();
	}

	/**
	 * Set a selected node.
	 * 
	 * @param aNode jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category selecting
	 */
	public void selection_(JunElementalNode aNode) {
		ArrayList aList = new ArrayList();
		aList.add(aNode);
		this.selections_(aList);
	}

	/**
	 * Answer my selection border color.
	 * 
	 * @return java.awt.Color
	 * @category selecting
	 */
	public Color selectionBorderColor() {
		return this.graph().selectionBorderColor();
	}

	/**
	 * Answer my selection border width.
	 * 
	 * @return int
	 * @category selecting
	 */
	public int selectionBorderWidth() {
		return this.graph().selectionBorderWidth();
	}

	/**
	 * Answer a node which is placed at the specified point.
	 * If none, return null.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.topology.graph.JunElementalNode
	 * @category selecting
	 */
	public JunElementalNode whichAt_(Point aPoint) {
		return this.graph().whichAt_(aPoint);
	}

	/**
	 * Answer the value holder for the selected nodes.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category selecting
	 */
	protected StValueHolder selectedNodesHolder() {
		if (selectedNodesHolder == null) {
			selectedNodesHolder = new StValueHolder(this.selections());
		}
		return selectedNodesHolder;
	}

	/**
	 * Update the grapher.
	 * 
	 * @category updating
	 */
	public void updateGrapher() {
		this.flushAll();
		this.changed_($("graph"));
		this.graphSettingsFor_((JunElementalGraph) this.graphSettings().stuff());
		this.nodeSettingsFor_((JunElementalNode) this.nodeSettings().stuff());
		this.arcSettingsFor_((JunElementalArc) this.arcSettings().stuff());
	}

}
