package jp.co.sra.jun.voronoi.twoD.diagram;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.goodies.display.JunDisplayModel;

/**
 * JunVoronoi2dDiagram class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/02/07 (by NISHIHARA Satoshi)
 *  @updated   2006/11/06 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun661 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVoronoi2dDiagram.java,v 8.12 2008/02/20 06:33:14 nisinaka Exp $
 */
public class JunVoronoi2dDiagram implements StDisplayable {

	protected Dimension diagramExtent;
	protected ArrayList diagramPoints;
	protected StImage diagramImage;
	protected JunVoronoi2dProcessor voronoiProcessor;
	protected boolean processingCondition;

	/**
	 * Create a new instance of JunVoronoi2dDiagram and initialize it.
	 *
	 * @param width int
	 * @param height int
	 * @category Instance creation
	 */
	public JunVoronoi2dDiagram(int width, int height) {
		this.initialize();
		this.extent_(width, height);
	}

	/**
	 * Create a new instance of JunVoronoi2dDiagram and initialize it.
	 *
	 * @param extent java.awt.Dimension
	 * @category Instance creation
	 */
	public JunVoronoi2dDiagram(Dimension extent) {
		this(extent.width, extent.height);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		diagramExtent = new Dimension(100, 100);
		diagramPoints = new ArrayList();
		diagramImage = null;
		voronoiProcessor = null;
		processingCondition = true;
	}

	/**
	 * Answer my current extent.
	 * 
	 * @return java.awt.Extent
	 * @category accessing
	 */
	public Dimension extent() {
		return diagramExtent;
	}

	/**
	 * Set my new extent.
	 * 
	 * @param width int
	 * @param height height
	 * @category accessing
	 */
	public void extent_(int width, int height) {
		diagramExtent = new Dimension(Math.max(width, 1), Math.max(height, 1));
		this.flushImage();
	}

	/**
	 * Answer the width of my current extent.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int width() {
		return this.extent().width;
	}

	/**
	 * Answer the height of my current extent.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int height() {
		return this.extent().height;
	}

	/**
	 * Answer my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		double factorX = (double) (this.width() + 1) / this.width();
		double factorY = (double) (this.height() + 1) / this.height();
		Jun2dPoint factor = new Jun2dPoint(factorX, factorY);

		Jun2dPoint[] points = (Jun2dPoint[]) diagramPoints.toArray(new Jun2dPoint[diagramPoints.size()]);
		for (int i = 0; i < points.length; i++) {
			points[i] = points[i].scaledBy_(factor);
		}
		return points;
	}

	/**
	 * Answer my current diagram image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage diagram() {
		if (diagramImage == null) {
			diagramImage = this.makeImage();
		}
		return diagramImage;
	}

	/**
	 * Answer my voronoi processor.
	 * 
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dProcessor
	 * @category accessing
	 */
	public JunVoronoi2dProcessor processor() {
		this.diagram();
		return voronoiProcessor;
	}

	/**
	 * Answer my current condition for processing. 
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean condition() {
		return processingCondition;
	}

	/**
	 * Answer my current area.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double area() {
		if (this.condition() == false) {
			return 0;
		}

		return this.processor().area();
	}

	/**
	 * Answer a Rectangle that represents the receiver's actual bounding rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		return this.preferredBounds();
	}

	/**
	 * Answer a Rectangle that represents the receiver's preferred bounding rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return new Rectangle(0, 0, this.width(), this.height());
	}

	/**
	 * Add the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category adding
	 */
	public void add_(Jun2dPoint aPoint) {
		if (this.condition() == false) {
			return;
		}

		double factorX = (double) this.width() / (this.width() + 1);
		double factorY = (double) this.height() / (this.height() + 1);
		Jun2dPoint factor = new Jun2dPoint(factorX, factorY);
		Jun2dPoint point = aPoint.scaledBy_(factor);

		Jun2dBoundingBox bounds = new Jun2dBoundingBox(this.bounds());
		if (bounds.containsPoint_(point) && diagramPoints.contains(point) == false) {
			diagramPoints.add(point);
			this.flushImage();
		}
	}

	/**
	 * Add the all points.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category adding
	 */
	public void addAll_(Jun2dPoint[] points) {
		for (int i = 0; i < points.length; i++) {
			this.add_(points[i]);
		}
	}

	/**
	 * Convert the receiver as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		return this.diagram();
	}

	/**
	 * Convert the receiver as JunVoronoi2dDiagram.
	 * 
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram
	 * @category converting
	 */
	public JunVoronoi2dDiagram asVoronoiDiagram() {
		return this;
	}

	/**
	 * Convert the receiver as JunDelaunay2dDiagram.
	 * 
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram
	 * @category converting
	 */
	public JunDelaunay2dDiagram asDelaunayDiagram() {
		JunDelaunay2dDiagram diagram = new JunDelaunay2dDiagram(this.extent());
		diagram.addAll_(this.points());
		return diagram;
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.asImage().displayOn_(aGraphics);
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		this.asImage().displayOn_at_(aGraphics, aPoint);
	}

	/**
	 * Show the diagram.
	 * 
	 * @category utilities
	 */
	public void show() {
		JunDisplayModel.OpenVisual_label_(this, this.labelString());
	}

	/**
	 * Answer my label string for showing.
	 * 
	 * @return java.lang.String
	 * @category utilities
	 */
	protected String labelString() {
		return "Voronoi Diagram";
	}

	/**
	 * Flush the diagram image.
	 * 
	 * @category private
	 */
	protected void flushImage() {
		diagramImage = null;
	}

	/**
	 * Create my current image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	protected StImage makeImage() {
		voronoiProcessor = new JunVoronoi2dProcessor(this.points());
		processingCondition = voronoiProcessor.compute();

		StImage anImage = new StImage(this.extent());
		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) anImage.image().getGraphics();
			aGraphics.setBackground(Color.white);

			Rectangle bounds = this.bounds();
			double factorX = (double) bounds.width / (bounds.width + 1);
			double factorY = (double) bounds.height / (bounds.height + 1);
			Jun2dPoint factor = new Jun2dPoint(factorX, factorY);

			// Draw the border.
			aGraphics.setColor(this.borderColor());
			aGraphics.drawRect(bounds.x, bounds.y, bounds.width - 1, bounds.height - 1);

			// Draw the dots.
			aGraphics.setColor(this.dotColor());
			Jun2dPoint[] dots = voronoiProcessor.dots();
			for (int i = 0; i < dots.length; i++) {
				StRectangle box = new StRectangle(dots[i].scaledBy_(factor)._toPoint(), new Dimension(1, 1));
				box = box.expandedBy_(new StRectangle(1, 1, 1, 1));
				voronoiProcessor.displayBoxOn_box_clip_(aGraphics, box, bounds);
			}

			// Draw the sides.
			aGraphics.setColor(this.sideColor());
			Jun2dPoint[][] sides = voronoiProcessor.sides();
			for (int i = 0; i < sides.length; i++) {
				Point point1 = sides[i][0].scaledBy_(factor)._toPoint();
				Point point2 = sides[i][1].scaledBy_(factor)._toPoint();
				voronoiProcessor.displayLineOn_from_to_clip_(aGraphics, point1, point2, bounds);
			}

		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return anImage;
	}

	/**
	 * Answer my border color.
	 * 
	 * @return java.awt.Color
	 * @category private
	 */
	protected Color borderColor() {
		return Color.green;
	}

	/**
	 * Answer my side color.
	 * 
	 * @return java.awt.Color
	 * @category private
	 */
	protected Color sideColor() {
		return Color.red;
	}

	/**
	 * Answer my dot color.
	 * 
	 * @return java.awt.Color
	 * @category private
	 */
	protected Color dotColor() {
		return Color.blue;
	}

}
