package jp.co.sra.jun.vrml.support;

import java.util.*;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.opengl.display.*;
import jp.co.sra.jun.opengl.objects.*;
import jp.co.sra.jun.vrml.node.abstracts.*;

/**
 * JunVrmlGenerator class
 * 
 *  @author    ASTI Beijing
 *  @created   UNKNOWN
 *  @updated   2000/03/17 (by nisinaka)
 *  @updated   2003/07/01 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVrmlGenerator.java,v 8.10 2008/02/20 06:33:17 nisinaka Exp $
 */
public abstract class JunVrmlGenerator extends JunVrmlNodeVisitor {

	/** The dictionary to hold some information about a node tree. */
	protected Hashtable informationDictionary = new Hashtable();

	/**
	 * Answer an instance of the default display model.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category Defaults
	 */
	public static JunOpenGLDisplayModel DefaultDisplayModel() {
		return new JunOpenGLDisplayModel();
	}

	/**
	 * Answer the collection of lights.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector lights() {
		return (Vector) this.informationAt_($("lights"));
	}

	/**
	 * Answer a 3D object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public JunOpenGL3dObject object() {
		Object myObject = this.informationAt_($("objects"));
		if (myObject == null) {
			return null;
		}

		// Then my object should be an instance of Vector.
		Vector objects = (Vector) myObject;
		if (objects.size() == 1) {
			return (JunOpenGL3dObject) objects.elementAt(0);
		}

		return new JunOpenGL3dCompoundObject(objects);
	}

	/**
	 * Set the JunOpenGL3dObject.
	 * 
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category accessing
	 */
	public void object_(JunOpenGL3dObject anObject) {
		this.informationAt_put_($("objects"), null);
		this.addObject_(anObject);
	}

	/**
	 * Add the information of the generator.
	 * 
	 * @param aGenerator jp.co.sra.jun.vrml.support.JunVrmlGenerator
	 * @category adding
	 */
	public void add_(JunVrmlGenerator aGenerator) {
		this.addObject_(aGenerator.object());

		Vector lights = aGenerator.lights();
		if (lights != null) {
			for (int i = 0; i < lights.size(); i++) {
				this.addLight_((JunOpenGLDisplayLight) lights.elementAt(i));
			}
		}
	}

	/**
	 * Add a new light to my information dictionary.
	 * 
	 * @param aLight jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category adding
	 */
	public void addLight_(JunOpenGLDisplayLight aLight) {
		if (aLight == null) {
			return;
		}

		Vector myLights = (Vector) this.informationAt_($("lights"));
		if (myLights == null) {
			myLights = new Vector();
		}
		myLights.addElement(aLight);
		this.informationAt_put_($("lights"), myLights);
	}

	/**
	 * Add a new 3D object to my information dictionary.
	 * 
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category adding
	 */
	public void addObject_(JunOpenGL3dObject anObject) {
		if (anObject == null) {
			return;
		}

		Vector myObjects = (Vector) this.informationAt_($("objects"));
		if (myObjects == null) {
			myObjects = new Vector();
		}
		myObjects.addElement(anObject);
		this.informationAt_put_($("objects"), myObjects);
	}

	/**
	 * Add a new property node to my information dictionary.
	 * 
	 * @param aNode jp.co.sra.jun.vrml.node.abstracts.JunVrmlAbstractPropertyNode
	 * @category adding
	 */
	public void addPropertyNode_(JunVrmlAbstractPropertyNode aNode) {
		if (aNode == null) {
			return;
		}

		Vector propertyNodes = (Vector) this.informationAt_($("propertyNodes"));
		if (propertyNodes == null) {
			propertyNodes = new Vector();
		}
		propertyNodes.addElement(aNode);
		this.informationAt_put_($("propertyNodes"), propertyNodes);
	}

	/**
	 * Process the geometry node.
	 * 
	 * @param aNode jp.co.sra.jun.vrml.node.abstracts.JunVrmlAbstractGeometryNode
	 * @category enumerating
	 */
	public void doGeneralGeometry_(JunVrmlAbstractGeometryNode aNode) {
		JunOpenGL3dObject newObject = aNode.asJunOpenGL3dObject();
		if (newObject != null) {
			this.propertyOn_(newObject);
			this.addObject_(newObject);
		}
	}

	/**
	 * Process the property node.
	 * 
	 * @param aNode jp.co.sra.jun.vrml.node.abstracts.JunVrmlAbstractPropertyNode
	 * @category enumerating
	 */
	public void doGeneralProperty_(JunVrmlAbstractPropertyNode aNode) {
		this.addPropertyNode_(aNode);
	}

	/**
	 * Enumerate the node tree and generate 3D objects.
	 * 
	 * @param nodeTree jp.co.sra.jun.vrml.node.abstracts.JunVrmlNode[]
	 * @category enumerating
	 */
	public void doNodes_(JunVrmlNode[] nodeTree) {
		for (int i = 0; i < nodeTree.length; i++) {
			nodeTree[i].nodeDo_(this);
		}
	}

	/**
	 * Generate a JunOpenGLDisplayModel from a node tree.
	 * 
	 * @param aNodeTree jp.co.sra.jun.vrml.node.abstracts.JunVrmlNode[]
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category code generation
	 */
	public JunOpenGLDisplayModel generate_(JunVrmlNode[] aNodeTree) {
		return this.generate_with_(aNodeTree, DefaultDisplayModel());
	}

	/**
	 * Generate a JunOpenGLDisplayModel from a node tree.
	 * 
	 * @param aNodeTree jp.co.sra.jun.vrml.node.abstracts.JunVrmlNode[]
	 * @param aDisplayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category code generation
	 */
	public JunOpenGLDisplayModel generate_with_(JunVrmlNode[] aNodeTree, JunOpenGLDisplayModel aDisplayModel) {
		this.doNodes_(aNodeTree);
		this.setObject_(aDisplayModel);
		this.setLights_(aDisplayModel);
		this.setProjection_(aDisplayModel);
		return aDisplayModel;
	}

	/**
	 * Set properties on the JunOpenGL3dObject.
	 * 
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category code generation
	 */
	public void propertyOn_(JunOpenGL3dObject anObject) {
		Vector propertyNodes = (Vector) this.informationAt_($("propertyNodes"));
		if (propertyNodes != null) {
			JunVrmlAbstractPropertyNode[] nodes = new JunVrmlAbstractPropertyNode[propertyNodes.size()];
			propertyNodes.copyInto(nodes);
			for (int i = 0; i < nodes.length; i++) {
				nodes[i].propertyOn_(anObject);
			}
		}
	}

	/**
	 * Get the information specified with the symbol.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return java.lang.Object
	 * @category private
	 */
	protected Object informationAt_(StSymbol aSymbol) {
		return informationDictionary.get(aSymbol);
	}

	/**
	 * Set the information specified with the symbol.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aValue java.lang.Object
	 * @category private
	 */
	protected void informationAt_put_(StSymbol aSymbol, Object aValue) {
		if (aValue == null) {
			informationDictionary.remove(aSymbol);
		} else {
			informationDictionary.put(aSymbol, aValue);
		}
	}

	/**
	 * Set the lights on the JunOpenGLDisplayModel.
	 * 
	 * @param aDisplayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category private
	 */
	protected void setLights_(JunOpenGLDisplayModel aDisplayModel) {
		JunOpenGLDisplayLight[] displayLights = aDisplayModel.displayLights();
		for (int i = 0; i < displayLights.length; i++) {
			displayLights[i].beOff();
		}

		// Use the ambient light anyway.
		aDisplayModel.displayLight5().beOn();

		Vector lights = this.lights();
		if (lights == null) {
			return;
		}

		if (lights.size() > 0) {
			aDisplayModel.displayLights()[0] = (JunOpenGLDisplayLight) lights.elementAt(0);
			aDisplayModel.displayLights()[0].beOn();
		}
		if (lights.size() > 1) {
			aDisplayModel.displayLights()[1] = (JunOpenGLDisplayLight) lights.elementAt(1);
			aDisplayModel.displayLights()[1].beOn();
		}
		if (lights.size() > 2) {
			aDisplayModel.displayLights()[2] = (JunOpenGLDisplayLight) lights.elementAt(2);
			aDisplayModel.displayLights()[2].beOn();
		}
		if (lights.size() > 3) {
			aDisplayModel.displayLights()[3] = (JunOpenGLDisplayLight) lights.elementAt(3);
			aDisplayModel.displayLights()[3].beOn();
		}
	}

	/**
	 * Set the 3D object to the JunOpenGLDisplayModel.
	 * 
	 * @param aDisplayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category private
	 */
	protected void setObject_(JunOpenGLDisplayModel aDisplayModel) {
		aDisplayModel.displayObject_(this.object());
	}

	/**
	 * Set the projection to the JunOpenGLDisplayModel.
	 * 
	 * @param aDisplayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category private
	 */
	protected void setProjection_(JunOpenGLDisplayModel aDisplayModel) {
		// Can do nothing as a default.
	}
}
