package jp.co.sra.jun.vrml.support;

import java.util.*;
import jp.co.sra.smalltalk.*;
import jp.co.sra.jun.vrml.node.abstracts.JunVrmlNode;

/**
 * JunVrmlParser97 class
 * 
 *  @author    nisinaka
 *  @created   2000/03/22 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVrmlParser97.java,v 8.10 2008/02/20 06:33:17 nisinaka Exp $
 */
public class JunVrmlParser97 extends JunVrmlParser {

	protected Hashtable nodeDefinitions = new Hashtable();
	protected Hashtable protoDefinitions = new Hashtable();
	protected Hashtable interfaceDefinitions = new Hashtable();

	/**
	 * Parse a source stream with the specified interface definitions.
	 * 
	 * @param anObject java.lang.Object
	 * @param aDictionary java.util.Hashtable
	 * @return java.lang.Object
	 * @category Utilities
	 */
	public static Object Parse_withInterfaceDefinitions_(Object anObject, Hashtable aDictionary) {
		JunVrmlParser97 newParser = new JunVrmlParser97();
		newParser.interfaceDefinitions = aDictionary;
		return newParser.parse_builder_(MakeStream_(anObject), newParser.defaultBuilder());
	}

	/**
	 * Answer an instance of the default syntax tree builder.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlSyntaxTreeBuilder
	 * @category defaults
	 */
	protected JunVrmlSyntaxTreeBuilder defaultBuilder() {
		return new JunVrmlSyntaxTreeBuilder97();
	}

	/**
	 * Answer the default keyword table.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlKeywordTable
	 * @category defaults
	 */
	protected JunVrmlKeywordTable defaultKeywordTable() {
		return JunVrmlKeywordTable97.KeywordTable();
	}

	/**
	 * Answer the default scanner table.
	 * 
	 * @return jp.co.sra.jun.vrml.support.JunVrmlScannerTable
	 * @category defaults
	 */
	protected JunVrmlScannerTable defaultScannerTable() {
		return JunVrmlScannerTable97.ScannerTable();
	}

	/**
	 * Answer true if the current token type is an externproto.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isExternproto() {
		return tokenType == $("EXTERNPROTO");
	}

	/**
	 * Answer true if the current token type is a field type.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isFieldType() {
		if (tokenType == $("MFColor")) {
			return true;
		}

		if (tokenType == $("MFFloat")) {
			return true;
		}

		if (tokenType == $("MFInt32")) {
			return true;
		}

		if (tokenType == $("MFNode")) {
			return true;
		}

		if (tokenType == $("MFRotation")) {
			return true;
		}

		if (tokenType == $("MFString")) {
			return true;
		}

		if (tokenType == $("MFTime")) {
			return true;
		}

		if (tokenType == $("MFVec2f")) {
			return true;
		}

		if (tokenType == $("MFVec3f")) {
			return true;
		}

		if (tokenType == $("SFBool")) {
			return true;
		}

		if (tokenType == $("SFColor")) {
			return true;
		}

		if (tokenType == $("SFFloat")) {
			return true;
		}

		if (tokenType == $("SFImage")) {
			return true;
		}

		if (tokenType == $("SFInt32")) {
			return true;
		}

		if (tokenType == $("SFNode")) {
			return true;
		}

		if (tokenType == $("SFRotation")) {
			return true;
		}

		if (tokenType == $("SFString")) {
			return true;
		}

		if (tokenType == $("SFTime")) {
			return true;
		}

		if (tokenType == $("SFVec2f")) {
			return true;
		}

		if (tokenType == $("SFVec3f")) {
			return true;
		}

		return false;
	}

	/**
	 * Answer true if the current token type is an interface declaration.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isInterfaceDeclaration() {
		return this.isRestrictedInterfaceDeclaration() || (tokenType == $("exposedField"));
	}

	/**
	 * Answer true if the current token type is a node.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNode() {
		if ((tokenType != $("identifier")) && (tokenType != $("Script"))) {
			return false;
		}

		this.nextToken();
		boolean answer = tokenType == $("leftBrace");
		this.unNextToken();
		return answer;
	}

	/**
	 * Answer true if the current token type is a node body element.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNodeBodyElement() {
		return (tokenType == $("identifier")) || this.isRouteStatement() || this.isProtoStatement();
	}

	/**
	 * Answer true if the current token type is a node statement.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNodeStatement() {
		return this.isNode() || (tokenType == $("DEF")) || (tokenType == $("USE"));
	}

	/**
	 * Answer true if the current token type is a proto.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isProto() {
		return tokenType == $("PROTO");
	}

	/**
	 * Answer true if the current token type is a proto statement.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isProtoStatement() {
		return this.isProto() || this.isExternproto();
	}

	/**
	 * Answer true if the current token type is a restricted interface declaration.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isRestrictedInterfaceDeclaration() {
		return (tokenType == $("eventIn")) || (tokenType == $("eventOut")) || (tokenType == $("field"));
	}

	/**
	 * Answer true if the current token type is a route statement.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isRouteStatement() {
		return tokenType == $("ROUTE");
	}

	/**
	 * Answer true if the current token type is a statement.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isStatement() {
		return this.isNodeStatement() || this.isProtoStatement() || this.isRouteStatement();
	}

	/**
	 * Parse a VRML scene and answer true if successfully parsed.
	 * 
	 * @return boolean
	 * @category parse general
	 */
	public boolean parseVrmlScene() {
		this.parseStatements();
		return true;
	}

	/**
	 * Parse statements.
	 * 
	 * @category parse general
	 */
	public void parseStatements() {
		Vector parseNodes = new Vector();
		this.nextToken();
		while (this.isStatement()) {
			this.unNextToken();
			this.parseStatement();
			if (parseNode != null) {
				parseNodes.addElement(parseNode);
			}
			this.nextToken();
		}
		this.unNextToken();
		parseNode = parseNodes;
	}

	/**
	 * Parse a statement.
	 * 
	 * @category parse general
	 */
	public void parseStatement() {
		this.nextToken();
		if (this.isNodeStatement()) {
			this.unNextToken();
			this.parseNodeStatement();
			return;
		}

		if (this.isProtoStatement()) {
			this.unNextToken();
			this.parseProtoStatement();
			return;
		}

		if (this.isRouteStatement()) {
			this.unNextToken();
			this.parseRouteStatement();
			return;
		}

		failBlock.value_("parseStatement: invalid statement");
	}

	/**
	 * Parse a node statement.
	 * 
	 * @category parse general
	 */
	public void parseNodeStatement() {
		this.nextToken();
		if (tokenType == $("DEF")) {
			this.nextToken();
			if (tokenType == $("identifier")) {
				StSymbol nodeNameId = (StSymbol) token;
				this.parseNode();
				nodeDefinitions.put(nodeNameId, parseNode);
				return;
			} else {
				failBlock.value_("parseNodeStatement: nodeNameId expected after DEF");
			}
		}

		if (tokenType == $("USE")) {
			this.nextToken();
			if (tokenType == $("identifier")) {
				parseNode = nodeDefinitions.get(token);
				if (parseNode == null) {
					failBlock.value_("parseNodeStatement: no nodeNameId defined to USE");
				}
				return;
			} else {
				failBlock.value_("parseNodeStatement: nodeNameId expected after USE");
			}
		}

		this.unNextToken();
		this.parseNode();
	}

	/**
	 * Parse a proto statement.
	 * 
	 * @category parse general
	 */
	public void parseProtoStatement() {
		this.nextToken();
		if (this.isProto()) {
			this.unNextToken();
			this.parseProto();
			return;
		}

		if (this.isExternproto()) {
			this.unNextToken();
			this.parseExternproto();
			return;
		}

		failBlock.value_("parseProtoStatement: invalid statement");
	}

	/**
	 * Parse a route statement.
	 * 
	 * @category parse general
	 */
	public void parseRouteStatement() {
		this.nextToken();
		if (tokenType != $("ROUTE")) {
			failBlock.value_("parseRouteStatement: invalid statement");
		}

		this.nextToken(); // to ignore nodeNameId.
		this.nextToken(); // to ignore #period.
		this.nextToken(); // to ignore eventOutId.
		this.nextToken(); // to ignore #TO.
		this.nextToken(); // to ignore nodeNameId.
		this.nextToken(); // to ignore #period.
		this.nextToken(); // to ignore eventInId.
	}

	/**
	 * Parse a node.
	 * 
	 * @category parse node
	 */
	public void parseNode() {
		this.nextToken();
		if (tokenType == $("identifier")) {
			StSymbol nodeTypeId = (StSymbol) token;
			this.nextToken();
			if (tokenType != $("leftBrace")) {
				failBlock.value_("parseNode: #leftBrace expected after the nodeTypeId");
			}
			this.parseNodeBody_(new Hashtable());
			parseNode = this.generateNewNode_with_(nodeTypeId, (Hashtable) parseNode);
			this.nextToken();
			if (tokenType != $("rightBrace")) {
				failBlock.value_("parseNode: #rightBrace expected after the nodeBody");
			}
			return;
		}

		if (tokenType == $("Script")) {
			this.nextToken();
			if (tokenType != $("leftBrace")) {
				failBlock.value_("parseNode: #leftBrace expected after Script");
			}
			this.parseScriptBody();
			parseNode = null; // builder.newNode_with_($("Script"), parseNode);
			this.nextToken();
			if (tokenType != $("rightBrace")) {
				failBlock.value_("parseNode: #rightBrace expected after the scriptBody");
			}
			return;
		}

		failBlock.value_("parseNode: invalid node");
	}

	/**
	 * Parse a node body.
	 * 
	 * @param aDictionary java.util.Hashtable
	 * @category parse node
	 */
	public void parseNodeBody_(Hashtable aDictionary) {
		this.nextToken();
		if (tokenType == $("rightBrace")) {
			// It's end of the nodeBody.
			this.unNextToken();
			parseNode = aDictionary;
			return;
		}

		if (this.isNodeBodyElement()) {
			this.unNextToken();
			this.parseNodeBodyElement_(aDictionary);
			this.parseNodeBody_(aDictionary);
			parseNode = aDictionary;
			return;
		}

		failBlock.value_("parseNodeBody: invalue nodeBody");
	}

	/**
	 * Parse a node body element.
	 * 
	 * @param aDictionary java.util.Hashtable
	 * @category parse node
	 */
	public void parseNodeBodyElement_(Hashtable aDictionary) {
		this.nextToken();
		if (tokenType == $("identifier")) {
			StSymbol identifier = (StSymbol) token;
			this.nextToken();
			if (tokenType == $("IS")) {
				this.nextToken();
				if (tokenType != $("identifier")) {
					failBlock.value_("parseNodeBodyElement: #identifier expected after #IS");
				}
				StSymbol anotherIdentifier = (StSymbol) token;
				Object fieldValue = interfaceDefinitions.get(anotherIdentifier);
				if (fieldValue != null) {
					aDictionary.put(identifier, fieldValue);
				}
				return;
			}

			// Assume this is a field value.
			this.unNextToken();
			this.parseFieldValue();
			if (parseNode != null) {
				aDictionary.put(identifier, parseNode);
			}
			parseNode = aDictionary;
			return;
		}

		if (this.isRouteStatement()) {
			this.unNextToken();
			this.parseRouteStatement();
			return;
		}

		if (this.isProtoStatement()) {
			this.unNextToken();
			this.parseProtoStatement();
			return;
		}

		failBlock.value_("parseNodeBody: invalue nodeBodyElement");
	}

	/**
	 * Parse a script body.
	 * 
	 * @category parse node
	 */
	public void parseScriptBody() {
		// not implemented yet. just ignoring.
		this.nextToken();
		while (tokenType != $("rightBrace")) {
			this.nextToken();
		}
		this.unNextToken();
	}

	/**
	 * Parse a proto.
	 * 
	 * @category parse proto
	 */
	public void parseProto() {
		this.nextToken();
		if (this.isProto() == false) {
			failBlock.value_("parseProto: inappropriate call");
			return;
		}

		this.nextToken();
		if (tokenType != $("identifier")) {
			failBlock.value_("parseProto: nodeTypeId is expected after PROTO");
			return;
		}

		StSymbol nodeTypeId = (StSymbol) token;
		this.nextToken();
		if (tokenType != $("leftBracket")) {
			failBlock.value_("parseProto: #leftBracket is expected after nodeTypeId");
			return;
		}

		this.parseInterfaceDeclarations_(new Hashtable());
		Hashtable interfaceDeclarations = (Hashtable) parseNode;
		this.nextToken();
		if (tokenType != $("rightBracket")) {
			failBlock.value_("parseProto: #rightBracket is expected after interfaceDeclarations");
			return;
		}

		this.nextToken();
		if (tokenType != $("leftBrace")) {
			failBlock.value_("parseProto: #leftBrace is expected after interfaceDeclarations");
			return;
		}

		// wrap up the prototype definition.
		int startPosition = source.position() + 1;
		this.parseProtoBody();
		int endPosition = source.position();
		source.position_(0);
		StReadStream protoBody = new StReadStream(source.contents().substring(startPosition, endPosition));
		source.position_(endPosition);
		protoDefinitions.put(nodeTypeId, new Object[] { interfaceDeclarations, protoBody });
		parseNode = null;

		this.nextToken();
		if (tokenType != $("rightBrace")) {
			failBlock.value_("parseProto: #rightBrace is expected after protoBody");
			return;
		}
	}

	/**
	 * Parse proto statements.
	 * 
	 * @category parse proto
	 */
	public void parseProtoStatements() {
		this.nextToken();
		while (this.isProtoStatement()) {
			this.parseProtoStatement();
		}
		this.unNextToken();
	}

	/**
	 * Parse a proto body.
	 * 
	 * @category parse proto
	 */
	public void parseProtoBody() {
		this.parseProtoStatements();
		this.parseRootNodeStatement();
		this.parseStatements();
	}

	/**
	 * Parse a root node statement.
	 * 
	 * @category parse proto
	 */
	public void parseRootNodeStatement() {
		this.nextToken();
		if (this.isNode()) {
			this.unNextToken();
			this.parseNode();
			return;
		}

		if (tokenType == $("DEF")) {
			this.nextToken(); // to ignore the nodeNameId
			this.parseNode();
			return;
		}

		failBlock.value_("parseRootNodeStatement: invalid rootNodeStatement");
	}

	/**
	 * Parse an external proto.
	 * 
	 * @category parse proto
	 */
	public void parseExternproto() {
		failBlock.value_("parseExternproto: not implemented");
	}

	/**
	 * Parse an interface declaration.
	 * 
	 * @param aDictionary java.util.Hashtable
	 * @category parse proto
	 */
	public void parseInterfaceDeclaration_(Hashtable aDictionary) {
		this.nextToken();
		if (this.isRestrictedInterfaceDeclaration()) {
			this.unNextToken();
			this.parseRestrictedInterfaceDeclaration_(aDictionary);
			return;
		}

		if (tokenType == $("exposedField")) {
			failBlock.value_("parseInterfaceDeclaration: not implemented to parse #exposeField");
			return;
		}

		failBlock.value_("parseInterfaceDeclaration: invalue interfaceDeclaration");
	}

	/**
	 * Parse interface declarations.
	 * 
	 * @param aDictionary java.util.Hashtable
	 * @category parse proto
	 */
	public void parseInterfaceDeclarations_(Hashtable aDictionary) {
		this.nextToken();
		if (tokenType == $("rightBracket")) {
			// It's end of the interfaceDeclarations.
			this.unNextToken();
			parseNode = aDictionary;
			return;
		}

		if (this.isInterfaceDeclaration()) {
			this.unNextToken();
			this.parseInterfaceDeclaration_(aDictionary);
			this.parseInterfaceDeclarations_(aDictionary);
			parseNode = aDictionary;
			return;
		}

		failBlock.value_("parseInterfaceDeclarations: invalue interfaceDeclarations");
	}

	/**
	 * Parse a restricted interface declaration.
	 * 
	 * @param aDictionary java.util.Hashtable
	 * @category parse proto
	 */
	public void parseRestrictedInterfaceDeclaration_(Hashtable aDictionary) {
		this.nextToken();
		if (tokenType == $("eventIn")) {
			this.nextToken(); // to ignore the fieldType.
			this.nextToken(); // to ignore the eventInId.
			failBlock.value_("parseRestrictedInterfaceDeclaration: not implemented to parse #eventIn");
			return;
		}

		if (tokenType == $("eventOut")) {
			this.nextToken(); // to ignore the fieldType.
			this.nextToken(); // to ignore the eventOutId.
			failBlock.value_("parseRestrictedInterfaceDeclaration: not implemented to parse #eventOut");
			return;
		}

		if (tokenType == $("field")) {
			this.nextToken();
			if (this.isFieldType() == false) {
				failBlock.value_("parseRestrictedInterfaceDeclaration: fieldType is exptected after #field");
				return;
			}

			Object fieldType = token;
			this.nextToken();
			if (tokenType != $("identifier")) {
				failBlock.value_("parseRestrictedInterfaceDeclaration: fieldId is expected after fieldType");
				return;
			}

			Object fieldId = token;
			this.parseFieldValue();
			aDictionary.put(fieldId, new Object[] { fieldType, parseNode });
			parseNode = aDictionary;
			return;
		}

		failBlock.value_("parseRestrictedInterfaceDeclaration: invalue restrictedInterfaceDeclaration");
	}

	/**
	 * Parse a field value.
	 * 
	 * @category parse field
	 */
	public void parseFieldValue() {
		this.nextToken();

		if (tokenType == $("leftBracket")) {
			this.unNextToken();
			this.parseFieldValueWithLeftBracket();
			return;
		}

		if ((tokenType == $("TRUE")) || (tokenType == $("FALSE"))) {
			parseNode = token;
			return;
		}

		if (tokenType == $("number")) {
			this.unNextToken();
			this.parseFieldValueWithNumbers();
			return;
		}

		if (tokenType == $("string")) {
			this.unNextToken();
			this.parseFieldValueWithStrings();
			return;
		}

		if (this.isNodeStatement()) {
			this.unNextToken();
			this.parseFieldValueWithNodeStatements();
			return;
		}

		failBlock.value_("parseFieldValue: invalid fieldValue");
	}

	/**
	 * Parse a field value with left bracket.
	 * 
	 * @category parse field
	 */
	public void parseFieldValueWithLeftBracket() {
		this.nextToken();
		if (tokenType != $("leftBracket")) {
			failBlock.value_("parseFieldValueWithLeftBracket: internal error");
			return;
		}

		this.nextToken();
		if (tokenType == $("rightBracket")) {
			parseNode = null;
			return;
		}

		this.unNextToken();
		this.parseFieldValue();

		this.nextToken();
		if (tokenType != $("rightBracket")) {
			this.unNextToken();
			failBlock.value_("parseFieldValueWithLeftBracket: does not end with #rightBracket");
			return;
		}
	}

	/**
	 * Parse a field value with node statements.
	 * 
	 * @category parse field
	 */
	public void parseFieldValueWithNodeStatements() {
		Vector fieldValues = new Vector();
		this.nextToken();
		while (this.isNodeStatement()) {
			this.unNextToken();
			this.parseNodeStatement();
			if (parseNode != null) {
				fieldValues.addElement(parseNode);
			}
			this.nextToken();
		}

		this.unNextToken();
		parseNode = fieldValues;
		if (fieldValues.size() == 0) {
			parseNode = null;
		}
		if (fieldValues.size() == 1) {
			parseNode = fieldValues.elementAt(0);
		}
	}

	/**
	 * Parse a field value with numbers.
	 * 
	 * @category parse field
	 */
	public void parseFieldValueWithNumbers() {
		Vector fieldValues = new Vector();
		this.nextToken();
		while (tokenType == $("number")) {
			fieldValues.addElement(token);
			this.nextToken();
		}

		this.unNextToken();
		parseNode = fieldValues;
		if (fieldValues.size() == 0) {
			parseNode = null;
		}
		if (fieldValues.size() == 1) {
			parseNode = fieldValues.elementAt(0);
		}
	}

	/**
	 * Parse a field value with strings.
	 * 
	 * @category parse field
	 */
	public void parseFieldValueWithStrings() {
		Vector fieldValues = new Vector();
		this.nextToken();
		while (tokenType == $("string")) {
			fieldValues.addElement(token);
			this.nextToken();
		}

		this.unNextToken();
		parseNode = fieldValues;
		if (fieldValues.size() == 0) {
			parseNode = null;
		}
		if (fieldValues.size() == 1) {
			parseNode = fieldValues.elementAt(0);
		}
	}

	/**
	 * Generate a new node with fields.
	 * 
	 * @param nodeTypeId jp.co.sra.smalltalk.StSymbol
	 * @param fieldsDictionary java.util.Hashtable
	 * @return java.lang.Object
	 * @category private
	 */
	protected Object generateNewNode_with_(StSymbol nodeTypeId, Hashtable fieldsDictionary) {
		Object[] anArray = (Object[]) protoDefinitions.get(nodeTypeId);
		if (anArray == null) {
			// It's an ordinary node.
			return builder.newNode_with_ifFailed_(nodeTypeId, fieldsDictionary, new StBlockClosure() {
				public Object value() {
					return JunVrmlNode.Undefined;
				}
			});
		} else {
			// It's a prototype.
			Hashtable aDictionary = (Hashtable) ((Hashtable) anArray[0]).clone();
			Enumeration keys = fieldsDictionary.keys();
			while (keys.hasMoreElements()) {
				Object key = keys.nextElement();
				aDictionary.put(key, fieldsDictionary.get(key));
			}
			parseNode = Parse_withInterfaceDefinitions_(anArray[1], aDictionary);
			parseNode = ((Object[]) parseNode)[0];
			return parseNode;
		}
	}
}
