package jp.co.sra.smalltalk;

import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Panel;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.WindowEvent;

import javax.swing.JPanel;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.UIManager.LookAndFeelInfo;

import jp.co.sra.smalltalk.menu.MenuEvent;
import jp.co.sra.smalltalk.menu.MenuListener;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StPopupMenu;
import jp.co.sra.smalltalk.menu.StRadioButtonGroup;
import jp.co.sra.smalltalk.menu.StRadioButtonMenuItem;

/**
 * StApplicationModel class
 * 
 *  @author    he weijie
 *  @created   1998/08/18 (by he weijie)
 *  @updated   1999/10/01 (by MATSUDA Ryouichi)
 *  @updated   2002/04/09 (by nisinaka)
 *  @updated   2005/02/22 (by nisinaka)
 *  @updated   2006/10/04 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StApplicationModel.java,v 8.13 2008/02/20 06:33:17 nisinaka Exp $
 */
public abstract class StApplicationModel extends StModel {

	/** view mode */
	public static final int VIEW_AWT = 0;
	public static final int VIEW_SWING = 1;
	public static final int VIEW_SWING_ONLY = 2;
	protected static int DefaultViewMode = VIEW_SWING;

	public static final StMenu DefaultViewModeMenu;

	static {
		DefaultViewModeMenu = CreateDefaultViewModeMenu();

		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
			StApplicationModel.SetDefaultViewMode(StApplicationModel.VIEW_SWING);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Create JPanel with border.
	 * 
	 * @return javax.swing.JPanel
	 * @category Utilities
	 */
	public static JPanel _JPanelWithBorder() {
		return StApplicationWindowSwing._PanelWithBorder();
	}

	/**
	 * Create Panel with border.
	 * 
	 * @return java.awt.Panel
	 * @category Utilities
	 */
	public static Panel _PanelWithBorder() {
		return StApplicationWindowAwt._PanelWithBorder();
	}

	/**
	 * Show the window at the specified point.
	 * 
	 * @param aWindow java.awt.Window
	 * @param aPoint java.awt.Point
	 * @category Utilities
	 */
	public static void _ShowAtPoint(Window aWindow, Point aPoint) {
		Rectangle screenBounds = _GetScreenBoundsContainsPoint(aPoint);
		if (screenBounds != null) {
			Dimension windowSize = aWindow.getSize();
			int x = aPoint.x - windowSize.width / 2;
			if (x < screenBounds.x) {
				x = screenBounds.x;
			} else if (screenBounds.x + screenBounds.width < x + windowSize.width) {
				x = screenBounds.x + screenBounds.width - windowSize.width;
			}

			int y = aPoint.y - windowSize.height / 2;
			if (y < screenBounds.y) {
				y = screenBounds.y;
			} else if (screenBounds.y + screenBounds.height < y + windowSize.height) {
				y = screenBounds.y + screenBounds.height - windowSize.height;
			}

			aWindow.setLocation(x, y);
		}

		aWindow.setVisible(true);
	}

	/**
	 * Show the window at the center of the display.
	 * 
	 * @param aWindow java.awt.Window
	 * @category Utilities
	 */
	public static void _ShowAtCenterPoint(Window aWindow) {
		Rectangle screenBounds = _GetScreenBoundsContainsPoint(SystemInterface._MousePoint());
		_ShowAtPoint(aWindow, new StRectangle(screenBounds).center());
	}

	/**
	 * Show the window at the current mouse point.
	 * 
	 * @param aWindow java.awt.Window
	 * @category Utilities
	 */
	public static void _ShowAtMousePoint(Window aWindow) {
		_ShowAtPoint(aWindow, SystemInterface._MousePoint());
	}

	/**
	 * Answer the screen bounds which contains the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Rectangle
	 * @category Utilities
	 */
	public static Rectangle _GetScreenBoundsContainsPoint(Point aPoint) {
		GraphicsEnvironment graphicsEnvironment = GraphicsEnvironment.getLocalGraphicsEnvironment();
		GraphicsDevice[] screenDevices = graphicsEnvironment.getScreenDevices();
		for (int i = 0; i < screenDevices.length; i++) {
			Rectangle bounds = screenDevices[i].getDefaultConfiguration().getBounds();
			if (bounds.contains(aPoint)) {
				return bounds;
			}
		}
		return null;
	}

	/**
	 * Answer the default view mode.
	 * 
	 * @return int
	 * @category View mode
	 */
	public static int GetDefaultViewMode() {
		return DefaultViewMode;
	}

	/**
	 * Set the default view mode.
	 * 
	 * @param mode int
	 * @category View mode
	 */
	public static void SetDefaultViewMode(int mode) {
		DefaultViewMode = mode;
		UpdateDefaultViewModeMenu();
	}

	/**
	 * Set the default view mode to AWT.
	 * 
	 * @category View mode
	 */
	public static void SetDefaultViewModeAwt() {
		SetDefaultViewMode(VIEW_AWT);
	}

	/**
	 * Set the default view mode to Swing.
	 * 
	 * @category View mode
	 */
	public static void SetDefaultViewModeSwing(LookAndFeelInfo aLookAndFeelInfo) {
		try {
			UIManager.setLookAndFeel(aLookAndFeelInfo.getClassName());
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (UnsupportedLookAndFeelException e) {
			e.printStackTrace();
		}

		SetDefaultViewMode(VIEW_SWING);
	}

	/**
	 * Set the default view mode to Swing Only.
	 * 
	 * @category View mode
	 */
	public static void SetDefaultViewModeSwingOnly() {
		SetDefaultViewMode(VIEW_SWING_ONLY);
	}

	/**
	 * Update the DefaultViewModeMenu.
	 * 
	 * @category View mode
	 */
	public static void UpdateDefaultViewModeMenu() {
		StSymbol key = null;
		switch (GetDefaultViewMode()) {
			case VIEW_AWT:
				key = $("viewAwt");
				break;
			case VIEW_SWING:
				key = $("viewSwing:" + UIManager.getLookAndFeel().getClass().getName());
				break;
			case VIEW_SWING_ONLY:
				key = $("viewSwingOnly");
				break;
		}
		if (key == null) {
			return;
		}

		StRadioButtonMenuItem aMenuItem = (StRadioButtonMenuItem) DefaultViewModeMenu.atNameKey_(key);
		if (aMenuItem != null && aMenuItem.isSelected() == false) {
			aMenuItem.beOn();
		}
	}

	/**
	 * Create a "Default View Mode" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category View mode
	 */
	protected static StMenu CreateDefaultViewModeMenu() {
		StRadioButtonGroup aGroup = new StRadioButtonGroup();

		StMenu swingModeMenu = new StMenu("Swing");
		UIManager.LookAndFeelInfo[] info = UIManager.getInstalledLookAndFeels();
		for (int i = 0; i < info.length; i++) {
			final LookAndFeelInfo lookAndFeelInfo = info[i];
			String label = lookAndFeelInfo.getName();
			StSymbol nameKey = $("viewSwing:" + lookAndFeelInfo.getClassName());
			swingModeMenu.add(new StRadioButtonMenuItem(label, nameKey, aGroup, new MenuListener() {
				public void menuPerformed(MenuEvent aMenuEvent) {
					SetDefaultViewModeSwing(lookAndFeelInfo);
				}
			}));
		}

		StMenu aMenu = new StMenu(new StUserMessage("Default View Mode"));
		aMenu.add(new StRadioButtonMenuItem("AWT", $("viewAwt"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultViewModeAwt();
			}
		}));
		aMenu.add(swingModeMenu);

		/*
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Swing only"), $("viewSwingOnly"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultViewModeSwingOnly();
			}
		}));
		*/

		return aMenu;
	}

	////////////////////////////////////////

	protected StUIBuilder builder = null;

	/**
	 * Create a new instance of the ApplicationModel and initialize it.
	 * 
	 * @category Instance creation
	 */
	public StApplicationModel() {
		super();
		this.initialize();
	}

	/**
	 * Initialize the ApplicationModel when created.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		// Nothing to do by default.
	}

	/**
	 * Answer my builder.
	 * 
	 * @return jp.co.sra.smalltalk.StUIBuilder
	 * @category accessing
	 */
	public final StUIBuilder builder() {
		if (builder == null) {
			builder = new StUIBuilder(this);
		}
		return builder;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category defaults
	 */
	public abstract StView defaultView();

	/**
	 * The application is checking to see if the application is in a state such that it can be closed.
	 * 
	 * @return boolean
	 * @category events
	 */
	protected boolean requestForWindowClose() {
		return true;
	}

	/**
	 * Answer whether a window resizable or not.
	 * 
	 * @return boolean
	 * @category testing
	 */
	protected boolean windowResizable() {
		return true;
	}

	/**
	 * Create an application window of the default view and open it.
	 * 
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame open() {
		return this.openView_(this.defaultView());
	}

	/**
	 * Create an application window of the view and open it.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @return jp.co.sra.smalltalk.StApplicationWindow
	 * @category interface opening
	 */
	public Frame openView_(StView aView) {
		Frame aFrame = this.allButOpenView_(aView);
		_ShowAtMousePoint(aFrame);
		return aFrame;
	}

	/**
	 * This method will be sent by the view when its top component is opened.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StUIBulider#windowOpened(WindowEvent)
	 * @category interface opening
	 */
	public void postOpenWith_(StView aView) {
		// nothing to do.
	}

	/**
	 * Create a Frame for the application model.
	 * Do everything but open the frame.
	 *
	 * @param aView jp.co.sra.smalltalk.StView
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	protected Frame allButOpenView_(StView aView) {
		StApplicationWindow anApplicationWindow = this._createWindow(aView);
		this.builder().addWindow_(anApplicationWindow, aView);
		aView._setupKeyboard(anApplicationWindow.toWindow());
		return anApplicationWindow.toFrame();
	}

	/**
	 * Create an application window.
	 * 
	 * @return jp.co.sra.smalltalk.StApplicationModel
	 * @category interface opening
	 */
	protected StApplicationWindow _createWindow(StView aView) {
		StApplicationWindow anApplicationWindow;
		if (GetDefaultViewMode() == VIEW_AWT) {
			anApplicationWindow = new StApplicationWindowAwt(this, aView);
		} else {
			anApplicationWindow = new StApplicationWindowSwing(this, aView);
		}
		return anApplicationWindow;
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return this.toString();
	}

	/**
	 * Answer whether the view is placed on a bordered panel or not.
	 * 
	 * @return boolean
	 * @category interface opening
	 */
	protected boolean _viewOnBorderedPanel() {
		return false;
	}

	/**
	 * Close all windows of the application.
	 * 
	 * @category interface closing
	 */
	public void closeRequest() {
		Window aWindow = this.builder().window();
		if (aWindow != null) {
			aWindow.dispatchEvent(new WindowEvent(aWindow, WindowEvent.WINDOW_CLOSING));
		}
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		Window[] windows = this.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			this.builder().removeWindow_(windows[i]);
			windows[i].dispose();
		}
	}

	/**
	 * Invoked when all windows are closed.
	 *  
	 * @category interface closing
	 */
	public void allWindowsClosed() {
		builder = null;
	}

	/**
	 * A utility method to make all windows expand and raise.
	 * 
	 * @category utilities
	 */
	public void _windowExpandAndRaise() {
		Window[] windows = this.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			if (windows[i] instanceof Frame) {
				((Frame) windows[i]).setState(Frame.NORMAL);
			}
			windows[i].setVisible(true);
		}
	}

	/**
	 * Show the view.
	 * If the view already exists, just expand and raise it.
	 * Otherwise open a new one.
	 * 
	 * @category utilities
	 */
	public void _show() {
		if (this.builder().windows().length > 0) {
			this._windowExpandAndRaise();
		} else {
			this.open();
		}
	}

	/**
	 * Update the title string of my windows.
	 * 
	 * @category updating
	 */
	public void updateWindowTitle() {
		Window[] windows = this.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			Window aWindow = windows[i];
			if (aWindow instanceof StApplicationWindow) {
				StApplicationWindow applicationWindow = (StApplicationWindow) aWindow;
				if (applicationWindow.application() == this) {
					applicationWindow.updateWindowTitle();
				}
			}
		}
	}

	/**
	 * Update the menu bar of my windows.
	 * 
	 * @category updating
	 */
	public void updateMenuBar() {
		Window[] windows = this.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			Window aWindow = windows[i];
			if (aWindow instanceof StApplicationWindow) {
				((StApplicationWindow) aWindow).updateMenuBar();
			}
		}
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		return null;
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		return null;
	}

}
