package jp.co.sra.smalltalk;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;

/**
 * StRectangle class
 * 
 *  @author    nisinaka
 *  @created   2002/10/28 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StRectangle.java,v 8.11 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StRectangle extends StObject {

	protected int originX;
	protected int originY;
	protected int cornerX;
	protected int cornerY;

	/**
	 * Create a new instance of StRectangle.
	 * 
	 * @param newOriginX int
	 * @param newOriginY int
	 * @param newCornerX int
	 * @param newCornerY int
	 * @category Instance creation
	 */
	public StRectangle(int newOriginX, int newOriginY, int newCornerX, int newCornerY) {
		originX = newOriginX;
		originY = newOriginY;
		cornerX = newCornerX;
		cornerY = newCornerY;
	}

	/**
	 * Create a new instance of StRectangle.
	 * 
	 * @param aPoint java.awt.Point
	 * @param aDimension java.awt.Dimension
	 * @category Instance creation
	 */
	public StRectangle(Point aPoint, Dimension aDimension) {
		this(aPoint.x, aPoint.y, aPoint.x + aDimension.width, aPoint.y + aDimension.height);
	}

	/**
	 * Create a new instance of StRectangle.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @category Instance creation
	 */
	public StRectangle(Dimension aDimension) {
		this(0, 0, aDimension.width, aDimension.height);
	}

	/**
	 * Create a new instance of StRectangle.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @category Instance creation
	 */
	public StRectangle(Rectangle aRectangle) {
		this(aRectangle.x, aRectangle.y, aRectangle.x + aRectangle.width, aRectangle.y + aRectangle.height);
	}

	/**
	 * Create a new instance of StRectangle with the origin and the corner.
	 * 
	 * @param newOrigin java.awt.Point
	 * @param newCorner java.awt.Point
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category Instance creation
	 */
	public static StRectangle Origin_corner_(Point newOrigin, Point newCorner) {
		return new StRectangle(newOrigin.x, newOrigin.y, newCorner.x, newCorner.y);
	}

	/**
	 * Create a new instance of StRectangle with the origin and the extent.
	 * 
	 * @param originPoint java.awt.Point
	 * @param extentPoint java.awt.Point
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category Instance creation
	 */
	public static StRectangle Origin_extent_(Point originPoint, Point extentPoint) {
		return new StRectangle(originPoint.x, originPoint.y, originPoint.x + extentPoint.x, originPoint.y + extentPoint.y);
	}

	/**
	 * Create a new instance of StRectangle with two points.
	 * 
	 * @param point1 java.awt.Point
	 * @param point2 java.awt.Point
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category Instance creation
	 */
	public static StRectangle Vertex_vertex_(Point point1, Point point2) {
		return new StRectangle(Math.min(point1.x, point2.x), Math.min(point1.y, point2.y), Math.max(point1.x, point2.x), Math.max(point1.y, point2.y));
	}

	/**
	 * Answer the receiver's area, the product of width and height.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int area() {
		return this.width() * this.height();
	}

	/**
	 * Answer my current bottom horizontal line.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int bottom() {
		return cornerY;
	}

	/**
	 * Answer the point at the center of the bottom horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point bottomCenter() {
		return new Point((originX + cornerX) / 2, cornerY);
	}

	/**
	 * Answer the point at the left edge of the bottom horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point bottomLeft() {
		return new Point(originX, cornerY);
	}

	/**
	 * Answer the point at the right edge of the bottom horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point bottomRight() {
		return this.corner();
	}

	/**
	 * Answer my center point.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point center() {
		return new Point((originX + cornerX) / 2, (originY + cornerY) / 2);
	}

	/**
	 * Answer the point at the right edge of the bottom horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point corner() {
		return new Point(cornerX, cornerY);
	}

	/**
	 * Answer my current x-coordinate of the corner point.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int cornerX() {
		return cornerX;
	}

	/**
	 * Answer my current y-coordinate of the corner point.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int cornerY() {
		return cornerY;
	}

	/**
	 * Answer the extent of the receiver.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension extent() {
		return new Dimension(this.width(), this.height());
	}

	/**
	 * Answer my height.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int height() {
		return cornerY - originY;
	}

	/**
	 * Answer my current left vertical line.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int left() {
		return originX;
	}

	/**
	 * Answer the point at the center of the receiver's left vertical line.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point leftCenter() {
		return new Point(this.left(), this.center().y);
	}

	/**
	 * Answer the point at the left edge of the top horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point origin() {
		return new Point(originX, originY);
	}

	/**
	 * Answer my current x-coordinate of the origin point.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int originX() {
		return originX;
	}

	/**
	 * Answer my current y-coordinate of the origin point.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int originY() {
		return originY;
	}

	/**
	 * Set my new origin and corner.
	 * 
	 * @param newOrigin java.awt.Point
	 * @param newCorner java.awt.Point
	 * @category accessing
	 */
	public void origin_corner_(Point newOrigin, Point newCorner) {
		originX = newOrigin.x;
		originY = newOrigin.y;
		cornerX = newCorner.x;
		cornerY = newCorner.y;
	}

	/**
	 * Answer my current right vertical line.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int right() {
		return cornerX;
	}

	/**
	 * Answer the point at the center of the receiver's right vertical line.
	 * 
	 * @return Point
	 * @category accessing
	 */
	public Point rightCenter() {
		return new Point(this.right(), this.center().y);
	}

	/**
	 * Answer my current top horizontal line.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int top() {
		return originY;
	}

	/**
	 * Answer the point at the center of the top horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point topCenter() {
		return new Point((originX + cornerX) / 2, originY);
	}

	/**
	 * Answer the point at the left edge of the top horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point topLeft() {
		return this.origin();
	}

	/**
	 * Answer the point at the right edge of the top horizontal line of the receiver.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point topRight() {
		return new Point(cornerX, originY);
	}

	/**
	 * Answer my width.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int width() {
		return cornerX - originX;
	}

	/**
	 * Answer my current x-coordinate of the origin point.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int x() {
		return this.originX();
	}

	/**
	 * Answer my current y-coordinate of the origin point.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int y() {
		return this.originY();
	}

	/**
	 * Answer the point specified with the symbol.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point _pointAt(StSymbol aSymbol) {
		if (aSymbol == $("bottomCenter")) {
			return this.bottomCenter();
		} else if (aSymbol == $("bottomLeft")) {
			return this.bottomLeft();
		} else if (aSymbol == $("bottomRight")) {
			return this.bottomRight();
		} else if (aSymbol == $("center")) {
			return this.center();
		} else if (aSymbol == $("corner")) {
			return this.corner();
		} else if (aSymbol == $("leftCenter")) {
			return this.leftCenter();
		} else if (aSymbol == $("origin")) {
			return this.origin();
		} else if (aSymbol == $("rightCenter")) {
			return this.rightCenter();
		} else if (aSymbol == $("topCenter")) {
			return this.topCenter();
		} else if (aSymbol == $("topLeft")) {
			return this.topLeft();
		} else if (aSymbol == $("topRight")) {
			return this.topRight();
		}
		return null;
	}

	/**
	 * Indicates whether some other object is "equal to" this one.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || anObject instanceof StRectangle == false) {
			return false;
		}
		StRectangle aRectangle = (StRectangle) anObject;
		return (this.originX == aRectangle.originX && this.originY == aRectangle.originY && this.cornerX == aRectangle.cornerX && this.cornerY == aRectangle.cornerY);
	}

	/**
	 * Returns a hash code value for the object.
	 * 
	 * @return int
	 * @see java.lang.Object#hashCode()
	 * @category comparing
	 */
	public int hashCode() {
		int hash = originX;
		hash = hash * 31 + originY;
		hash = hash * 31 + cornerX;
		hash = hash * 31 + cornerY;
		return hash;
	}

	/**
	 * Answer a Collection of Rectangles comprising the parts of the receiver that do not lie within aRectangle.
	 * 
	 * @param aRectangle
	 * @return array of rectangle
	 * @category rectangle functions
	 */
	public StRectangle[] areasOutside_(StRectangle aRectangle) {
		// Make sure the intersection is non-empty
		ArrayList areas;
		int yOrigin;
		int yCorner;

		if (!this.intersects_(aRectangle)) {
			return new StRectangle[] { this };
		}
		areas = new ArrayList();
		if (aRectangle.origin().y > originY) {
			yOrigin = aRectangle.origin().y;
			areas.add(new StRectangle(originX, originY, cornerX, yOrigin));
		} else {
			yOrigin = originY;
		}
		if (aRectangle.corner().y < cornerY) {
			yCorner = aRectangle.corner().y;
			areas.add(new StRectangle(originX, yCorner, cornerX, cornerY));
		} else {
			yCorner = cornerY;
		}
		if (aRectangle.origin().x > originX) {
			areas.add(new StRectangle(originX, yOrigin, aRectangle.origin().x, yCorner));
		}
		if (aRectangle.corner().x < cornerX) {
			areas.add(new StRectangle(aRectangle.corner().x, yOrigin, corner().x, yCorner));
		}
		StRectangle[] array = new StRectangle[areas.size()];
		for (int i = 0; i < areas.size(); i++) {
			array[i] = (StRectangle) areas.get(i);
		}
		return array;
	}

	/**
	 * Answer an StRectangle that is outset from the receiver by delta.
	 * 
	 * @param delta int
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle expandedBy_(int delta) {
		return new StRectangle(originX - delta, originY - delta, cornerX + delta, cornerY + delta);
	}

	/**
	 * Answer an StRectangle that is outset from the receiver by delta.
	 * 
	 * @param deltaX int
	 * @param deltaY int
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle expandedBy_(int deltaX, int deltaY) {
		return new StRectangle(originX - deltaX, originY - deltaY, cornerX + deltaX, cornerY + deltaY);
	}

	/**
	 * Answer an StRectangle that is outset from the receiver by delta.
	 * 
	 * @param delta java.awt.Point
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle expandedBy_(Point delta) {
		return new StRectangle(originX - delta.x, originY - delta.y, cornerX + delta.x, cornerY + delta.y);
	}

	/**
	 * Answer an StRectangle that is outset from the receiver by delta.
	 * 
	 * @param delta jp.co.sra.smalltalk.StRectangle
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle expandedBy_(StRectangle delta) {
		return new StRectangle(originX - delta.originX, originY - delta.originY, cornerX + delta.cornerX, cornerY + delta.cornerY);
	}

	/**
	 * Answer an StRectangle that is inset from the receiver by delta.
	 * 
	 * @param delta int
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle insetBy_(int delta) {
		return new StRectangle(originX + delta, originY + delta, cornerX - delta, cornerY - delta);
	}

	/**
	 * Answer an StRectangle that is inset from the receiver by delta.
	 * 
	 * @param delta java.awt.Point
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle insetBy_(Point delta) {
		return new StRectangle(originX + delta.x, originY + delta.y, cornerX - delta.x, cornerY - delta.y);
	}

	/**
	 * Answer an StRectangle that is inset from the receiver by delta.
	 * 
	 * @param delta jp.co.sra.smalltalk.StRectangle
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle insetBy_(StRectangle delta) {
		return new StRectangle(originX + delta.originX, originY + delta.originY, cornerX - delta.cornerX, cornerY - delta.cornerY);
	}

	/**
	 * Answer the intersect with the rectangle.
	 * 
	 * @param aRectangle jp.co.sra.smalltalk.StRectangle
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle intersect_(StRectangle aRectangle) {
		return new StRectangle(Math.max(originX, aRectangle.originX), Math.max(originY, aRectangle.originY), Math.min(cornerX, aRectangle.cornerX), Math.min(cornerY, aRectangle.cornerY));
	}

	/**
	 * Answer whether aRectangle intersects the receiver anywhere.
	 * 
	 * @param aRectangle jp.co.sra.smalltalk.StRectangle
	 * @return boolean
	 * @category rectangle functions
	 */
	public boolean intersects_(StRectangle aRectangle) {
		return (this.originX() < aRectangle.cornerX() && this.originY() < aRectangle.cornerY()) && (aRectangle.originX() < this.cornerX() && aRectangle.originY() < this.cornerY())
				&& (this.originX() < aRectangle.cornerX() && this.originY() < aRectangle.cornerY()) && (aRectangle.originX() < aRectangle.cornerX() && aRectangle.originY() < aRectangle.cornerY());
	}

	/**
	 * Answer a new StRectangle that contains both the receiver and the argument rectangle.
	 * 
	 * @param aRectangle jp.co.sra.smalltalk.StRectangle
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category rectangle functions
	 */
	public StRectangle merge_(StRectangle aRectangle) {
		if (aRectangle != null) {
			return new StRectangle(Math.min(originX, aRectangle.originX), Math.min(originY, aRectangle.originY), Math.max(cornerX, aRectangle.cornerX), Math.max(cornerY, aRectangle.cornerY));
		} else {
			return new StRectangle(this.left(), this.top(), this.right(), this.bottom());
		}
	}

	/**
	 * Answer true if the point is contained in the rectangle, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Point aPoint) {
		return (originX <= aPoint.x) && (originY <= aPoint.y) && (aPoint.x <= cornerX) && (aPoint.y <= cornerY);
	}

	/**
	 * Answer a new StRectangle that is a translated by aPoint2 - aPoint1.
	 * 
	 * @param aPoint1 java.awt.Point
	 * @param aPoint2 java.awt.Point
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category transforming
	 */
	public StRectangle align_with_(Point aPoint1, Point aPoint2) {
		return this.translatedBy_(aPoint2.x - aPoint1.x, aPoint2.y - aPoint1.y);
	}

	/**
	 * Answer a new StRectangle scaled by the amount.
	 * 
	 * @param amountX double
	 * @param amountY double
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category transforming
	 */
	public StRectangle scaledBy_(double amountX, double amountY) {
		Point origin = new Point((int) Math.round(originX * amountX), (int) Math.round(originY * amountY));
		Point corner = new Point((int) Math.round(cornerX * amountX), (int) Math.round(cornerY * amountY));
		return StRectangle.Vertex_vertex_(origin, corner);
	}

	/**
	 * Answer a new StRectangle translated by delta.
	 * 
	 * @param deltaX int
	 * @param deltaY int
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category transforming
	 */
	public StRectangle translatedBy_(int deltaX, int deltaY) {
		return new StRectangle(originX + deltaX, originY + deltaY, cornerX + deltaX, cornerY + deltaY);
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @category printing 
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write(String.valueOf(originX));
		aWriter.write('@');
		aWriter.write(String.valueOf(originY));
		aWriter.write(" corner: ");
		aWriter.write(String.valueOf(cornerX));
		aWriter.write('@');
		aWriter.write(String.valueOf(cornerY));
	}

	/**
	 * Convert to a Rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @category converting
	 */
	public Rectangle toRectangle() {
		return new Rectangle(originX, originY, cornerX - originX, cornerY - originY);
	}

}
