package jp.co.sra.smalltalk;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.Insets;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.event.WindowEvent;

import javax.swing.SwingUtilities;

/**
 * StSimpleDialog class
 * 
 *  @author    nisinaka
 *  @created   2003/03/07 (by nisinaka)
 *  @updated   2006/04/19 (by nisinaka)
 *  @updated   2007/10/04 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StSimpleDialog.java,v 8.13 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StSimpleDialog extends StApplicationModel {

	protected StValueHolder accept;
	protected StValueHolder cancel;
	protected StValueHolder close;
	protected boolean escapeIsCancel;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		accept = new StValueHolder(false);
		cancel = new StValueHolder(false);
		close = new StValueHolder(false);
		escapeIsCancel = true;
	}

	/**
	 * Control whether the escape character cancels the dialog.
	 *
	 * @param aBoolean boolean
	 * @category initialize-release
	 */
	public void escapeIsCancel_(boolean aBoolean) {
		escapeIsCancel = aBoolean;
	}

	/**
	 * Answer the value holder for the accept status.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder accept() {
		return accept;
	}

	/**
	 * Answer the value holder for the cancel status.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder cancel() {
		return cancel;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new StSimpleDialogViewAwt(this) {
				public Insets getInsets() {
					return new Insets(5, 5, 5, 5);
				}
			};
		} else {
			return new StSimpleDialogViewSwing(this) {
				public Insets getInsets() {
					return new Insets(5, 5, 5, 5);
				}
			};
		}
	}

	/**
	 * Do everything short of actually opening the window.
	 * 
	 * @category interface construction
	 */
	protected void preOpen() {
		accept.onChangeSend_to_("closeAccept", this);
		cancel.onChangeSend_to_("closeCancel", this);
	}

	/**
	 * Answer the string for the accept button label.
	 *
	 * @return java.lang.String
	 * @category interface construction
	 */
	protected String _acceptButtonLabelString() {
		return "OK";
	}

	/**
	 * Answer the string for the cancel button label.
	 *
	 * @return java.lang.String
	 * @category interface construction
	 */
	protected String _cancelButtonLabelString() {
		return "Cancel";
	}

	/**
	 * Open a dialog.
	 *
	 * @return java.lang.Object
	 * @category interface opening
	 */
	public Object openDialog() {
		return this.openDialogFor(null);
	}

	/**
	 * Open a dialog for the parent view.
	 * 
	 * @param aComponentOrNull jp.co.sra.smalltalk.StView
	 * @return java.lang.Object
	 * @category interface opening
	 */
	public Object openDialogFor(Component aComponentOrNull) {
		Dialog aDialog = this.allButOpenDialogFor(aComponentOrNull);
		_ShowAtMousePoint(aDialog);
		return accept.value();
	}

	/**
	 * Do everything short of actually opening the window.
	 *
	 * @return java.awt.Dialog
	 * @category interface opening
	 */
	protected Dialog allButOpenDialog() {
		return this.allButOpenDialog((StSimpleDialogView) this.defaultView());
	}

	/**
	 * Do everything short of actually opening the dialog with the view.
	 * 
	 * @param aSimpleDialogView jp.co.sra.jun.smalltalk.StSimpleDialogView
	 * @return java.awt.Dialog
	 * @category interface opening
	 */
	protected Dialog allButOpenDialog(StSimpleDialogView aSimpleDialogView) {
		return this.allButOpenDialogFor(aSimpleDialogView, null);
	}

	/**
	 * Do everything short of actually opening the dialog with the view.
	 * 
	 * @param aComponentOrNull java.awt.Component
	 * @return java.awt.Dialog
	 * @category interface opening
	 */
	protected Dialog allButOpenDialogFor(Component aComponentOrNull) {
		return this.allButOpenDialogFor((StSimpleDialogView) this.defaultView(), aComponentOrNull);
	}

	/**
	 * Do everything short of actually opening the dialog with the view.
	 * 
	 * @param aSimpleDialogView jp.co.sra.jun.smalltalk.StSimpleDialogView
	 * @param aComponentOrNull java.awt.Component
	 * @return java.awt.Dialog
	 * @category interface opening
	 */
	protected Dialog allButOpenDialogFor(StSimpleDialogView aSimpleDialogView, Component aComponentOrNull) {
		Dialog aDialog = aSimpleDialogView.defaultDialog(aComponentOrNull);
		aSimpleDialogView._setupKeyboard(aDialog);
		this.builder().addWindow_(aDialog, aSimpleDialogView);
		this.preOpen();
		return aDialog;
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return "";
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * Replacement for the ApplicationDialogController>>closeNoTerminate in Smalltalk.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose()
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);

		this.simulateEscapeKey();
	}

	/**
	 * A utility method to open a dialog.
	 * Ask the user a question.
	 * Let the user pick form a row of buttons made up to match the labels collection.
	 * Return the response from the corresponding item from the values collection.
	 * 
	 * @param messageString java.lang.String
	 * @param list java.lang.String[]
	 * @param listValues java.lang.Object[]
	 * @param buttons java.lang.String[]
	 * @param buttonValues java.lang.Object[]
	 * @param maxLines int
	 * @param cancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category utility
	 */
	public Object choose_fromList_values_buttons_values_lines_cancel_(String messageString, String[] list, Object[] listValues, String[] buttons, Object[] buttonValues, int maxLines, StBlockClosure cancelBlock) {
		StValueHolder result = new StValueHolder();

		StSimpleDialogView aSimpleDialogView = (StSimpleDialogView) this.defaultView();
		aSimpleDialogView._initializeToChooseFromList(messageString, list, listValues, buttons, buttonValues, maxLines, result);
		Dialog aDialog = aSimpleDialogView.defaultDialog();
		this.builder().addWindow_(aDialog, aSimpleDialogView);
		this.preOpen();
		_ShowAtMousePoint(aDialog);

		if (accept._booleanValue()) {
			if (listValues != null) {
				int index = result._intValue();
				if (0 <= index && index < listValues.length) {
					return listValues[index];
				}
			}
			return null;
		} else if (cancel._booleanValue() && cancelBlock != null) {
			return cancelBlock.value();
		} else {
			return result.value();
		}
	}

	/**
	 * A utility method to open a dialog.
	 * Ask the user a question.
	 * Let the user pick from a row of buttons made up to match the labels collection.
	 * Return the response from the corresponding item from the values collection.
	 *
	 * @param messageString java.lang.String
	 * @param labels java.lang.String[]
	 * @param values java.lang.Object[]
	 * @param defaultValue java.lang.Object
	 * @return java.lang.Object
	 * @category utility
	 */
	public Object choose_labels_values_default_(String messageString, String[] labels, Object[] values, Object defaultValue) {
		return this.choose_labels_values_default_equalize_(messageString, labels, values, defaultValue, true);
	}

	/**
	 * A utility method to open a dialog.
	 * Ask the user a question.
	 * Let the user pick from a row of buttons made up to match the labels collection.
	 * Return the response from the corresponding item from the values collection.
	 *
	 * @param messageString java.lang.String
	 * @param labels java.lang.String[]
	 * @param values java.lang.Object[]
	 * @param defaultValue java.lang.Object
	 * @param eqBoolean boolean
	 * @return java.lang.Object
	 * @category utility
	 */
	public Object choose_labels_values_default_equalize_(String messageString, String[] labels, Object[] values, Object defaultValue, boolean eqBoolean) {
		this.escapeIsCancel_(labels.length < 2);
		StValueHolder result = new StValueHolder();
		if (labels.length == 1) {
			result.value_(defaultValue);
		}

		StSimpleDialogView aSimpleDialogView = (StSimpleDialogView) this.defaultView();
		aSimpleDialogView._initializeToChoose(messageString, labels, values, defaultValue, result, eqBoolean);
		Dialog aDialog = aSimpleDialogView.defaultDialog();
		this.builder().addWindow_(aDialog, aSimpleDialogView);
		this.preOpen();
		_ShowAtMousePoint(aDialog);
		return result.value();
	}

	/**
	 * A utility method to open a dialog.
	 * Ask the user to enter a string and return it.
	 *
	 * @param messageString java.lang.String
	 * @param aString java.lang.String
	 * @param aBlockOrNull jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category utility
	 */
	public Object request_initialAnswer_onCancel_(String messageString, String aString, StBlockClosure aBlockOrNull) {
		StValueHolder string = new StValueHolder(aString);

		StSimpleDialogView aSimpleDialogView = (StSimpleDialogView) this.defaultView();
		aSimpleDialogView._initializeToRequest(messageString, string);
		Dialog aDialog = aSimpleDialogView.defaultDialog();
		this.builder().addWindow_(aDialog, aSimpleDialogView);
		this.preOpen();
		_ShowAtMousePoint(aDialog);

		Object result = accept._booleanValue() ? string.value() : (aBlockOrNull == null ? new String() : aBlockOrNull.value());
		return result;
	}

	/**
	 * Close the dialog.
	 * 
	 * @category private
	 */
	public void close() {
		close.value_(true);

		// This is done by ApplicationDialogController indirectly in Smalltalk by checking isFinishedWithEvent:.
		super.noticeOfWindowClose(null);
	}

	/**
	 * Called when the accept value is changed.
	 * 
	 * @category private
	 */
	public void closeAccept() {
		if (accept._booleanValue()) {
			if (this.requestForWindowClose() == false) {
				accept.value_(false);
				return;
			}
			this.close();
		}
	}

	/**
	 * Called when the accept value is changed.
	 * 
	 * @category private
	 */
	public void closeCancel() {
		this.close();
	}

	/**
	 * Do the cancellation.
	 * 
	 * @category private
	 */
	protected void doCancel() {
		cancel.value_(true);
	}

	/**
	 * Simulate as if the escape key is pressed.
	 * 
	 * @category private
	 */
	protected void simulateEscapeKey() {
		if (escapeIsCancel) {
			this.doCancel();
		}
	}

	/**
	 * Answer the default parent window.
	 * 
	 * @return java.awt.Window
	 * @category Utility
	 */
	public static Window DefaultParentWindow() {
		return KeyboardFocusManager.getCurrentKeyboardFocusManager().getActiveWindow();
	}

	/**
	 * Answer the default parent window of the component.
	 * 
	 * @param aComponentOrNull java.awt.Component
	 * @return java.awt.Window
	 * @category Utility
	 */
	public static Window DefaultParentWindow(Component aComponentOrNull) {
		Window parent = null;
		if (aComponentOrNull != null) {
			parent = SwingUtilities.getWindowAncestor(aComponentOrNull);
		}
		if (parent == null) {
			parent = DefaultParentWindow();
		}
		return parent;
	}

}
