package jp.co.sra.smalltalk;

import java.awt.Component;
import java.awt.Frame;
import java.awt.Window;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * StUIBuilder class
 * 
 *  @author    He Weijie
 *  @created   1998/08/27 (by He Weijie)
 *  @updated   2003/03/04 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StUIBuilder.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StUIBuilder extends StObject implements WindowListener {

	protected StApplicationModel model;
	protected ArrayList windows;
	protected HashMap views;
	protected HashMap namedComponents;

	/**
	 * Create a new instance of StUIBuilder and initialize it with the StApplicationModel.
	 * 
	 * @param aModel jp.co.sra.smalltalk.StApplicationModel
	 * @category Instance creation
	 */
	public StUIBuilder(StApplicationModel aModel) {
		super();

		model = aModel;
		windows = new ArrayList();
		views = new HashMap();
		namedComponents = new HashMap();
	}

	/**
	 * Answer a window.
	 * 
	 * @return java.awt.Window
	 * @category accessing
	 */
	public Window window() {
		return windows.isEmpty() ? null : (Window) windows.get(0);
	}

	/**
	 * Answer the current windows.
	 * 
	 * @return java.awt.Window[]
	 * @category accessing
	 */
	public Window[] windows() {
		return (Window[]) windows.toArray(new Window[windows.size()]);
	}

	/**
	 * Answer the current frames, not windows.
	 * 
	 * @return java.awt.Frame[]
	 * @category accessing
	 */
	public Frame[] frames() {
		Window[] windows = this.windows();
		ArrayList frames = new ArrayList(windows.length);
		for (int i = 0; i < windows.length; i++) {
			if (windows[i] instanceof Frame) {
				frames.add(windows[i]);
			}
		}
		return (Frame[]) frames.toArray(new Frame[frames.size()]);
	}

	/**
	 * Answer my StApplicationModel.
	 * 
	 * @return jp.co.sra.smalltalk.StApplicationModel
	 * @category accessing
	 */
	protected StApplicationModel myModel() {
		return model;
	}

	/**
	 * Add a frame.
	 * 
	 * @param aFrame java.awt.Frame
	 * @deprecated since Jun379, use addWindow_(Window)
	 * @category adding
	 */
	public void addFrame_(Frame aFrame) {
		this.addWindow_(aFrame, null);
	}

	/**
	 * Add the window as a member of the builder.
	 *
	 * @param aWindow java.awt.Window
	 * @param aView jp.co.sra.smalltalk.StView
	 * @category adding
	 */
	public void addWindow_(Window aWindow, StView aView) {
		aWindow.addWindowListener(this);
		windows.add(aWindow);
		views.put(aWindow, aView);
	}

	/**
	 * Add the application window as a member of the builder.
	 * 
	 * @param anApplicationWindow jp.co.sra.smalltalk.StApplicationWindow
	 * @param aView jp.co.sra.smalltalk.StView
	 * @category adding
	 */
	public void addWindow_(StApplicationWindow anApplicationWindow, StView aView) {
		this.addWindow_(anApplicationWindow.toWindow(), aView);
	}

	/**
	 * Remove the frame from the members of the builder.
	 * 
	 * @param aFrame java.awt.Frame
	 * @deprecated since Jun379, use removeWindow_(Window)
	 * @category removing
	 */
	public void removeFrame_(Frame aFrame) {
		this.removeWindow_(aFrame);
	}

	/**
	 * Remove the window from the members of the builder.
	 *
	 * @param aWindow java.awt.Window
	 * @category removing
	 */
	public void removeWindow_(Window aWindow) {
		aWindow.removeWindowListener(this);
		windows.remove(aWindow);
		views.remove(aWindow);
	}

	/**
	 * Retrieve the Component of the indicated component.
	 *
	 * @return java.awt.Component
	 * @param aKey jp.co.sra.smalltalk.StSymbol
	 * @category component handling
	 */
	public Component componentAt_(StSymbol aKey) {
		return (Component) namedComponents.get(aKey);
	}

	/**
	 * Associate aKey with aComponent.
	 *
	 * @param aKey jp.co.sra.smalltalk.StSymbol
	 * @param aComponent java.awt.Component
	 * @category component handling
	 */
	public void componentAt_put_(StSymbol aKey, Component aComponent) {
		namedComponents.put(aKey, aComponent);
	}

	/**
	 * Invoked when a window is activated.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowActivated(WindowEvent e) {
	}

	/**
	 * Invoked when a window has been closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowClosed(WindowEvent e) {
		this.removeWindow_(e.getWindow());

		if (windows.isEmpty()) {
			this.myModel().allWindowsClosed();
		}
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * The close operation can be overridden at this point.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowClosing(WindowEvent e) {
		this.myModel().noticeOfWindowClose(e);
	}

	/**
	 * Invoked when a window is de-activated.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowDeactivated(WindowEvent e) {
	}

	/**
	 * Invoked when a window is de-iconified.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowDeiconified(WindowEvent e) {
	}

	/**
	 * Invoked when a window is iconified.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowIconified(WindowEvent e) {
	}

	/**
	 * Invoked when a window has been opened.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @category event handling
	 */
	public void windowOpened(WindowEvent e) {
		Window aWindow = e.getWindow();
		if (aWindow == null) {
			return;
		}
		StView aView = (StView) views.get(aWindow);
		if (aView == null) {
			return;
		}

		this.myModel().postOpenWith_(aView);
	}

}
