package jp.co.sra.smalltalk;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Window;

import javax.swing.JPanel;
import javax.swing.JPopupMenu;

import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuBarView;
import jp.co.sra.smalltalk.menu.StMenuBarViewSwing;
import jp.co.sra.smalltalk.menu.StPopupMenu;
import jp.co.sra.smalltalk.menu.StPopupMenuViewSwing;

/**
 * StViewJPanel class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1999/11/11 (by MATSUDA Ryouichi)
 *  @updated   2002/04/12 (by nisinaka)
 *  @updated   2002/11/11 (by nisinaka)
 *  @updated   2002/11/21 (by nisinaka)
 *  @updated   2006/10/04 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StViewJPanel.java,v 8.12 2008/02/20 06:33:18 nisinaka Exp $
 */
public abstract class StViewJPanel extends JPanel implements StViewSwing, StDisplayable {

	protected StModel model;
	protected StController controller;
	protected JPopupMenu popupMenuView;
	protected StPopupMenuViewSwing _popupMenuView;
	protected boolean isLightWeightPopupEnabled;

	/**
	 * Create a new instance of StViewJPanel and initialize it.
	 * 
	 * @category Instance creation
	 */
	public StViewJPanel() {
		this(null);
	}

	/**
	 * Create a new instance of StViewJPanel and initialize it.
	 * 
	 * @param newModel jp.co.sra.smalltalk.StModel
	 * @category Instance creation
	 */
	public StViewJPanel(StModel newModel) {
		super();
		this.initialize();
		model = (newModel == null) ? this.defaultModel() : newModel;
		model.addDependent_(this);
		this.buildComponent();
	}

	/**
	 * Replacement for the # notation of Smalltalk. :-)
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Utilities
	 */
	public static final StSymbol $(String aString) {
		return StObject.$(aString);
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key) {
		return StObject.$String(key);
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString) {
		return StObject.$String(key, defaultString);
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @param argument java.lang.Object
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString, Object argument) {
		return StObject.$String(key, defaultString, argument);
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @param argument1 java.lang.Object
	 * @param argument2 java.lang.Object
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString, Object argument1, Object argument2) {
		return StObject.$String(key, defaultString, argument1, argument2);
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString, Object[] arguments) {
		return StObject.$String(key, defaultString, arguments);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		model = null;
		controller = null;
		popupMenuView = null;
		isLightWeightPopupEnabled = true;
	}

	/**
	 * Remove the receiver as a dependent of its model.
	 * 
	 * @see jp.co.sra.smalltalk.StView#release()
	 * @category initialize-release
	 */
	public void release() {
		this.model().removeDependent_(this);

		if (controller != null) {
			controller.release();
			controller = null;
		}
	}

	/**
	 * Make sure to release the receiver.
	 * 
	 * @see javax.swing.JComponent#removeNotify()
	 * @category initialize-release
	 */
	public void removeNotify() {
		this.release();
		super.removeNotify();
	}

	/**
	 * Build this component.
	 * 
	 * @category initialize-release
	 */
	protected void buildComponent() {
		// Do nothing as a default.
	}

	/**
	 * Rebuild this component.
	 * 
	 * @param oldModel jp.co.sra.smalltalk.StModel
	 * @param newModel jp.co.sra.smalltalk.StModel
	 * @category initialize-release
	 */
	protected void rebuildComponent(StModel oldModel, StModel newModel) {
		// Do nothing as a default.
	}

	/**
	 * Answer my current model.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StView#model()
	 * @category model accessing
	 */
	public StModel model() {
		if (model == null) {
			this.model_(this.defaultModel());
		}

		return model;
	}

	/**
	 * Set my new model.
	 * 
	 * @param newModel jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StView#model_(jp.co.sra.smalltalk.StModel)
	 * @category model accessing
	 */
	public void model_(StModel newModel) {
		if (model != newModel) {
			if (model != null) {
				model.removeDependent_(this);
			}

			StModel oldModel = model;
			model = newModel;

			if (model != null) {
				model.addDependent_(this);
			}

			this.rebuildComponent(oldModel, newModel);
			this.repaint();
		}
	}

	/**
	 * Answer a default model. Subclasses may override this.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 * @category model accessing
	 */
	protected StModel defaultModel() {
		return new StModel();
	}

	/**
	 * Answer my current controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StView#controller()
	 * @category controller accessing
	 */
	public StController controller() {
		if (controller == null) {
			this.controller_(this.defaultController());
		}

		return controller;
	}

	/**
	 * Set my new controller.
	 * 
	 * @param newController jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StView#controller_(jp.co.sra.smalltalk.StController)
	 * @category controller accessing
	 */
	public void controller_(StController newController) {
		if (controller != null) {
			controller.release();
		}

		controller = newController;

		if (controller != null) {
			controller.view_(this);
		}
	}

	/**
	 * Answer my default controller. Subclasses may override this.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new StController();
	}

	/**
	 * Make sure to have both a model and a controller before showing.
	 * 
	 * @see java.awt.Component#addNotify()
	 * @category actions
	 */
	public void addNotify() {
		if (model == null) {
			this.model_(this.defaultModel());
		}

		if (controller == null) {
			this.controller_(this.defaultController());
		}

		super.addNotify();
	}

	/**
	 * Answer the receiver as StImage.
	 *
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		int width = this.getWidth();
		int height = this.getHeight();
		StImage anImage = new StImage(width, height);
		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) anImage.image().getGraphics();
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Answer the view as a java component.
	 *
	 * @return java.awt.Component
	 * @see jp.co.sra.smalltalk.StView#toComponent()
	 * @category converting
	 */
	public Component toComponent() {
		return (Component) this;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		// Do nothing as a default.
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		aGraphics.translate(aPoint.x, aPoint.y);
		this.displayOn_(aGraphics);
		aGraphics.translate(-aPoint.x, -aPoint.y);
	}

	/**
	 * Paints the receiver.
	 * 
	 * @param g java.awt.Graphics
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 * @category displaying
	 */
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		this.displayOn_(g);
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument anAspectSymbol is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		repaint();
	}

	/**
	 * Answer the top component of the view.
	 *
	 * @return java.awt.Window
	 * @see jp.co.sra.smalltalk.StView#topComponent()
	 * @category user interface
	 */
	public Window topComponent() {
		Component aComponent = this.toComponent();
		do {
			if (aComponent instanceof Window) {
				return (Window) aComponent;
			}
			aComponent = aComponent.getParent();
		} while (aComponent != null);

		return null;
	}

	/**
	 * Answer my popup menu view.
	 *
	 * @return javax.swing.JPopupMenu
	 * @see jp.co.sra.smalltalk.StViewSwing#popupMenuView()
	 * @category popup menu
	 */
	public JPopupMenu popupMenuView() {
		if (popupMenuView == null) {
			if (model instanceof StApplicationModel) {
				StPopupMenu aPopupMenu = ((StApplicationModel) model)._popupMenu();
				if (aPopupMenu != null) {
					this.popupMenu_(aPopupMenu);
				}
			}
		}
		return popupMenuView;
	}

	/**
	 * Set my popup menu.
	 * 
	 * @param aPopupMenu javax.swing.JPopupMenu
	 * @see jp.co.sra.smalltalk.StViewSwing#popupMenu_(javax.swing.JPopupMenu)
	 * @category popup menu
	 */
	public void popupMenuView_(JPopupMenu aPopupMenu) {
		if (popupMenuView != null) {
			this.remove(popupMenuView);
		}

		popupMenuView = aPopupMenu;

		if (popupMenuView != null) {
			popupMenuView.setLightWeightPopupEnabled(this.isLightWeightPopupEnabled());
			this.add(popupMenuView);
		}
	}

	/**
	 * Set my popup menu view from the StPopupMenu.
	 * 
	 * @param aPopupMenu jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StView#popupMenu_(jp.co.sra.smalltalk.menu.StPopupMenu)
	 * @category popup menu
	 */
	public void popupMenu_(StPopupMenu aPopupMenu) {
		if (_popupMenuView != null) {
			_popupMenuView.release();
		}

		if (aPopupMenu == null) {
			_popupMenuView = null;
			this.popupMenuView_(null);
		} else {
			_popupMenuView = new StPopupMenuViewSwing(aPopupMenu);
			this.popupMenuView_(_popupMenuView.toPopupMenu());
		}
	}

	/**
	 * Show the popup menu at the specified point on the view.
	 *
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.smalltalk.StView#_showPopupMenu(int, int)
	 * @category popup menu
	 */
	public void _showPopupMenu(int x, int y) {
		JPopupMenu popupMenu = this.popupMenuView();
		if (popupMenu != null) {
			popupMenu.show(this, x, y);
		}
	}

	/**
	 * Answer true if a menu is specified to be a light weight one.
	 * 
	 * @return boolean
	 * @see jp.co.sra.smalltalk.StViewSwing#isLightWeightPopupEnabled()
	 * @category popup menu
	 */
	public boolean isLightWeightPopupEnabled() {
		return isLightWeightPopupEnabled;
	}

	/**
	 * Set whether a menu is created as a light weight one or not.
	 * 
	 * @param aFlag boolean
	 * @see jp.co.sra.smalltalk.StViewSwing#setLightWeightPopupEnabled(boolean)
	 * @category popup menu
	 */
	public void setLightWeightPopupEnabled(boolean aFlag) {
		isLightWeightPopupEnabled = aFlag;
	}

	/**
	 * Create my menu bar view.
	 * My subclasses may override this method to change properties of the menu.
	 * 
	 * @param aMenuBar jp.co.sra.smalltalk.menu.StMenuBar
	 * @return jp.co.sra.smalltalk.menu.StMenuBarView
	 * @see jp.co.sra.smalltalk.StView#_createMenuBarView()
	 * @category resources
	 */
	public StMenuBarView _createMenuBarView(StMenuBar aMenuBar) {
		return new StMenuBarViewSwing(aMenuBar);
	}

	/**
	 * Set up the keyboard for the view on the window.
	 * 
	 * @param aWindow
	 * @see jp.co.sra.smalltalk.StView#_setupKeyboard(java.awt.Window)
	 * @category keyboard 
	 */
	public void _setupKeyboard(Window aWindow) {
		// No keyboard handling.
	}

	/**
	 * Answer the bordered panel which contains the receiver.
	 *
	 * @return java.awt.Container
	 * @category interface opening
	 */
	public Container _onBorderedPanel() {
		JPanel aPanel = StApplicationModel._JPanelWithBorder();
		aPanel.setLayout(new BorderLayout());
		aPanel.add(this, BorderLayout.CENTER);
		return aPanel;
	}

}
