/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/GProgram.h"
#include "glib/sys/GSystem.h"
#include "glib/gui/GAbstractTextField.h"

GAbstractTextField::GAbstractTextField ( GWindow& parentWin,
                                         const GString& name,
                                         const GString& constraints,
                                         long winStyle,
                                         long winStyle2 )
                   :GWindow(name, 
                            constraints, 
                            &parentWin, 
                            &parentWin, 
                            winStyle, 
                            winStyle2 | WS2_OS2Y | WS2_NOT_STATIC | WS2_AUTO_PAINT_BACKGROUND),
                    bckColorEnabled(GColor::WHITE),
                    bckColorDisabled(GColor::GRAY),
                    frgColorEnabled(GColor::BLACK),
                    frgColorDisabled(GColor::BLACK)
{
}

GAbstractTextField::~GAbstractTextField ()
{
}

bool GAbstractTextField::onBackgroundColorChanged ( const GColor& color )
{
   if (isEnabled())
      bckColorEnabled = color;
   else
      bckColorDisabled = color;
   GWindow& peer = getPeer();
   peer.setBackgroundColor(color);
   return true;
}

bool GAbstractTextField::onForegroundColorChanged ( const GColor& color )
{
   if (isEnabled())
      frgColorEnabled = color;
   else
      frgColorDisabled = color;
   GWindow& peer = getPeer();
   peer.setForegroundColor(color);
   return true;
}

bool GAbstractTextField::onFontNameSizeChanged ( const GString& fontNameSize )
{
   GWindow& peer = getPeer();
   peer.setFontNameSize(fontNameSize);
   return true;
}

void GAbstractTextField::writeProfile ( const GString& sectName, bool force )
{
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   if (!(winStyle2 & WS2_IGNORE_COLORS_PROFILE))
   {
      bool onlyIf = force || prg.isAutoSaveColorOptions();
      ini.putColor(sectName, "BckColorEnabled", bckColorEnabled, onlyIf);
      ini.putColor(sectName, "FrgColorEnabled", frgColorEnabled, onlyIf);
      ini.putColor(sectName, "BckColorDisabled", bckColorDisabled, onlyIf);
      ini.putColor(sectName, "FrgColorDisabled", frgColorDisabled, onlyIf);
   }
   GWindow::writeProfile(sectName, force);
}

void GAbstractTextField::queryProfile ( const GString& sectName )
{
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();

   if (!(winStyle2 & WS2_IGNORE_COLORS_PROFILE))
   {
      bckColorEnabled = ini.getColor(sectName, "BckColorEnabled", bckColorEnabled);
      frgColorEnabled = ini.getColor(sectName, "FrgColorEnabled", frgColorEnabled);
      bckColorDisabled = ini.getColor(sectName, "BckColorDisabled", bckColorDisabled);
      frgColorDisabled = ini.getColor(sectName, "FrgColorDisabled", frgColorDisabled);
   }

   GWindow::queryProfile(sectName);

   if (!(winStyle2 & WS2_IGNORE_COLORS_PROFILE))
   {
      if (isEnabled())
      {
         setBackgroundColor(bckColorEnabled);
         setForegroundColor(frgColorEnabled);
      }
      else
      {
         setBackgroundColor(bckColorDisabled);
         setForegroundColor(frgColorDisabled);
      }
   }
}

bool GAbstractTextField::getInsertMode () const
{
   return GSystem::GetSystemMetrics(GSystem::SMID_INSERTMODE) != 0;
}

void GAbstractTextField::setInsertMode ( bool flag )
{
   if (flag != getInsertMode())
   {
      ::WinSetSysValue(HWND_DESKTOP, SV_INSERTMODE, flag);

      // Due to a bug (?) in OS/2 (at least Warp 4) I had to implement the
      // below code. Or else the insert mode was not activated until the
      // user moved the keyboard focus and then went back to the MLE.
      if (containsFocusedWindow())
      {
         GWindow* parent = getParentWindow();
         parent->grabFocus();
         grabFocus();
      }
   }
}

void GAbstractTextField::grabFocus ( bool force )
{
   GWindow& peer = getPeer();
   peer.grabFocus(force);
}

void GAbstractTextField::setEnabled ( bool flag, bool repaint )
{
   if (flag == isEnabled())
      return;

   GWindow::setEnabled(flag, false);
   if (isEnabled())
   {
      setBackgroundColor(bckColorEnabled);
      setForegroundColor(frgColorEnabled);
   }
   else
   {
      setBackgroundColor(bckColorDisabled);
      setForegroundColor(frgColorDisabled);
   }

   if (repaint)
      invalidateAll(true);
}

void GAbstractTextField::setOily ( bool flag )
{
   GWindow& peer = getPeer();
   peer.setOily(flag);
   GWindow::setOily(flag);
}

void GAbstractTextField::moveCursorLeft ()
{
   GWindowMessage msg(WM_CHAR, GWindowMessage::Param1(1661014022), GWindowMessage::Param2(1395680));
   GWindow& peer = getPeer();
   peer.callDefaultMsgProc(msg);
}

void GAbstractTextField::moveCursorRight ()
{
   GWindowMessage msg(WM_CHAR, GWindowMessage::Param1(1677791238), GWindowMessage::Param2(1527264));
   GWindow& peer = getPeer();
   peer.callDefaultMsgProc(msg);
}

void GAbstractTextField::moveCursorHome ()
{
   GWindowMessage msg(WM_CHAR, GWindowMessage::Param1(1610682374), GWindowMessage::Param2(1329120));
   GWindow& peer = getPeer();
   peer.callDefaultMsgProc(msg);
}

void GAbstractTextField::moveCursorEnd ()
{
   GWindowMessage msg(WM_CHAR, GWindowMessage::Param1(1694564358), GWindowMessage::Param2(1265632));
   GWindow& peer = getPeer();
   peer.callDefaultMsgProc(msg);
}

void GAbstractTextField::moveCursorWordLeft ()
{
   GWindowMessage msg(WM_CHAR, GWindowMessage::Param1(1661009938), GWindowMessage::Param2(1405920));
   GWindow& peer = getPeer();
   peer.callDefaultMsgProc(msg);
}

void GAbstractTextField::moveCursorWordRight ()
{
   GWindowMessage msg(WM_CHAR, GWindowMessage::Param1(1677787158), GWindowMessage::Param2(1537248));
   GWindow& peer = getPeer();
   peer.callDefaultMsgProc(msg);
}

void GAbstractTextField::selectAll ()
{
   int textLen = getTextLength();
   setSelection(0, textLen);
}

GString GAbstractTextField::getText () const
{
   GWindow& peer = getPeer();
   HWND hwnd = peer.getHWND();
   int len = ::WinQueryWindowTextLength(hwnd);
   char* buff = new char[len+2];
   ::WinQueryWindowText(hwnd, len+2, buff);
   GString ret = GString(buff);
   delete buff;
   return ret;
}

bool GAbstractTextField::isAnySelectedText () const
{
   int startSel = getSelectionStart();
   int endSel = getSelectionEnd();
   return endSel > startSel && startSel >= 0 && endSel >= 1;
}

bool GAbstractTextField::isEmpty () const
{
   return getTextLength() <= 0;
}

GString GAbstractTextField::queryValue () const
{
   return getText();
}

void GAbstractTextField::changeValue ( const GString& newValue, bool notify )
{
   enterCtrlChangedFilter();
   setText(newValue);
   exitCtrlChangedFilter();
   if (notify)
      fireValueChangeListeners();
}
