/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GColor.h"
#include "glib/primitives/GInteger.h"
#include "glib/util/GTokenizer.h"
#include "glib/util/GMath.h"

// Low density RGB Colors (normal).
const GColor GColor::BLACK       = 0x000000;
const GColor GColor::DBLUE       = 0x00007F;
const GColor GColor::DGREEN      = 0x007F00;
const GColor GColor::DCYAN       = 0x007F7F;
const GColor GColor::DRED        = 0x7F0000;
const GColor GColor::DMAGENTA    = 0x7F007F;
const GColor GColor::DYELLOW     = 0x7F7F00;
const GColor GColor::GRAY        = 0xCCCCCC;

// High density RGB Colors (bold/bright).
const GColor GColor::DGRAY       = 0x7F7F7F;
const GColor GColor::BLUE        = 0x0000FF;
const GColor GColor::GREEN       = 0x00FF00;
const GColor GColor::CYAN        = 0x00FFFF;
const GColor GColor::RED         = 0xFF0000;
const GColor GColor::MAGENTA     = 0xFF00FF;
const GColor GColor::YELLOW      = 0xFFFF00;
const GColor GColor::WHITE       = 0xFFFFFF;

// Standard palettes.
GVector<int> GColor::Std16Pal(16);
GVector<int> GColor::Std256Pal(256);

GColor::GColor () 
       :rgb(0)
{ 
}

GColor::GColor ( int red, int green, int blue ) 
       :rgb(((red & 0xFF) << 16) | ((green & 0xFF) << 8) | (blue & 0xFF))
{ 
}

GColor::GColor ( int rgb ) 
       :rgb(rgb)
{ 
}

GColor::GColor ( const GColor& c ) 
       :rgb(c.rgb) 
{ 
}

GColor::~GColor () 
{
}

const GVector<int>& GColor::GetStd16ColorPalette ()
{
   if (Std16Pal.getCount() == 0)
   {
      Std16Pal.ensureCapacity(16);
      Std16Pal.add(GColor::BLACK);
      Std16Pal.add(GColor::DBLUE);
      Std16Pal.add(GColor::DGREEN);
      Std16Pal.add(GColor::DCYAN);
      Std16Pal.add(GColor::DRED);
      Std16Pal.add(GColor::DMAGENTA);
      Std16Pal.add(GColor::DYELLOW);
      Std16Pal.add(GColor::GRAY);
      Std16Pal.add(GColor::DGRAY);
      Std16Pal.add(GColor::BLUE);
      Std16Pal.add(GColor::GREEN);
      Std16Pal.add(GColor::CYAN);
      Std16Pal.add(GColor::RED);
      Std16Pal.add(GColor::MAGENTA);
      Std16Pal.add(GColor::YELLOW);
      Std16Pal.add(GColor::WHITE);
   }

   return Std16Pal;
}

const GVector<int>& GColor::GetStd256ColorPalette ()
{
   if (Std256Pal.getCount() == 0)
   {
      Std256Pal.ensureCapacity(256);

      int index = 0;

      // The 16 first colors are the "standard" 16 color set.
      const GVector<int>& std16 = GColor::GetStd16ColorPalette();
      for (int i=0; i<16; i++, index++)
         Std256Pal.add(std16[i]);

      index = 16;
      float graystep = 256.0f / (24 + 2);
      for (int i=1; i<=24; i++, index++)
      {
         int tone = (int) (i * graystep);
         Std256Pal.add(GColor(tone, tone, tone));
      }

      index = 40;
      int redset[] = { 0, 51, 102, 153, 204, 255 };
      int greenset[] = { 0, 51, 102, 153, 204, 255 };
      int blueset[] = { 0, 51, 102, 153, 204, 255 };
      for (int red=0; red<6; red++)
         for (int green=0; green<6; green++)
            for (int blue=0; blue<6; blue++, index++)
               Std256Pal.add(GColor(redset[red], greenset[green], blueset[blue]));
   }

   return Std256Pal;
}

bool GColor::isGrayTone () const
{
   int blue = getBlue();
   if (blue != getGreen())
      return false;
   if (blue != getRed())
      return false;
   return true;
}

GColor GColor::getDarker ( int step, bool spin ) const
{
   int r, g, b;

   if (spin && (this->rgb & 0xFFFFFF) == 0)
   {
      r = 0xFF - step;
      g = 0xFF - step;
      b = 0xFF - step;
   }
   else
   {
      r = GMath::Max(getRed() - step, 0);
      g = GMath::Max(getGreen() - step, 0);
      b = GMath::Max(getBlue() - step, 0);
   }

   return GColor(r, g, b);
}

GColor GColor::getLighter ( int step, bool spin ) const
{
   int r, g, b;

   if (spin && (this->rgb & 0xFFFFFF) == 0xFFFFFF)
   {
      r = step;
      g = step;
      b = step;
   }
   else
   {
      r = GMath::Min(getRed() + step, 0xFF);
      g = GMath::Min(getGreen() + step, 0xFF);
      b = GMath::Min(getBlue() + step, 0xFF);
   }

   return GColor(r, g, b);
}

GString GColor::toString () const
{
   if (*this == GColor::BLACK)
      return "Black";
   else
   if (*this == GColor::DBLUE)
      return "DarkBlue";
   else
   if (*this == GColor::DGREEN)
      return "DarkGreen";
   else
   if (*this == GColor::DCYAN)
      return "DarkCyan";
   else
   if (*this == GColor::DRED)
      return "DarkRed";
   else
   if (*this == GColor::DMAGENTA)
      return "DarkMagenta";
   else
   if (*this == GColor::DYELLOW)
      return "Brown";
   else
   if (*this == GColor::GRAY)
      return "PaleGray";
   else
   if (*this == GColor::DGRAY)
      return "DarkGray";
   else
   if (*this == GColor::BLUE)
      return "LightBlue";
   else
   if (*this == GColor::GREEN)
      return "LightGreen";
   else
   if (*this == GColor::CYAN)
      return "LightCyan";
   else
   if (*this == GColor::RED)
      return "LightRed";
   else
   if (*this == GColor::MAGENTA)
      return "LightMagenta";
   else
   if (*this == GColor::YELLOW)
      return "Yellow";
   else
   if (*this == GColor::WHITE)
      return "White";
   else
      return "0x" + GInteger::ToString(rgb & 0xFFFFFF, 16, 6);
}

GColor GColor::ParseColor ( const GString& strColor, const GColor& def )
{
   GString str(strColor);
   str.trim(GString::TrimAll);
   if (str == "")
      return def;

   if (str.equalsIgnoreCase("Black"))
      return GColor::BLACK;
   else
   if (str.equalsIgnoreCase("DBlue") || str.equalsIgnoreCase("DarkBlue"))
      return GColor::DBLUE;
   else
   if (str.equalsIgnoreCase("DGreen") || str.equalsIgnoreCase("DarkGreen"))
      return GColor::DGREEN;
   else
   if (str.equalsIgnoreCase("DCyan") || str.equalsIgnoreCase("DarkCyan"))
      return GColor::DCYAN;
   else
   if (str.equalsIgnoreCase("DRed") || str.equalsIgnoreCase("DarkRed"))
      return GColor::DRED;
   else
   if (str.equalsIgnoreCase("DMagenta") || str.equalsIgnoreCase("DarkMagenta"))
      return GColor::DMAGENTA;
   else
   if (str.equalsIgnoreCase("Brown") || str.equalsIgnoreCase("DYellow") || str.equalsIgnoreCase("DarkYellow"))
      return GColor::DYELLOW;
   else
   if (str.equalsIgnoreCase("Gray") || str.equalsIgnoreCase("PaleGray"))
      return GColor::GRAY;
   else
   if (str.equalsIgnoreCase("DGray") || str.equalsIgnoreCase("DarkGray"))
      return GColor::DGRAY;
   else
   if (str.equalsIgnoreCase("Blue") || str.equalsIgnoreCase("LightBlue"))
      return GColor::BLUE;
   else
   if (str.equalsIgnoreCase("Green") || str.equalsIgnoreCase("LightGreen"))
      return GColor::GREEN;
   else
   if (str.equalsIgnoreCase("Cyan") || str.equalsIgnoreCase("LightCyan"))
      return GColor::CYAN;
   else
   if (str.equalsIgnoreCase("Red") || str.equalsIgnoreCase("LightRed"))
      return GColor::RED;
   else
   if (str.equalsIgnoreCase("Magenta") || str.equalsIgnoreCase("LightMagenta"))
      return GColor::MAGENTA;
   else
   if (str.equalsIgnoreCase("Yellow"))
      return GColor::YELLOW;
   else
   if (str.equalsIgnoreCase("White"))
      return GColor::WHITE;

   // ---
   if (GInteger::IsInteger(str))
      return GColor(GInteger::ParseInt(str));

   // ---
   GTokenizer tokenizer(str, " \t\r\n,.;:/", "", false);
   GString red = tokenizer.getNextToken()->toString();
   GString green = tokenizer.getNextToken()->toString();
   GString blue = tokenizer.getNextToken()->toString();
   if (GInteger::IsInteger(red) && GInteger::IsInteger(green) && GInteger::IsInteger(blue))
      return GColor(GInteger::ParseInt(red), GInteger::ParseInt(green), GInteger::ParseInt(blue));

   return def;
}

int GColor::getBlue () const 
{ 
   return rgb & 0xFF; 
}

int GColor::getGreen () const 
{ 
   return (rgb & 0xFF00) >> 8; 
}

int GColor::getRed () const 
{ 
   return (rgb & 0xFF0000) >> 16; 
}

int GColor::getRGB () const 
{ 
   return rgb; 
}

GColor& GColor::setBlue ( int blue ) 
{ 
   rgb |= (blue & 0xFF); 
   return *this; 
}

GColor& GColor::setGreen ( int green ) 
{ 
   rgb |= (green & 0xFF00); 
   return *this; 
}

GColor& GColor::setRed ( int red ) 
{ 
   rgb |= (red & 0xFF0000); 
   return *this; 
}

GColor& GColor::setRGB ( int rgb ) 
{ 
   this->rgb = rgb; 
   return *this; 
}

GColor& GColor::operator= ( int c ) 
{ 
   return setRGB(c); 
}

GColor& GColor::operator= ( const GColor& c ) 
{ 
   return setRGB(c.getRGB()); 
}

bool GColor::operator== ( int c ) const 
{ 
   return c == rgb; 
}

bool GColor::operator== ( const GColor& c ) const 
{ 
   return c.rgb == rgb; 
}

bool GColor::operator!= ( int c ) const 
{ 
   return c != rgb; 
}

bool GColor::operator!= ( const GColor& c ) const 
{ 
   return c.rgb != rgb; 
}

GColor::operator int () const 
{ 
   return rgb & 0xFFFFFF; 
}

