/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GDecoratedWindow.h"
#include "glib/gui/GContainerWindow.h"
#include "glib/gui/GScrollbar.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/sys/GSystem.h"

GDecoratedWindow::GDecoratedWindow ( const GString& name,
                                     const GString& constraints,
                                     GWindow* parentWin,
                                     const GString& titleStr,
                                     long winStyle,
                                     long winStyle2 )
                 :GWindow(GWindowClass::GENERIC, constraints, true),
                  containerWin(null),
                  vScrollbar(null),
                  hScrollbar(null),
                  frameWTop(0),
                  frameWBottom(0),
                  frameWLeft(0),
                  frameWRight(0),
                  vscrollVisible(false),
                  hscrollVisible(false)
{
   containerWin = new GContainerWindow(name, GBorderLayout::CENTER, *this, parentWin, ((winStyle & WS_VISIBLE) != 0) ? true : false);
   winStyle2 |= WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT; // Don't "expose" the client-area in the ini-file!
   GWindow::init("ClientArea", containerWin, containerWin, winStyle, winStyle2);
   setText(titleStr);

   // Make sure that the colors are actually activated 
   // on the HWND, because we postponed the creation of the HWND.
   setBackgroundColor(defaultBackgroundColor = containerWin->getBackgroundColor());
   setForegroundColor(defaultForegroundColor = containerWin->getForegroundColor());

   vScrollbar = new GScrollbar("VScroll", GBorderLayout::EAST, *containerWin, true, 0);
   hScrollbar = new GScrollbar("HScroll", GBorderLayout::SOUTH, *containerWin, false, 0);
}

GDecoratedWindow::~GDecoratedWindow ()
{
   delete hScrollbar;
   delete vScrollbar;
   delete containerWin;
}

bool GDecoratedWindow::isVisible () const 
{ 
   return containerWin->isVisible(); 
}

void GDecoratedWindow::setEnabled ( bool flag, bool repaint ) 
{ 
   containerWin->setEnabled(flag, repaint); 
}

void GDecoratedWindow::setVisible ( bool flag ) 
{ 
   containerWin->setVisible(flag); 
}

void GDecoratedWindow::setWindowPos ( int xpos, int ypos ) 
{ 
   containerWin->setWindowPos(xpos, ypos); 
}

void GDecoratedWindow::setWindowSize ( int width, int height ) 
{ 
   containerWin->setWindowSize(width, height); 
}

void GDecoratedWindow::setWindowBounds ( int xpos, int ypos, int width, int height ) 
{ 
   containerWin->setWindowBounds(xpos, ypos, width, height); 
}

bool GDecoratedWindow::isHScrollbarEnabled () const 
{ 
   int range = hScrollbar->getScrollRange();
   return range > 0;
}

bool GDecoratedWindow::isHScrollbarVisible () const 
{ 
   return hscrollVisible; 
}

bool GDecoratedWindow::isVScrollbarEnabled () const 
{ 
   int range = vScrollbar->getScrollRange();
   return range > 0;
}

bool GDecoratedWindow::isVScrollbarVisible () const 
{ 
   return vscrollVisible; 
}

GContainerWindow& GDecoratedWindow::getContainerWin () 
{ 
   return *containerWin;
}

GScrollbar& GDecoratedWindow::getHScrollbar () 
{ 
   return *hScrollbar; 
}

GScrollbar& GDecoratedWindow::getVScrollbar () 
{ 
   return *vScrollbar; 
}

int GDecoratedWindow::getFrameWTop () const 
{ 
   return frameWTop; 
}

int GDecoratedWindow::getFrameWBottom () const 
{ 
   return frameWBottom; 
}

int GDecoratedWindow::getFrameWLeft () const 
{ 
   return frameWLeft; 
}

int GDecoratedWindow::getFrameWRight () const 
{ 
   return frameWRight; 
}

void GDecoratedWindow::setHScrollPos ( int pos ) 
{ 
   hScrollbar->setScrollPos(pos); 
}

void GDecoratedWindow::setHScrollPosAndRange ( int pos, int scrolLen, int thumbLen, bool dontLayout ) 
{ 
   hScrollbar->setScrollPosAndRange(pos, scrolLen, thumbLen); 
   if (hscrollVisible && !dontLayout)
   {
      int range = hScrollbar->getScrollRange();
      bool oldVisible = hScrollbar->isVisible();
      bool newVisible = (range > 0);
      if (newVisible != oldVisible)
      {
         hScrollbar->setVisible(newVisible);
         containerWin->layout();
      }
   }
}

void GDecoratedWindow::setHScrollRange ( int scrolLen, int thumbLen, bool dontLayout ) 
{ 
   hScrollbar->setScrollRange(scrolLen, thumbLen); 
   if (hscrollVisible && !dontLayout)
   {
      int range = hScrollbar->getScrollRange();
      bool oldVisible = hScrollbar->isVisible();
      bool newVisible = (range > 0);
      if (newVisible != oldVisible)
      {
         hScrollbar->setVisible(newVisible);
         containerWin->layout();
      }
   }
}

void GDecoratedWindow::setVScrollPos ( int pos ) 
{ 
   vScrollbar->setScrollPos(pos); 
}

void GDecoratedWindow::setVScrollPosAndRange ( int pos, int scrolLen, int thumbLen, bool dontLayout ) 
{ 
   vScrollbar->setScrollPosAndRange(pos, scrolLen, thumbLen); 
   if (vscrollVisible && !dontLayout)
   {
      int range = vScrollbar->getScrollRange();
      bool oldVisible = vScrollbar->isVisible();
      bool newVisible = (range > 0);
      if (newVisible != oldVisible)
      {
         vScrollbar->setVisible(newVisible);
         containerWin->layout();
      }
   }
}

void GDecoratedWindow::setVScrollRange ( int scrolLen, int thumbLen, bool dontLayout ) 
{ 
   vScrollbar->setScrollRange(scrolLen, thumbLen); 
   if (vscrollVisible && !dontLayout)
   {
      int range = vScrollbar->getScrollRange();
      bool oldVisible = vScrollbar->isVisible();
      bool newVisible = (range > 0);
      if (newVisible != oldVisible)
      {
         vScrollbar->setVisible(newVisible);
         containerWin->layout();
      }
   }
}

void GDecoratedWindow::setHScrollVisible ( bool flag )
{
   hscrollVisible = flag;
   hScrollbar->setVisible(flag);
   containerWin->layout();
   invalidateAll(true);
}

void GDecoratedWindow::setVScrollVisible ( bool flag )
{
   vscrollVisible = flag;
   vScrollbar->setVisible(flag);
   containerWin->layout();
   invalidateAll(true);
}

void GDecoratedWindow::setFrameW ( int left, int right, int top, int bottom )
{
   frameWTop = top;
   frameWBottom = bottom;
   frameWLeft = left;
   frameWRight = right;
   containerWin->layout();
}

GWindowMessage::Answer GDecoratedWindow::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_VSCROLL:
      {
         int pos = 0;
         int event = msg.getParam2HiShort();
         if (event == SB_SLIDERPOSITION || event == SB_SLIDERTRACK)
         {
            pos = msg.getParam2LoShort();
            pos = vScrollbar->convertFromSysPos(pos);
         }
         onVScroll(event, pos);
         return 0; 
      } 

      case WM_HSCROLL: 
      {
         int pos = 0;
         int event = msg.getParam2HiShort();
         if (event == SB_SLIDERPOSITION || event == SB_SLIDERTRACK)
         {
            pos = msg.getParam2LoShort();
            pos = hScrollbar->convertFromSysPos(pos);
         }
         onHScroll(event, pos);
         return 0;
      } 

      default:
         return GWindow::handleWindowMessage(msg);
   }
}

bool GDecoratedWindow::onHScroll ( int event, int pos )
{
   int oldpos = hScrollbar->getScrollPos();
   int range = hScrollbar->getScrollRange();
   switch (event)
   {
      case SB_LINEUP:
         if (oldpos > 0)
            return onHScrollLineUp();
         return true;

      case SB_LINEDOWN:
         if (oldpos < range)
            return onHScrollLineDown();
         return true;

      case SB_PAGEUP:
         if (oldpos > 0)
            return onHScrollPageUp();
         return true;

      case SB_PAGEDOWN:
         if (oldpos < range)
            return onHScrollPageDown();
         return true;

      case SB_SLIDERPOSITION:
      case SB_SLIDERTRACK:
         if (range > 0 && pos >= 0 && pos <= range)
            return onHScrollSliderTrack(pos);
         return true;

      default:
         return false;
   }
}

bool GDecoratedWindow::onVScroll ( int event, int pos )
{
   int oldpos = vScrollbar->getScrollPos();
   int range = vScrollbar->getScrollRange();
   switch (event)
   {
      case SB_LINEUP:
         if (oldpos > 0)
            return onVScrollLineUp();
         return true;

      case SB_LINEDOWN:
         if (oldpos < range)
            return onVScrollLineDown();
         return true;

      case SB_PAGEUP:
         if (oldpos > 0)
            return onVScrollPageUp();
         return true;

      case SB_PAGEDOWN:
         if (oldpos < range)
            return onVScrollPageDown();
         return true;

      case SB_SLIDERPOSITION:
      case SB_SLIDERTRACK:
         if (range > 0 && pos >= 0 && pos <= range)
            return onVScrollSliderTrack(pos);
         return true;

      default:
         return false;
   }
}
