/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DECORATEDWINDOW
#define __GLIB_DECORATEDWINDOW

#include "glib/gui/GWindow.h"

/**
 * A window with support for a frame and a horizontal and/or a vertical
 * scrollbar.
 *
 * The window it self is the client, and the parent is the
 * container window of which contains the frame and/or scrollbars.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
class GDecoratedWindow : public GWindow
{
   friend class GContainerWindow;

   private:

      /** The parent window, which is the frame of ours and contains the scrollbar(s). */
      class GContainerWindow* containerWin;

      /** Vertical scrollbar window (if WINDOWSTYLE2_VSCROLL was specified in winStyle2). */
      class GScrollbar* vScrollbar;

      /** Horizontal scrollbar window (if WINDOWSTYLE2_HSCROLL was specified in winStyle2). */
      class GScrollbar* hScrollbar;

      /** Width of the top margin between container window and its sub-windows. */
      int frameWTop;

      /** Width of the bottom margin between container window and its sub-windows. */
      int frameWBottom;

      /** Width of the left margin between container window and its sub-windows. */
      int frameWLeft;

      /** Width of the right margin between container window and its sub-windows. */
      int frameWRight;

      /** True if the vertical scrollbar is to be visible, or else false. */
      bool vscrollVisible;

      /** True if the horizontal scrollbar is to be visible, or else false. */
      bool hscrollVisible;

   public:

      GDecoratedWindow ( const GString& name,
                         const GString& constraints,
                         GWindow* parentWin = null,
                         const GString& titleStr = GString::Empty,
                         long winStyle = WS_VISIBLE,
                         long winStyle2 = 0 );

      virtual ~GDecoratedWindow ();

   private:

      /** Disable the copy constructor. */
      GDecoratedWindow ( const GDecoratedWindow& src ) {}

      /** Disable the assignment operator. */
      GDecoratedWindow& operator= ( const GDecoratedWindow& ) { return *this; }

   public:

      /**
       * Overridest <i>GWindow::isVisible()</i> in order to perform some
       * special logic.
       * @see GWindow#isVisible
       */
      virtual bool isVisible () const;

      /**
       * Overridest <i>GWindow::setEnabled()</i> in order to perform some
       * special logic.
       * @see GWindow#setEnabled
       */
      virtual void setEnabled ( bool flag = true, bool repaint = false );

      /**
       * Overridest <i>GWindow::setVisible()</i> in order to perform some
       * special logic.
       * @see GWindow#setVisible
       */
      virtual void setVisible ( bool flag = true );

      /**
       * Overridest <i>GWindow::setWindowPos()</i> in order to perform some
       * special logic.
       * @see GWindow#setWindowPos
       */
      virtual void setWindowPos ( int xpos, int ypos );

      /**
       * Overridest <i>GWindow::setWindowSize()</i> in order to perform some
       * special logic.
       * @see GWindow#setWindowSize
       */
      virtual void setWindowSize ( int width, int height );

      /**
       * Overridest <i>GWindow::setWindowBounds()</i> in order to perform
       * some special logic.
       * @see GWindow#setWindowBounds
       */
      virtual void setWindowBounds ( int xpos, int ypos, int width, int height );

      /** 
       * Returns true if the horizontal scrollbar is not disabled.
       * That is if it is currently defined with a scrollable range.
       * This is not the same as testing if the scrollbar is visible.
       * This method might return true even if the scrollbar is hidden.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.26
       */
      virtual bool isHScrollbarEnabled () const;
      virtual bool isHScrollbarVisible () const;
      virtual bool isVScrollbarEnabled () const;
      virtual bool isVScrollbarVisible () const;

      class GContainerWindow& getContainerWin ();
      class GScrollbar& getHScrollbar ();
      class GScrollbar& getVScrollbar ();
      int getFrameWTop () const;
      int getFrameWBottom () const;
      int getFrameWLeft () const;
      int getFrameWRight () const;
      void setFrameW ( int left, int right, int top, int bottom );
      void setHScrollPos ( int pos );
      void setHScrollPosAndRange ( int pos, int scrolLen, int thumbLen, bool dontLayout = false );
      void setHScrollRange ( int scrolLen, int thumbLen, bool dontLayout = false );
      void setHScrollVisible ( bool flag );
      void setVScrollPos ( int pos );
      void setVScrollPosAndRange ( int pos, int scrolLen, int thumbLen, bool dontLayout = false );
      void setVScrollRange ( int scrolLen, int thumbLen, bool dontLayout = false );
      void setVScrollVisible ( bool flag );

   protected:

      /**
       * Override {@link GWindow#handleWindowMessage} in order to handle
       * the WM_VSCROLL and the WM_HSCROLL events, in case they are
       * programatically sent by some program or driver that emulates
       * the scrollbars (e.g. a mouse wheel tracker). Such drivers tends
       * to send the scrollbar events to the window directly underneath 
       * the mouse cursor, which is this window, even if that window 
       * does not contain any scrollbar window(s) in it self.
       *
       * @author  Leif Erik Larsen
       * @since   2002.04.09
       */
      virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );

   public:

      virtual bool onHScroll ( int event, int pos );
      virtual bool onHScrollLineUp () { return false; }
      virtual bool onHScrollLineDown () { return false; }
      virtual bool onHScrollPageUp () { return false; }
      virtual bool onHScrollPageDown () { return false; }
      virtual bool onHScrollSliderTrack ( int pos ) { return false; }
      virtual bool onVScroll ( int event, int pos );
      virtual bool onVScrollLineUp () { return false; }
      virtual bool onVScrollLineDown () { return false; }
      virtual bool onVScrollPageUp () { return false; }
      virtual bool onVScrollPageDown () { return false; }
      virtual bool onVScrollSliderTrack ( int pos ) { return false; }
};

#endif
