/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FOCUSMANAGER
#define __GLIB_FOCUSMANAGER

#include "glib/primitives/GObject.h"

/**
 * This is the Focus Manager class used to manage keyboard focus in
 * Dialog Panels.
 *
 * The source was originally taken from the source
 * of the Swing default Focus Manager. I hacked it to get better support
 * for Oily Components and make sure that TAB and SHIFT+TAB walks
 * through the Dialog Components in the same order as they was defined in
 * the Resource Script. In addition, this class should take care of
 * handling the <b>NEXT</b> and <b>PREVIOUS</b> optional Resource Parameters
 * on each Component as well.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.26
 */
class GFocusManager
{
   public:

      /**
       * Cause the focus manager to set the focus on the next focusable
       * component.
       */
      virtual void focusNextComponent ( class GWindow* aComponent, bool skipOily = false );

      /**
       * Cause the focus manager to set the focus on the previous
       * focusable component.
       */
      virtual void focusPreviousComponent ( class GWindow* aComponent, bool skipOily = false );

   private:

      /**
       * Cause the focus manager to set the focus on the next focusable
       * component, either foreward or backward.
       */
      class GWindow* getFocusableComponentAfter ( class GWindow* focusedComponent,
                                                  class GWindow* rootContainer,
                                                  bool moveForward,
                                                  bool skipOily );

      /**
       */
      class GWindow* getNextComponent ( class GWindow* component,
                                        class GWindow* root, 
                                        bool canGoDown );

      /**
       */
      class GWindow* getPreviousComponent ( class GWindow* component, 
                                            class GWindow* root, 
                                            bool skipOily );

      /**
       */
      class GWindow* getDeepestLastComponent ( class GWindow* component );

      /**
       */
      class GWindow* getFocusRoot ( class GWindow* c );

      /**
       * Return the first component that should receive the focus.
       */
      class GWindow* getFirstComponent ( class GWindow* aContainer );

      /**
       * Return the last component that should receive the focus.
       */
      class GWindow* getLastComponent ( class GWindow* aContainer );

      /**
       * Return the component that should receive the focus before aComponent.
       */
      class GWindow* getComponentBefore ( class GWindow* aContainer, 
                                          class GWindow* aComponent, 
                                          bool skipOily );

      /**
       * Return the component the should receive the focus after aComponent.
       */
      class GWindow* getComponentAfter ( class GWindow* aContainer, 
                                         class GWindow* aComponent );
};

#endif


