/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FRAMEWINDOW
#define __GLIB_FRAMEWINDOW

#include "glib/gui/GWindow.h"
#include "glib/util/GStack.h"

/**
 * The class used to represent a standard frame window.
 *
 * The window object it self is the client window, while the
 * frame window that contains the client area can be accessed
 * with {@link #getFrame}.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
class GFrameWindow : public GWindow
{
   public:

      /**
       * The different frame window states.
       * Used with {@link #restoreWindowPosImpl}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       */
      enum State
      {
         Minimized = 0,
         Normalized,
         Maximized
      };

   private:

      /**
       * This inner class is needed in order to support the 
       * "keepCaptionActive" property on OS/2.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.27
       */
      class FrameWin : public GWindow
      {
         private:

            GFrameWindow& owner;

         public:

            FrameWin ( GFrameWindow& owner, HWND hWndFrame );
            virtual ~FrameWin ();
            virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );
      };

      friend class FrameWin;


      /** The parent window containg the titlebar, border, menubar, etc. */
      class GWindow* frameWin;

      /** The titlebar window (FID_TITLEBAR). */
      class GWindow* titleBar;
      HWND hWndTitlebar;

      HWND hWndFrame;

      /** The remaining area that is to be filled with {@link #client}. */
      GRectangle clientArea;

      /** @see #pushStatusbarText */
      GStack<GString> statbarTextStack;

      /** The client window possibly set by {@link #setAutoLayoutClientWin}. */
      class GWindow* clientWin;

      /** The menubar window. */
      class GMenu* menuBar;

      /** The toolbar window, or null if we have no toolbar. */
      class GToolbar* toolbar;

      /** The keybar (F1 - F10), or null if we have no toolbar. */
      class GKeyBar* keybar;

      /** The statusbar window. */
      class GStatusbar* statbar;

      /** @see #setAutoManageWindowPos */
      bool autoManageWindowPos;

      /** @see #setKeepCaptionActive */
      bool keepCaptionActive;

      /** True if this frame is used as a dialog frame. */
      bool isdlg;

   public:

      explicit GFrameWindow ( const GString& name = GString::Empty,
                              const GString& titleStr = GString::Empty,
                              GWindow* ownerWin = null,
                              long winStyle = 0,
                              long winStyle2 = 0,
                              bool isdlg = false );

      virtual ~GFrameWindow ();

   private:

      /** Disable the copy constructor. */
      GFrameWindow ( const GFrameWindow& src ) {}

      /** Disable the assignment operator. */
      GFrameWindow& operator= ( const GFrameWindow& ) { return *this; }

   public:

      /**
       * Make sure that the keyboard focus is set on the correct component.
       *
       * This function is called whenever the frame window gets activated
       * and whenever any of the window components (such as the toolbar,
       * keybar, statusbar, etc.) are toggled on or off.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.10
       */
      virtual void ensureFocusOK () {}

      const class GRectangle& getClientAreaRect () const;
      const class GWindow& getFrame () const;
      class GWindow& getFrame ();
      class GKeyBar* getKeybar ();
      class GMenu* getMenubar ();
      int getPreferredKeybarHeight () const;
      int getPreferredStatusbarHeight () const;
      int getPreferredToolbarHeight () const;
      class GStatusbar* getStatusbar ();

      /**
       * Get a pointer to the titlebar window of the frame, or null if
       * the frame has no titlebar. This method will always return
       * null on Win32, because the titlebar area of a frame window
       * on Windows is not implemented as an HWND. At least not from
       * the user code point of view.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.31
       */
      class GWindow* getTitlebar ();

      class GToolbar* getToolbar ();
      GPoint getWindowPos () const;
      bool isActive () const;

      /**
       * Test if the auto window pos management is currently on.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       * @see     #setAutoManageWindowPos
       */
      bool isAutoManageWindowPos () const;

      bool isKeybarVisible () const;
      bool isMaximized () const;
      bool isMinimized () const;
      bool isResizable () const;
      bool isStatusbarVisible () const;
      bool isToolbarVisible () const;
      bool isVisible () const;

      virtual void layout ();

      /**
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       * @see     #pushStatusbarText
       * @see     #setStatusbarText
       */
      void popStatusbarText ();

      /**
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       * @see     #popStatusbarText
       * @see     #setStatusbarText
       */
      void pushStatusbarText ( const GString& txt );

      virtual void setActive ( bool force = false );

      /**
       * Set whether or not we shall automatically 
       * call {@link #restoreWindowPos} upon {@link #queryProfile}, 
       * and {@link #storeWindowPos} upon {@link #writeProfile}. 
       * This property is on by default.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.17
       * @param   yes  True to set the property on, false to set it off.
       * @see     #isAutoManageWindowPos
       */
      void setAutoManageWindowPos ( bool yes );

      void setEnabled ( bool flag = true, bool repaint = false );

      void setKeybarVisible ( bool show = true );

      void setMaximized ( bool flag = true );
      void setMinimized ( bool flag = true );
      void setVisible ( bool flag = true );
      void setWindowBounds ( int xpos, int ypos, int width, int height );
      void setWindowPos ( int xpos, int ypos );
      void setWindowSize ( int width, int height );

      /**
       * Show or hide the statusbar window.
       */
      void setStatusbarVisible ( bool show = true );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @see     GStatusbar#setHint
       * @see     #setStatusbarText
       */
      void setStatusbarHint ( const GString& hint );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.23
       * @see     GStatusbar#setText
       * @see     #setStatusbarHint
       * @see     #pushStatusbarText
       * @see     #popStatusbarText
       */
      void setStatusbarText ( const GString& text );

      void setText ( const GString& text );

      void setToolbar ( const GString& resID, const GString& name = GString::Empty );

      /**
       * Show or hide the toolbar window.
       */
      void setToolbarVisible ( bool show = true );

      virtual void updateHintString ( const GString& idString );

      /**
       * Automatically called by {@link #onInitMenu} to let the
       * sub-class initialize the menu command states.
       */
      virtual void updateCommandStates () {}

      /**
       * Read and activate settings of this window (and all its sibling parts,
       * such as the title-, tool- and status-bar) from the profile of the
       * owner program.
       *
       * @see GWindow#queryProfile
       */
      void queryProfile ( const GString& sectName );

      /**
       * Write all current settings of this window (and all its sibling parts,
       * such as the title-, tool- and status-bar) to the profile of the
       * owner program.
       *
       * @see GWindow#writeProfile
       */
      void writeProfile ( const GString& sectName, bool force = false );

      /**
       * Restore the size and position as well as minimize/maximize/restore
       * state of the Frame Window, as of upon the most recent call to
       * {@link #storeWindowPos}.
       *
       * The window will be set visible as well.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.07
       * @param   sectionName The name of the section (within the INI-file) 
       *                      of where to lookup <i>entryName</i>.
       * @param   entryName   The name of the entry within <i>sectionName</i>
       *                      of where to lookup the variables containing the
       *                      window properties of which to restore.
       * @param   preventMinimized True if we shall prevent the window from 
       *                      being restored to a minimized state. That is,
       *                      if this parameter is true and the stored state
       *                      of the window was {@link #Minimized} then we 
       *                      will restore the window into a 
       *                      {@link #Normalized} state instead.
       * @see     #storeWindowPos
       */
      void restoreWindowPos ( const GString& sectionName, 
                              const GString& entryName,
                              bool preventMinimized = true );

      /**
       * Restore the size and position as well as minimize/maximize/restore
       * state of the Frame Window, as specified.
       *
       * The window will be set visible as well.
       *
       * <b>Note:</b> The specified position and size must come from a 
       * recent call to {@link #storeWindowPosImpl} and not from 
       * e.g. {@link GWindow#getWindowPos} or {@link GWindow#getWindowSize}.
       * This is important in order to guarantee that the window is restored
       * at the same position/size as is intended.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       * @see     #storeWindowPosImpl
       */
      void restoreWindowPosImpl ( const class GPoint& pos, 
                                  const class GDimension& size, 
                                  GFrameWindow::State state = GFrameWindow::Normalized );

      /**
       * Store the size and position as well as minimize/maximize/restore
       * state of the Frame Window, into the specified variable references.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       * @see     #restoreWindowPosImpl
       */
      void storeWindowPosImpl ( class GPoint& pos, 
                                class GDimension& size, 
                                GFrameWindow::State& state );

      /**
       * Get the current state of this frame window.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       */
      GFrameWindow::State getWindowState () const;

      /**
       * Remove the current menubar from the window, if any.
       */
      void removeMenubar ();

      /**
       * If this method is called with a non-null argument then the
       * specified window will automatically be layed out by {@link #layout}
       * to take up the window space defined by {@link #getClientAreaRect}.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.11
       */
      void setAutoLayoutClientWin ( class GWindow* clientWin );

      /**
       * Set the enable/disable state of all menu- and toolbar items
       * that have an ID that equals the specified command ID.
       */
      virtual void setCommandEnableState ( const char* cmdID, bool state );

      /**
       * Set the toggle state of all menuitems and toolbar lockbuttons
       * that have an ID that equals the specified command ID.
       */
      virtual void setCommandToggleState ( const char* cmdID, bool state );

      /**
       * Use the specified icon in the system menu area of the frame window.
       *
       * @param iconName The Icon Resource of which to activate on the Window.
       */
      void setIcon ( const GString& iconName );

      /**
       * Set whether or not we should keep the caption (title bar) in the 
       * "activated color" even when the frame window is not actually 
       * the active window. This is espessially useful when temporarily
       * displaying some kind of popup window, such as e.g. a dropdown list.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.27
       */
      void setKeepCaptionActive ( bool b );

      /**
       * Set the menubar of the frame window.
       *
       * @param  menuID   The menu resource of which to form the new menu.
       * @param  useFancy True if we shall use fancy menues with icons, etc.
       */
      void setMenubar ( const GString& menuID, 
                        bool useFancy = true, 
                        bool topMnemonicsOn = true );

      void setResizable ( bool flag );

      /**
       * Activate the specified menuitem in the current menubar.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.10
       * @param   cmdID   The ID of which submenu in the menubar of which 
       *                  to activate.
       */
      void startMenubarMode ( const GString& cmdID );

      /**
       * Store the current size and position of the Frame Window.
       * The data will be saved to the user profile (in RAM only) of
       * the owner program.
       *
       * @see #restoreWindowPos
       */
      void storeWindowPos ( const char* sectionName, const char* entryName );

   protected:

      virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );

      virtual bool onCommand ( int cmd );
      virtual bool onDeactivate ();
      virtual bool onInitMenu ();
      virtual bool onKey ( const GKeyMessage& key );
      virtual bool onMenuSelect ( int id );

      /**
       * Overrides {@link GWindow#updateKeybarCommands}.
       * Will update the keybar (see {@link #getKeyBar}) if any.
       * Will also call recursively to next parent, if any, in case 
       * the parent also has a keybar (not likely, though).
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.10
       */
      virtual void updateKeybarCommands ( class GWindow* focusWin );
};

#endif
