/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ICON
#define __GLIB_ICON

#include "glib/util/GKeyBag.h"

/**
 * Class to represent a single icon image that can have transparent 
 * pixels.
 *
 * An icon consist of two bitmaps, one that is the bitmap it self on a
 * black background. The other bitmap is the so-called mask bitmap where
 * each transparent pixel are represented by a white color and all other
 * pixels are black.
 *
 * In order to help preventing the same icon bitmaps being duplicated
 * in memory this class has no public constructor to create icon objects 
 * from icon resources. Instead it provides a static factory method, 
 * {@link #getIcon}, that can be used to get access to a named icon 
 * from an automatically managed icon pool.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
class GIcon : public GObject
{
   friend class GGraphics;

   private:

      class A : public GObject
      {
         friend class GIcon;
         friend class GGraphics;
         class GBitmap* bmImage;
         class GBitmap* bmMask;
         A ();
         virtual ~A ();
      };

      class B : public GObject 
      {
         friend class GIcon;
         friend class GGraphics;
         HPOINTER hicon;
         bool wasLoadedFromFile;
         B ();
         virtual ~B ();
      };

      /** 
       * An icon can be in two forms, or both:
       *
       * A) Managed and painted completely under the control of GLib in 
       *    the form of two bitmaps. Icons of this type typically 
       *    originates from an icon resource in the resouirce table
       *    of the application.
       *
       * B) In the form of a platform dependent icon handle. The underlying 
       *    system will paint the icon for us. Icons of this type typically 
       *    originates from an exe-, dll- or ico-file on the file system,
       *    or from some system dependent Shell API that provides an icon 
       *    handle that can be used e.g. to represent some Shell Desktop 
       *    Object, etc.
       */
      A* a;
      mutable B* b;

      /** The width of the icon, if known. */
      int width;

      /** The height of the icon, if known. */
      int height;

   private:

      /**
       * Whenever we paint a normal (not disabled-) icon in memory we will 
       * keep it in this bag, where the key is the resource name of the icon.
       *
       * This is for speed optimization, so that we don't need to paint
       * pixel-by-pixel each time the same icon is to be drawn. Instead we
       * can perform a fast BitBlt-operation, which is more likely to exploit
       * the hardware graphics accelerator features of the underlying graphics
       * system.
       *
       * @author  Leif Erik Larsen
       * @since   2003.09.30
       * @see     IconPoolDisabled
       */
      static GKeyBag<GIcon> IconPoolNormal;

      /**
       * Whenever we paint a disabled (grayed-) icon in memory we will 
       * keep it in this bag, where the key is the resource name of the icon.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.14
       * @see     IconPoolNormal
       */
      static GKeyBag<GIcon> IconPoolDisabled;

   public:

      /**
       * Usually you want to use the static factory method {@link #getIcon} 
       * instead of using this constructor directly. Because the factory
       * method will automatically keep each referenced icon in an internal 
       * icon pool and reuse icons from that pool.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.03.05
       */
      explicit GIcon ( const class GIconResource& ires, bool grayed );

   private:

      /** Disable the copy constructor. */
      GIcon ( const GIcon& ) {}

      /** Disable the assigment operator. */
      GIcon& operator= ( const GIcon& ) { return *this; }

      /** Get the grayed version of the specified RGB color. */
      static int GetGrayedRgb ( int rgb, int darkGray, int darkerGray );

   public:

      /**
       * Load the system dependent icon that is used by the system to
       * represent the specified file. The file can be of any type,
       * including executables and DLL's.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.08
       * @param   path      The path of any document or executable for which 
       *                    the associated icon respresentation is to 
       *                    be loaded.
       * @param   smallIcon True if we should load the small icon version 
       *                    of the document, or else false to load the 
       *                    default (large) icon size.
       */
      explicit GIcon ( const GString& path, bool smallIcon );

      virtual ~GIcon ();

   public:

      /**
       * Get a reference to the specified icon. The icon must exist 
       * in the default resource table of the running application.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.07
       * @param   iconName  The resource name of which icon to get.
       * @param   grayed    True if we should return a grayed version 
       *                    of the icon, or else false to return the 
       *                    normal version of the icon. Grayed icons 
       *                    are typically used on disabled components,
       *                    and disabled menu commands.
       * @return  A pointer to the icon, or null if the specified icon 
       *          does not exist in the resource table of the application.
       */
      static const GIcon* GetIcon ( const GString& iconName, bool grayed = false );

      /** Get the height of the icon, in pixels. */
      int getHeight () const;

      /** Get the width of the icon, in pixels. */
      int getWidth () const;

      /**
       * Get the system dependent handle for the icon. On Win32 this is 
       * a HICON, and on OS/2 it is a HPOINTER.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.08
       */
      int getSystemHandle () const;
};

#endif
