/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_KEYBAR
#define __GLIB_KEYBAR

#include "glib/gui/GAbstractToolbarWindow.h"
#include "glib/gui/GToolbarButton.h"

/**
 * Kind of a toolbar window but with a fixed number of buttons that
 * each represents a single function key (F1..F10) on keyboard and
 * which containing command ID is dynamically changed with respect to
 * the current combination of pressed down shift-keys (Shift/Ctrl/Alt).
 *
 * @author  Leif Erik Larsen
 * @since   2002.03.09
 */
class GKeyBar : public GAbstractToolbarWindow
{
   public:

      class Button : public GAbstractCommand
      {
         private:

            /** The name of which key this button face represents. */
            GString keyName;

            /** Pointer to our owner Keybar window. */
            GKeyBar& keyBar;

         public:

            Button ( GKeyBar& keyBar, const GString& keyName );
            virtual ~Button ();

         public:

            const GString& getKeyName () const { return keyName; }
      };

   private:

      /**
       * Block of eight button faces that are all representing a single
       * function key, F1 for instance.
       *
       * The first face is used when neither SHIFT, CONTROL or
       * ALT are held down (KB_DEF), the second face is used when
       * only SHIFT is held down (KB_SHIFT), etc.
       */
      class Key : public GToolbarButton
      {
         friend class GKeyBar;

         private:

            /** The owner Key Bar window. */
            GKeyBar& keyBar;

            /**
             * To simplify the task of accessing the correct keys with respect to the
             * current shift-key state (that is, combinations of the keys ALT, CTRL
             * and SHIFT that are currently down), which is being tracked by the
             * variable <i>shiftBitMap</i> in the class {@link GKeyBar}.
             *
             * <b><i><u>Possible combinations are:</u></i></b><br>
             * <pre>
             * KB_DEF                   ==> Index 0
             * KB_SHIFT                 ==> Index 1
             * KB_ALT                   ==> Index 2
             * KB_ALT|KB_SHIFT          ==> Index 3
             * KB_CTRL                  ==> Index 4
             * KB_CTRL|KB_SHIFT         ==> Index 5
             * KB_CTRL|KB_ALT           ==> Index 6
             * KB_CTRL|KB_ALT|KB_SHIFT  ==> Index 7
             * </pre>
             */
            GArray<Button> key; // Can be indexed by or'ed combinations of KB_*

         public:

            Key ( GKeyBar& keyBar, const GString& iconName, const GString& keyName );
            virtual ~Key ();

         private:

            /** Disable the copy constructor. */
            Key ( const Key& src ) 
               : GToolbarButton(GString::Empty, GString::Empty, src.keyBar, GString::Empty), keyBar(src.keyBar) {}

            /** Disable the assignment operator. */
            Key& operator= ( const Key& ) { return *this; }

         public:

            virtual void performAction ();
            virtual const GString& getIDString () const;

         protected:

            virtual void paintButtonFaceImpl ( GGraphics& g, const GRectangle& rect );
            virtual void paintButtonBorderImpl ( GGraphics& g, const GRectangle& rect, bool force );
      };

   friend class Key;

   public:

      enum
      {
         KB_DEF            = (0x0),                     // Index #0
         KB_SHIFT          = (0x1),                     // Index #1
         KB_ALT            = (0x2),                     // Index #2
         KB_ALT_SHIFT      = (KB_ALT|KB_SHIFT),         // Index #3
         KB_CTRL           = (0x4),                     // Index #4
         KB_CTRL_SHIFT     = (KB_CTRL|KB_SHIFT),        // Index #5
         KB_CTRL_ALT       = (KB_CTRL|KB_ALT),          // Index #6
         KB_CTRL_ALT_SHIFT = (KB_CTRL|KB_ALT|KB_SHIFT)  // Index #7
      };

      /** Height of this keybar (pixels). */
      int preferredHeight;

      /** Time stamp of when the ALT-flag was last set. */
      ulonglong tmstWhenAltWasSet;

   private:

      /** Each of the ten keybar buttons. Possibly more buttons in the future. */
      GArray<Key> keys;

      /** Or'ed combinations of KB_* to form the current index N of key[].key[N]. */
      int shiftBitMap;

      GWindow* previousFocusWin;

   public:

      GKeyBar ( const GString& name,
                const GString& constraints,
                GWindow& parentWin );

   private:

      /** Disable the copy constructor. */
      GKeyBar ( const GKeyBar& src ) 
         : GAbstractToolbarWindow(GString::Empty, GString::Empty, *this) {}

      /** Disable the assignment operator. */
      GKeyBar& operator= ( const GKeyBar& ) { return *this; }

   private:

      void addKey ( const char* iconName, const char* keyName );

   public:

      /**
       * Get the preferred height of the window, in pixels.
       */
      int getPreferredHeight () const;

      bool isAltFlag () { return (shiftBitMap & KB_ALT) != 0; }
      bool isCtrlFlag () { return (shiftBitMap & KB_CTRL) != 0; }
      bool isShiftFlag () { return (shiftBitMap & KB_SHIFT) != 0; }
      void layout ();
      void setAltFlag ( bool on );
      void setCtrlFlag ( bool on );
      void setShiftFlag ( bool on );

      /**
       * This method is automatically called 
       * by {@link GWindow#updateKeybarCommands} 
       * whenever another window has received the keyboard input focus.
       * We will update the current keyboard shortcut map on the keybar 
       * with respect to the newly focused window, which is the one 
       * specified. If the specified focusWin is null then we will 
       * assume the parenmt window of ours, which is usually an 
       * instance of {@link GFrameWindow}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.10
       */
      void updateKeyCommands ( GWindow* focusWin );
};

#endif
