/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_LISTBOX
#define __GLIB_LISTBOX

#include "glib/gui/GObjectWindow.h"
#include "glib/gui/event/GAbstractCommand.h"
#include "glib/gui/event/GListSelectionListener.h"

/**
 * This is the low level window class that implements a list box control.
 *
 * @author  Leif Erik Larsen
 * @since   2000.05.09
 */
class GListBox : public GWindow, 
                 public GAbstractListBox
{
   private:

      /**
       * The class that implements the system dependent list box. A default
       * instance of this class is created by the constructor of the listbox.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.06.30
       */
      class Peer : public GWindow
      {
         friend class GListBox;

         private:

            /** The owner list box. */
            GListBox& listb;

         public:

            Peer ( GListBox& listb );

            virtual ~Peer ();

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) : listb(src.listb) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }

         private:

            virtual bool onKeyDown ( const GKeyMessage& key );
            virtual void onFocusKill ();
      };

   friend class Peer;

   private:

      GArray<GListBoxItem> items;
      GVector<GListSelectionListener*> selectionListeners;

      /** The height, in pixels, of the heighest icon used by any item. */
      int heighestIcon;

      /** Keeps track of the width of the widest item. */
      int widestItemInPixels;

      /** True if the MULTISEL mode is on for this list-box. */
      bool multisel;

      /** 
       * Used in order not to send selection notification if selction 
       * index has not actually changed.
       */
      int prevIdx;

      const GIcon* tglIconOn;
      const GIcon* tglIconOff;
      int tglIconWidth;
      int tglIconHeight;

      mutable Peer peer;

   public:

      GListBox ( const GString& name,
                 const GString& constraints,
                 GWindow& parentWin,
                 bool multisel = false,
                 long winStyle = WS_VISIBLE,
                 long winStyle2 = WS2_DEFAULTPAINT );

      virtual ~GListBox ( void );

   private:

      /** Disable the copy constructor. */
      GListBox ( const GListBox& src ) : peer(*this) {}

      /** Disable the assignment operator. */
      GListBox& operator= ( const GListBox& ) { return *this; }

   private:

      void drawItemImpl ( GGraphics& g,
                          int index, 
                          const GRectangle& rect,
                          bool selected );

      /** Calculate the width of the specified list box item. */
      int measureItemWidth ( GGraphics& g, 
                             const GListBoxItem& item  ) const;

      /**
       * This method is to be called once for each item that is
       * added or inserted into the listbox.
       */
      void updateItemHeight ();

      /**
       * Called when the list box peer has lost focus.
       */
      void peerHasLostFocus ();

   protected:

      virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );
      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );

   public:

      GWindow& getPeer () { return peer; }

      virtual bool isEmpty () const;
      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual GString queryValue () const;

      virtual int getPreferredHeight () const;
      virtual int getPreferredWidth () const;

      virtual void grabFocus ( bool force = false );

      virtual void setEnabled ( bool flag = true, bool repaint = true );
      virtual void setOily ( bool flag );

      /**
       * Add the specified listener to the set of object to be called
       * whenever the list box selection is changed.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.25
       * @param   l    A reference to the listener object to add.
       */
      void addListBoxSelectionListener ( GListSelectionListener* l );

      /**
       * Get the height, in pixels, of the heighest icon used by any item.
       */
      int getHeighestIcon () const;

      /**
       * Set the height, in pixels, of the heighest icon used by any item.
       * This method is normally not to be called by user- or application-
       * code. It was initially implemented to be called internally by
       * G-Lib only.
       */
      void setHeighestIcon ( int height );

      /**
       * This method is normally not to be called by user- or application-
       * code. It was initially implemented to be called internally by
       * G-Lib only.
       */
      int getDefaultItemHeight () const;

      /**
       * Implements {@link GAbstractListBox#addItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#addItem
       */
      virtual void addItem ( const GString& text, 
                             const GString& iconName = GString::Empty, 
                             GObject* userData = null, 
                             bool autoDelUD = false );

      /**
       * Implements {@link GAbstractListBox#insertItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#insertItem
       */
      virtual void insertItem ( int index, 
                                const GString& text, 
                                const GString& iconName = GString::Empty, 
                                GObject* userData = null, 
                                bool autoDelUD = false );

      /**
       * Implements {@link GAbstractListBox#getSelectedIndex}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getSelectedIndex
       */
      virtual int getSelectedIndex () const;

      /**
       * Implements {@link GAbstractListBox#getItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItem
       * @throws  GArrayIndexOutOfBoundsException if the specified index is
       *                                          out of bounds.
       */
      virtual const GListBoxItem& getItem ( int index ) const;

      /**
       * Implements {@link GAbstractListBox#setItemIcon}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.06.07
       * @see     GAbstractListBox#setItemIcon
       */
      virtual void setItemIcon ( int index, const GString& iconName );

      /**
       * Implements {@link GAbstractListBox#setItemTextAndIcon}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.06.07
       * @see     GAbstractListBox#setItemTextAndIcon
       */
      virtual void setItemTextAndIcon ( int index, const GString& text, const GString& iconName );

      /**
       * Implements {@link GAbstractListBox#setItemText}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#setItemText
       */
      virtual void setItemText ( int index, const GString& text );

      /**
       * Implements {@link GAbstractListBox#getItemText}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItemText
       */
      virtual GString getItemText ( int index ) const;

      /**
       * Implements {@link GAbstractListBox#setItemUserData}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#setItemUserData
       */
      virtual void setItemUserData ( int index, GObject* userData, bool autoDelete );

      /**
       * Implements {@link GAbstractListBox#getItemUserData}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItemUserData
       */
      virtual GObject* getItemUserData ( int index ) const;

      /**
       * Implements {@link GAbstractListBox#removeItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#removeItem
       */
      virtual bool removeItem ( int index );

      /**
       * Implements {@link GAbstractListBox#removeAllItems}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#removeAllItems
       */
      virtual void removeAllItems ();

      /**
       * Implements {@link GAbstractListBox#getItemCount}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItemCount
       */
      virtual int getItemCount () const;

      /** 
       * We override this method in order to call every 
       * registered {@link GListSelectionListener} in addition to
       * sending the GM_CTRLCHANGED message.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.03
       */
      virtual void sendCtrlChanged ();

      /**
       * Implements {@link GAbstractListBox#setSelectedIndex}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#setSelectedIndex
       */
      virtual void setSelectedIndex ( int index );

      /**
       * Request the indexed item to be repainted as soon as possible.
       * If -1 is specified then we will assume to repaint the current
       * selected item.
       */
      void repaintItem ( int index ) const;

      /**
       * Select the next item which first character match the specified
       * character, ignoring case.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.27
       * @return  True if there was some match and a new selection has been
       *          done, or else false.
       */
      bool selectNextCharMatch ( char chr );

      /**
       * Check if the indexed item is selected or not.
       *
       * This method probably makes sense only for list-boxes with the MULTISEL
       * parameter on.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.25
       */
      bool isItemSelected ( int index );

      /**
       * Set the selection state of the indexed item.
       *
       * This method probably makes sense only for list-boxes with the MULTISEL
       * parameter on. If this parameter is not on then we will do
       * the same as {@link #setSelectedIndex}.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.25
       */
      void setItemSelected ( int index, bool sel );
};

#endif
