/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_MENUPOPUP
#define __GLIB_MENUPOPUP

#include "glib/gui/GMenuPopupItem.h"
#include "glib/gui/GWindow.h"
#include "glib/resource/GMenuPopupParams.h"

/**
 * Sub-menues will be an instance of this class.
 *
 * This class is not to be used by user program code. The user
 * code should use the class <i>GMenu</i> instead.
 *
 * @author  Leif Erik Larsen
 * @since   1998.09.13
 */
class GMenuPopup : public GObject
{
   friend class GMenu;

   public:

      class TheWindow : public GWindow
      {
         friend class GMenuPopup;

         private:

            GMenuPopup& menuObj;

         private:

            TheWindow ( GMenuPopup& menuObj,
                        const GString& name,
                        GWindow& ownerWin,
                        const GString& constraints,
                        GMenu& topLevelMenu,
                        bool isMenuBar,
                        int level );

            virtual ~TheWindow ();

         private:

            /** Disable the copy constructor. */
            TheWindow ( const TheWindow& src ) : menuObj(src.menuObj) {}

            /** Disable the assignment operator. */
            TheWindow& operator= ( const TheWindow& ) { return *this; }

         protected:

            /**
             * Try to convert the keyboard event to a corresponding WM_COMMAND
             * event, in case the pressed key corresponds to the mnemonic
             * character of one of the menu items of this popup menu.
             */
            virtual bool onKeyDown ( const GKeyMessage& key );

            /**
             * Because of a bug in OS/2 Warp PM that causes the system to always
             * draw the sub-menu arrow upon WM_PAINT we need to override WM_PAINT
             * from the system and emulate the system's behaviour of sending
             * WM_DRAWITEN once for each item manually.
             */
            virtual bool onPaint ( GGraphics& g, const GRectangle& rect );
      };

   friend class TheWindow;

   private:

      static int MenuIDCounter;

      int level;
      GMenu& topLevelMenu;
      GArray<GMenuPopupItem> items;
      GWindow& parentWin; // The parent window of ours.
      const GIcon* iconMenuItemTlgMark;
      const GIcon* iconMenuItemSubMenu;

      TheWindow theWin;

   public:

      /**
       * Make an initially empty menu window of where to add menu
       * items and/or sub-menues.
       */
      GMenuPopup ( const GString& name,
                   const GString& constraints,
                   GMenu& topLevelMenu,
                   GWindow& parentWin,
                   GMenuPopupParams* popup,
                   bool mnemonicsOn,
                   int level,
                   bool dontInit,
                   bool isMenuBar );

      virtual ~GMenuPopup ();

   private:

      /** Disable the copy constructor. */
      GMenuPopup ( const GMenuPopup& src ) :
         theWin(*this, GString::Empty, src.parentWin, GString::Empty, src.topLevelMenu, false, 0),
         topLevelMenu(src.topLevelMenu), parentWin(parentWin) {}

      /** Disable the assignment operator. */
      GMenuPopup& operator= ( const GMenuPopup& ) { return *this; }

   private:

      void init_ ( GWindow& ownerWin,
                   GMenuPopupParams* popup,
                   bool mnemonicsOn );

   public:

      TheWindow& getTheWindow ();

      /** Called by owner window upon WM_MEASUREITEM. */
      GDimension measureMenuItem ( GGraphics& g, class GMenuPopupItem& item );

      /** Called by owner window upon WM_DRAWITEM. */
      void drawMenuItem ( GGraphics& g,
                          GMenuPopupItem& item,
                          const GRectangle& rect,
                          bool itemDisabled,
                          bool itemHilited,
                          bool itemChecked );

      GWindow& getParentWindow ();

      void setVisible ( bool flag = true );
      bool isVisible () const;

      /**
       * Get a reference to the indexed menu item.
       * @see #getItemCount
       */
      GMenuPopupItem& getIndexedItem ( int idx );

      /**
       * Get the current number of menu items that are contained in this
       * menu window.
       * @see #getIndexedItem
       */
      int getItemCount () const;

      /**
       * Get a reference to the top level menu window.
       */
      GMenu& getTopLevelMenu ();

      /**
       * Return true if and only if this menu window is represented
       * on screen as a menubar window. This default implementation will
       * always return false, but {@link GMenu} overrides this
       * method in order to return true if it is a menubar rather than
       * a popup.
       */
      virtual bool isMenubar () const;

   protected:

      /**
       * Set whether to use fancy items or not for this popup menu
       * and its sub-menues (if any).
       *
       * This method is to be overridden by {@link GMenu},
       * and only the overriding method should call this super implementation.
       *
       * @param fancy     True if we shall use fancy items, or else false to
       *                  use the default as of which is defined by the
       *                  underlying system.
       */
      virtual void setUseFancyMenues ( bool fancy );
};

#endif
