/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TEXTENTRY
#define __GLIB_TEXTENTRY

#include "glib/gui/GAbstractTextField.h"

/**
 * This is the low level window class that implements a text entry component.
 * <p>
 * This class supports value change listeners.
 * Whenever the contained text is changed (either programatically or by user)
 * we will automatically call {@link GWindow#fireValueChangeListeners}.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.29
 */
class GTextEntry : public GAbstractTextField
{
   public:

      enum ADJUST
      {
         LEFT,
         RIGHT,
         CENTER
      };

   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2004.08.27
       */
      class Peer : public GWindow
      {
         friend class GTextEntry;

         private:

            GTextEntry& theEntry;

         public:

            Peer ( GTextEntry& theEntry );
            virtual ~Peer ();

         protected:

            virtual void onFocusKill ();
            virtual void onFocusSet ();
            virtual bool onKeyDown ( const GKeyMessage& key );

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) : theEntry(src.theEntry) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }
      };

      mutable Peer peer;

   public:

      GTextEntry ( const GString& name,
                   const GString& constraints,
                   GWindow& parentWin,
                   long winStyle = WS_VISIBLE,
                   long winStyle2 = WS2_DEFAULTPAINT,
                   int maxTextLength = 0,
                   ADJUST adjust = LEFT,
                   bool border = true,
                   bool password = false,
                   bool readOnly = false );

      virtual ~GTextEntry ();

   private:

      /** Disable the copy constructor. */
      GTextEntry ( const GTextEntry& src ) 
         : GAbstractTextField(*src.getParentWindow()), peer(*this) {}

      /** Disable the assignment operator. */
      GTextEntry& operator= ( const GTextEntry& ) { return *this; }

   protected:

      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );

   public:

      // Methods that implements the pure virtual methods of our parent class.

      virtual void appendText ( const GString& text );
      virtual void copy () const;
      virtual void cut ();
      virtual int getPreferredHeight () const;
      virtual int getPreferredWidth () const;
      virtual int getSelectionEnd () const;
      virtual int getSelectionStart () const;
      virtual int getTextLength () const;
      virtual void paste ();
      virtual void setMaxTextLength ( int maxLength );
      virtual void setSelection ( int selStart, int selEnd );
      virtual void setText ( const GString& text );
      virtual GWindow& getPeer () const;

      void setReadOnly ( bool flag );
      void setSelectionAll () { setSelection(0, getTextLength()); }
};

#endif
