/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TEXTVIEWER
#define __GLIB_TEXTVIEWER

#include "glib/gui/GDecoratedWindow.h"
#include "glib/gui/event/GCommandMap.h"
#include "glib/util/GArray.h"

/**
 * This is the Text Viewer class.
 *
 * @author  Leif Erik Larsen
 * @since   1998.10.11
 */
class GTextViewer : public GDecoratedWindow
{
   public:

      /** The default font for text viewer windows. */
      static const GString DefaultFont;

   public:

      /**
       * Concrete class to manage a single caret position inside the text
       * of {@link GTextViewer}.
       *
       * @author  Leif Erik Larsen
       * @since   1999.09.14
       */
      class Pos : public GObject
      {
         public:

            int x;
            int y;

            Pos ();
            Pos ( int x, int y );
            Pos ( const Pos &pos );
            virtual ~Pos ();

            int getX () const;
            int getY () const;
            Pos& set ( int x, int y );
            void setX ( int x );
            void setY ( int y );

            virtual GString toString () const;

            Pos& operator= ( const Pos& pos );
            bool operator== ( const Pos& pos ) const;
            bool operator!= ( const Pos& pos ) const;
            bool operator> ( const Pos& pos ) const;
            bool operator< ( const Pos& pos ) const;
            bool operator>= ( const Pos& pos ) const;
            bool operator<= ( const Pos& pos ) const;
      };

   public:

      /**
       * Concrete class to manage the current selection area of the text
       * of {@link GTextViewer}.
       *
       * @author  Leif Erik Larsen
       * @since   1999.09.14
       */
      class Selection : public GObject
      {
         public:

            /**
             * The anchor position of the selection.
             *
             * This is the selection origo that is used to remember where
             * the mouse button was initially pressed down when the user
             * began dragging a selection area.
             *
             * This position is needed in order to know if the current drag
             * position is above or below the start position and thus decide
             * if the selction is to be extended or shrinked upwards or
             * downwards.
             */
            GTextViewer::Pos anchor;

            /** Position of selection start. */
            GTextViewer::Pos start;

            /** Position of selection end. */
            GTextViewer::Pos end;

         public:

            Selection ();
            virtual ~Selection ();

         public:

            void clear ( const GTextViewer::Pos& newOrigo );
            void clear ();
            bool isAnySelection () const;
            bool isEmpty () const;
      };

   public:

      /**
       * Abstract class to represent all the configurable parameters for the
       * "search" and the "search next" command.
       *
       * For this class to be usable, a subclass must be defined that also
       * implements the code needed for the user to be able to edit the
       * parameters in something like a typical "find text" dialog box.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.18
       */
      class SearchParams : public GObject
      {
         public:

            /** The most recent search string given by user in {@link cmdSearch}. */
            GString searchString;

            /** True if {@link #findNextText} shall search forward. */
            bool searchForward;

            /** True if {@link #findNextText} shall use case sensitive search. */
            bool searchCaseSen;

            /** True if {@link #findNextText} shall search for whole words only. */
            bool searchWord;

         public:

            SearchParams ();
            virtual ~SearchParams ();

         public:

            /**
             * This method is called by the text viewer whenever the user has
             * requested a new text find operation. It is up to the concrete
             * implementing subclass to override this method and implement the
             * code needed to let the user specify the parameters.
             *
             * Typically, this will open a "search" dialog of where the user
             * can edit the various public variables of this abstract class.
             *
             * The default implementation does nothing but return false.
             * Thus, this method should normally be overridden 
             * by the subclass.
             *
             * @author  Leif Erik Larsen
             * @since   2000.09.18
             * @return  True if the user selected "OK", or else false if the
             *          user cancelled the operation.
             */
            virtual bool userEditSearchParams ( GTextViewer& owner );

            /**
             * This method is called by {@link GTextViewer#cmdSearch} and
             * {@link GTextViewer#cmdSearchNext} if no matching text 
             * was found. It is up to the subclass to override this method 
             * and present something like a message box to the user.
             *
             * The default implementation does nothing but return.
             *
             * @author  Leif Erik Larsen
             * @since   2005.02.03
             */
            virtual void onSearchStringNotFound ( GTextViewer& owner );
      };

   private:

      class ViewerOptions : public GObject
      {
         public:

            /** Width of the tabulator character. */
            int tabWidth;

            /** In case of any extremely long lines, force EOL at this position in order to prevent serious CPU-hogs. */
            int forceEOLPos;

            /** True if we are used as a "console monitor" with some special scrolling behaviour. */
            bool useAsConsoleMonitor;

            /** True if we shall paint the small possible free space at top of window rather than at bottom. */
            bool paintFreeTextSpaceAtTop;

            /** True if we shall paint a thin vertical line at the "maximum line width" limit ({@link #forceEOLPos}). */
            bool showForceEOLPosVLine;

            /** The color of the thin vertical line painted if {@link #showForceEOLPosVLine} is true. */
            GColor colorOfForceEOLPosVLine;

            /** True if we shall paint tabulator characters (if any) with special arrows. */
            bool showTabs;

            /** True if we shall paint space characters (if any) with special dots. */
            bool showSpaces;

            /** True if we shall paint the linefeed (if any) character () at end of each line. */
            bool showLinefeeds;

            /** The character used for tab characters when {@link #showTabs} is true. */
            char chrShowTab;

            /** The character used for space characters when {@link #showSpaces} is true. */
            char chrShowSpace;

            /** The character used for linefeed characters when {@link #showLinefeeds} is true. */
            char chrShowLinefeed;

            /** True if we shall scroll content of window directly on up-, down-, left- and right- arrow commands, instead of moving the caret as in an editor. */
            bool quickScrollMode;

            /** True if word-wrap mode is currently on. */
            bool wordWrap;

            /** The characters that defines "white space" for the word wrapping algorithm, in addition to all ASCII characters <= 32. */
            GString wordWrapChars;

         public:

            ViewerOptions ();
            virtual ~ViewerOptions ();
      };

   private:

      /**
       * Vector of contained text lines.
       *
       * Each element is an instance of {@link GString}, and contains the
       * raw character string exacly as specified to {@link #appendLine},
       * including TAB- and EOL- characters.
       *
       * Text painting algorithms in this class uses {@link #makeVirtualLine}
       * to convert TAB(s) to spaces and to strip out any trailing EOL(s)
       * before painting the text lines contained in this vector.
       */
      GArray<GString> lines;

      /** 
       * User configurable word separator characters.
       * This string doesn't need to contain standard whitespace 
       * characters, since we tests explicitly on whitespace in addition 
       * to the word separator characters in this string.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.10
       */
      GString wordSeparatorCharacters;

      /** All the parameters used by the "search" and the "search next" command. */
      GTextViewer::SearchParams* searchParams;

      /** The user configurable options of the text viewer. */
      ViewerOptions opts;

      /** Index of current leftmost visible character (0..). */
      int idxFirstVisibleColumn;

      /** Pointer to start of 1. line in window. */
      int idxFirstVisibleLine;

      /** Number of columns in width of window. */
      int winColumnNr;

      /** Number of lines in height of window. */
      int winLineNr;

      /**
       * Number of characters in the longest line, after it has been
       * virtualized with {@link #makeVirtualLine}.
       */
      int widestLine;

      /** Average width of each character (in pixels). */
      int fontw;

      /** Height of each text line (in pixels). */
      int fonth;

      /** The height of the remaining vertical area after all visible lines are painted. */
      int verticalFreeSpace;

      /** The rectangle that defines the "paintable" area. */
      GRectangle rectScrollClip;

      /** The window rectangle area as of the last {@link #onSize}. */
      GRectangle previousSize;

      /**
       * This object defines the current selected physical text area.
       * The <i>start</i> is inclusive and the <i>end</i> is exclusive.
       */
      GTextViewer::Selection selection;

      /** Current position of the blinking caret. */
      GTextViewer::Pos curPos;

      /** Used to keep track of original x-pos when user scrolls up/down using arrow keys. */
      int lastExplicitXPos;

      /** True if current {@link #lastExplicitXPos} is due to an {@link #navigateEnd} command. */
      bool lastExplicitXPosWasEol;

      /** Mouse position. Only legal while mouse is captured. */
      GPoint mousePos;

      /** True when selction mode is ON. */
      bool selectionMode;

      /** Buffer used by {@link #makeVirtualLine}. */
      mutable GString vlineBuff;

   public:

      GTextViewer ( GTextViewer::SearchParams* searchParams,
                    const GString& name,
                    const GString& constraints,
                    GWindow& parentWin,
                    bool useAsConsoleMonitor,
                    bool paintFreeTextSpaceAtTop );

      virtual ~GTextViewer ();

   private:

      /** Disable the copy constructor. */
      GTextViewer ( const GTextViewer& src ) 
         : GDecoratedWindow(GString::Empty, GString::Empty) {}

      /** Disable the assignment operator. */
      GTextViewer& operator= ( const GTextViewer& ) { return *this; }

      /**
       * Calculate the physical painting rectangle of the 
       * indexed line, regardless if the indexed line is within visible 
       * area or not.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.09
       */
      GRectangle calcRectOfIndexedLine ( int idx ) const;

      /**
       * Convert the specified physical position into its corresponding
       * virtual position, where "physical position" means the absolute
       * index of a positioned character in memory and "virtual position"
       * means the location of that character on screen after the tab
       * characters (if any) have been respected.
       *
       * @author  Leif Erik Larsen
       * @since   2002.05.03
       */
      Pos calcXPosFromPhysicalToVirtual ( const Pos& pos ) const;
      Pos calcXPosFromPhysicalToVirtual ( int x, int y ) const;

      /**
       * Convert the specified virtual position into its corresponding
       * physical position, where "virtual position" means the location
       * of a character on screen and "physical position" means the
       * absolute index of that character in text buffer in memory.
       *
       * @author  Leif Erik Larsen
       * @since   2002.05.03
       */
      Pos calcXPosFromVirtualToPhysical ( const Pos& pos ) const;

      /**
       * Calculate the physical y-position of where to paint the 
       * indexed line, regardless if the indexed line is within visible 
       * area or not.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.09
       */
      int calcYPosOfIndexedLine ( int idx ) const;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       */
      char getCharAt ( const GTextViewer::Pos& pos ) const;

      /** Common handling of mouse positioning events. */
      bool handleMouseMovement ( int xpos, int ypos, bool updateMousePosRecorder = true );

      /**
       * Increment the specified position with automatic wrapping to 
       * "start of next line" at "end of line".
       *
       * If the parameter "forward" is false then we will instead 
       * decrement the specified position with automatic wrapping to 
       * "end of previous line" at "stard of line".
       *
       * This method is useful when searching forward or backward for text 
       * within the text body of the viewer, since it allows for using a
       * single "position object" as a virtual global text "index counter".
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @param   pos     The position of which to increment.
       * @param   forward True if we shall increment forward, or else false
       *                  if we shall decrement backward.
       * @return  True if the position has not reached start- or end-
       *               of-text-body, or else false.
       */
      bool incrementPos ( GTextViewer::Pos& pos, bool forward = true ) const;

      bool isCharactersTheSame ( char chr1, char chr2, bool caseSen ) const;

      bool isWordSeparatorChar ( char chr ) const;

      /**
       * Create and return a tab-expanded version of the specified text
       * line. That is a string that contains a copy of the specified line
       * but with all tab characters expanded to spaces 
       * (or {@link ViewerOptions#chrShowTab} in case
       * variable {@link ViewerOptions#showTabs} is true).
       *
       * In addition this method will transform the trailing CR/LF (if
       * any) into one or more space(s) representing each CR/LF respectively,
       * or {@link ViewerOptions#chrShowLinefeed} in case 
       * variable {@link ViewerOptions#showLinefeeds} is true.
       *
       * @author  Leif Erik Larsen
       * @since   2002.04.09
       * @param   line    The text of which to virtualize.
       * @param   fromX   The horizontal position of where the first character
       *                  of the specified text is to be painted. This is used
       *                  when calculating the number of spaces that is to
       *                  replace any tab-character(s) in the text.
       * @return  A reference to an internal string buffer that will
       *          be overwritten on next call to this method.
       */
      const GString& makeVirtualLine ( const GString& line, int fromX ) const;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @param   str   The string to search for. This must be a raw format 
       *                character string that will be compared byte-by-byte
       *                to the content of the text viewer.
       * @paream  pos   The starting search position.
       * @param   searchWord True if we should search "whole words" only.
       * @param   searchCaseSen True if we should match character case.
       * @param   searchForeward True if we search forward. If this 
       *                argument is false the specified search-string must 
       *                be reverted before calling this method. This is in
       *                order for us to use the the same algorithm code
       *                regardless if we search foreward or backward.
       */
      bool matchStringAt ( const GString& str, 
                           const GTextViewer::Pos& pos,
                           bool searchWord, 
                           bool searchCaseSen,
                           bool searchForeward ) const;

      /** Helper method used by {@link #paintLineImpl} only. */
      static void PaintText ( GGraphics& g, 
                              GRectangle& r, 
                              const GString& txt, 
                              const GColor& bck, 
                              const GColor& frg );

      /**
       * Paint specified single line, at specified Y-position.
       *
       * The line will be painted using the current font and colors of
       * the specified graphics object.
       *
       * @author  Leif Erik Larsen
       * @since   1994.04.26
       * @param   g         The graphics context of where to paint.
       * @param   lineIndex The index of which line to paint.
       * @param   width     Pixels width of viewer window client area.
       * @param   ypos      Y-position of where to paint the line in the device.
       */
      void paintLineImpl ( GGraphics& g, int lineIndex, int width, int ypos );

      /**
       * Paint the indexed line.
       *
       * This method is to be called when user has scrolled
       * one line down or up.
       *
       * @author  Leif Erik Larsen
       * @since   1994.04.26
       * @param   idx  Index of which line to paint.
       * @param   nr   Number of lines to paint (1..).
       */
      void paintLine ( int idx, int nr = 1 );

      /**
       * Automatically divide the indexed line if needed (to respect the
       * "maximum line length" setting).
       * <p>
       * This will cut the indexed line only if needed, and the remaining
       * part of the line will be inserted as a new line at position index+1.
       * Thus, the inserted line must be divided too, since that is not
       * done automatically by this method.
       */
      bool respectForceEOLPosAt ( int lineIndex );

      void setCurrentPos ( const GTextViewer::Pos& pos );

      void setWordWrapImpl ( bool flag, bool temp );

      void updateWidestLine ( int index );

      /**
       * Word wrap the indexed line. This will cut the indexed line at
       * best word-wrap position if needed, and the remaining part of the
       * line will be inserted as a new line at position index+1. Thus,
       * the inserted line must be word wrapped too, since that is not
       * done automatically by this method.
       */
      bool wordWrapLineAt ( int index );

      bool wrapLineIfNeededAt ( int lineIndex );

   protected:

      /**
       * Select the word underneath the mouse double click position.
       */
      virtual bool onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

      virtual bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton1Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

      /**
       * This method is automatically called whenever the caret has 
       * change position either horizontally and/or vertically. It is 
       * intended to be overridden.
       *
       * The default implementation of this method does nothing 
       * but return.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       * @see     #getCurrentPos
       */
      virtual void onCaretPositionChanged ();

      virtual void onFocusSet ();
      virtual bool onFontNameSizeChanged ( const GString& fontNameSize );
      virtual bool onInitMenu ();
      virtual bool onKeyDown ( const GKeyMessage& key );
      virtual bool onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onPaint ( GGraphics& g, const GRectangle& rect );
      virtual bool onSize ( int width, int height );
      virtual bool onTimer ( const GString& timerID, GObject* userData );

   public:

      virtual bool onHScrollLineUp ();
      virtual bool onHScrollLineDown ();
      virtual bool onHScrollPageUp ();
      virtual bool onHScrollPageDown ();
      virtual bool onHScrollSliderTrack ( int pos );
      virtual bool onVScrollLineUp ();
      virtual bool onVScrollLineDown ();
      virtual bool onVScrollPageUp ();
      virtual bool onVScrollPageDown ();
      virtual bool onVScrollSliderTrack ( int pos );

   protected:

      // Cursor navigation (arrow) commands.
      // If the shift-flag is true then we will select text as part of
      // the movement.

      void navigateUp ( bool shift );
      void navigateDown ( bool shift );
      void navigateLeft ( bool shift );
      void navigateRight ( bool shift );
      void navigateHome ( bool shift );
      void navigateEnd ( bool shift );
      void navigatePrevWord ( bool shift );
      void navigateNextWord ( bool shift );
      void navigatePageUp ( bool shift );
      void navigatePageDown ( bool shift );
      void navigateTop ( bool shift );
      void navigateBottom ( bool shift );

   public:

      /** @see #getSelectedText */
      void cmdCopy ( GAbstractCommand* cmd = null );

      void cmdSearch ( GAbstractCommand* cmd = null );
      void cmdSearchNext ( GAbstractCommand* cmd = null );
      void cmdGoto ( GAbstractCommand* cmd = null );
      void cmdSelectAll ( GAbstractCommand* cmd = null );
      void cmdUnselectAll ( GAbstractCommand* cmd = null );
      void cmdNavigateUp ( GAbstractCommand* cmd = null );
      void cmdNavigateDown ( GAbstractCommand* cmd = null );
      void cmdNavigateLeft ( GAbstractCommand* cmd = null );
      void cmdNavigateRight ( GAbstractCommand* cmd = null );
      void cmdNavigateHome ( GAbstractCommand* cmd = null );
      void cmdNavigateEnd ( GAbstractCommand* cmd = null );
      void cmdNavigatePrevWord ( GAbstractCommand* cmd = null );
      void cmdNavigateNextWord ( GAbstractCommand* cmd = null );
      void cmdNavigatePageUp ( GAbstractCommand* cmd = null );
      void cmdNavigatePageDown ( GAbstractCommand* cmd = null );
      void cmdNavigateTop ( GAbstractCommand* cmd = null );
      void cmdNavigateBottom ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectUp ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectDown ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectLeft ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectRight ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectHome ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectEnd ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectPrevWord ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectNextWord ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectPageUp ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectPageDown ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectTop ( GAbstractCommand* cmd = null );
      void cmdNavigateSelectBottom ( GAbstractCommand* cmd = null );

      /**
       * Append the specified string as a single line of text.
       * The specified string might (but does not have to-) contain 
       * a trailing platform dependent linefeed character.
       * 
       * This version of the method will make a new clone of the 
       * specified string, and keep that new clone internally.
       * Thus, the caller owns the ownership of the specified 
       * string, and can safely destroy it as soon as this method 
       * has finished. If runtime speed is critical you should 
       * consider using the other overloaded version of this 
       * method; {@link #appendLine(GString*, bool, bool)}.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.04.26
       * @see     #appendLine(GString*, bool, bool)
       */
      void appendLine ( const GString& text );

      /**
       * Append the specified string as a single line of text.
       * The specified string might (but does not have to-) contain 
       * a trailing platform dependent linefeed character.
       * 
       * This version of the method will make a new clone of the 
       * specified string only if <i>autoDelete</i> is false.
       * It also takes a richer set of parameters that can be used 
       * to control whether or not to update certain GUI related stuff
       * when appending the line. For speed optimizing reasons, this 
       * method is therefore often the best alternative to use.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.04.26
       * @param   text    The line of which text to add.
       * @param   autoDel True if we should grab the ownership of the 
       *                  specified text line, or else false. If false 
       *                  is specified then we will create a clone copy,
       *                  and use the created clone internally. In that 
       *                  case the caller keeps the ownership.
       * @param   updtGui True if we shall automatically update certain 
       *                  GUI related stuff after the line has been 
       *                  appended. E.g. the scrollbar ranges, and 
       *                  synchronously painting the appended text if 
       *                  needed. If false is specified then it is up 
       *                  to the caller to make sure that the method 
       *                  {@link #updateGuiAfterContentChange} is 
       *                  manually called at some later time, probably 
       *                  as soon as all the text lines has been appended.
       * @see     #appendLine(const GString&)
       */
      void appendLine ( GString* text, bool autoDel, bool updtGui );

      void continueSelection ( bool shift );

      /**
       * Set current position and drag the selection to it.
       *
       * However, the selection will be dragged only if the selection mode
       * is currently on. No painting will take place. That has to be done
       * by the caller. We assume that the selection origo position has
       * already been set.
       */
      void dragSelection ( const GTextViewer::Pos& newPos, 
                           const GTextViewer::Pos& prevPos );

      /**
       * Perform a new search using the same properties and search string as
       * defined by the user in the most recent call to {@link #cmdSearch}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @param   pos  The position of where to start the search.
       *               We will return the position of the next found 
       *               occurence of the text in this argument, but it is 
       *               valid only if we return true. If the search is 
       *               forward then the returned position will point to the
       *               start of the found text. If the search is backward 
       *               then the returned position will point to the end of 
       *               the found text.
       * @return  True if a match was found, or else false. If we return
       *          false then the caller can assume that the value in <i>pos</i>
       *          has not been changed by us.
       * @see     #searchNextImpl
       */
      bool findNextText ( GTextViewer::Pos& pos ) const;

      char getChrShowLinefeed () const;
      char getChrShowSpace () const;
      char getChrShowTab () const;

      /**
       * Get a copy of the current position of the caret, within global text.
       * 
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       * @see     #getCurrentPosX
       * @see     #getCurrentPosY
       */
      GTextViewer::Pos getCurrentPos () const;

      int getCurrentPosX () const;
      int getCurrentPosY () const;
      int getFontHeight () const;

      /**
       * Get a reference to the indexed line.
       */
      const GString& getIndexedLine ( int index ) const;
      
      /**
       * @see #getIndexOfFirstVisibleLine
       */
      int getIndexOfFirstVisibleColumn () const;

      /**
       * @see #getIndexOfFirstVisibleColumn
       */
      int getIndexOfFirstVisibleLine () const;
      
      int getLinesCount () const;

      /**
       * Get a copy of the currently selected text, if any.
       * The returned text will have exactly the same format as 
       * is placed on the system clipboard upon {@link #cmdCopy}.
       * 
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @param   returnStr The string of where to return a copy of the 
       *                    currently selected text. If no text is currently 
       *                    selected we will return an empty string in here.
       * @return  A reference to the same string object as is specified 
       *          in parameter <i>returnStr</i>.
       * @see     #isSelectedTextMultipleLines
       */
      GString& getSelectedText ( GString& returnStr ) const;
      
      int getWidestLine () const;
      
      int getWindowVisibleColumnsCount () const;
      
      int getWindowVisibleLinesCount () const;

      /**
       * Activate the specified caret position and scroll that location
       * into view if required.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.02
       */
      void gotoPos ( int absLine, int absColumn, bool updtLastExplicitXPos = true );
      void gotoPos ( const Pos& pos, bool updtLastExplicitXPos = true );

      /**
       * Add the area of the window where the specified line is contained
       * to the region that needs to be repainted as soon as possible (when
       * the message queue is idle).
       */
      void invalidateLine ( int lineIdx );

      /**
       * Add the area of the window where the specified lines range are
       * contained to the region that needs to be repainted as soon as
       * possible (when the message queue is idle).
       */
      void invalidateLines ( int firstLine, int lastLine );

      bool isAnySelectedText () const;

      /**
       * Return true if the "find next" command can currently be executed.
       * That is if and only if there are currently any text in the search string.
       * The search string is set upon each call to {@link #cmdSearch}.
       */
      bool isFindNextReady () const;

      /**
       * Test if the so-called "quick-scroll" mode is currently on.
       * @see #setQuickScrollMode
       */
      bool isQuickScrollMode () const;

      /**
       * Test if the currently selected text contains more than one linefeed.
       *
       * Mark that even if this method returns true {@link #getSelectedText}
       * can still contain a trailing linefeed, but never more than one
       * and never any linefeeds in the "body" of the text.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @return  True if and only if there is some current selection and 
       *          that selection contains no more than one line of text.
       * @see     #getSelectedText
       */
      bool isSelectedTextSingleLine () const;

      bool isSelectionMode () const;

      bool isShowLinefeeds () const;
      bool isShowSpaces () const;
      bool isShowTabs () const;

      bool isVisibleVirtualX ( int virtualX ) const;
      bool isVisibleY ( int absY ) const;
      bool isVisiblePos ( const GTextViewer::Pos& absPos ) const;

      bool isWordWrap () const;

      void removeAllText ();

      /**
       * Try to restore the specified caret position and view.
       * We will check the validity of every specified parameter.
       * Those that are found to be invalid will silently be ignored.
       * The specified position will always be scrolled into view 
       * if needed, and if valid.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       * @param   pos  Position of the caret, as returned from a 
       *               recent call to {@link #getCurrentPos}.
       * @param   firstVisibleLine  Index of first visible line, 
       *               as returned from a recent call to 
       *               {@link #getIndexOfFirstVisibleLine}.
       * @param   firstVisibleColumn  Index of first visible column, 
       *               as returned from a recent call to 
       *               {@link #getIndexOfFirstVisibleColumn}.
       */
      void restoreCaretPositioningAndView ( const GTextViewer::Pos& pos, 
                                            int firstVisibleLine,
                                            int firstVisibleColumn );

      /**
       * Find the next occurence of the current search text by calling 
       * {@link #findNextText} and then automatically select the found text 
       * and put the caret at the correct position with respect to forward 
       * or backward search.
       *
       * If we don't find another occurence we will do nothing 
       * but return false.
       * 
       * @author  Leif Erik Larsen
       * @since   2005.02.03
       * @return  True if we did find another occurence of the text,
       *          or else false.
       */
      bool searchNextImpl ();

      void setChrShowLinefeed ( char chr );
      void setChrShowSpace ( char chr );
      void setChrShowTab ( char chr );

      /**
       * Activate or deactivate the so-called "quick-scroll" mode.
       *
       * If the quick-scroll mode is on then the content of the whole
       * text area will be scrolled directly (without waiting for the
       * caret to reach the edges of the visible area) when the user
       * preses the various caret navigation keys such
       * as Up/Down/Left/Right/PageUp/PageDown/etc.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.09
       * @see     #isQuickScrollMode
       */
      void setQuickScrollMode ( bool flag );

      void setSelectionEndPos ( const GTextViewer::Pos& pos );
      void setSelectionStartPos ( const GTextViewer::Pos& pos );

      void setShowLinefeeds ( bool flag );
      void setShowSpaces ( bool flag );
      void setShowTabs ( bool flag );

      void setWordWrap ( bool flag );

      void startSelectionMode ();
      void stopSelectionMode ();
      void unselectAll ();

      /**
       * @author  Leif Erik Larsen
       * @since   2004.10.06
       * @see     #appendLine(GString*, bool, bool)
       */
      void updateGuiAfterContentChange ();

      void updateScrollBarPos ();
      void updateScrollBarPosAndRange ();
      void updateScrollBarRange ();

      /**
       * Update the content of the indexed line with the specified new text.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.26
       * @param   index   Index of which line to update.
       * @param   text    The new text of the indexed line.
       */
      void updateLine ( int index, const GString& text );

   protected:

      /**
       * This method will be called by {@link #cmdGoto} to let the user
       * specify a line number of which to go to.
       *
       * It is up to the implementing subclass to override this method and
       * present the dialog to the user. If this method is not overridden then
       * the user will not be able to select a line and the command will
       * simply do nothing but return.
       *
       * It is also up to the implementing subclass to show any error message
       * if needed and if preferred.
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.18
       * @return  The selected line number, or zero if cancelled or unsupported.
       */
      virtual int getUserSelectedLineNumber ();

   public:

      DECLARE_COMMAND_TABLE(GTextViewer);
};

#endif
