/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TOGGLEBUTTON
#define __GLIB_TOGGLEBUTTON

#include "glib/gui/GWindow.h"

/**
 * This is the low level window class that implements a toggle button control.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.30
 */
class GToggleButton : public GWindow
{
   private:

      /**
       * The class representing the system provided toggle button control.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      class Peer : public GWindow
      {
         public:

            Peer ( GToggleButton& parentWin, bool is3State, bool noFocusPaint );
            virtual ~Peer ();

         private:

            /** Disable the copy constructor. */
            Peer ( const Peer& src ) {}

            /** Disable the assignment operator. */
            Peer& operator= ( const Peer& ) { return *this; }
      };

      bool is3State;
      bool useHotKey;
      mutable Peer peer;

   public:

      GToggleButton ( const GString& name,
                      const GString& constraints,
                      GWindow& parentWin,
                      long winStyle = WS_VISIBLE,
                      long winStyle2 = WS2_DEFAULTPAINT,
                      bool is3State = false,
                      bool noFocusPaint = false,
                      bool useHotKey = true );

      virtual ~GToggleButton ();

   private:

      /** Disable the copy constructor. */
      GToggleButton ( const GToggleButton& src ) : peer(*this, false, false) {}

      /** Disable the assignment operator. */
      GToggleButton& operator= ( const GToggleButton& ) { return *this; }

   protected:

      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );

   public:

      virtual bool isEmpty () const;
      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual GString queryValue () const;

      /**
       * Get a reference to the peer window.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      GWindow& getPeer ();

      /**
       * Get the current toggle state. Possible return values are:
       * 1=on, 0=off or 2=undefined. The last one can possibly be returned
       * only if the toggle button is defined to support three states.
       */
      int getToggleState () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      virtual void grabFocus ( bool force = false );

      /**
       * This method is called when the user has pressed the mnemonic
       * keyboard code (Alt+X, where X is the hot mnemonic character).
       *
       * We will perform just as if the toggle button has been clicked by
       * the mouse, and toggle the state of the button.
       */
      virtual void hotKeyAction ();

      /**
       * Return true if and only if the toggle state is currently undefined.
       * This can possibly happen only if the toggle button is defined to
       * support three states.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.27
       * @see     #getToggleState
       * @see     #setToggleStateUndefined
       */
      bool isToggleStateUndefined () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      virtual void setEnabled ( bool flag = true, bool repaint = true );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.07
       */
      virtual void setOily ( bool flag );

      /**
       * Set the toggle state on (1) or off (0), or undefined (2) if
       * the toggle button is defined to support three states.
       */
      void setToggleState ( int state );

      /**
       * Set the text of the toggle button.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.28
       */
      virtual void setText ( const GString& text );

      /**
       * Set the toggle state to be undefined, but only if
       * the toggle button is defined to support three states.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.27
       * @see     #setToggleState
       * @see     #isToggleStateUndefined
       */
      void setToggleStateUndefined ();
};

#endif
