/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GMenu.h"
#include "glib/gui/GDecoratedWindow.h"
#include "glib/gui/GScrollbar.h"
#include "glib/gui/GAddComponentException.h"
#include "glib/gui/GNoSuchComponentException.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GMouseEvent.h"
#include "glib/gui/event/GMouseListener.h"
#include "glib/gui/border/GBorder.h"
#include "glib/gui/dragndrop/GDragDropHandler.h"
#include "glib/gui/dragndrop/GDragInfo.h"
#include "glib/gui/dragndrop/GDragOverAnswer.h"
#include "glib/gui/layout/GLayoutManager.h"
#include "glib/primitives/GBoolean.h"
#include "glib/util/GMath.h"
#include "glib/util/GIsHereTracker.h"
#include "glib/util/GValueChangeListener.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/sys/GSystem.h"
#include "glib/GProgram.h"

/**
 * \class GWindow
 * This is the base window class that is extended by all window classes
 * in the GLib class library.
 *
 * <h2>Focus Management & Activation</h2>
 * {@link #getFocusedWindow}<br>
 * {@link #grabFocus}<br>
 * {@link #grabFocusAsynchronously}<br>
 * {@link #hasFocus}<br>
 * {@link #isActive}<br>
 * {@link #isFocusable}<br>
 * {@link #isFocusCycleRoot}<br>
 * {@link #isOily}<br>
 * {@link #setActive}<br>
 * {@link #setEnabled}<br>
 * {@link #setFocusable}<br>
 * {@link #setOily}<br>
 *
 * <h2>Layout Management</h2>
 * {@link #getFontHeight}<br>
 * {@link #getInsets}<br>
 * {@link #getLayoutConstraints}<br>
 * {@link #getLayoutManager}<br>
 * {@link #getMaximumHeight}<br>
 * {@link #getMaximumSize}<br>
 * {@link #getMaximumWidth}<br>
 * {@link #getMinimumHeight}<br>
 * {@link #getMinimumSize}<br>
 * {@link #getMinimumWidth}<br>
 * {@link #getPreferredHeight}<br>
 * {@link #getPreferredSize}<br>
 * {@link #getPreferredWidth}<br>
 * {@link #getWidthOfString}<br>
 * {@link #layout}<br>
 * {@link #postLayout}<br>
 * {@link #setBorder}<br>
 * {@link #setInsets}<br>
 * {@link #setLayoutConstraints}<br>
 * {@link #setLayoutManager}<br>
 * {@link #setMaximumSize}<br>
 * {@link #setMinimumSize}<br>
 * {@link #setPreferredSize}<br>
 *
 * <h2>Naming & Logging</h2>
 * {@link #getClassName}<br>
 * {@link #getDebugInfoString}<br>
 * {@link #getDebugName}<br>
 * {@link #getFullName}<br>
 * {@link #getName}<br>
 * {@link #getWindowID}<br>
 * {@link #getWindowTreeString}<br>
 *
 * TODO: Add more groups of GWindow-methods here.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */

/**
 * \typedef GWindow::WindowProc
 * The platform dependent message handler procedure type.
 *
 * @author  Leif Erik Larsen
 * @since   2001.10.26
 * @see     GWindow#SubclassWindow
 */

/**
 * \typedef GWindow::SysRect
 * The platform dependent rectangle structure.
 *
 * @author  Leif Erik Larsen
 * @since   2004.06.02
 * @see     GWindow#MakeSysRect
 */

/** 
 * \var GWindow::wrapperOnly
 * Will be true if initiated with constructor #1, or else false. 
 */

/** 
 * \var GWindow::name
 * The case sensitive name of the window. 
 */

/** 
 * \var GWindow::fullName
 * This variable is initiated by the very first call 
 * to {@link #getFullName}. 
 */

/** 
 * \var GWindow::nameCounter
 * This counter will increase with one for each child window name that is 
 * generated due to empty name string given. 
 */

/** 
 * \var GWindow::text
 * A copy of the current window text. 
 */

/** 
 * \var GWindow::tooltipText
 * A copy of the current tooltip text of the window. 
 */

/** 
 * \var GWindow::resources
 * The resource table to be used by "this" window, or null if we are to 
 * use the same resource table as of our parent window. 
 */

/** 
 * \var GWindow::userData
 * Optional program defined user data for the window. 
 */

/** 
 * \var GWindow::autoDeleteUserData
 * True if and only if we has the ownership of the {@link #userData}. 
 */

/** 
 * \var GWindow::mouseListeners
 * The array of registered mouse listeners of the window. 
 */

/** 
 * \var GWindow::valueChangeListeners
 * The array of registered value change listeners of the window. 
 */

/** 
 * \var GWindow::defaultMsgProc
 * The default message handler routine, as of before we "subclassed" 
 * the window. 
 */

/** 
 * \var GWindow::winClass
 * The class (type) of this window. 
 */

/** 
 * \var GWindow::winentry
 * The window entry object. 
 */

/**
 * \var GWindow::hWnd
 * The system dependent window handle of ours. 
 */

/** 
 * \var GWindow::parentWin
 * The parent window, or null if no parent. 
 */

/** 
 * \var GWindow::ownerWin
 * The owner window, or null if no owner. 
 */

/** 
 * \var GWindow::tooltipPos
 * The default tooltip position for this window. 
 */

/** 
 * \var GWindow::tooltipDelayShowMillis
 * The default number of milliseconds to wait until the tooltip box is 
 * actually displayed, for this window. A value of zero means "no delay".
 * A value less than zero means "use the same value as our parent window".
 */

/** 
 * \var GWindow::tooltipAutoHideMillis
 * The default number of milliseconds to wait until automatically hiding the
 * tooltip box after it has been displayed. A value of zero means "never".
 * A value less than zero means "use the same value as our parent window".
 */

/** 
 * \var GWindow::ownedModalDialogs
 * Contains references to the windows currently running as a modal dialog 
 * on this window. 
 */

/** 
 * \var GWindow::deleteParentWin
 * True if we shall delete <i>parentWin</i> when we are destroyed. 
 */

/** 
 * \var GWindow::deleteOwnerWin
 * True if we shall delete <i>ownerWin</i> when we are destroyed. 
 */

/** 
 * \var GWindow::childrenByName
 * A reference to all of our current child windows, sorted by name. 
 */

/** 
 * \var GWindow::childrenByIndex
 * A reference to all of our current child windows, unsorted. 
 */

/** 
 * \var GWindow::activeTimersBySysID
 * Used to map user defined timerID (string) to their corresponding 
 * system timerID (int). 
 */

/** 
 * \var GWindow::dragDropHandler
 * The current drag-n-drop handler for the window. Is null by default.
 */

/** 
 * \var GWindow::autoDeleteDragDropHandler
 * True if we shall automatically delete {@link #dragDropHandler}
 * upon destruction.
 */

/** 
 * \var GWindow::caretIsOn
 * True when the blinking caret is ON. 
 */

/** 
 * \var GWindow::os2y
 * True if this window uses OS/2'ish Y-coordinates. 
 */

/** 
 * \var GWindow::mouseIsCaptured
 * True while the mouse cursor is captured. 
 */

/**
 * \var GWindow::accelTable
 * Pointer to the accelerator table that defines which keyboard events
 * we should automatically translate into command messages (GM_COMMAND).
 *
 * Can be null.
 *
 * @author  Leif Erik Larsen
 * @see     #setAccelTable
 */

/**
 * \var GWindow::closeWinOnDestroyObject
 * True if we shall automatically close the window when the
 * GWindow-object is destroyed.
 */

/**
 * \var GWindow::layoutManager
 * The layout manager of the window, or null if no layout manager has
 * been set.
 */

/** 
 * \var GWindow::border
 * The border of the window (if != null) gets automatically painted. 
 */

/** 
 * \var GWindow::deleteBorder
 * True if we shall delete <i>border</i> when we are destroyed. 
 */

/**
 * \var GWindow::autoDeleteLayoutManager
 * True if we shall automatically delete <i>layoutManager</i> upon
 * window object destruction.
 */

/**
 * \var GWindow::layoutConstraints
 * The layout constraints name/id of the window.
 *
 * Can be used by <i>layoutManager</i> in order to identify where to
 * place the window within its layout scheme.
 */

/**
 * \var GWindow::insets
 * The insets of the window defines the frame free space area of the
 * window.
 *
 * The insets is to contain no child windows, and it can for instance
 * be used as the space of where to paint something like a border.
 *
 * It is up to the layout manager to respect the insets, however.
 */

/**
 * \var GWindow::oily
 * @see #isOily
 */

/**
 * \var GWindow::focusable
 * @see #isFocusable
 */

/**
 * \var GWindow::layoutPosted
 * True when we have already posted a message for later 
 * asynchronous execution of {@link #layout}, and which has not 
 * yet been handled. Or else false. This is used to prevent the 
 * GUI message queue to be filled up with more than one single 
 * instance of that message.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.05
 * @see     #postLayout
 */

/**
 * \var GWindow::AboutToHandleKey
 * Will be true when we are about to handle keyboard input.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.03
 * @see     #IsAboutToHandleKey
 */

/**
 * \var GWindow::SysDesktopWindow
 * This object represents the system dependent desktop window.
 *
 * @author  Leif Erik Larsen
 * @since   2003.11.10
 * @see     #getSysDesktopWindow
 */

/** 
 * \var GWindow::HiddenDummy
 * This hidden window can be used as a "dummy" parent window for anyone. 
 */

/**
 * \var GWindow::preferredSize
 * The preferred size of "this" window, or null if the preferred size
 * has not been explisitly set by a call to {@link #setPreferredSize}.
 */

/**
 * \var GWindow::autoDeletePreferredSize
 * True if we shall delete {@link #preferredSize} upon destruction.
 */

/**
 * \var GWindow::minimumSize
 * The minimum size of "this" window, or null if the minimum size
 * is unknown.
 */

/**
 * \var GWindow::maximumSize
 * The maximum size of "this" window, or null if the maximum size
 * is unknown.
 */

/** 
 * \var GWindow::defaultBackgroundColor
 * Default background color of this window. The background color profile 
 * will not be stored for the window, if the current background color is 
 * the same as this default. 
 */

/** 
 * \var GWindow::defaultForegroundColor
 * Default foreground color of this window. The boreground color profile 
 * will not be stored for the window, if the current foreground color is 
 * the same as this default. 
 */

/** 
 * \var GWindow::defaultFontNameSize
 * Default fontNameSize of this window. The fontNameSize profile will not 
 * be stored for the window, if the current fontNameSize is the same as 
 * this default. 
 */

/** 
 * \var GWindow::winStyle2
 * Window style flags that are independent of the underlying system.
 */

/** 
 * \var GWindow::FontDimTemplateStr
 * String used by {@link GDialogPanel} and {@link GStatusbar} to 
 * calculate average character width. 
 */

/**
 * \var GWindow::DESKTOP_WINDOW_NAME
 * This is the name of the top level desktop window, known as
 * HWND_DESKTOP in the OS/2 API. This name is reserved. That is,
 * no application defined window can have this name.
 */

/** 
 * \var GWindow::SysDefBckColor
 * The system default background color for windows. 
 */

/** 
 * \var GWindow::SysDefFrgColor
 * The system default foreground color for windows. 
 */

/** 
 * \var GWindow::SysDefFontNameSize
 * The system default variable-width fontNameSize for windows. 
 */

/** 
 * \var GWindow::SysDefFixedFontNameSize
 * The system default fixed-width fontNameSize for windows. 
 */

/** 
 * \var GWindow::PrfItemName_BckColor
 * The name of the profile item used to store and restore the background 
 * color of the window. 
 */

/** 
 * \var GWindow::PrfItemName_FrgColor
 * The name of the profile item used to store and restore the foreground 
 * color of the window. 
 */

/** 
 * \var GWindow::PrfItemName_FontNameSize
 * The name of the profile item used to store and restore the fontNameSize 
 * of the window. 
 */

/**
 * \var GWindow::recentSearchIndex
 * Varaiable used in conjunction with <i>getComponentIndex()</i> to
 * speed optimize the search by guessing that the requested Component
 * is the same as was requested by the previous call
 * to <i>getComponentIndex()</i>.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */

/**
 * \var GWindow::ctrlChangedFilter
 * The <i>DM_CTRLCHANGED</i> message will be sent only if this
 * variable is zero.
 *
 * @see     #sendCtrlChanged
 * @see     #enterCtrlChangedFilter
 * @see     #exitCtrlChangedFilter
 * @see     #getCtrlChangedFilter
 */

/**
 * \var GWindow::noInitCtrlChanged
 * True if no initial GM_CTRLCHANGED messages.
 */

/**
 * \var GWindow::hotChar
 * The keyboard character of the Hot Key that is associated to the
 * Component by a call to {@link #autoAssocAHotKey}. This will be an
 * uppercase character, not lowercase.
 *
 * @see #hotKeyAction
 * @see #getHotKeyChar
 */

/**
 * \var GWindow::hasSent_GM_INITDIALOG
 * This is a flag used internally by this class to test if the
 * GM_INITDIALOG message has been sent to the user message
 * handler or not.
 */

/**
 * \var GWindow::userMsgProc
 * A pointer to the user defined message handler (if any).
 * This is just an interface pointer, so the memory ownership
 * of the referenced object is not ours.
 *
 * @see #setDialogMessageHandler
 */

/**
 * \var GWindow::filterDlgMessages
 * True if we shall filter out all dialog messages from
 * being sent to the user program registered dialog message handler.
 */

/**
 * \var GWindow::isIn_DmInitDialog
 * This boolean is true when, and only when, the <i>GM_INITDIALOG</i>
 * message is about being processed.
 */

/**
 * \var GWindow::isInDismiss
 * True when inside <i>dismiss()</i>.
 * This flag is set by <i>GDialogFrame.dismiss()</i> and
 * <i>GDialogPanel.dismiss()</i>.
 */

/**
 * \var GWindow::recentFocus
 * A pointer to the child window of ours that was the last
 * one that had the keyboard input focus when we was active.
 * This component will get the focus back when we are activated
 * again (if ever).
 */

/**
 * \enum GWindow::WS2_STYLES
 * GLib-spesific window style flags.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_NULL
 * Dummy style that contains "no flags".
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_DEFAULTPAINT
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to prevent the {@link #onPaint} method from
 * being called upon the <i>WM_PAINT</i> message.
 *
 * Instead, the window will be painted directly by the system
 * dependent default message handler only.
 *
 * This flag is typically used by various window classes that are
 * used to wrap system specific dialog controls, such as text entry,
 * push button, scroll bar, etc.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_IGNORE_COLORS_PROFILE
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to prevent the default implementation of
 * <i>writeProfile()</i> and <i>queryProfile()</i> from attempting
 * to write or read the color options.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_IGNORE_FONT_PROFILE
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to prevent the default implementation of
 * <i>writeProfile()</i> and <i>queryProfile()</i> from attempting
 * to write or read the font options.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_IGNORE_COLORS_AND_FONT_PROFILE
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to prevent the default implementation of
 * <i>writeProfile()</i> and <i>queryProfile()</i> from attempting
 * to write or read the font- and color- options.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to prevent <i>layout</i> from being called on the
 * parent window when the font has been changed on "this" window.
 * One example of where this flag is used is on popup menu windows.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_AUTO_PAINT_BACKGROUND
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to make the window automatically clear the
 * background area using the current background color, before
 * calling {@link #onPaint}.
 *
 * @see #onPaintBackground
 * @see #WS2_AUTO_PAINT_TEXT
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_DONT_CREATE_HWND
 * Add this flag to the <i>winStyle2</i> parameter of the method
 * {@link #init} if you want to postpone the creation of the
 * system dependent window handle even more.
 *
 * If this flag is specified then the sub-class code is
 * responsible for setting the HWND explicitly as soon as
 * possible with a call to {@link #setHWND}.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_OS2Y
 * Add this flag if you want the window to use OS/2'ish Y-coordinates.
 * That is, where "ypos=0" means "bottom" instead of "top", and 
 * positive values points up while negative values points down.
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_NO_RECURSIVE_KEY_HANDLING
 * Unhandled keyboard events are automatically given recursively 
 * to the parent window by default. Use this flag to switch off
 * this behaviour. If this flag is specified then the default 
 * implementation of {@link #onKey}/{@link #callDefaultKeyHandler} 
 * will return false instead of calling the parent window 
 * recursively, if there is no accelerator command associated 
 * with the key. This flag is used e.g. by pupup windows in OS/2, 
 * to prevent the parent of the popup from stealing keyboard 
 * events from the system that manages the keyboard navigation 
 * of the menu.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.13
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_NOT_STATIC
 * Specify this flag for windows that should not be treated 
 * as "static" components. Static components will never receive
 * keyboard focus by the default focus manager, and they will
 * never send any {@link GDialogMessageHandler#GM_CTRLCHANGED}
 * message at all since they are "static" (that is; they contain
 * no changeable value).
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @see     #isStatic
 * @see     #WS2_DONT_SEND_CTRL_CHANGE_MSG
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_DONT_INCLUDE_VALUE_IN_CTRL_CHANGE_MSG
 * Specify this flag for windows that should not 
 * include its component value in parameter #2 of 
 * {@link GDialogMessageHandler#GM_CTRLCHANGED}, which is sent 
 * upon {@link #sendCtrlChanged}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @see     #isStatic
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_DONT_SEND_CTRL_CHANGE_MSG
 * Specify this flag in order to prevent {@link #sendCtrlChanged}
 * from sending any {@link GDialogMessageHandler#GM_CTRLCHANGED}
 * messages at all.
 *
 * Note that static windows will never send that message 
 * anyway, regardless if this flag is specified or not.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.07
 * @see     #WS2_NOT_STATIC
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_AUTO_PAINT_TEXT
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to make the window automatically paint its
 * containing window text, before calling {@link #onPaint}. The text 
 * painting is done by calling {@link #onPaintText}.
 *
 * This flag may be usable e.g. for windows created during testing,
 * but it is also used e.g. internally by {@link GTooltip}.
 *
 * @see #onPaintText
 * @see #WS2_AUTO_PAINT_BACKGROUND
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_AUTO_SHOW_TOOLTIP
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to make the window automatically show a tooltip 
 * box when the mouse is captured. The tooltip box is shown only if 
 * {@link #isShowTooltip} returns true and {@link #getTooltipText}
 * returns some non-empty text.
 *
 * This flag is used e.g. by {@link GToolbarButton} 
 * {@link GPushButton} and {@link GStaticText}.
 *
 * @author  Leif Erik Larsen
 * @since   2005.12.13
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT
 * Add this flag to the <i>winStyle2</i> parameter of the constructor
 * of <i>GWindow</i> to make the window use the same profile section 
 * name as of ots parent window. That is, if this flag is specified 
 * then the default implementation of {@link #getProfileSectionName}
 * will return a copy of the received parent section name.
 *
 * This flag is typically used for "Peer" and "ClientArea" windows,
 * since these are typically platform dependent and the application 
 * probably want the ini-file to be portable between platforms.
 * Another reason is to get shorter section names for deeply nested
 * windows.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.27
 */

/**
 * \var GWindow::WS2_STYLES GWindow::WS2_HIDDEN
 * If this flag is specified then the window will be initially 
 * hidden, even if the <i>winStyle</i> is defined with the system 
 * dependent WS_VISIBLE-flag.
 *
 * @author  Leif Erik Larsen
 * @since   2006.05.03
 */

/** 
 * \enum GWindow::MessageID
 * These message-IDs should normally not be used directly by 
 * application code. 
 */

/**
 * \var GWindow::MessageID GWindow::GM_ABSTRACTCOMMAND
 * This is the message identifier used to identify GLib command
 * messages. Such command messages are usually posted asynchronosuly
 * and therefore uses the underlying window message queue to expose
 * the command execution until the GUI thread of the program is idle.
 */

/** 
 * \var GWindow::MessageID GWindow::GM_PRESPARAMCHANGED
 * Used internally for Win32 only, to emulate OS/2's 
 * WM_PRESPARAMCHANGED. 
 */

/**
 * \var GWindow::MessageID GWindow::GM_USERMESSAGE
 * This enumeration constant identifies a user defined window 
 * message that is provided via an instance of {@link GUserMessage} 
 * and which the application code usually handles by overriding 
 * {@link #onUserMessage}.
 * 
 * Here is a sample of how to send a user defined messages to a window:
 * 
 * <pre>
 *    win.sendUserMessage("MyMessage1", (GObject*) myMsgArgument);
 *    win.sendUserMessage("MyMessage2", (GObject*) myMsgArgument);
 * </pre>
 * 
 * Handling user defined messages:
 * 
 * <pre>
 *    class MyClass : public GWindow
 *    {
 *       bool onUserMessage ( GUserMessage& msg ) 
 *       {
 *          GString id = msg.getParam1String();
 *          if (id == "MyMessage1")
 *          {
 *             GObject* arg = msg.getParam2();
 *             // ... do something with the argument.
 *             return true;
 *          }
 *          else
 *          if (id == "MyMessage2")
 *          {
 *             GObject* arg = msg.getParam2();
 *             // ... do something with the argument.
 *             return true;
 *          }
 *          return GWindow::onUserMessage(msg);
 *       }
 *    }
 * </pre>
 * 
 * <b>Note</b> that this message is also used by GDialogPanel when 
 * sending and handling {@link GDialogMessage}'s. The way this works 
 * is by {@link GDialogPanel#onUserMessage} checks if the received 
 * user message is an instance of {@link GDialogMessage} and if it 
 * is then it will call {@link #onDialogMessage}.
 *
 * @author  Leif Erik Larsen
 * @since   2003.12.01
 * @see     #sendUserMessage
 * @see     #postUserMessage
 */

/**
 * \var GWindow::MessageID GWindow::GM_USERMESSAGE_FROM_THREAD
 * Same as {@link #GM_USERMESSAGE} except this identifyer is used 
 * when sending such an user message from some non-GUI background 
 * thread via {@link GThread#sendGuiUserMessage}. The point is to 
 * automate the notification back to the waiting thread when the 
 * GUI has finihed handling the message.
 *
 * @author  Leif Erik Larsen
 * @since   2004.02.01
 * @see     GThread#guiUserMessageHandlerHasFinished
 */

/**
 * \var GWindow::MessageID GWindow::GM_ASYNC_PAINT
 * Special message used to perform asynchronous painting of 
 * of window even if window is defined by the system to perform 
 * synchronous painting e.g. upon {@link #invalidateRect}.
 * This is the case e.g. for OS/2 push button controls.
 * By posting this message instead of just calling e.g.
 * {@link #invalidateRect} we fool the system to get 
 * asynchronous painting anyway.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.16
 */

/**
 * \var GWindow::MessageID GWindow::GM_CREATE_CARET
 * Used internally by GLib (e.g. by GPushButton in OS/2) to 
 * asynchronously create and display the system dependent caret.
 * This might be needed in order not to conflict with the 
 * default caret management of the system.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.16
 */

/**
 * \var GWindow::MessageID GWindow::GM_ASYNC_GRAB_FOCUS
 * Special message used to perform asynchronous focus change 
 * requests. By posting this message instead of just calling 
 * e.g. {@link #grabFocus} the focus change is not actually 
 * done until the calling GUI handling code has returned from 
 * its message handler. This also makes it possible to request
 * focus change from other threads than the GUI-thread it self.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.27
 * @see     #grabFocusAsynchronously
 */

const GString GWindow::DESKTOP_WINDOW_NAME = "!DESKTOP!";
const GString GWindow::FontDimTemplateStr = "0123456789AaaBbbCccDddEeeFffGggHhhIiiJjjKkkLllMmmNnnOooPppQqqRrrSssTttUuuVvvWwwXxxYyyZzz";
GWindow::LeaderStyle GWindow::CommonLeaderStyle = GWindow::LeaderStyle_FANCY1;
bool GWindow::AboutToHandleKey = 0;
GWindow* GWindow::SysDesktopWindow = null;
GWindow* GWindow::HiddenDummy = null;

/** 
 * Map of all instances of GWindow*, using the GInteger(int(HWND)) as key. 
 *
 * @author  Leif Erik Larsen
 * @since   2006.05.08
 */
static GHashtable<GInteger, GWindow> WindowsMap(256);

const GColor GWindow::SysDefBckColor = GSystem::GetSystemColor(GSystem::SCID_DIALOGBCK);
const GColor GWindow::SysDefFrgColor = GSystem::GetSystemColor(GSystem::SCID_DIALOGTXT);
const GString GWindow::SysDefFontNameSize = "9.WarpSans";
const GString GWindow::SysDefFixedFontNameSize = "11.System VIO";
const GString GWindow::PrfItemName_BckColor = "BckColor";
const GString GWindow::PrfItemName_FrgColor = "FrgColor";
const GString GWindow::PrfItemName_FontNameSize = "Font";

/**
 * \class GWindowEntry
 * This class is used to map window messages from the underlying system to
 * the message handler method of the window object.
 *
 * There is one instance of this class for each <i>GWindow</i>.
 *
 * This class is probably not to be used by any user code. It is for
 * internal usage by <i>GWindow</i> only.
 *
 * @author  Leif Erik Larsen
 * @since   2001.02.20
 * @see     GWindow#getWindowEntry
 */
class GWindowEntry : public GObject
{
   friend class GWindow;

   /**
    * Needed for compataibility with the system, because an instance 
    * of this class is contained in the "window  user data" property
    * of the system.
    */
   USHORT structSize;

   /** 
    * The window that is being wrapped by this window entry. 
    */
   GWindow* theWindow;

   explicit GWindowEntry ( GWindow* theWindow )
      : structSize(sizeof(*this)),
        theWindow(theWindow)
   {
   }

   virtual ~GWindowEntry ()
   {
   }

   GWindowEntry ( const GWindowEntry& ); //!< Disable the copy constructor.
   void operator= ( const GWindowEntry& ); //!< Disable the assignment operator.

   /** 
    * Will call the {@link GWindow#handleWindowMessage} on {@link #theWindow}. 
    */
   GWindowMessage::Answer dispatchMessage ( GWindowMessage& msg )
   {
      GString excptWhat;
      GString stackTrace;

      // Let the window handle the message. Any unhandled exception thrown by 
      // the user code will be catched by us so the user can choose to 
      // continue normally or abort the application in that case.
      try {
         return theWindow->handleWindowMessage(msg);
      } catch (GException& e) {
         excptWhat = e.toString();
         stackTrace = e.getStackTrace(excptWhat);
      } catch (std::exception& e) {
         excptWhat = e.what();
      } catch (APIRET& rc) {
         GString str(" (APIRET=%d)", GVArgs(rc));
         excptWhat = GSystem::GetApiRetString(rc) + str;
      } catch (...) {
         excptWhat = GStringl("%TxtUnhandledExceptionUnknownType");
      }

      // If we reach here it is because of some exception catched above.
      // The user description of the exception is in "excptWhat".
      if (stackTrace == "")
         stackTrace = "No stacktrace available.";
      GLog::Log(this, stackTrace);
      GStringl errmsg("%TxtUnhandledException", GVArgs(excptWhat).add(msg.toString()));
      GProgram& oprg = GProgram::GetProgram();
      GWindow& mainWin = oprg.getMainWindow();
      GMessageBox::Answer answ = mainWin.showMessageBox(errmsg, GMessageBox::TYPE_ERROR, "aId", GString::Empty, false, stackTrace);
      if (answ == GMessageBox::IDABORT)
         throw;
      
      // User has choosen to ignore the exception, so just silently continue
      // and hope for the exception not to be critical.
      return 0;
   }
};

/**
 * This is constructor #1.
 *
 * The specified window handle will not be "subclassed" when using
 * this constructor. That is, the constructed window object will
 * just wrap the specified window handle so that various methods can
 * be used on the object in order to manipulate the window state.
 * The window object will not receive window message events, because
 * they are always sent directly to the default message handler.
 */
GWindow::GWindow ( HWND win,
                   bool os2y )
        :winClass(GWindowClass::GENERIC),
         ownedModalDialogs(1),
         childrenByName(4),
         childrenByIndex(4)
{
   GInteger* key = new GInteger(int(win));
   WindowsMap.put(key, this, true, false);
   clearInstanceVariables();
   setOS2Y(os2y);
   wrapperOnly = true;
   initiateTheObject(true, null, null, 0, winStyle2);
   initiateTheName(GString::Empty);
   hWnd = win;
}

/**
 * This is constructor #2.
 *
 * @throws GAddComponentException if we fail to create the window or if we 
 *                      fail to add the window to the parent. One typical 
 *                      reason is that the window name specified already is
 *                      in use by some other child window of the same parent.
 */
GWindow::GWindow ( const GString& name,
                   HWND win,
                   int winStyle2 )
        :winClass(GWindowClass::GENERIC),
         ownedModalDialogs(1),
         childrenByName(4),
         childrenByIndex(4)
{
   GInteger* key = new GInteger(int(win));
   WindowsMap.put(key, this, true, false);
   clearInstanceVariables();
   this->name = name; // Just temporary, until calling initiateTheName() later on.
   setOS2Y((winStyle2 & WS2_OS2Y) != 0);
   initiateTheObject(true, null, null, 0, winStyle2);
   initiateTheName(name);
   hWnd = win;

   // We must subclass the window so that our {@link #handleWindowMessage}
   // gets called the same way as for other window classes.
   WindowProc mproc = SubclassWindow(hWnd, GenericWindowProc);
   setDefaultMsgProc(mproc);
   ::WinSetWindowPtr(hWnd, QWL_USER, (void*) winentry);
}

/**
 * Constructor #3.
 *
 * This constructor is intended for most user code, either it is 
 * called by a subclass constructor or directly by some end user 
 * code. The parameters of this constructor are placed in a succession 
 * that should require as few arguments as possible (using as many 
 * default arguments as possible) in most cases.
 *
 * @author  Leif Erik Larsen
 * @since   2006.05.03
 * @param   parentWin   A pointer to the parent window of the new
 *                      window. All windows must have a parent, but
 *                      you can specify null as the parent in order
 *                      to use the system dependent desktop as the
 *                      parent window.
 * @param   name        The case sensitive name/id of the window.
 *                      All child windows within the same parent must
 *                      have unique names. If an empty string is
 *                      specified then we will automatically
 *                      construct an unique name to the window.
 * @param   constraints The layout constraints name/id can be used by
 *                      layout managers to identify where to place
 *                      the window within its layout scheme. The
 *                      meaning of the constraints is specific to
 *                      each layout manager.
 * @param   winStyle2   GLib defined window flags.
 * @param   defBckColor The default background color of the new 
 *                      window. If {@link GGraphics#DUMMY_COLOR} is 
 *                      specified then we will use the same default
 *                      background color as of the specified 
 *                      parent window (if <i>parentWin</i> is null
 *                      we will use the system default, which 
 *                      is defined by {@link #SysDefBckColor}).
 * @param   defFrgColor The default foreground color of the new 
 *                      window. If {@link GGraphics#DUMMY_COLOR} is 
 *                      specified then we will use the same default 
 *                      foreground color as of the specified 
 *                      parent window (if <i>parentWin</i> is null
 *                      we will use the system default, which 
 *                      is defined by {@link #SysDefFrgColor}).
 * @param   defFont     The default fontNameSize of the new 
 *                      window. If an empty string is 
 *                      specified then we will use the same default
 *                      fontNameSize as of the specified 
 *                      parent window (if <i>parentWin</i> is null
 *                      we will use the system default, which 
 *                      is defined by {@link #SysDefFontNameSize}).
 * @param   ownerWin    A pointer to the owner window of the new
 *                      window. All windows must have an owner, but
 *                      you can specify null as the owner in order
 *                      to use the parent window as the owner
 *                      window. This is the most typical as well.
 * @param   winStyle    System dependent window flags.
 * @param   winClass    The class of the window, as of which to be
 *                      seen by the underlying system. This can
 *                      be one of the predefined system classes,
 *                      such as <i>GWindowClass::MLE</i>,
 *                      <i>GWindowClass::SCROLLBAR</i>, etc.
 * @throws  GAddComponentException if we fail to create the window
 *                      or if we fail to add the window to the parent. 
 *                      One typical reason is that the window name 
 *                      specified already is in use by some other
 *                      child window of the same parent.
 */
GWindow::GWindow ( GWindow* parentWin,
                   const GString& name,
                   const GString& constraints,
                   int winStyle2,
                   const GColor& defBckColor,
                   const GColor& defFrgColor,
                   const GString& defFont,
                   GWindow* ownerWin,
                   long winStyle,
                   const GWindowClass& winClass )
        :winClass(winClass),
         ownedModalDialogs(1),
         childrenByName(4),
         childrenByIndex(4),
         layoutConstraints(constraints)
{
   clearInstanceVariables();
   init(name, parentWin, ownerWin, winStyle, winStyle2, defBckColor, defFrgColor, defFont);
}

/**
 * Constructor #4.
 *
 * To be used for window classes whos creation of the actual HWND is
 * postponed until the {@link #init} method is explicitly called by
 * application code, usually the sub-class.
 *
 * @author  Leif Erik Larsen
 * @since   2001.02.02
 */
GWindow::GWindow ( const GWindowClass& winClass,
                   const GString& constraints,
                   bool os2y )
        :winClass(winClass),
         ownedModalDialogs(1),
         childrenByName(4),
         childrenByIndex(4),
         layoutConstraints(constraints)
{
   clearInstanceVariables();
   setOS2Y(os2y);
}

/** 
 * Constructor #5.
 * Intended for calls by internal GLib-subclasses.
 *
 * @author  Leif Erik Larsen
 * @since   2006.05.03
 */
GWindow::GWindow ( const GString& name,
                   const GString& constraints,
                   GWindow* parentWin,
                   GWindow* ownerWin,
                   long winStyle,
                   int winStyle2,
                   const GWindowClass& winClass,
                   const GColor& defBckColor,
                   const GColor& defFrgColor,
                   const GString& defFont )
        :winClass(winClass),
         ownedModalDialogs(1),
         childrenByName(4),
         childrenByIndex(4),
         layoutConstraints(constraints)
{
   clearInstanceVariables();
   init(name, parentWin, ownerWin, winStyle, winStyle2, defBckColor, defFrgColor, defFont);
}

void GWindow::init ( const GString& name,
                     GWindow* parentWin,
                     GWindow* ownerWin,
                     long winStyle,
                     int winStyle2,
                     const GColor& defBckColor,
                     const GColor& defFrgColor,
                     const GString& defFont )
{
   if (&defBckColor == &GGraphics::DUMMY_COLOR)
   {
      if (parentWin == null)
         defaultBackgroundColor = GWindow::SysDefBckColor;
      else
         defaultBackgroundColor = parentWin->defaultBackgroundColor;
   }
   else
   {
      defaultBackgroundColor = defBckColor;
   }

   if (&defFrgColor == &GGraphics::DUMMY_COLOR)
   {
      if (parentWin == null)
         defaultForegroundColor = GWindow::SysDefFrgColor;
      else
         defaultForegroundColor = parentWin->defaultForegroundColor;
   }
   else
   {
      defaultForegroundColor = defFrgColor;
   }

   if (defFont == "")
   {
      if (parentWin == null)
         defaultFontNameSize = GWindow::SysDefFontNameSize;
      else
         defaultFontNameSize = parentWin->defaultFontNameSize;
   }
   else
   {
      defaultFontNameSize = defFont;
   }

   setOS2Y((winStyle2 & WS2_OS2Y) != 0);
   bool postpone = ((winStyle2 & WS2_DONT_CREATE_HWND) != 0);
   closeWinOnDestroyObject = true;
   this->name = name; // Just temporary, until calling initiateTheName() later on.
   initiateTheObject(postpone, parentWin, ownerWin, winStyle, winStyle2);
   initiateTheName(name);

   if (parentWin != null && !parentWin->wrapperOnly)
      parentWin->addChildWindow(*this);
}

void GWindow::initiateTheObject ( bool postponeWinCrea,
                                  GWindow* parentWin,
                                  GWindow* ownerWin,
                                  long winStyle,
                                  int winStyle2 )
{
   this->nameCounter = 0;
   this->winStyle2 = winStyle2;
   this->parentWin = parentWin;
   this->ownerWin = ownerWin;

   if (!wrapperOnly)
      winentry = new GWindowEntry(this);

   if (parentWin != null)
      resources = &parentWin->getResourceTable();
   else
      resources = null;

   if (!postponeWinCrea)
      createTheWindow(parentWin, ownerWin, winStyle);
}

/**
 * Create the OS-specific HWND instance and assign it to <i>hWnd</i>.
 *
 * We will also initiate the overloaded window procedure in case the
 * class of the window is a system class, such as
 * <i>GWindowClass::MLE</i>, <i>GWindowClass::SCROLLBAR</i>, etc.
 */
void GWindow::createTheWindow ( GWindow* parentWin,
                                GWindow* ownerWin,
                                long winStyle )
{
   // For several system defined window classes the control-data parameter
   // to ::WinCreateWindow() is expected to be a pointer to a specific
   // window class specific structure, if not null (see the OS/2
   // documentation of the pCtlData parameter of the WinCreateWindow API).
   // Therefore we can not give a pointer to our own GWindowEntry object
   // via this parameter for such types of windows. I had a hard and long
   // time of debugging until I finally experienced the importance of this on
   // july the 19th, 2000. Until then, a pointer to the GWindowEntry object
   // was always given to this parameter, and therfore caused several
   // annoying bugs with memory being overwritten. Espessially, the bug that
   // caused the Larsen Commander command line window from not being created
   // correctly (in Larsen Commander version <= 1.1.6) was due to this.
   bool issysclass = winClass.isSystemClass();
   void* ctldata = null;
   if (!issysclass)
      ctldata = (void*) winentry;

   // ---
   HWND hWndParent = HWND_DESKTOP;
   if (parentWin != null)
   {
      hWndParent = parentWin->getHWND();
      if (hWndParent == null)
      {
         if (GLog::Filter(GLog::PROD))
         {
            GString thisName = getDebugName();
            GString parentName = parentWin->getDebugName();
            GString msg("Possible error; parentWin->getHWND() returned null, this=\"%s\", parentWin=\"%s\"", GVArgs(thisName).add(parentName));
            GLog::Log(this, msg);
         }
         hWndParent = HWND_DESKTOP;
      }
   }

   // ---
   if ((winStyle2 & WS2_HIDDEN) != 0)
   {
      // This flags means that we should create the window initially 
      // invisible, even if the winStyle is defined with the system 
      // dependent WS_VISIBLE-flag. So make sure that system flag is off.
      winStyle &= ~WS_VISIBLE;
   }

   if (ownerWin == null)
      ownerWin = parentWin;
   HWND hWndOwner = HWND_DESKTOP;
   if (ownerWin != null)
   {
      hWndOwner = ownerWin->getHWND();
      if (hWndOwner == null)
      {
         if (GLog::Filter(GLog::PROD))
         {
            GString thisName = getDebugName();
            GString ownerName = ownerWin->getDebugName();
            GLog::Log(this, "Possible error; ownerWin->getHWND() returned null, this=\"%s\", ownerWin=\"%s\"", GVArgs(thisName).add(ownerName));
         }
         hWndOwner = HWND_DESKTOP;
      }
   }

   // ---
   static int WindowIDCounter = 0;
   WindowIDCounter++;
   const char* sysClassID = winClass.getSystemClassID();
   hWnd = ::WinCreateWindow(hWndParent, sysClassID, "", winStyle,
                            0, 0, 0, 0, hWndOwner, HWND_TOP,
                            WindowIDCounter, ctldata, null);

   if (hWnd == null)
   {
      gthrow_(GIllegalStateException("Internal error creating the window!"));
   }
   else
   {
      GInteger* key = new GInteger(int(hWnd));
      WindowsMap.put(key, this, true, false);
   }

   if (issysclass)
   {
      // The specified class is a system class that is already registered.
      // We must, however, subclass it, so that our {@link #handleWindowMessage}
      // gets called the same way as for other window classes.
      WindowProc mproc = SubclassWindow(hWnd, GenericWindowProc);
      setDefaultMsgProc(mproc);
   }

   ::WinSetWindowPtr(hWnd, QWL_USER, (void*) winentry);

   if (&winClass != &GWindowClass::SCROLLBAR)
   {
      setBackgroundColor(defaultBackgroundColor);
      setForegroundColor(defaultForegroundColor);

      // Since we are part of construction, temporarily set that we should
      // not call {@link #layout} on our parent window (from
      // {@link #onFontNameSizeChanged}). This is in order to prevent 
      // various possible NullPointerException's due to child-windows 
      // not yet being created and/or properly initialized.
      int oldWinStyle2 = winStyle2;
      try {
         winStyle2 |= WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED;
         setFontNameSize(defaultFontNameSize);
         winStyle2 = oldWinStyle2;
      } catch (...) {
         winStyle2 = oldWinStyle2;
         throw;
      }
   }
}

/**
 * Initiate the name of "this" window and make sure that the name is
 * not already in use by some other sibling.
 *
 * If the name specified is an empty string then we will automatically
 * assign an unique name to the window.
 *
 * @throws GAddComponentException if the name specified is already
 *                                in use by one of the sibling windows.
 */
void GWindow::initiateTheName ( const GString& name )
{
   if (name == GWindow::DESKTOP_WINDOW_NAME)
      gthrow_(GAddComponentException("Window-name is reserved: " + name));

   GString nameToUse(name);
   if (nameToUse == "")
   {
      if (parentWin == null)
      {
         nameToUse = GWindow::DESKTOP_WINDOW_NAME;
      }
      else
      {
         nameToUse = GString("##%d", GVArgs(++(parentWin->nameCounter)));
         while(parentWin->childrenByName.containsKey(nameToUse))
            nameToUse = GString("##%d", GVArgs(++(parentWin->nameCounter)));
      }
   }
   this->name = nameToUse;
}

/**
 * The window will be automatically closed if needed, and all
 * resources used by the window object will be released.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
GWindow::~GWindow ()
{
   if (hWnd != null)
   {
      GInteger key = int(hWnd);
      WindowsMap.remove(key);
   }

   // Deregister our self from our parents list of children.
   if (parentWin != null && !parentWin->wrapperOnly)
      parentWin->removeChildWindow(*this);

   // Deregister every mouse listener, if any.
   if (mouseListeners != null)
   {
      mouseListeners->removeAll();
      delete mouseListeners;
      mouseListeners = null;
   }

   // Deregister every value change listener, if any.
   if (valueChangeListeners != null)
   {
      valueChangeListeners->removeAll();
      delete valueChangeListeners;
      valueChangeListeners = null;
   }

   // Deactivate the layout manager.
   setLayoutManager(null, false);

   if (autoDeletePreferredSize)
      delete preferredSize;
   delete minimumSize;
   delete maximumSize;

   if (deleteBorder)
      delete border;

   if (deleteParentWin)
      delete parentWin;

   if (deleteOwnerWin)
      delete ownerWin;

   border = null;
   parentWin = null;
   preferredSize = null;
   autoDeletePreferredSize = false;
   minimumSize = null;
   maximumSize = null;

   if (hWnd != null)
   {
      // Destroy the caret if needed.
      destroyCaret_();

      // ---
      if (isMouseCapture())
         captureMouse(false);

      // "Undock" our self from the window. This needs two steps.
      // 1) Restore to the original default message handler as of before we
      // subclassed the window (from the underlying systems point of view).
      if (defaultMsgProc != null && !wrapperOnly)
         GWindow::SubclassWindow(hWnd, defaultMsgProc);
      defaultMsgProc = null;
      // 2) Remove the user data from the window, which we used to
      // associate the window handle with this window object.
      if (winentry != null && !wrapperOnly)
         ::WinSetWindowPtr(hWnd, QWL_USER, null);

      // Set hWnd to zero _before_ we destroy the window handle.
      // This is in order to prevent {@link #genericWindowProc} from
      // dispatching any GUI messages to our message handler while
      // the window handle is about being destroyed.
      HWND myWinH = hWnd;
      hWnd = null;

      // Close/destroy the window handle.
      if (closeWinOnDestroyObject)
      {
         // A WM_DESTROY message will not be received by our message handler,
         // because we have already "undocked" our self from the window by
         // setting the "window user data" to null.
         ::WinDestroyWindow(myWinH);
      }
   }

   delete activeTimersBySysID;
   delete winentry;
   winentry = null;
   defaultMsgProc = null;
   activeTimersBySysID = null;

   if (autoDeleteDragDropHandler)
      delete dragDropHandler;

   if (autoDeleteUserData)
      delete userData;
}

/**
 * This method is called initially by all constructors of the
 * <i>GWindow</i> class in order to clear all instance variables.
 */
void GWindow::clearInstanceVariables ()
{
   userData = null;
   autoDeleteUserData = false;
   wrapperOnly = false;
   defaultBackgroundColor = GWindow::SysDefBckColor;
   defaultForegroundColor = GWindow::SysDefFrgColor;
   defaultFontNameSize = GWindow::SysDefFontNameSize;
   oily = false;
   focusable = false;
   nameCounter = 0;
   resources = null;
   defaultMsgProc = null;
   mouseListeners = null;
   valueChangeListeners = null;
   winentry = null;
   hWnd = null;
   parentWin = null;
   deleteParentWin = false;
   deleteOwnerWin = false;
   activeTimersBySysID = null;
   dragDropHandler = null;
   autoDeleteDragDropHandler = false;
   caretIsOn = false;
   caretXPos = 0;
   caretYPos = 0;
   caretWidth = 0;
   caretHeight = 0;
   os2y = false;
   mouseIsCaptured = false;
   leaderStyle = GWindow::LeaderStyle_DEFAULT;
   popupMenu = null;
   accelTable = null;
   closeWinOnDestroyObject = false;
   winStyle2 = WS2_NULL;
   layoutManager = null;
   autoDeleteLayoutManager = false;
   border = null;
   deleteBorder = false;
   preferredSize = null;
   autoDeletePreferredSize = false;
   minimumSize = null;
   maximumSize = null;
   recentSearchIndex = 0;
   filterDlgMessages = false;
   ctrlChangedFilter = 0;
   noInitCtrlChanged = false;
   hasSent_GM_INITDIALOG = false;
   isIn_DmInitDialog = false;
   isInDismiss = false;
   recentFocus = null;
   hotChar = '\0';
   userMsgProc = null;
   layoutPosted = false;
   tooltipText = "";
   tooltipPos = GTooltip::PosInheritFromParent;
   tooltipDelayShowMillis = -1; // Inherit value from parent window by default.
   tooltipAutoHideMillis = -1; // Ditto.
}

/**
 * Get a reference to the case sensitive window name/id.
 * All child windows within the same parent are guaranteed to have
 * an unique name.
 *
 * @see #getDebugName
 * @see #getFullName
 */
GString GWindow::getName () const
{
   return name;
}

/** 
 * Convert to system dependent Y-coordinate, if needed. 
 */
int GWindow::sysY ( int y ) const
{
   if (os2y)
      return y;
   else
      return getWindowSize().height - y - 1;
}

/**
 * Return true if and only if we are about to handle keyboard input.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.03
 */
bool GWindow::IsAboutToHandleKey ()
{
   return GWindow::AboutToHandleKey;
}

/**
 * Get a reference to the special GLib window that can be 
 * used by anyone that needs some dummy window to use as 
 * a parent window.
 *
 * This static method is typically used by the dialog creation
 * code of GLib in Windows, and by code that creates some 
 * window that is to be added to a statusbar via method
 * {@link GStatusbar#addCell} (which will automatically assign
 * the added window to a new parent window anyway).
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.23
 */
GWindow& GWindow::GetHiddenDummy ()
{
   if (GWindow::HiddenDummy == null)
      GWindow::HiddenDummy = new GWindow(null, "{HiddenDummy}", GString::Empty, WS2_HIDDEN);
   return *GWindow::HiddenDummy;
}

/** 
 * Convert to library rectangle from the 
 * specified system rectangle.
 *
 * @author  Leif Erik Larsen
 * @since   2006.02.13
 * @param   r    The rectangle of which to convert from.
 * @param   conv The system Y-converter of which to callback.
 * @see     #MakeSysRect
 */
GRectangle GWindow::MakeLibRect ( const SysRect& r, const GSystemYConverter& conv )
{
   GRectangle rect;
   rect.x = r.xLeft;
   rect.width = r.xRight - r.xLeft;
   if (conv.isOS2Y())
      rect.y = r.yBottom;
   else
      rect.y = conv.sysY(r.yTop + 1);
   rect.height = r.yTop - r.yBottom;
   return rect;
}

/** 
 * Convert to system dependent rectangle from the 
 * specified {@link GRectangle}. 
 *
 * @author  Leif Erik Larsen
 * @since   2004.06.08
 * @param   r    The rectangle of which to convert to system rectangle.
 * @param   conv The system Y-converter of which to callback.
 * @param   incl True if the caller needs a rectangle that is 
 *               "inclusive". That is; if the returned rectangle 
 *               should not include any additional space to the right
 *               and bottom/top (depending on Y coordinate system.
 *               This parameter is false e.g. when called by Windows 
 *               code that is to paint a filled rectangle, because 
 *               the Win32 API ::FillRect() is documented not to fill
 *               the right and the bottom edge of the rectangle.
 *               Else this parameter should be true.
 * @see     #MakeLibRect
 */
GWindow::SysRect GWindow::MakeSysRect ( const GRectangle& r, 
                                        const GSystemYConverter& conv,
                                        bool incl )
{
   RECTL sr;
   sr.xLeft = r.x;
   sr.xRight = r.x + r.width;
   if (incl)
      sr.xRight -= 1;
   if (conv.isOS2Y())
   {
      sr.yBottom = r.y;
      sr.yTop = r.y + r.height;
      if (incl)
         sr.yTop -= 1;
   }
   else
   {
      sr.yTop = conv.sysY(r.y);
      sr.yBottom = sr.yTop - r.height + 1;
      if (!incl)
         sr.yTop += 1;
   }
   return sr;
}

/**
 * Get a pointer to the first modal dialog that is owned by this 
 * window, and that is currently visible. If this window has no
 * such dialog we will return null.
 *
 * This method is used by {@link GFrameWindow} as part of the 
 * algorithm used to raise the modal dialog to the top of the 
 * Z-order above the window that owns it, when the owner window 
 * is attempted to be activated.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.22
 * @see     #addOwnedModalDialog
 */
GWindow* GWindow::getFirstOwnedModalDialog ()
{
   int count = ownedModalDialogs.getCount();
   for (int i=0; i<count; i++)
   {
      GWindow& w = ownedModalDialogs[i];
      if (w.isVisible())
         return &w;
   }
   return null;
}

/**
 * Remove the specified window from the list of modal dialogs 
 * that are currently owned by this window.
 *
 * This method is probably to be called by 
 * {@link GDialogFrame#modalMessageLoop} only.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.09.24
 * @see     #getFirstOwnedModalDialog
 * @see     #addOwnedModalDialog
 */
void GWindow::removeOwnedModalDialog ( GWindow& win )
{
   for (int i=ownedModalDialogs.getCount()-1; i>=0; i--)
   {
      GWindow& w = ownedModalDialogs[i];
      if (&win == &w)
      {
         ownedModalDialogs.remove(i, 1, false);
         break;
      }
   }
}

/**
 * Add the specified window to the list of modal dialogs that 
 * are currently executing as a modal dialog on this window. 
 * The specified window will be automatically forced to be on top
 * of this window, and the user will not be able to activate this
 * (the owning) window, as long as there is any such executing 
 * modal dialogs.
 *
 * This method is probably to be called by 
 * {@link GDialogFrame#modalMessageLoop} only.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.09.22
 * @see     #getFirstOwnedModalDialog
 * @see     #removeOwnedModalDialog
 */
void GWindow::addOwnedModalDialog ( GWindow& win )
{
   // First, make sure that the specified window is not already added.
   for (int i=ownedModalDialogs.getCount()-1; i>=0; i--)
   {
      GWindow& w = ownedModalDialogs[i];
      if (&win == &w)
         return; // gthrow_(GIllegalStateException("The specified window is already owned by this window!"));
   }
   ownedModalDialogs.add(win);
}

/**
 * Set the owner window object.
 * Probably to be used by {@link GDialogFrame#execute} only.
 *
 * @author  Leif Erik Larsen
 * @since   2004.07.29
 * @see     #setParentWindow
 */
void GWindow::setOwnerWindow ( GWindow* owner, bool autoDelete )
{
   if (deleteOwnerWin)
      delete ownerWin;
   ownerWin = owner;
   deleteOwnerWin = autoDelete;
   HWND newOwner = (owner == null ? null : owner->getHWND());
   HWND prevOwner = ::WinQueryWindow(hWnd, QW_OWNER);
   if (newOwner != prevOwner)
      ::WinSetOwner(getHWND(), newOwner);
}

/**
 * Make the specified window the new parent window of this window.
 * This method should be called with great care. Usually a window 
 * should be assigned to a parent as part of window construction.
 * This method was initially implemented to be used by 
 * {@link GStatusbar} only, in order to be able to dynamically 
 * add statusbar cells that contains application defined 
 * windows.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.09.23
 * @param   newParent The new parent window. Can be null.
 * @throws  GIllegalArgumentException if the window is already a 
 *                    descendant of the specified new parent window.
 */
void GWindow::setParentWindow ( GWindow* newParent )
{
   if (newParent == parentWin)
      return; // The specified window is already our parent.
   if (newParent != null && isDescendant(*newParent))
      gthrow_(GIllegalArgumentException("Already a descendant of the specified parent."));
   bool hadOldParent = (parentWin != null);
   if (parentWin != null)
      parentWin->removeChildWindow(*this);
   if (deleteParentWin)
      delete parentWin;
   deleteParentWin = false;
   parentWin = newParent;
   if (parentWin != null)
      parentWin->addChildWindow(*this);
   HWND parentHwnd = (parentWin == null ? HWND_DESKTOP : parentWin->getHWND());
   ::WinSetParent(getHWND(), parentHwnd, TRUE);
}

/**
 * Build and return a string object containing a debug tree of window
 * information, including all of the child windows of the window.
 */
GString GWindow::getWindowTreeString ( int level ) const
{
   GString childrenInfo(128);
   int num = getChildWindowCount();
   for (int i=0; i<num; i++)
   {
      GWindow& win = const_cast<GWindow*>(this)->getChildWindow(i);
      GString str = win.getWindowTreeString(level + 1);
      childrenInfo += str;
   }

   GString ret(childrenInfo.length() + 128);
   ret.append(' ', 3 * level);
   ret += getDebugInfoString();
   ret += '\n';
   ret += childrenInfo;
   return ret;
}

/**
 * This method is called on the parent window object (if any) by the
 * constructor of the child window in order to add the child window
 * to the parents reference list of children.
 *
 * @author  Leif Erik Larsen
 * @param   win  The window of which to be added.
 * @throws  GAddComponentException if we fail to add the window.
 *                                One typical reason is that the
 *                                name of the specified window
 *                                is already in use by some other
 *                                child window.
 */
void GWindow::addChildWindow ( GWindow& win )
{
   GString name(win.getName());
   if (!childrenByName.put(name, &win, false))
      gthrow_(GAddComponentException("Window-name already in use: " + name));
   childrenByIndex.add(win);
}

/**
 * This method is called on the parent window object (if any) by the
 * destructor of the child window in order to deregister the child 
 * window from the parents reference list of children.
 *
 * If the specified window is not actually registered as a child 
 * of this window then we will do nothing but silently return.
 *
 * @author  Leif Erik Larsen
 * @since   2004.07.27
 * @param   win  The window of which to be removed/deregistered.
 */
void GWindow::removeChildWindow ( GWindow& win )
{
   GString name(win.getName());
   childrenByName.remove(name);
   for (int i=0, num=childrenByIndex.getCount(); i<num; i++)
   {
      GWindow& child = childrenByIndex.get(i);
      if (&child == &win)
      {
         childrenByIndex.remove(i);
         break;
      }
   }
}

/**
 * Get a reference to the specified child window.
 *
 * @see       #getChildWindowCount
 * @throws    GNoSuchComponentException if the specified index is
 *                                      out of range.
 */
GWindow& GWindow::getChildWindow ( int index )
{
   if (index < 0 || index >= getChildWindowCount())
      gthrow_(GNoSuchComponentException(GString("Index: ") + index));
   return childrenByIndex.get(index);
}

/**
 * Get a reference to the specified child window.
 *
 * @see       #getChildWindowCount
 * @throws    GNoSuchComponentException if there is no child window
 *                                      with the specified name.
 */
GWindow& GWindow::getChildWindow ( const GString& name )
{
   GWindow* win = childrenByName.get(name);
   if (win == null)
      gthrow_(GNoSuchComponentException("Name: " + name));
   return *win;
}

/**
 * Get a pointer to the first child window that has the specified
 * layout constraints.
 *
 * This method is typically used by layout managers only.
 *
 * @return A pointer to the child window, or null if no such child window.
 */
GWindow* GWindow::getChildWindowByLayoutConstraints ( const GString& constraints )
{
   int num = getChildWindowCount();
   for (int i=0; i<num; i++)
   {
      GWindow& child = getChildWindow(i);
      if (child.getLayoutConstraints() == constraints)
         return &child;
   }

   return null;
}

/**
 * Set wheteher or not this window is to use an OS/2 based Y-coordinate 
 * system. 
 *
 * If this property is set true then the origo position [0,0]
 * is located at the bottom left corner of the window, and a positive 
 * y-position counts upwards in the window. This is the default 
 * coordinate system on OS/2.
 *
 * If this property is set false then the origo position [0,0]
 * is located at the top left corner of the window, and a positive 
 * y-position counts downwards in the window. This is the default 
 * coordinate system on Windows.
 *
 * @author  Leif Erik Larsen
 * @since   2004.01.17
 */
void GWindow::setOS2Y ( bool flag )
{
   os2y = flag;
   if (os2y)
      winStyle2 |= WS2_OS2Y;
   else
      winStyle2 &= ~WS2_OS2Y;
}

/**
 * Read and activate all default settings of this window panel from
 * the profile of the window owner program.
 *
 * @author  Leif Erik Larsen
 * @since   1999.12.28
 * @param   sectName  Name of the profile section of where to read
 *                    all the default settings.
 */
void GWindow::queryProfile ( const GString& sectName )
{
   GProgram& oprg = GProgram::GetProgram();
   GSectionBag& ini = oprg.getIniProperties();

   // Query and activate the window properties, but only if the
   // window object has not been requested to ignore them.
   if (!(winStyle2 & WS2_IGNORE_FONT_PROFILE))
   {
      GString def = getFontNameSize();
      GString fontns = ini.getString(sectName, GWindow::PrfItemName_FontNameSize, def);
      if (fontns != "")
         setFontNameSize(fontns);
   }
   if (!(winStyle2 & WS2_IGNORE_COLORS_PROFILE))
   {
      GColor defBck = getBackgroundColor();
      GColor bck = ini.getColor(sectName, GWindow::PrfItemName_BckColor, defBck);
      setBackgroundColor(bck);
      GColor defFrg = getForegroundColor();
      GColor frg = ini.getColor(sectName, GWindow::PrfItemName_FrgColor, defFrg);
      setForegroundColor(frg);
   }

   // Call the same method on all child windows.
   const int num = getChildWindowCount();
   for (int i=0; i<num; i++)
   {
      GWindow& childWin = getChildWindow(i);
      GString subName = childWin.getProfileSectionName(sectName);
      childWin.queryProfile(subName);
   }
}

/**
 * Write all current settings of this window to the profile of the
 * window owner program.
 *
 * We will call this method recirsively on all of our child windows
 * as well. On each recursive call we will automatically append a comma
 * and the name of the corresponding child window to the specified
 * section name. This way each window will have its own section name
 * of where the settings of the corresponding window can be safely
 * written without overwriting the settings of any other window of the
 * program.
 *
 * @author  Leif Erik Larsen
 * @since   1999.12.28
 * @param   sectName  Name of the profile section of where to write
 *                    all the current settings.
 * @param   force     True if we shall write all properties regardless
 *                    of the user customized options of what to write
 *                    upon exit, or else false if we shall respect the
 *                    user selected options. Typically, the force-flag
 *                    is true if we are called due to a menu command
 *                    such as "Save options now" or if the user has
 *                    selected "Save all options on exit". Else this
 *                    flag is probably to be false.
 * @see     #queryProfile
 */
void GWindow::writeProfile ( const GString& sectName, bool force )
{
   GProgram& oprg = GProgram::GetProgram();
   GSectionBag& ini = oprg.getIniProperties();

   // Write the window properties, but only if the
   // window object has not been requested to ignore them.
   if (!(winStyle2 & WS2_IGNORE_FONT_PROFILE))
   {
      GString fontns = getFontNameSize();
      if (fontns != "" && fontns != defaultFontNameSize)
         ini.putString(sectName, GWindow::PrfItemName_FontNameSize, fontns, force || oprg.isAutoSaveFontOptions());
      else
         ini.deleteItem(sectName, GWindow::PrfItemName_FontNameSize);
   }
   if (!(winStyle2 & WS2_IGNORE_COLORS_PROFILE))
   {
      GColor bck = getBackgroundColor();
      if (bck != defaultBackgroundColor)
         ini.putColor(sectName, GWindow::PrfItemName_BckColor, bck, force || oprg.isAutoSaveColorOptions());
      else
         ini.deleteItem(sectName, GWindow::PrfItemName_BckColor);
   }
   if (!(winStyle2 & WS2_IGNORE_COLORS_PROFILE))
   {
      GColor frg = getForegroundColor();
      if (frg != defaultForegroundColor)
         ini.putColor(sectName, GWindow::PrfItemName_FrgColor, frg, force || oprg.isAutoSaveColorOptions());
      else
         ini.deleteItem(sectName, GWindow::PrfItemName_FrgColor);
   }

   // Call the same method on all child windows.
   const int num = getChildWindowCount();
   for (int i=0; i<num; i++)
   {
      GWindow& childWin = getChildWindow(i);
      GString subName = childWin.getProfileSectionName(sectName);
      childWin.writeProfile(subName, force);
   }
}

/**
 * Get the default section name to use for the profile settings of this
 * window. The returned name is used as the argument to the automated 
 * calls to {@link #queryProfile} and {@link #writeProfile}.
 *
 * The default implementation returns a string that equals 
 * <code>parentSectName + "." + getName()</code>. This ensures 
 * all windows will get and use their own unique section name
 * in the user profile, by default. However, if the window has 
 * the flag {@link #WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT}
 * set then the default implementation will return the same 
 * name as specified in <code>parentSectName</code>.
 *
 * However, this method can be overridden to give each window 
 * full control of which section they use to manage their 
 * respective user profile settings.
 *
 * Return an empty string in order to use the default (nameless)
 * section at the top of the ini-file.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.27
 * @param   parentSectName  The section name of the parent window.
 *                          For top-level windows, this parameter will
 *                          be an empty string.
 * @return  The section name as is to be given to 
 *          {@link #queryProfile} and {@link #writeProfile}. 
 */
GString GWindow::getProfileSectionName ( const GString& parentSectName )
{
   if ((winStyle2 & WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT) != 0)
      return parentSectName;
   else
      return parentSectName + "." + getName();
}

/**
 * Get the low level ID of the window.
 * This is the ID as seen by the underlying system.
 */
int GWindow::getWindowID () const
{
   return WinQueryWindowUShort(getHWND(), QWS_ID);
}

/**
 * Get a pointer to the top-level window that contains this window
 * at some level.
 *
 * The top-level window is usually the frame window, either an
 * instance of {@link GFrameWindow} or an instance
 * of {@link GDialogFrame}.
 *
 * @author  Leif Erik Larsen
 * @since   2001.03.02
 * @return  A reference to the top-level window, or a reference to our
 *          self if the top-level window could not be determined of
 *          some reason.
 * @see     #getFocusedWindow
 */
GWindow& GWindow::getTopLevelWindow ()
{
   for (GWindow* win=this; win!=null; win=win->getParentWindow())
   {
      GDialogPanel* dlg = dynamic_cast<GDialogPanel*>(win);
      if (dlg != null)
         return dlg->getOwnerFrame();
      else
      if (dynamic_cast<GFrameWindow*>(win) != null)
         return *win;
   }

   return *this;
}

/**
 * Set or clear the {@link #WS2_DEFAULTPAINT} flag on this window.
 */
void GWindow::setDefaultPaint ( bool defaultPaint )
{
   if (defaultPaint)
      winStyle2 |= WS2_DEFAULTPAINT;
   else
      winStyle2 &= ~WS2_DEFAULTPAINT;
}

/**
 * Get a pointer to the GWindow that currently wraps the specified 
 * system dependent window handle, or null if the application 
 * currently has no such window.
 *
 * @author  Leif Erik Larsen
 * @since   2004.03.11
 */
GWindow* GWindow::GetWindowFromHandle ( HWND win )
{
   GWindowEntry* we = (GWindowEntry*) ::WinQueryWindowPtr(win, QWL_USER);
   if (we == null)
      return null;
   GInteger key = int(win);
   GWindow* w = WindowsMap.get(key);
   if (w == null)
      return null;
   // Just in order to make sure that we return a valid window,
   // return it only if its entry is (still-) associated with the
   // system defined user data of the HWND.
   if (w->winentry == we)
      return w;
   else
      return null;
}

/**
 * Get a reference to the <i>GWindow</i> that currently has the keyboard
 * input focus and which is a child window of ours at some level.
 *
 * If the current focused window is not an instance of <i>GWindow</i>
 * then we will return <i>null</i> even if some window actually has
 * the focus. In that case the focused window is probably not
 * contained in this window.
 *
 * This method is usually to be used on instances of
 * {@link GDialogFrame} and {@link GFrameWindow} only. It is
 * implemented here first and foremost for the sake of
 * this method as a general interface.
 *
 * @author  Leif Erik Larsen
 * @since   2001.03.02
 * @see     #getTopLevelWindow
 */
GWindow* GWindow::getFocusedWindow () const
{
   HWND hw = ::WinQueryFocus(HWND_DESKTOP);
   if (isParentOf(hw))
      return GWindow::GetWindowFromHandle(hw);
   else
      return null;
}

/**
 * Adds the specified mouse listener to receive mouse events from
 * this window when the user uses the mouse on this window.
 *
 * @author  Leif Erik Larsen
 * @since   2004.06.30
 * @param   l    The listener object to add.
 * @see     #removeMouseListener
 */
void GWindow::addMouseListener ( GMouseListener* l )
{
   if (l == null)
      gthrow_(GIllegalArgumentException("l == null"));
   if (mouseListeners == null)
      mouseListeners = new GVector<GMouseListener*>(3);
   if (!mouseListeners->contains(l)) // If not already registered.
      mouseListeners->add(l);
}

/**
 * Removes the specified mouse listener so that it no longer receives
 * mouse events from this window. This method performs no function,
 * nor does it throw any exception, if the specified listener 
 * was not previously added to this window.
 *
 * @author  Leif Erik Larsen
 * @since   2004.06.30
 * @see     #addMouseListener
 */
void GWindow::removeMouseListener ( GMouseListener* l )
{
   if (mouseListeners == null)
      return; // No focus listeners are registered.
   int idx = mouseListeners->indexOf(l);
   if (idx >= 0)
      mouseListeners->remove(idx);
}

/**
 * Adds the specified listener to receive notification when the 
 * component value of the window has been changed. The component 
 * value is dependent of the window subclass. By default, the window 
 * has not component value and the specified listener will never be 
 * called. But for some window classes adding component value change 
 * listeners would make sense. One example is for {@link GTextEntry},
 * which will call the specified listener every time its text has 
 * been changed. Another example is for {@link GSpinner}, which will 
 * call the specified listener every time its contained value has 
 * changed.
 *
 * However, it is up to the window subclass to implement 
 * the code needed to invoke the registered listeners when its 
 * corresponding value has changed. In order to invoke every value 
 * change listener that has been registered, the subclass should 
 * call {@link #fireValueChangeListeners}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.05.27
 * @param   l    The listener object to add.
 * @see     #removeValueChangeListener
 * @see     #fireValueChangeListeners
 */
void GWindow::addValueChangeListener ( GValueChangeListener* l )
{
   if (l == null)
      gthrow_(GIllegalArgumentException("l == null"));
   if (valueChangeListeners == null)
      valueChangeListeners = new GVector<GValueChangeListener*>(3);
   if (!valueChangeListeners->contains(l)) // If not already registered.
      valueChangeListeners->add(l);
}

/**
 * Removes the specified value change listener so that it no longer 
 * receives any value change notifications from this window. This 
 * method performs no function, nor does it throw any exception, 
 * if the specified listener was not previously added to this window.
 *
 * @author  Leif Erik Larsen
 * @since   2004.05.27
 * @see     #addValueChangeListener
 */
void GWindow::removeValueChangeListener ( GValueChangeListener* l )
{
   if (valueChangeListeners == null)
      return; // No value change listeners are registered.
   int idx = valueChangeListeners->indexOf(l);
   if (idx >= 0)
      valueChangeListeners->remove(idx);
}

/**
 * Invoke all value change listeners that are currently registered
 * on this window.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.05.27
 * @see     #addValueChangeListener
 */
void GWindow::fireValueChangeListeners ()
{
   sendCtrlChanged();
   if (valueChangeListeners == null)
      return; // No value change listeners are registered.

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = valueChangeListeners->getCount();
   GVector<GValueChangeListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(valueChangeListeners->elementAt(i));

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GValueChangeListener* l = llist[i];
      l->valueHasChanged(*this);
   }
}

/**
 * Part of {@link GChangeableValueContainer}.
 * This default implementation does nothing but return.
 *
 * @author  Leif Erik Larsen
 * @since   2004.05.27
 * @see     #queryValue
 */
void GWindow::changeValue ( const GString& newValue, bool notify )
{
}

/**
 * Part of {@link GChangeableValueContainer}.
 * The default implementation does nothing but return an empty string.
 *
 * @author  Leif Erik Larsen
 * @since   2004.05.27
 * @see     #changeValue
 */
GString GWindow::queryValue () const
{
   return GString::Empty;
}

/**
 * Return true if and only if the specified window is a child at some
 * level of this window. Will return true also if the specified 
 * window handle is the same as the window handle represented by
 * this window it self.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.28
 */
bool GWindow::isParentOf ( HWND win ) const
{
   bool ret = false;
   while (win != null && !ret)
   {
      if (win == this->hWnd)
         ret = true;
      else
         win = ::WinQueryWindow(win, QW_PARENT);
   }

   return ret;
}

/**
 * Return true if and only if the specified window is a child at some
 * level of this window. Will return true also if the specified 
 * window points to this window it self.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.28
 */
bool GWindow::isParentOf ( GWindow* win ) const
{
   HWND hw = win->getHWND();
   return isParentOf(hw);
}

/**
 * Get the default style to be used by {@link GStaticText} components
 * that are contained in this window.
 *
 * @see #setLeaderStyle
 */
GWindow::LeaderStyle GWindow::getLeaderStyle () const
{
   if (leaderStyle != GWindow::LeaderStyle_DEFAULT)
      return leaderStyle;
   else
   if (parentWin != null)
      return parentWin->getLeaderStyle();
   else
   if (GWindow::CommonLeaderStyle != GWindow::LeaderStyle_DEFAULT)
      return GWindow::CommonLeaderStyle;
   else
      return GWindow::LeaderStyle_FANCY1;
}

/**
 * Set the default style to be used by {@link GStaticText} components
 * that are contained in this window.
 *
 * @see #getLeaderStyle
 * @see #setCommonLeaderStyle
 */
void GWindow::setLeaderStyle ( GWindow::LeaderStyle style )
{
   this->leaderStyle = style;
}

/**
 * Let the user perform a modal vertical track operation, typically
 * used by something like a Split Panel when the user clicks on the
 * vertical split bar.
 *
 * @param  minX   The minimum X-position.
 * @param  maxX   The maximum X-position.
 * @param  startX The X-position of where the track is initated.
 * @param  cancel The value of which to return in case the track
 *                operation is canceled or if it fails.
 * @return The X-position as was positioned by the user when the
 *         mouse button was released.
 */
int GWindow::trackVSplit ( int minX, int maxX, int startX, int cancel )
{
   const int trackW = 4; // The thickness of the "drag frame".
   const int trackWx2 = 2 * trackW;
   const int trackW_2 = trackW / 2;

   TRACKINFO ti;
   memset(&ti, 0, sizeof(ti));
   GRectangle wrect = getWindowRect();
   ti.rclBoundary.xRight = wrect.x + wrect.width - 1 + trackW_2;
   ti.rclBoundary.xLeft = wrect.x - trackW;
   ti.rclBoundary.yTop = wrect.y + wrect.height - 1 + trackW;
   ti.rclBoundary.yBottom = wrect.y - trackW;
   ti.ptlMinTrackSize.x = minX + trackWx2 - trackW_2 + 1;
   ti.ptlMinTrackSize.y = ti.rclBoundary.yTop + trackWx2;
   ti.ptlMaxTrackSize.x = maxX + trackWx2 - trackW_2 + 1;
   ti.ptlMaxTrackSize.y = ti.rclBoundary.yTop + trackWx2;
   ti.cxBorder = trackW;
   ti.cyBorder = trackW;
   ti.cxGrid = 1; // Smooth tracking with mouse
   ti.cyGrid = 1;
   ti.cxKeyboard = 8; // Faster tracking using cursor keys
   ti.cyKeyboard = 8;

   // Starting point
   ti.rclTrack = ti.rclBoundary;
   ti.rclTrack.xRight = startX + trackW_2;
   ti.fs = TF_ALLINBOUNDARY | TF_RIGHT;

   int ret = cancel;
   if (::WinTrackRect(getHWND(), null, &ti))
      ret = ti.rclTrack.xRight - trackW_2;

   if (ret == startX)
      ret = cancel;

   return ret;
}

/**
 * Let the user perform a modal horizontal track operation, typically
 * used by something like a Split Panel when the user clicks on the
 * horizontal split bar.
 *
 * @param  minY   The minimum Y-position.
 * @param  maxY   The maximum Y-position.
 * @param  startY The Y-position of where the track is initated.
 * @param  cancel The value of which to return in case the track
 *                operation is canceled or if it fails.
 * @return The Y-position as was positioned by the user when the
 *         mouse button was released.
 */
int GWindow::trackHSplit ( int minY, int maxY, int startY, int cancel )
{
   const int trackW = 4; // The thickness of the "drag frame".
   const int trackWx2 = 2 * trackW;
   const int trackW_2 = trackW / 2;

   TRACKINFO ti;
   memset(&ti, 0, sizeof(ti));
   GRectangle wrect = getWindowRect();
   ti.rclBoundary.xRight = wrect.x + wrect.width - 1 + trackW;
   ti.rclBoundary.xLeft = wrect.x - trackW;
   ti.rclBoundary.yTop = wrect.y + wrect.height - 1 + trackW_2;
   ti.rclBoundary.yBottom = wrect.y - trackW;
   ti.ptlMinTrackSize.y = minY + trackWx2 - trackW_2 + 1;
   ti.ptlMinTrackSize.x = ti.rclBoundary.xRight + trackWx2;
   ti.ptlMaxTrackSize.y = maxY + trackWx2 - trackW_2 + 1;
   ti.ptlMaxTrackSize.x = ti.rclBoundary.xRight + trackWx2;
   ti.cxBorder = trackW;
   ti.cyBorder = trackW;
   ti.cxGrid = 1; // Smooth tracking with mouse
   ti.cyGrid = 1;
   ti.cxKeyboard = 8; // Faster tracking using cursor keys
   ti.cyKeyboard = 8;

   // Starting point
   ti.rclTrack = ti.rclBoundary;
   ti.rclTrack.yTop = startY + trackW_2;
   ti.fs = TF_ALLINBOUNDARY | TF_TOP;

   int ret = cancel;
   if (::WinTrackRect(getHWND(), null, &ti))
      ret = ti.rclTrack.yTop - trackW_2 + 1;

   if (ret == startY)
      ret = cancel;

   return ret;
}

GResourceTable& GWindow::getResourceTable () const
{
   if (resources != null)
      return *resources;
   if (parentWin != null)
      return parentWin->getResourceTable();
   GProgram& oprg = GProgram::GetProgram();
   return oprg.getResourceTable();
}

void GWindow::createCaret_ ()
{
   if (hWnd && isCaretOn())
   {
      int y = caretYPos;
      if (!isOS2Y())
         y = sysY(y) + caretHeight - 1;
      ::WinCreateCursor(hWnd, caretXPos, y, caretWidth, caretHeight, CURSOR_SOLID | CURSOR_FLASH, null);
      ::WinShowCursor(hWnd, true);
   }
}

void GWindow::destroyCaret_ ()
{
   if (hWnd && isCaretOn())
      ::WinDestroyCursor(hWnd);
}

void GWindow::setCaretPos_ ()
{
   int y = caretYPos;
   if (!isOS2Y())
      y = sysY(y) + caretHeight - 1;
   ::WinCreateCursor(hWnd, caretXPos, y, 0, 0, CURSOR_SETPOS, null);
}

void GWindow::showCaret ( bool flag )
{
   if (isCaretOn() && hWnd)
   {
      ::WinShowCursor(hWnd, flag);
      if (flag)
         sendMessage(WM_TIMER, (MPARAM) TID_CURSOR);
   }
}

void GWindow::setCaretOn ( bool flag )
{
   if (flag == isCaretOn())
   {
      // Nothing to do!
      return; 
   }
   else
   if (flag)
   {
      caretIsOn = true;
      createCaret_();
   }
   else
   {
      destroyCaret_();
      caretIsOn = false;
   }
}

void GWindow::setCaretPos ( int x, int y )
{
   caretXPos = x;
   caretYPos = y;
   if (isCaretOn() && hWnd)
      setCaretPos_();
}

void GWindow::setCaretSize ( int width, int height )
{
   caretWidth = width;
   caretHeight = height;

   if (isCaretOn())
   {
      // Recreate the caret, if it is currently on
      setCaretOn(false);
      setCaretOn(true);
   }
}

/**
 * Enable or disable the window.
 * @param flag    True to enable the window, or else false to disable it.
 * @param repaint True if we shall invalidate the window, or else false.
 */
void GWindow::setEnabled ( bool flag, bool repaint )
{
   if (flag == isEnabled())
      return;
   ::WinEnableWindow(hWnd, flag);
   if (repaint)
      invalidateAll(true);
}

bool GWindow::isEnabled () const
{
   return ::WinIsWindowEnabled(hWnd);
}

/**
 * Test if the specified window is a descendant of this window.
 * The window is a descendant if it is a a child or grand-child of this
 * window. It will be reported to be a descendant also in case the 
 * specified window points to this window it self.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */
bool GWindow::isDescendant ( const GWindow& parent ) const
{
   HWND hwnd = getHWND();
   HWND hwndParent = parent.getHWND();
   if (hwnd == hwndParent)
      return true;
   return ::WinIsChild(hwnd, hwndParent);
}

/**
 * @author  Leif Erik Larsen
 * @since   2004.01.24
 */
class GWindowTimerData : public GObject
{
   friend class GWindow;
   GString userTimerID;
   int systemTimerID;
   int millis;
   GObject* userData;
   bool autoDeleteUserData;

   GWindowTimerData ( const GString& userTimerID, 
                      int systemTimerID, 
                      int millis, 
                      GObject* userData, 
                      bool autoDeleteUserData )
      : userTimerID(userTimerID),
        systemTimerID(systemTimerID),
        millis(millis),
        userData(userData),
        autoDeleteUserData(autoDeleteUserData)
   {
   }

   virtual ~GWindowTimerData ()
   {
      if (autoDeleteUserData)
         delete userData;
   }
};

/**
 * Create and start a new timer event generator that will cause the 
 * GUI-thread to automatically call {@link #onTimer} with aproximately 
 * the specified number of milliseconds between each call.
 *
 * @author  Leif Erik Larsen
 * @since   2004.01.24
 * @param   timerID   The ID of the new timer.
 * @param   millis    Number of milliseconds between each timer event.
 * @param   userData  Optional user data that will be associated with
 *                    the timer and given to every corresponding 
 *                    call to {@link #onTimer}.
 * @param   autoDeleteUserData  True if we shall automatically destroy 
 *                    the userData when the timer is stopped with 
 *                    {@link #stopTimer} or this window object is 
 *                    destroyed, whatever comes first.
 * @see     #resetTimer
 * @see     #stopTimer
 * @see     #isTimerStarted
 * @throws  GIllegalArgumentException if the specified timer is 
 *                    already part of this window.
 * @see     GIllegalStateException if we was unable to create the timer
 *                    for some system dependent reason, such as e.g. if
 *                    there are too many timers active on the system.
 */
void GWindow::startTimer ( const GString& timerID,
                           int millis,
                           GObject* userData,
                           bool autoDeleteUserData )
{
   if (activeTimersBySysID == null)
      activeTimersBySysID = new GKeyBag<GWindowTimerData>(3);

   // Make sure that the specified timerID is not already in use.
   if (getTimerData(timerID, false) != null)
      gthrow_(GIllegalArgumentException(GStringl("TimerID already in use: '%s'", GVArgs(timerID))));

   // Find a system timer ID that is not already in use.
   const int max = GMath::Min(TID_USERMAX, 1000);
   int sysTimerID = 0;
   for (int i=1; i<max; i++)
   {
      GString key = GInteger::ToString(i);
      if (!activeTimersBySysID->containsKey(key))
      {
         sysTimerID = i;
         break;
      }
   }

   if (sysTimerID == 0)
      gthrow_(GIllegalStateException(GStringl("Too many active timers. Max is %d.", GVArgs(max))));

   // Add the system timer ID to our bags so that we know that it is in use.
   GWindowTimerData* td = new GWindowTimerData(timerID, sysTimerID, millis, userData, autoDeleteUserData);
   GString idStr = GInteger::ToString(sysTimerID);
   activeTimersBySysID->put(idStr, td);

   ::WinStartTimer(GProgram::hAB, hWnd, sysTimerID, millis);
}

/**
 * @author  Leif Erik Larsen
 * @since   2004.07.29
 * @param   timerID The ID of which timer to get.
 * @param   throw_  True if we shall throw a GIllegalArgumentException
 *                  if the specified timer is not part of this 
 *                  window. Else we will silently return null.
 * @return  A pointer to the requested timer data, or null if 
 *          the window has no such timer and <i>throw_</i> is false.
 * @throws  GIllegalArgumentException if the specified timer is not 
 *                  part of this window and <i>throw_</i> is true.
 */
GWindowTimerData* GWindow::getTimerData ( const GString& timerID, bool throw_ )
{
   GWindowTimerData* timerData = null;
   if (activeTimersBySysID != null)
   {
      for (int i=activeTimersBySysID->getCount()-1; i>=0; i--)
      {
         GWindowTimerData& td = activeTimersBySysID->getIndexedItem(i);
         if (td.userTimerID == timerID)
         {
            timerData = &td;
            break;
         }
      }
   }
   if (timerData == null && throw_)
      gthrow_(GIllegalArgumentException(GStringl("Unknown timerID: '%s'", GVArgs(timerID))));
   return timerData;
}

/**
 * Test if the specified timer is registered on this window.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.06
 * @param   timerID The ID of which timer to test.
 * @return  True if the specified timer is registered on this window,
 *          or else false.
 * @see     #startTimer
 */
bool GWindow::isTimerStarted ( const GString& timerID )
{
   GWindowTimerData* t = getTimerData(timerID, false);
   return t != null;
}

/**
 * Reset the specified timer so that it will not occur again 
 * until its full delay time has passed again.
 *
 * @author  Leif Erik Larsen
 * @since   2004.07.29
 * @see     #startTimer
 * @see     #stopTimer
 * @throws  GIllegalArgumentException if the specified timer is not 
 *                                    part of this window.
 */
void GWindow::resetTimer ( const GString& timerID )
{
   GWindowTimerData* timerData = getTimerData(timerID, true);
   int sysTimerID = timerData->systemTimerID;
   int millis = timerData->millis;
   ::WinStartTimer(GProgram::hAB, hWnd, sysTimerID, millis);
}

/**
 * Stop the timer which ID was returned from a recent call to
 * <i>startTimer()</i>.
 *
 * @author  Leif Erik Larsen
 * @see     #startTimer
 * @see     #resetTimer
 * @throws  GIllegalArgumentException if the specified timer is not 
 *                                    part of this window.
 */
void GWindow::stopTimer ( const GString& timerID )
{
   // Find the timer info object.
   GWindowTimerData* timerData = getTimerData(timerID, true);

   // Remove the timer from our bags so that we know that
   // it is no longer in use.
   int sysTimerID = timerData->systemTimerID;
   activeTimersBySysID->remove(GInteger::ToString(sysTimerID));

   ::WinStopTimer(GProgram::hAB, hWnd, sysTimerID);
}

/**
 * Show a modal dialog box, using "this" window as the parent window, of
 * where the user can select a file path.
 *
 * @param  titleText   The string to use as title text of the file
 *                     dialog.
 * @param  initialPath The path of which file is to be initially
 *                     selected, or an empty string to use the default.
 * @return The full path of the user selected file, or an empty string
 *         if the user selected the cancel button.
 */
GString GWindow::userChooseFile ( const GString& titleText,
                                  const GString& initialPath )
{
   // Load the text, in case it is loadable.
   GString loadedTitleText = GProgram::LoadText(titleText);

   FILEDLG fild;
   memset(&fild, 0, sizeof(fild));
   fild.cbSize = sizeof(fild);
   fild.fl = FDS_CENTER | FDS_OPEN_DIALOG;
   fild.pszTitle = (char *) loadedTitleText.cstring();
   if (initialPath != "")
   {
      memset(fild.szFullFile, 0, sizeof(fild.szFullFile));
      strncpy(fild.szFullFile, initialPath, sizeof(fild.szFullFile)-1);
   }
   HWND hWnd = ::WinFileDlg(HWND_DESKTOP, getHWND(), &fild);
   if (hWnd && (fild.lReturn == DID_OK))
      return fild.szFullFile;
   else
      return GString::Empty;
}

/** Helper class used by {@link GWindow#userChooseFont} only. */
class FontAttr : public GObject
{
   public:
      const GString name;
      int flag;
      FontAttr ( const GString& name, int flag ) : name(name), flag(flag) {}
      virtual ~FontAttr () {}
   private:
      /** Prevent this operator from being accessed. */
      FontAttr& operator= ( const FontAttr& ) { return *this; }
};

/**
 * Show a modal dialog box, using "this" window as the parent window, of
 * where the user can select any font name and size.
 *
 * @param  titleText   The string to use as title text of the font
 *                     dialog.
 * @param  setFont     True if we shall automatically activate the
 *                     selected font on "this" window when the user
 *                     select the OK button.
 * @param  parentWin   Pointer to which window will be used as the
 *                     parent of the font selection dialog. If null
 *                     is specified then we will use "this" window
 *                     it self as the parent.
 * @param  initialFont The name/size of which font to initially
 *                     activate in the font dialog. If an empty string
 *                     is specified then we will use the current font
 *                     name/size of the window as the initial and
 *                     default font.
 * @param  fixedOnly   True if the user should be able to choose
 *                     only fixed-pitch (monospace) fonts.
 * @return The user selected Font name/size string. If the user
 *         selects no font (e.g. by selecting the cancel button) then
 *         we will return an empty string.
 */
GString GWindow::userChooseFont ( const GString& titleText,
                                  bool setFont,
                                  GWindow* parentWin,
                                  const GString& initialFont,
                                  bool fixedOnly )
{
   if (parentWin == null)
      parentWin = this; // Use our self as the parent window, by default.

   GString fullFaceName;
   if (initialFont == "")
      fullFaceName = getFontNameSize();
   else
      fullFaceName = initialFont;

   // Load the text, in case it is loadable.
   GString loadedTitleText = GProgram::LoadText(titleText);

   FONTMETRICS fm;
   GGraphics g(*this);
   GpiQueryFontMetrics(g.getHandle(), sizeof(fm), &fm);

   FONTDLG fd;
   memset(&fd, 0, sizeof(fd));
   fd.cbSize = sizeof(fd);
   fd.hpsScreen = g.getHandle();
   fd.pszFamilyname = fm.szFamilyname;
   fd.usFamilyBufLen = sizeof(fm.szFamilyname);
   fd.clrFore = CLR_BLACK;
   fd.clrBack = CLR_WHITE;
   fd.fl = FNTS_INITFROMFATTRS | FNTS_CENTER | FNTS_RESETBUTTON | FNTS_HELPBUTTON;
   if (fixedOnly)
      fd.fl |= FNTS_FIXEDWIDTHONLY;
   fd.pszPreview = "Larsen Commander";
   fd.pszTitle = (char*) loadedTitleText.cstring();
   fd.fAttrs.usRecordLength = sizeof(FATTRS);
   fd.fAttrs.lMatch = fm.lMatch;
   memset(fd.fAttrs.szFacename, 0, sizeof(fd.fAttrs.szFacename));
   strncpy(fd.fAttrs.szFacename, fm.szFacename, sizeof(fd.fAttrs.szFacename)-1);

   // Fetch the point size of the font
   GString strSize(33); // Size part of font face name
   int i = 0;
   for (; GCharacter::IsDigit(fullFaceName[i]); i++)
      strSize += fullFaceName[i];

   int pointSize = 12;
   if (strSize != "")
      pointSize = GInteger::ParseInt(strSize);
   fd.fxPointSize = MAKEFIXED(pointSize, 0);

   if (fullFaceName[i] == '.')
      i++; // Skip the dot character

   // ---
   // Fetch BOLD, ITALIC, etc. from the font face name.

   GArray<FontAttr> attrs(5);
   if (attrs.getCount() <= 0)
   {
      attrs.add(new FontAttr(".Bold", FATTR_SEL_BOLD));
      attrs.add(new FontAttr(".Italic", FATTR_SEL_ITALIC));
      attrs.add(new FontAttr(".Underscore", FATTR_SEL_UNDERSCORE));
      attrs.add(new FontAttr(".Outline", FATTR_SEL_OUTLINE));
      attrs.add(new FontAttr(".Strikeout", FATTR_SEL_STRIKEOUT));
   }

   GString fontName = fullFaceName.substring(i); // Get face name without size and dot.
   for (int ii=0; ii<attrs.getCount(); ii++)
   {
      if (fontName.endsWith(attrs[ii].name, true)) // Ignore case in test.
      {
         fd.fAttrs.fsSelection |= USHORT(attrs[ii].flag);
         int l = attrs[ii].name.length();
         fontName.cutTailFrom(fontName.length() - l);
      }
   }

   // ---
   ::WinFontDlg(HWND_DESKTOP, parentWin->getHWND(), &fd);

   if (fd.lReturn != DID_OK)
      return GString::Empty;

   fullFaceName = GInteger::ToString(FIXEDINT(fd.fxPointSize));
   fullFaceName += ".";
   fullFaceName += fd.fAttrs.szFacename;

   for (int i3=0; i3<attrs.getCount(); i3++)
      if (fd.flStyle & attrs[i3].flag)
         fullFaceName += attrs[i3].name;

   if (setFont)
      setFontNameSize(fullFaceName);

   return fullFaceName;
}

/**
 * Get a reference to the full window name/id.
 *
 * The returned string is a dot-separated list of the name
 * of the whole parent window chain of the window, including the
 * name of the window it self.
 *
 * For instance, if "this" window C is a child of window B, which
 * in turn is a child of window A, then this method will
 * return "A.B.C".
 *
 * This method is typically to be used by applications for
 * debugging purposes only. But it can also be used to get the default
 * profile section name of the window.
 *
 * @see #getDebugName
 * @see #getFullName
 */
const GString& GWindow::getFullName () const
{
   if (fullName == "")
   {
      const GWindow* win = this;
      while (win != null)
      {
         const GString& winName = win->getName();
         if (winName == GWindow::DESKTOP_WINDOW_NAME)
            break;
         if (fullName != "")
            fullName.insert('.');
         fullName.insert(winName);
         win = win->getParentWindow();
      }
   }

   return fullName;
}

/**
 * Get a string that can be used to identify the window in debug
 * logging, etc. The returned string is to be used for debugging
 * purposes only.
 *
 * @see #getName
 * @see #getDebugInfoString
 */
GString GWindow::getDebugName () const
{
   const GString& fname = getFullName();
   GString ret(fname.length() + 64);
   ret = "<Class=";
   // ret += winClass.getClassName();
   ret += typeid(*this).name();
   ret += ", Name=";
   if (name == "")
      ret += "?";
   else
      ret += name;
   ret += ", FullName=";
   ret += fname;
   ret += ">";
   return ret;
}

/**
 * Return a string representation of the window, for debugging
 * purposes only.
 *
 * @see #getDebugName
 */
GString GWindow::getDebugInfoString () const
{
   GString layoutName(32);
   if (layoutManager == null)
      layoutName = "<null>";
   else
      layoutName = typeid(*layoutManager).name();
   GString str(128);
   str += "Class=" + winClass.getClassName() + ", ";
   str += "Name=" + getName() + ", ";
   str += "Pos=" + getWindowPos().toString() + ", ";
   str += "Size=" + getWindowSize().toString() + ", ";
   str += "Insets=" + insets.toString() + ", ";
   str += "Constraints=" + layoutConstraints + ", ";
   str += "Font=" + getFontNameSize() + ", ";
   str += "BckColor=" + getBackgroundColor().toString() + ", ";
   str += "FrgColor=" + getForegroundColor().toString() + ", ";
   str += "Visible=" + GBoolean::ToString(isVisible()) + ", ";
   str += "Layout=" + layoutName;
   return str;
}

/**
 * Get a reference to the system dependent desktop window, which is 
 * typically used as an argument to the "parentWin" parameter of one of 
 * the constructors of this GWindow class.
 *
 * Mark that the returned object is constructed just as a wrapper for 
 * the system dependent HWND of the desktop window. The window will not 
 * hook into the system by any means. E.g. we will not subclass the 
 * desktop window and will therefore not get any messages from it. 
 * That is; the {@link #handleWindowMessage} method of the system 
 * window object will never get called automatically.
 *
 * @author  Leif Erik Larsen
 * @since   2003.11.10
 */
GWindow& GWindow::GetSysDesktopWindow ()
{
   if (SysDesktopWindow == null)
      SysDesktopWindow = new GWindow(HWND_DESKTOP, true);
   return *SysDesktopWindow;
}

/**
 * Get the window face area rectangle, excluding insets (if any).
 * This method is the same as {@link #getWindowRect}, except the 
 * insets (as returned by {@link #getInsets}) is excluded from the 
 * returned rectangle.
 *
 * This method should be used to get the painting rectangle area 
 * of the window.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.18
 */
GRectangle GWindow::getWindowFaceRect () const
{
   const GInsets& ins = getInsets();
   GRectangle r = getWindowRect();
   r.x += ins.left;
   r.width -= ins.left + ins.right;
   if (isOS2Y())
      r.y += ins.bottom;
   else
      r.y += ins.top;
   r.height -= ins.bottom + ins.top;
   return r;
}

GRectangle GWindow::getWindowRect () const
{
   if (hWnd == null)
      return GRectangle();

   RECTL r;
   ::WinQueryWindowRect(hWnd, &r);
   return GWindow::MakeLibRect(r, *this);
}

/**
 * Get the current size of the window, in pixels.
 */
GDimension GWindow::getWindowSize () const
{
   if (hWnd == null)
      return GDimension();
   RECTL r;
   ::WinQueryWindowRect(hWnd, &r);
   int w = r.xRight;
   int h = r.yTop;
   return GDimension(w, h);
}

/**
 * Scroll the content of the window horizontally and/or vertically
 * with as many pixels as specified.
 *
 * @author  Leif Erik Larsen
 * @since   2001.11.30
 * @param   hscroll     Numbers of pixels to scroll horizontally.
 *                      If negative then the content will be scrolled
 *                      to the left, or else scroll to the right.
 * @param   vscroll     Numbers of pixels to scroll vertically.
 *                      If negative then the content will be scrolled
 *                      down, or else scroll up.
 * @param   invalidate  True if the invalid region created as a result
 *                      of the scroll should be added to the update
 *                      regions of those windows affected. This may
 *                      result in sending WM_PAINT messages to
 *                      CS_SYNCPAINT windows before the call returns.
 * @param   scrollRect  The rectangle of which window area to scroll.
 *                      If null is specified then the whole client area
 *                      of the window will be scrolled.
 * @param   clipRect    Rectangle area that should not be affected by
 *                      the scroll, even if it overlaps all or part of
 *                      the rectangle specified by <i>scrollRect</i>.
 */
void GWindow::scrollWindow ( int hscroll,
                             int vscroll,
                             bool invalidate,
                             const GRectangle* scrollRect,
                             const GRectangle* clipRect )
{
   RECTL sr, cr;
   if (scrollRect == null)
      ::WinQueryWindowRect(hWnd, &sr);
   else
      sr = MakeSysRect(*scrollRect, *this, false);
   if (clipRect == null)
      cr = sr;
   else
      cr = MakeSysRect(*clipRect, *this, false);
   if (!os2y)
      vscroll = -vscroll;
   int options = (invalidate ? SW_INVALIDATERGN : 0);
   ::WinScrollWindow(hWnd, hscroll, vscroll, &sr, &cr, 0, 0, options);
}

/**
 * Get a reference to the insets of the window.
 *
 * The insets defines the frame free space area of the window.
 * It is to contain no child windows, and it can for instance
 * be used as the space of where to paint something like a border.
 *
 * It is up to the layout manager to respect the insets, however.
 *
 * @see #setInsets
 */
const GInsets& GWindow::getInsets () const
{
   return insets;
}

/**
 * Get a reference to the constraints string that can be used
 * by layout managers to identify where to place the window within
 * its layout scheme.
 *
 * @see #setLayoutConstraints
 */
const GString& GWindow::getLayoutConstraints () const
{
   return layoutConstraints;
}

/**
 * Get a pointer to the owner window of the window.
 *
 * If the window has no owner then we will return null.
 */
GWindow* GWindow::getOwnerWindow () const
{
   return ownerWin;
}

/**
 * Get a pointer to the parent window of the window.
 *
 * If the window has no parent then we will return null.
 */
GWindow* GWindow::getParentWindow () const
{
   return parentWin;
}

/**
 * Get a pointer to the current popup menu, or null if the window
 * has no popup menu.
 *
 * @see #setPopupMenu
 */
GMenu* GWindow::getPopupMenu ()
{
   return popupMenu;
}

/**
 * Get the current optional program defined user data object
 * of the window, or null if the window has no such user data.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.04
 * @see     #setUserData
 */
GObject* GWindow::getUserData () const
{
   return userData;
}

bool GWindow::isCaretOn () const
{
   return caretIsOn;
}

/**
 * Root panels should override this method is order to return
 * true. False is returned if the window has a parent that contains
 * focusable child window(s).
 */
bool GWindow::isFocusCycleRoot () const
{
   return parentWin == null;
}

bool GWindow::isMouseCapture () const
{
   return mouseIsCaptured;
}

/**
 * Test if this window is focusable or not.
 * Usually only non-static components are to be focusable.
 * The default is that every window is not focusable, if not 
 * set otherwise.
 *
 * This method is usually to be used by the focus manager only.
 *
 * @see    #setFocusable
 */
bool GWindow::isFocusable () const
{
   return focusable;
}

/**
 * Set whether or not this windows is to be focusable.
 *
 * @author  Leif Erik Larsen
 * @since   2004.07.28
 * @see     isFocusable
 */
void GWindow::setFocusable ( bool b )
{
   focusable = b;
}

/**
 * Test if the window is oily or not. An oily window is a window that
 * is so slippery that it will not catch the cursor when pressing the
 * <b><i>up</i></b> or <b><i>down</i></b> key in dialogs. To activate
 * it, the user must either press <b><i>tab</i></b> or he must click
 * on it with the mouse.
 *
 * @return  True if the window is oily, or else false.
 * @see     #setOily
 */
bool GWindow::isOily () const
{
   return oily;
}

/**
 * Return true if this window uses OS/2'ish y-coordinates.
 *
 * @author  Leif Erik Larsen
 * @since   2003.09.29
 * @see     #WS2_OS2Y
 */
bool GWindow::isOS2Y () const
{
   return os2y;
}

const GString& GWindow::getClassName () const
{
   return winClass.getClassName();
}

/**
 * Returns the system dependent handle of the window.
 */
HWND GWindow::getHWND () const
{
   return hWnd;
}

GWindowEntry* GWindow::getWindowEntry () const
{
   return winentry;
}

void GWindow::setDefaultMsgProc ( WindowProc msgProc )
{
   defaultMsgProc = msgProc;
}

/**
 * To be called by constructor of subclass if <i>postponeWinCrea</i>
 * was true.
 */
void GWindow::setHWND ( HWND hWin )
{
   if (this->hWnd != null)
   {
       // TODO: What about the old entry in GWindow::WindowsMap? Should we remove it?
   }
   this->hWnd = hWin;
   GInteger* key = new GInteger(int(hWin));
   WindowsMap.put(key, this, true, false);
}

/**
 * This is a special method that is to be used in conjunction with
 * the command table macros only.
 *
 * @return True if the command was actually executed, or else false.
 */
bool GWindow::__executeAbstractCommand ( GAbstractCommand* /*cmd*/ )
{
   return false;
}

int GWindow::getCaretHeight () const
{
   return caretHeight;
}

int GWindow::getCaretWidth () const
{
   return caretWidth;
}

int GWindow::getCaretXPos () const
{
   return caretXPos;
}

int GWindow::getCaretYPos () const
{
   return caretYPos;
}

/**
 * Get the number of child windows that are contained by "this"
 * window object.
 *
 * @see #getChildWindow
 */
int GWindow::getChildWindowCount () const
{
   return childrenByIndex.getCount();
}

/**
 * @see #getTextLength
 */
GString GWindow::getText () const
{
   return text;
}

/**
 * Set the common default leader style for all windows of which
 * has a leader style of {@link GWindow#LeaderStyle_DEFAULT}.
 *
 * @see #setLeaderStyle
 */
void GWindow::SetCommonLeaderStyle ( GWindow::LeaderStyle style )
{
   GWindow::CommonLeaderStyle = style;
}

/**
 * Get the position of the window, with respect to parent window.
 */
GPoint GWindow::getWindowPos () const
{
   SWP wp;
   ::WinQueryWindowPos(hWnd, &wp);
   GPoint ret(wp.x, wp.y);
   return ret;
}

/**
 * Get the position of the window, with respect to the physical screen.
 *
 * @author  Leif Erik Larsen
 * @see     #setWindowPosOnScreen
 */
GPoint GWindow::getWindowPosOnScreen () const
{
   GPoint pos;
   SWP wp;
   HWND hw = hWnd;
   while (hw != null)
   {
      ::WinQueryWindowPos(hw, &wp);
      pos.x += wp.x;
      pos.y += wp.y;
      hw = ::WinQueryWindow(hw, QW_PARENT);
   }
   return pos;
}

bool GWindow::isMaximized () const
{
   SWP swp;
   WinQueryWindowPos(getHWND(), &swp);
   bool ret = ((swp.fl & SWP_MAXIMIZE) != 0);
   return ret;
}

bool GWindow::isMinimized () const
{
   SWP swp;
   WinQueryWindowPos(getHWND(), &swp);
   bool ret = ((swp.fl & SWP_MINIMIZE) != 0);
   return ret;
}

/**
 * Maximize or de-maximize (restore) the window.
 * @see #setMaximized
 */
void GWindow::setMaximized ( bool flag )
{
   HWND hw = getHWND();
   WinSetWindowPos(hw, null, 0, 0, 0, 0, flag ? SWP_MAXIMIZE : SWP_RESTORE);
}

/**
 * Minimize or de-minimize (restore) the window.
 * @see #setMaximized
 */
void GWindow::setMinimized ( bool flag )
{
   HWND hw = getHWND();
   WinSetWindowPos(hw, null, 0, 0, 0, 0, flag ? SWP_MINIMIZE : SWP_RESTORE);
}

/**
 * Request the keyboard focus to go to this window, but in an 
 * asynchronous way. That is, the focus change will not take place 
 * until the calling thread (if it is the GUI-thread) has returned 
 * out of its current message handler, or (if the calling thread is 
 * not the GUI-thread) until the GUI-thread is idle.
 *
 * This method does nothing but post a {@link #GM_ASYNC_GRAB_FOCUS}
 * message to the window.
 *
 * This method can be called by any thread, not just the GUI-thread!
 *
 * @author  Leif Erik Larsen
 * @see     2004.10.27
 * @see     #grabFocus
 */
void GWindow::grabFocusAsynchronously ()
{
   postMessage(GM_ASYNC_GRAB_FOCUS);
}

/**
 * Invalidate all parts of the window client area so that it will 
 * be repainted as soon as possible. For most windows this will 
 * cause the client area to be repainted asynchronosuly, but for 
 * some windows (depending on their class and properties) on some 
 * systems the repainting might also be synchronous.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.05.06
 * @param   inclChildren  True if we shall also invalidate all the 
 *                        child windows of ours (if any). If this 
 *                        parameter is true then we will request 
 *                        all children to be repainted recursively.
 *                        That is; all descendants of ours.
 * @see     #invalidateRect
 * @see     #validateAll
 */
void GWindow::invalidateAll ( bool inclChildren ) const
{
   if (hWnd == null)
      return;
   ::WinInvalidateRect(hWnd, null, inclChildren);
}

/**
 * Validate all parts of the window client area so that no parts 
 * of it will wait to be repainted by now.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.10.14
 * @param   inclChildren  True if we shall also validate all the 
 *                        child windows of ours (if any) recursively.
 *                        That is; all descendants of ours.
 * @see     #validateRect
 * @see     #invalidateAll
 */
void GWindow::validateAll ( bool inclChildren ) const
{
   if (hWnd == null)
      return;
   ::WinValidateRect(hWnd, null, inclChildren);
}

/**
 * Same as {@link #invalidateAll} but only for the specified 
 * rectangle (if not null), and not for any child windows.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.03.21
 * @param   rect    The rectangle of which are to invalidate.
 * @see     #invalidateAll
 * @see     #validateRect
 */
void GWindow::invalidateRect ( const GRectangle& rect ) const
{
   if (hWnd == null)
      return;
   RECTL r = MakeSysRect(rect, *this, false);
   ::WinInvalidateRect(hWnd, &r, false);
}

/**
 * Same as {@link #validateAll} but only for the specified 
 * rectangle (if not null), and not for any child windows.
 *
 * @author  Leif Erik Larsen
 * @since   2004.03.21
 * @param   rect    The rectangle of which are to invalidate.
 * @see     #validateAll
 * @see     #invalidateRect
 */
void GWindow::validateRect ( const GRectangle& rect ) const
{
   if (hWnd == null)
      return;
   RECTL r = MakeSysRect(rect, *this, true);
   ::WinValidateRect(hWnd, &r, false);
}

void GWindow::updateWindow ()
{
   if (hWnd == null)
      return;
   ::WinUpdateWindow(hWnd);
}

/**
 * Set the window position, with respect to its parent window.
 *
 * @param  xpos        New horizontal position, in pixels.
 * @param  ypos        New vertical position, in pixels.
 * @see    #setWindowSize
 * @see    #setWindowBounds
 */
void GWindow::setWindowPos ( int xpos, int ypos )
{
   if (hWnd == null)
      return;
   GWindow* parent = getParentWindow();
   if (parent != null && !parent->os2y)
      ypos = parent->getWindowSize().height - ypos - getWindowSize().height;
   ::WinSetWindowPos(hWnd, 0, xpos, ypos, 0, 0, SWP_MOVE);
}

/**
 * Set the window position, with respect to the physical screen.
 *
 * @param  xpos        New horizontal position, in pixels.
 * @param  ypos        New vertical position, in pixels.
 * @param  autoAdjust  True if we shall automatically adjust the
 *                     specified position so that the window will not be
 *                     clipped outside of the screen area.
 * @see    #getWindowPosOnScreen
 */
void GWindow::setWindowPosOnScreen ( int xpos, int ypos, bool autoAdjust )
{
   if (hWnd == null)
      return;

   if (autoAdjust)
   {
      GDimension screenSize = GSystem::GetScreenSize();
      GDimension winSize = getWindowSize();
      xpos = GMath::Clip(xpos, 0, screenSize.width - winSize.width);
      ypos = GMath::Clip(ypos, 0, screenSize.height - winSize.height);
   }

   GWindow* parent = getParentWindow();
   if (parent != null)
   {
      GPoint parentPos = parent->getWindowPosOnScreen();
      xpos -= parentPos.x;
      ypos -= parentPos.y;
   }
   ::WinSetWindowPos(hWnd, 0, xpos, ypos, 0, 0, SWP_MOVE);
}

/**
 * Set the size of the window, in pixels.
 *
 * @param  width       New width, in pixels.
 * @param  height      New height, in pixels.
 * @see    #setWindowPos
 * @see    #setWindowBounds
 */
void GWindow::setWindowSize ( int width, int height )
{
   if (hWnd == null)
      return;

   ::WinSetWindowPos(hWnd, 0, 0, 0, width, height, SWP_SIZE);
}

/**
 * Set the window position and size.
 * The position is with respect to its parent window.
 *
 * @param  xpos        New horizontal position, in pixels.
 * @param  ypos        New vertical position, in pixels.
 * @param  width       New width, in pixels.
 * @param  height      New height, in pixels.
 * @see    #setWindowSize
 * @see    #setWindowPos
 */
void GWindow::setWindowBounds ( int xpos, int ypos, int width, int height )
{
   if (hWnd == null)
      return;

   GWindow* parent = getParentWindow();

   if (parent == null && !os2y)
      ypos = GSystem::GetScreenSize().height - ypos - height;
   else
   if (parent != null && !parent->os2y)
      ypos = parent->getWindowSize().height - ypos - height;
   ::WinSetWindowPos(hWnd, 0, xpos, ypos, width, height, SWP_MOVE | SWP_SIZE);
}

/**
 * Set the insets of the window.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.26
 * @param   ins  The new insets. Can be null.
 * @param   autoDelete True if we shall automatically delete the 
 *               specified insets object when we are finished with it.
 * @see     #getInsets
 * @see     #setBorder
 */
void GWindow::setInsets ( GInsets* ins, bool autoDelete )
{
   GInsets newIns(0, 0, 0, 0);
   if (ins != null)
      newIns = *ins;
   if (newIns != insets)
   {
      insets = newIns;
      layout();
   }
   if (autoDelete)
      delete ins;
}

/**
 * Show a modal message box with the specified properties and wait
 * for the response from the user, using this window as its
 * parent window.
 *
 * @author  Leif Erik Larsen
 * @since   2000.09.28
 * @param   msg       The main message string. Can contain linefeed
 *                    characters. This text can also be a reference
 *                    to an externally defined text resource (using
 *                    the percent prefix character) and the text will
 *                    be automatically loaded for the native language.
 * @param   type      The type of the message. This flag is used to
 *                    define the default properties of the message box,
 *                    such as its icon and title string, in case they
 *                    are not defined in the <i>flags</i> parameter.
 * @param   flags     A string defining the properties of the message 
 *                    box. See {@link GMessageBox#GMessageBox}
 *                    for a list of supported flag characters.
 * @param   title     The text to show in the title bar of the message
 *                    box. Can be empty, and in that case we will use
 *                    a default title text with respect to the 
 *                    specified message <i>type</i>.
 * @param   monoFont  True to use monospace font to draw the message
 *                    text, or else false. Should usually be false.
 * @param   userText1 The text of the first user defined button. This 
 *                    text is used only if the <i>flags</i> contains 
 *                    the tag for the first user defined button.
 * @param   userText2 The text of the second user defined button. This 
 *                    text is used only if the <i>flags</i> contains 
 *                    the tag for the second user defined button.
 * @param   userText3 The text of the third user defined button. This 
 *                    text is used only if the <i>flags</i> contains 
 *                    the tag for the third user defined button.
 * @return  See {@link GMessageBox#show} for a list of possible 
 *          return values.
 * @see     GMessageBox
 * @see     GProgram#showMessageBox
 */
GMessageBox::Answer GWindow::showMessageBox ( const GString& msg, 
                                              GMessageBox::Type type, 
                                              const GString& flags, 
                                              const GString& title, 
                                              bool monoFont,
                                              const GString& userText1,
                                              const GString& userText2,
                                              const GString& userText3 )
{
   GMessageBox mbox(type, msg, title, flags, monoFont, userText1, userText2, userText3);
   GMessageBox::Answer answ = mbox.show(this);
   return answ;
}

/**
 * Set the border of which to automatically paint on the window
 * insets area, just before calling {@link #onPaintBackground} and 
 * before {@link #onPaint}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.18
 * @param   border     The new border of which to activate, 
 *                     or null to remove the current border.
 * @param   autoDelete True if we shall automatically delete 
 *                     the specified border when the window 
 *                     is destroyed.
 * @see     #setInsets
 */ 
void GWindow::setBorder ( GBorder* border, bool autoDelete )
{
   if (this->border != null)
      if (this->deleteBorder)
         delete this->border;
   this->border = border;
   this->deleteBorder = autoDelete;
}

/**
 * This method can be overridden in order to layout the child window
 * panels as is preferred by the sub-class.
 *
 * It is automatically called by the default implementation of
 * <i>onSize()</i>.
 *
 * The default implementation of <i>layout()</i> does nothing but
 * return, except if a layout manager object has been set on the
 * window object. If a layout manager has been set, then we will
 * call the <i>layoutWindow()</i> method on that object.
 *
 * To request an asynchronous call to this method, 
 * use {@link #postLayout}.
 *
 * @see #setLayoutManager
 * @see GLayoutManager#layoutWindow
 * @see #onSize
 * @see #postLayout
 */
void GWindow::layout ()
{
   invalidateAll(true);
   if (layoutManager != null)
      layoutManager->layoutWindow(*this);
}

/**
 * Get the layout manager currently used by "this" window.
 *
 * If null is specified then we will deactivate the current
 * layout manager, if any. In that case we will also delete the
 * current layout manager object, if needed.
 *
 * @author  Leif Erik Larsen
 * @since   2006.04.22
 * @return  The current layout manager for this window, or null if the
 *          window currently has no layout manager.
 * @see     #setLayoutManager
 */
GLayoutManager* GWindow::getLayoutManager ()
{
   return layoutManager;
}

/**
 * Set the layout manager to be used by "this" window panel.
 *
 * If null is specified then we will deactivate the current
 * layout manager, if any. In that case we will also delete the
 * current layout manager object, if needed.
 *
 * @param lm         A pointer to the layout manager of which to use.
 * @param autoDelete True if we shall automatically delete the specified
 *                   layout manager object upon window object
 *                   destruction.
 * @see   #getLayoutManager
 * @see   #layout
 * @see   #layoutManager
 */
void GWindow::setLayoutManager ( GLayoutManager* lm, bool autoDelete )
{
   if (layoutManager != null)
   {
      // Deactivate the previous layout manager. Delete it if needed.
      if (autoDeleteLayoutManager)
         delete layoutManager;
      layoutManager = null;
   }

   layoutManager = lm;
   autoDeleteLayoutManager = autoDelete;
}

/**
 * Set the constraints string that can be used by layout managers
 * to identify where to place the window within its layout scheme.
 *
 * @param lc       The new layout constraints name.
 * @param doLayout True if we shall call <i>layout()</i> (on our parent
 *                 window), or else false.
 * @see   #getLayoutConstraints
 */
void GWindow::setLayoutConstraints ( const GString& lc, bool doLayout )
{
   layoutConstraints = lc;
   if (doLayout && parentWin != null)
      parentWin->layout();
}

/**
 * Set the preferred size of the window, in pixels.
 *
 * @see #getPreferredSize
 */
void GWindow::setPreferredSize ( const GDimension* size, bool autoDelete )
{
   // Delete the old "preferred size" (if any).
   if (autoDeletePreferredSize)
      delete preferredSize;

   // Set the new "preferred size" (can be null).
   preferredSize = size;
   autoDeletePreferredSize = autoDelete;
}

/**
 * Set the maximum size of the window, in pixels.
 * @see #getMaximumSize
 */
void GWindow::setMaximumSize ( const GDimension& size )
{
   this->maximumSize = new GDimension(size);
}

/**
 * Set the minimum size of the window, in pixels.
 *
 * @see #getMinimumSize
 */
void GWindow::setMinimumSize ( const GDimension& size )
{
   this->minimumSize = new GDimension(size);
}

/**
 * Get the preferred size of the window, in pixels.
 *
 * The default implementation will call <i>getPreferredWidth()</i>
 * and <i>getPreferredHeight()</i> to get the values of of which to
 * return as the preferred dimension.
 *
 * @see #setPreferredSize
 * @see #getPreferredWidth
 * @see #getPreferredHeight
 */
GDimension GWindow::getPreferredSize () const
{
   int width = getPreferredWidth();
   int height = getPreferredHeight();
   return GDimension(width, height);
}

/**
 * Get the maximum size of the window, in pixels.
 *
 * The default implementation will call <i>getMaximumWidth()</i>
 * and <i>getMaximumHeight()</i> to get the values of of which to
 * return as the maximum dimension.
 *
 * @see #setMaximumSize
 * @see #getMaximumWidth
 * @see #getMaximumHeight
 */
GDimension GWindow::getMaximumSize () const
{
   int width = getMaximumWidth();
   int height = getMaximumHeight();
   return GDimension(width, height);
}

/**
 * Get the minimum size of the window, in pixels.
 *
 * The default implementation will call <i>getMinimumWidth()</i>
 * and <i>getMinimumHeight()</i> to get the values of of which to
 * return as the minimum dimension.
 *
 * @see #setMinimumSize
 * @see #getMinimumWidth
 * @see #getMinimumHeight
 */
GDimension GWindow::getMinimumSize () const
{
   int width = getMinimumWidth();
   int height = getMinimumHeight();
   return GDimension(width, height);
}

/**
 * Get the preferred width of the window, in pixels.
 *
 * @see #getPreferredSize
 * @see #getPreferredHeight
 */
int GWindow::getPreferredWidth () const
{
   int ret;
   if (preferredSize != null)
      ret = preferredSize->width;
   else
   if (layoutManager != null)
      ret = layoutManager->preferredLayoutWidth(const_cast<GWindow&>(*this));
   else
      ret = 250;
   return ret + insets.left + insets.right;
}

/**
 * Get the maximum width of the window, in pixels.
 *
 * @see #getMaximumSize
 * @see #getMaximumHeight
 */
int GWindow::getMaximumWidth () const
{
   double ret;

   if (maximumSize != null)
      ret = maximumSize->width;
   else
      ret = GInteger::MAX_VALUE;
   ret += insets.left + insets.right;
   if (ret > GInteger::MAX_VALUE)
      ret = GInteger::MAX_VALUE;

   return (int) ret;
}

/**
 * Get the minimum width of the window, in pixels.
 *
 * @see #getMinimumSize
 * @see #getMinimumHeight
 */
int GWindow::getMinimumWidth () const
{
   int ret;
   if (minimumSize != null)
      ret = minimumSize->width;
   else
      ret = 0;
   return ret + insets.left + insets.right;
}

/**
 * Get the preferred height of the window, in pixels.
 *
 * @see #getPreferredSize
 * @see #getPreferredWidth
 */
int GWindow::getPreferredHeight () const
{
   int ret;
   if (preferredSize != null)
      ret = preferredSize->height;
   else
   if (layoutManager != null)
      ret = layoutManager->preferredLayoutHeight(const_cast<GWindow&>(*this));
   else
      ret = 25;
   return ret + insets.top + insets.bottom;
}

/**
 * Get the maximum height of the window, in pixels.
 *
 * @see #getMaximumSize
 * @see #getMaximumWidth
 */
int GWindow::getMaximumHeight () const
{
   double ret;
   if (maximumSize != null)
      ret = maximumSize->height;
   else
      ret = GInteger::MAX_VALUE;
   ret += insets.top + insets.bottom;
   if (ret > GInteger::MAX_VALUE)
      ret = GInteger::MAX_VALUE;
   return (int) ret;
}

/**
 * Get the minimum height of the window, in pixels.
 *
 * @see #getMinimumSize
 * @see #getMinimumWidth
 */
int GWindow::getMinimumHeight () const
{
   int ret;
   if (minimumSize != null)
      ret = minimumSize->height;
   else
      ret = 0;
   return ret + insets.top + insets.bottom;
}

/**
 * Request the keyboard focus to go to this window.
 *
 * If the window is not part of the active window (usually a frame)
 * then we will not actually grab the focus except of the <i>force</i>
 * parameter is specified to be true. If it is false and the window
 * is not part of the active window then we will instead just set an
 * internal flag on the top level window (usually a frame) telling
 * it that we want to be focused as soon as the top level window
 * is activated.
 *
 * This method must be called by the GUI-thread!
 *
 * @author  Leif Erik Larsen
 * @see     2001.02.27
 * @see     #setActive
 * @see     #grabFocusAsynchronously
 */
void GWindow::grabFocus ( bool force )
{
   // TODO: What about the specified "force"-flag?

   ::WinSetFocus(HWND_DESKTOP, hWnd);
}

/**
 * Set the optional program defined user data object of the window.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.04
 * @param   userData    The user data object.
 * @param   autoDelete  True if we should have the ownership of the
 *                      specified object and should automatically
 *                      delete it upon destruction of the window
 *                      object. Or else false.
 * @see     #getUserData
 */
void GWindow::setUserData ( GObject* userData, bool autoDelete )
{
   this->userData = userData;
   this->autoDeleteUserData = autoDelete;
}

void GWindow::setVisible ( bool flag )
{
   ::WinShowWindow(hWnd, flag);
   if (!flag)
   {
      // Make sure to release the mouse in case it is currently captured by a
      // window that is part of this now hidden window. I think OS/2 should 
      // have done this automatically, but it doesn't. I have seen a few 
      // situations when this could cause a WPS "hang", and this is a
      // dirty fix for such potential problems.
      HWND hw = ::WinQueryCapture(HWND_DESKTOP);
      if (isParentOf(hw))
         ::WinSetCapture(HWND_DESKTOP, null);
   }
}

bool GWindow::isVisible () const
{
   // Don't use WinIsWindowVisible() because that function will
   // return false also if window size is [0,0] even if the
   // WS_VISIBLE style flag is set.
   ULONG style = ::WinQueryWindowULong(hWnd, QWL_STYLE);
   bool ret = ((style & WS_VISIBLE) != 0);
   return ret;
}

/**
 * Returns true if and only if this window currently has the keyboard
 * input focus, from the system point of view. This method typically 
 * returns true only if the window currently owns the system 
 * dependent caret (e.g. the blinking caret in a text entry window).
 * 
 * @author  Leif Erik Larsen
 * @since   2004.09.06
 * @see     #containsFocusedWindow
 */
bool GWindow::hasFocus () const
{
   bool ret = ::WinQueryFocus(HWND_DESKTOP) == getHWND();
   return ret;
}

/**
 * Returns true if the currently focused window is a descendant of 
 * this window, or if it is this window it self.
 * 
 * @author  Leif Erik Larsen
 * @since   2001.03.04
 * @see     #hasFocus
 */
bool GWindow::containsFocusedWindow () const
{
   HWND hw = ::WinQueryFocus(HWND_DESKTOP);
   if (hw == hWnd || isParentOf(hw))
      return true;
   else
      return false;
}

/**
 * Get the length of the text string that would have been
 * returned if {@link #getText} was called.
 *
 * This method will usually be more memory and speed optimizing
 * to use instead of first calling {@link #getText} and then
 * fetch the length of the returned string.
 */
int GWindow::getTextLength () const
{
   return text.length(); // WinQueryWindowTextLength(getHWND());
}

void GWindow::setText ( const GString& t )
{
   text = GProgram::LoadText(t, GResourceTable::LT_PREFER_NONE, 0);
   ::WinSetWindowText(hWnd, text);
   invalidateAll(false);
}

/**
 * Set the foreground color of the window.
 */
void GWindow::setForegroundColor ( const GColor& color )
{
   int c = color;
   HWND hw = getHWND();
   ::WinSetPresParam(hw, PP_FOREGROUNDCOLOR, sizeof(c), &c);
}

/**
 * Get a copy of the current foreground color of the window.
 *
 * If a foreground color has not been explicitly set on this window
 * then we will return the foreground color of our parent window or 
 * a default in case this window has no parent.
 */
GColor GWindow::getForegroundColor () const
{
   int c = 0;
   HWND hw = getHWND();
   ::WinQueryPresParam(hw, PP_FOREGROUNDCOLOR, 0, null, sizeof(c), &c, QPF_PURERGBCOLOR);
   return GColor(c);
}

void GWindow::setBackgroundColor ( const GColor& color )
{
   int c = color;
   HWND hw = getHWND();
   ::WinSetPresParam(hw, PP_BACKGROUNDCOLOR, sizeof(c), &c);
}

/**
 * Get a copy of the current background color of the window.
 *
 * If a background color has not been explicitly set on this window
 * then we will return the background color of our parent window or 
 * a default in case this window has no parent.
 */
GColor GWindow::getBackgroundColor () const
{
   int c = 0;
   HWND hw = getHWND();
   ::WinQueryPresParam(hw, PP_BACKGROUNDCOLOR, 0, null, sizeof(c), &c, QPF_PURERGBCOLOR);
   return GColor(c);
}

/**
 * Set the font of which to be used by default on the window.
 */
void GWindow::setFontNameSize ( const GString& fontNameSize )
{
   if (fontNameSize == "")
      return;

   HWND hw = getHWND();
   ::WinSetPresParam(hw, PP_FONTNAMESIZE, fontNameSize.length()+1, PVOID(fontNameSize.cstring()));
}

/**
 * Get a copy of the current font face name and size of the window.
 *
 * The returned string has the following format:
 *
 * "12.Helv.Bold.Italic.Underscore.Strikeout"
 *
 * Where 12 is the point size of the font, Helv is the font face
 * name and the remaining substrings describes any optional font
 * attributes such as those in the above example.
 *
 * If a font name size has not been explicitly set on this window
 * then we will return the font name size of our parent window or 
 * a default in case this window has no parent.
 *
 * @author   Leif Erik Larsen
 */
GString GWindow::getFontNameSize () const
{
   char buff[128] = "";
   ::WinQueryPresParam(getHWND(), PP_FONTNAMESIZE, 0, null, sizeof(buff), buff, 0);
   return GString(buff);
}

/**
 * Get the width of the specified string in pixels, with respect
 * to the current window font.
 *
 * @see #getHeightOfString
 * @see #getStringDimension
 */
int GWindow::getWidthOfString ( const GString& str ) const
{
   GGraphics g(this);
   return g.getWidthOfString(str);
}

/**
 * Get the height (in screen device pixels) of the current
 * font of the window.
 */
int GWindow::getFontHeight () const
{
   GGraphics g(this);
   return g.getFontHeight();
}

/**
 * Get the width of the specified string in pixels, with respect
 * to the current window font.
 *
 * @see #getWidthOfString
 * @see #getStringDimension
 */
int GWindow::getHeightOfString ( const char* str ) const
{
   GGraphics g(this);
   return g.getHeightOfString(str);
}

/**
 * Get the width of the specified string in pixels, with respect
 * to the current window font.
 *
 * @see #getWidthOfString
 * @see #getStringDimension
 */
int GWindow::getHeightOfString ( const GString& str ) const
{
   GGraphics g(this);
   return g.getHeightOfString(str);
}

/**
 * Get the width of the specified string in pixels, with respect
 * to the current window font.
 *
 * @see #getHeightOfString
 * @see #getWidthOfString
 */
GDimension GWindow::getStringDimension ( const GString& str ) const
{
   GGraphics g(this);
   return g.getTextDim(str);
}

/**
 * Get the width of the specified string in pixels, with respect
 * to the current window font.
 *
 * @see #getHeightOfString
 * @see #getWidthOfString
 */
GDimension GWindow::getStringDimension ( const char* str ) const
{
   GGraphics g(this);
   return g.getTextDim(str);
}

/**
 * Set the popup menu of the window.
 *
 * @param  newMenu  The menu resource of which to form the new menu,
 *                  or null to remove the current menu (if any).
 * @param  useFancy True if we shall use fancy menues with icons, etc,
 *                  if any.
 * @see    #getPopupMenu
 */
void GWindow::setPopupMenu ( GMenuResource* newMenu, bool useFancy )
{
   // Delete the previous popup menu object (if any)
   if (popupMenu != null)
   {
      delete popupMenu;
      popupMenu = null;
   }

   if (newMenu != null)
      popupMenu = new GMenu("PopupMenu", GString::Empty, *this, newMenu, useFancy, false);
}

void GWindow::setPopupMenu ( const char* menuID, bool useFancy )
{
   GResourceTable& res = getResourceTable();
   GMenuResource* menuDef = res.getMenuResource(menuID);
   if (menuDef != null)
      setPopupMenu(menuDef, useFancy);
}

/**
 * Show or hide the popup menu of the window.
 *
 * This method is typically called to show or hide the context menu
 * of the window, upon mouse single click events.
 */
void GWindow::setVisiblePopupMenu ( bool flag, int xpos, int ypos )
{
   if (popupMenu != null)
   {
      if (flag)
         popupMenu->showPopupMenu(xpos, ypos);
      else
         popupMenu->setVisible(false);
   }
}

bool GWindow::isVisiblePopupMenu () const
{
   if (popupMenu != null)
      return popupMenu->isVisible();
   else
      return false;
}

/**
 * Set the enable/disable state of all menuitems that have an ID
 * that equals the specified command ID.
 */
void GWindow::setCommandEnableState ( const char* cmdID, bool state )
{
   if (popupMenu != null)
      popupMenu->setCommandEnableState(cmdID, state);
}

/**
 * Set the toggle state of all menuitems that have an ID that equals
 * the specified command ID.
 */
void GWindow::setCommandToggleState ( const char* cmdID, bool state )
{
   if (popupMenu != null)
      popupMenu->setCommandToggleState(cmdID, state);
}

/**
 * Set the Oily flag on the window.
 *
 * @see     #isOily
 */
void GWindow::setOily ( bool flag )
{
   this->oily = flag;
}

/**
 * Set the drag-n-drop handler for this window.
 *
 * @author  Leif Erik Larsen
 * @since   2007.01.07
 */
void GWindow::setDragDropHandler ( GDragDropHandler* handler, bool autoDelete )
{
   if (autoDeleteDragDropHandler)
      delete dragDropHandler;
   dragDropHandler = handler;
   autoDeleteDragDropHandler = autoDelete;
}

/**
 * Set the accelerator table that defines which keyboard events we
 * should automatically translate into command messages (GM_COMMAND).
 *
 * The specified accelerator table object must not be destroyed
 * before the window is destroyed, because we keeps a pointer
 * directly to it for as long as the window is living. This is in
 * order to save memory. We will not automatically destroy the
 * specified accelerator table upon window destruction either.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.24
 * @see     #setAccelTableImpl
 */
void GWindow::setAccelTable ( GAccelResource* newAccel )
{
   setAccelTableImpl(newAccel);
}

void GWindow::setAccelTable ( const GString& accelID )
{
   GResourceTable& res = getResourceTable();
   GAccelResource* mainAccel = res.getAcceleratorResource(accelID);
   setAccelTable(mainAccel);
}

/**
 * This method is automatically called by the overloaded method 
 * {@link #setAccelTable} and is intended for being overridden
 * by subclasses that need to perform some action whenever 
 * the accelerator table is changed.
 * <p>
 * Subclassses that overrides this method must always remember 
 * to call the super implementation first.
 *
 * @author  Leif Erik Larsen
 * @since   2003.11.10
 * @param   newAccel  The new accelerator table resource, or 
 *                    null if the accelerator table is switched off.
 */
void GWindow::setAccelTableImpl ( GAccelResource* newAccel )
{
   accelTable = newAccel;
}

/**
 * Get a reference to the accelerator table that should be used when
 * translating keyboard events into WM_COMMAND's for this window panel.
 *
 * If this window has no accelerator table then we will return the
 * accelerator table of our parent window. Thus, this method
 * will search recursively upward in the window tree until it finds
 * an accelerator table. If we have still not found an accelerator
 * table even when the top level window has been reached then
 * we will return null.
 *
 * @author  Leif Erik Larsen
 */
GAccelResource* GWindow::getAccelTable () const
{
   if (accelTable != null)
      return accelTable;
   else
   if (parentWin != null)
      return parentWin->getAccelTable();
   else
      return null;
}

/**
 * Get a reference to the accelerator key item for the
 * specified command.
 *
 * If this window has no accelerator table then we will search
 * recursively upward in the window tree until we find an accelerator
 * key item for the specified command. If we have still not found such
 * an item even when the top level window panel has been reached then
 * we will return null.
 *
 * @author  Leif Erik Larsen
 */
GAccelItem* GWindow::getAccelItemByCommandID ( const GString& commandID ) const
{
   if (accelTable != null)
   {
      GAccelItem *ai = accelTable->getAccelItemByCommandID(commandID);
      if (ai != null)
         return ai;
   }

   if (parentWin != null)
      return parentWin->getAccelItemByCommandID(commandID);

   return null;
}

/**
 * Get a reference to the accelerator key item for the specified key.
 * 
 * If this window has no accelerator table then we will search
 * recursively upward in the window tree until we find an accelerator
 * key item for the specified key. If we have still not found such
 * an item even when the top level window panel has been reached then
 * we will return null.
 *
 * @author  Leif Erik Larsen
 */
GAccelItem* GWindow::getAccelItemByKeyName ( const GString& keyName )
{
   if (accelTable != null)
   {
      GAccelItem *ai = accelTable->getAccelItemByKeyName(keyName);
      if (ai != null)
         return ai;
   }

   if (parentWin != null)
      return parentWin->getAccelItemByKeyName(keyName);

   return null;
}

/**
 * Activate the window so it contains the keyboard input focus.
 * This method is usually to be called on the top level frame 
 * window only, in order to request the application window to 
 * be activated.
 *
 * The default implementation will call this method recursively
 * on the parent window until the top level window is reached.
 * Then the top level window is attempted to be activated.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.09.14
 * @param   force   True if we shall force activation of the window,
 *                  even if the window is not part of the active 
 *                  process from the operating system point of view.
 *                  If false is specified then the window might be 
 *                  activated only if the application already owns 
 *                  the active window.
 * @see     #grabFocus
 */
void GWindow::setActive ( bool force )
{
   if (parentWin != null)
   {
      parentWin->setActive();
      return;
   }

   GWindow* modalDlg = getFirstOwnedModalDialog();
   if (modalDlg != null)
      return modalDlg->setActive(force);

   // Don't try to activate a disabled window!
   if (!isEnabled())
      return;

   // TODO: What about the "force" parameter in OS/2?
   GFrameWindow* fw = dynamic_cast<GFrameWindow*>(this);
   HWND hw = (fw != null ? fw->getFrame().getHWND() : hWnd);
   ::WinSetActiveWindow(HWND_DESKTOP, hw);
   // ::WinSetWindowPos(hw, NULL, 0, 0, 0, 0, SWP_ACTIVATE);
}

/**
 * This method was implemented in order to get a way to check if a 
 * dialog window is to be considered alive or not. This default 
 * implementation returns true as long as the window handle has not 
 * yet been destroyed (which is done by the window destructor).
 * 
 * The reason why we needed this method is to have an abstract way
 * for {@link #GenericWindowProc} to check if it should actually 
 * dispatch the GUI message or not. It is important for dialog box
 * windows not to dispatch such events while the dialog is about 
 * to be dismissed, in order to prevent various possible system 
 * dependent exceptions during dialog window destruction.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.01.24
 * @see     GDialogPanel#isAlive
 */
bool GWindow::isAlive () const
{
   return hWnd != null;
}

/**
 * Return true if and only if this is the currently active window,
 * from the underlying system point of view.
 *
 * <b>Mark</b> that this method will return true also if the window
 * is a descendant of the active window. Therefore, this method can
 * be used on non-frame windows, even if it is typically to be called
 * on frame windows only.
 */
bool GWindow::isActive () const
{
   // Invisible windows can never actually be "active".
   // This test is probably unnecessary on most platforms, 
   // but I have seen the need for it at least on OS/2 Warp. 
   // Anyway, this doesn't hurt.
   if (!isVisible())
      return false;

   // ---
   HWND activeWin = ::WinQueryActiveWindow(HWND_DESKTOP);
   GWindow* awin = GWindow::GetWindowFromHandle(activeWin);
   if (awin != null) // Should always be true.
      return isDescendant(*awin);

   // Should never reach here, but in case, use this dirty method.
   GWindow* self = const_cast<GWindow*>(this); // Throw away const.
   const GWindow& top = self->getTopLevelWindow();
   HWND topWin = top.getHWND();
   return activeWin == topWin;
}

/**
 * Check if the specified point lies within the window rectangle.
 *
 * @author  Leif Erik Larsen
 * @since   2003.11.10
 */
bool GWindow::isPointInWindowRect ( int xpos, int ypos ) const
{
   GRectangle r = getWindowRect();
   return r.isPointInRect(xpos, ypos);
}

/**
 * Send the specified message to the message handler of the window.
 * The message will be sent synchronously.
 *
 * @see   #handleWindowMessage
 * @see   #postMessage
 * @see   #sendUserMessage
 */
GWindowMessage::Answer GWindow::sendMessage ( int id,
                                              GWindowMessage::Param1 mp1,
                                              GWindowMessage::Param2 mp2 )
{
   return ::WinSendMsg(hWnd, id, mp1, mp2);
}

/**
 * Post the specified message to the window.
 *
 * @author  Leif Erik Larsen
 * @see     #isAnyWaitingMessage
 * @see     #sendMessage
 * @see     #postUserMessage
 */
void GWindow::postMessage ( int id,
                            GWindowMessage::Param1 mp1,
                            GWindowMessage::Param2 mp2 )
{
   ::WinPostMsg(hWnd, id, mp1, mp2);
}

/** 
 * Send a {@link #GM_USERMESSAGE} to this window, which can be handled 
 * by the subclass by overriding {@link #onUserMessage}.
 *
 * @author  Leif Erik Larsen
 * @since   2003.12.01
 * @return  True if the message was handled, or else false.
 * @see     #postUserMessage
 * @see     #sendMessage
 */
bool GWindow::sendUserMessage ( GUserMessage& msg )
{
   GWindowMessage::Param1 mp1 = GWindowMessage::Param1(&msg);
   GWindowMessage::Param2 mp2 = GWindowMessage::Param2(false); // Don't auto-delete!
   GWindowMessage wm(GM_USERMESSAGE, mp1, mp2);
   GWindowMessage::Answer answ = handleWindowMessage(wm);
   return answ != 0 ? true : false;
}

bool GWindow::sendUserMessage ( GObject* param1, GObject* param2 )
{
   GUserMessage msg(param1, param2);
   return sendUserMessage(msg);
}

bool GWindow::sendUserMessage ( const GString& str1, GObject* param2 )
{
   GUserMessage msg(str1, param2);
   return sendUserMessage(msg);
}

bool GWindow::sendUserMessage ( const GString& str1, const GString& str2 )
{
   GUserMessage msg(str1, str2);
   return sendUserMessage(msg);
}

/** 
 * Post a {@link #GM_USERMESSAGE} to this window, which can be handled 
 * by the subclass by overriding {@link #onUserMessage}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.27
 * @see     #sendUserMessage
 * @see     #postMessage
 */
void GWindow::postUserMessage ( GUserMessage* msg, bool autoDeleteMsg )
{
   GWindowMessage::Param1 mp1 = GWindowMessage::Param1(msg);
   GWindowMessage::Param2 mp2 = GWindowMessage::Param2(autoDeleteMsg);
   postMessage(GM_USERMESSAGE, mp1, mp2);
}

void GWindow::postUserMessage ( const GString& str1 )
{
   GUserMessage* msg = new GUserMessage(str1);
   postUserMessage(msg, true);
}

void GWindow::postUserMessage ( const GString& str1, GObject* param2, bool autoDelParam2 )
{
   GUserMessage* msg = new GUserMessage(str1, param2);
   msg->setAutoDeleteParam2(autoDelParam2);
   postUserMessage(msg, true);
}

void GWindow::postUserMessage ( const GString& str1, const GString& str2 )
{
   GUserMessage* msg = new GUserMessage(str1, str2);
   postUserMessage(msg, true);
}

/**
 * Check if the message queue of the window contains
 * any queued messages.
 *
 * @author  Leif Erik Larsen
 * @since   2001.05.05
 * @return  True if there is any waiting messages, or else false.
 * @see     #postMessage
 */
bool GWindow::isAnyWaitingMessages () const
{
   QMSG msg;
   return ::WinPeekMsg(GProgram::hAB, &msg, hWnd, 0, 0, PM_NOREMOVE);
}

/**
 * Request {@link #layout} to be automatically called at a later time,
 * when the GUI-thread is idle. This method can be called by any 
 * thread (not only the GUI-thread). It is handy e.g. for code that 
 * possibly needs to call {@link #layout} many times in sequence.
 * It is often better to call this method then, so that the 
 * {@link #layout} method is in fact only called just once.
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.05
 * @see     #layout
 */
void GWindow::postLayout ()
{
   GObject::Synchronizer synch(this); // Only one thread at a time here!
   if (!layoutPosted)
   {
      layoutPosted = true;
      if (getTimerData("_PostLayout", false) == null)
         startTimer("_PostLayout", 250);
   }
}

/**
 * Post the specified abstract command object to the event queue of the
 * window so that the command will be performed asynchronously. The
 * command is posted as an <i>GM_COMMAND</i> message. When the message
 * is received at a later time then the <i>GWindow</i> object will
 * automatically make a call to method <i>executeAbstractCommand()</i>.
 * @see #executeAbstractCommand
 */
void GWindow::postCommand ( GAbstractCommand* cmd )
{
   postMessage(GM_ABSTRACTCOMMAND, GWindowMessage::Param1(cmd));
}

/**
 * Used to prevent the tooltip box to occur just because a window 
 * underneath the mouse has been resized or moved (which might 
 * produce a WM_MOUSEMOVE message that in turn could cause the 
 * tooltip box to be shown if the window has the WS2_AUTO_SHOW_TOOLTIP
 * flag set on.
 *
 * @author  Leif Erik Larsen
 * @since   2006.11.06
 */
static bool disableAutoShowTooltip = false;

bool GWindow::captureMouse ( bool flag )
{
   if (flag)
   {
      if (mouseIsCaptured)
         return true; // Mouse is already captured!
      mouseIsCaptured = ::WinSetCapture(HWND_DESKTOP, hWnd);
      if (!disableAutoShowTooltip && (winStyle2 & WS2_AUTO_SHOW_TOOLTIP) != 0)
      {
         if (isActive()) // If we are part of the active (frame-) window.
         {
            GString tt = getTooltipText();
            if (tt == "")
            {
               GTooltip::Hide();
            }
            else
            {
               int delayShowMillis = getTooltipDelayShowMillis();
               int autoHideMillis = getTooltipAutoHideMillis();
               GTooltip::Show(*this, tt, delayShowMillis, autoHideMillis);
            }
         }
      }
      disableAutoShowTooltip = false;
      return mouseIsCaptured;
   }
   else
   {
      if (mouseIsCaptured)
      {
         ::WinSetCapture(HWND_DESKTOP, null);
         mouseIsCaptured = false;
         GTooltip::Hide();
         return true;
      }
      else
      {
         return false;
      }
   }
}

/**
 * This method is called whenever the window size has changed.
 *
 * The default implementation will automatically call <i>layout()</i>,
 * which in turn will invoke the layout manager of the window, if any.
 *
 * Therefore, most sub-classes should not override the <i>onSize</i>
 * method, but rather use some layout manager in order to adjust the
 * size and position of its children when the window has been resized.
 *
 * @see #layout
 */
bool GWindow::onSize ( int /*width*/, int /*height*/ )
{
   layout();
   return true;
}

void GWindow::onFocusKill ()
{
   if (isCaretOn())
      destroyCaret_();
}

void GWindow::onFocusSet ()
{
   if (isCaretOn())
      createCaret_();
}

void GWindow::onFocus ( bool set )
{
   if (set)
   {
      onFocusSet();

      // Let GDialogFrame handle it before any user registered focus listener.
      GDialogPanel* dlg = getOwnerDialogPanel();
      if (dlg != null)
      {
         GDialogFrame& dlgf = dlg->getOwnerFrame();
         dlgf.trackContainedComponentFocusGain(*this);
      }
   }
   else
   {
      onFocusKill();
   }
}

bool GWindow::onPresParamChanged ( GWindow::PresParamID paramID )
{
   switch (paramID)
   {
      case PP_FONTNAMESIZE:
      {
         invalidateAll(false);
         GString fontDef = getFontNameSize();
         return onFontNameSizeChanged(fontDef);
      }

      case PP_BACKGROUNDCOLOR:
      {
         invalidateAll(false);
         GColor color = getBackgroundColor();
         return onBackgroundColorChanged(color);
      }

      case PP_FOREGROUNDCOLOR:
      {
         invalidateAll(false);
         GColor color = getForegroundColor();
         return onForegroundColorChanged(color);
      }

      default:
         return false;
   }
}

bool GWindow::onFontNameSizeChanged ( const GString& /*fontNameSize*/ )
{
   GWindow* p = parentWin;
   if (p != null && 
       p->getHWND() != HWND_DESKTOP &&
       !(winStyle2 & WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED))
   {
      p->layout();
   }
   else
   {
      // Do layout the window, but only if the window has at least some 
      // dimension. This is to prevent possible NullPointerExcepotion's 
      // during window creation. There is no need to layout a zero-
      // dimension window anyway.
      GDimension dim = getWindowSize();
      if (dim.width > 0 || dim.height > 0)
         layout();
   }

   return true;
}

bool GWindow::onForegroundColorChanged ( const GColor& color ) 
{ 
   return false; 
}

/**
 * Check if this window is to show push buttons with a flat look.
 * We will call our parent window recursively until one of them
 * overrides this method and returns the answer. If none of them 
 * overrides this method then we will return the default, 
 * which is "true".
 *
 * @author  Leif Erik Larsen
 * @since   2005.11.23
 * @see     #setTooltipPosition
 * @see     GAbstractToolbarWindow#isFlatButtonBorders
 */
bool GWindow::isFlatButtonBorders () const
{
   if (parentWin != null)
      return parentWin->isFlatButtonBorders();
   else
      return true;
}

/**
 * Check if this window is to show tooltips at all. 
 * We will call our parent window recursively until one of them 
 * overrides this method and returns the answer. If none of them 
 * overrides this method then we will return the default, 
 * which is "true".
 *
 * @author  Leif Erik Larsen
 * @since   2005.11.23
 * @see     #setTooltipPosition
 */
bool GWindow::isShowTooltip () const
{
   if (parentWin != null)
      return parentWin->isShowTooltip();
   else
      return true;
}

/**
 * Set the tooltip position for this window.
 * The default is to ask our parent window.
 *
 * @author  Leif Erik Larsen
 * @since   2005.12.21
 * @see     #getTooltipPosition
 * @see     #isShowTooltip
 * @see     #isFlatButtonBorders
 * @see     GAbstractToolbarWindow#isFlatButtonBorders
 */
void GWindow::setTooltipPosition ( GTooltip::Position pos )
{
   this->tooltipPos = pos;
}

/**
 * Get the tooltip position for this window (if any tooltip at all).
 * The default is to inherit the tooltip position from our 
 * parent window.
 *
 * @author  Leif Erik Larsen
 * @since   2005.12.21
 * @see     #setTooltipPosition
 */
GTooltip::Position GWindow::getTooltipPosition () const
{
   if (tooltipPos != GTooltip::PosInheritFromParent)
      return tooltipPos;
   if (parentWin != null)
      return parentWin->getTooltipPosition();
   return GTooltip::PosBelow; // Very default.
}

/**
 * Get the value to be specified for the "autoHideMillis" parameter of 
 * the {@link GTooltip#Show} static method, when a tooltip box for this 
 * window is to be shown.
 *
 * If a value less then zero is returned it is to be treated as if to 
 * use the very default value, which is specified by a recent call
 * to {@link GTooltip#SetDefaultAutoHideMillis}.
 *
 * @author  Leif Erik Larsen
 * @since   2006.11.08
 * @see     #setTooltipAutoHideMillis
 */
int GWindow::getTooltipAutoHideMillis () const
{
   if (tooltipAutoHideMillis >= 0)
      return tooltipAutoHideMillis;
   if (parentWin != null)
      return parentWin->getTooltipAutoHideMillis();
   // Use the very default, as defined by a recent call 
   // to {@link GTooltip#SetDefaultAutoHideMillis}.
   return -1; 
}

/**
 * Set the default number of milliseconds to wait until automatically 
 * hiding the tooltip box after it has been displayed for this window. 
 * A value of zero means "never". A value less than zero means 
 * "use the same value as our parent window".
 *
 * @author  Leif Erik Larsen
 * @since   2006.11.08
 * @see     #getTooltipAutoHideMillis
 */
void GWindow::setTooltipAutoHideMillis ( int value )
{
   tooltipAutoHideMillis = value;
}

/**
 * Get the value to be specified for the "delayShowMillis" parameter of 
 * the {@link GTooltip#Show} static method, when a tooltip box for this 
 * window is to be shown.
 *
 * If a value less then zero is returned it is to be treated as if to 
 * use the very default value, which is specified by a recent call
 * to {@link GTooltip#SetDefaultDelayShowMillis}.
 *
 * @author  Leif Erik Larsen
 * @since   2006.11.08
 * @see     #setTooltipDelayShowMillis
 */
int GWindow::getTooltipDelayShowMillis () const
{
   if (tooltipDelayShowMillis >= 0)
      return tooltipDelayShowMillis;
   if (parentWin != null)
      return parentWin->getTooltipDelayShowMillis();
   // Use the very default, as defined by a recent call 
   // to {@link GTooltip#SetDefaultDelayShowMillis}.
   return -1;
}

/**
 * Set the default number of milliseconds to wait until the tooltip box is 
 * actually displayed, for this window. A value of zero means "no delay".
 * A value less than zero means "use the same value as our parent window".
 *
 * @author  Leif Erik Larsen
 * @since   2006.11.08
 * @see     #getTooltipDelayShowMillis
 */
void GWindow::setTooltipDelayShowMillis ( int value )
{
   tooltipDelayShowMillis = value;
}

/**
 * This method is called by {@link GTooltip::Show} to get the 
 * concrete text of which to display in the tooltip of the window.
 *
 * @author  Leif Erik Larsen
 * @since   2005.12.13
 * @return  The text of which to display in the tooltip for the
 *          window. If the window has no tooltip then just return
 *          an empty string.
 * @see     #setTooltipText
 */
GString GWindow::getTooltipText () const
{
   return tooltipText;
}

/**
 * Set the tooltip text of the window.
 * The window has no tooltip text by default.
 *
 * @author  Leif Erik Larsen
 * @since   2005.12.13
 * @see     #getTooltipText
 * @see     #updateHintString
 */
void GWindow::setTooltipText ( const GString& tooltipText )
{
   this->tooltipText = tooltipText;
}

/**
 * Override this method in order to display the hint text for the
 * specified command ID, in the statusbar of the window.
 *
 * Clear the hint if specified command ID is empty or unknown.
 *
 * The default implementation of this method will call the same
 * method recursively on the parent window object (if any).
 *
 * @author Leif Erik Larsen
 * @since  2000.01.17
 * @see    #setTooltipText
 */
void GWindow::updateHintString ( const GString& cmdID )
{
   if (parentWin != null)
      parentWin->updateHintString(cmdID);
}

bool GWindow::onMenuSelect ( int id )
{
   // Check if the command comes from the popup menu
   if (popupMenu != null)
   {
      GMenuPopupItem *mi = popupMenu->getCommand(id);
      if (mi != null)
      {
         const GString& idString = mi->getIDString();
         updateHintString(idString);
         return true;
      }
   }

   bool ret = false;

   if (parentWin != null)
      ret = parentWin->onMenuSelect(id);

   return ret;
}

bool GWindow::onCommand ( int cmd )
{
   // Check if the command comes from the popup menu.
   if (popupMenu != null)
   {
      GMenuPopupItem* mi = popupMenu->getCommand(cmd);
      if (executeAbstractCommand(mi))
         return true;
   }

   bool ret = false;

   if (parentWin != null)
      ret = parentWin->onCommand(cmd);

   return ret;
}

bool GWindow::executeAbstractCommand ( GAbstractCommand* cmd )
{
   bool ret;

   if (cmd == null)
      ret = false;
   else
   if (__executeAbstractCommand(cmd))
      ret = true;
   else
   if (parentWin != null)
      ret = parentWin->executeAbstractCommand(cmd);
   else
      ret = false;

   return ret;
}

/**
 * This method is called only if the {@link #WS2_AUTO_PAINT_TEXT}
 * was specified upon window creation. It is called just after 
 * {@link #onPaintBackground} and just before {@link #onPaint}.
 * 
 * The default implementation will paint the current window text 
 * centered inside the window face. The text will be painted with 
 * transparent background, using the current foreground color and
 * the current font of the window. 
 *
 * @author  Leif Erik Larsen
 * @since   2004.10.22
 */
bool GWindow::onPaintText ( GGraphics& g, const GRectangle& rect )
{
   GRectangle r = getWindowRect();
   GColor frg = getForegroundColor();
   g.drawText(text, r, frg, GGraphics::DUMMY_COLOR, GGraphics::HCENTER);
   return true;
}

/**
 * Paint the background of the window.
 *
 * If the window is created with the {@link #WS2_AUTO_PAINT_BACKGROUND}
 * flag set then this method is automatically called just
 * before {@link #onPaint} by the default message handler when
 * it receives the WM_PAINT message.
 *
 * @see #onPaint
 * @see #onPaintText
 */
bool GWindow::onPaintBackground ( GGraphics& g, const GRectangle& rect )
{
   if ((winStyle2 & WS2_AUTO_PAINT_BACKGROUND) == 0)
      return false;

   GColor bck = getBackgroundColor();
   g.drawFilledRectangle(rect, bck);

   // TODO: Implement support for wallpaper images.
   /*
   // Do not attempt to paint the wallpaper image before it is 100% loaded.
   if (dlgWallPaper != null && dlgWallPaper.on && wallPaperImage != null)
   {
      Graphics g = (Graphics) msg.getParam1();
      Color bckColor = IStandardRGBColors.RGB_GRAY;
      g.setColor (bckColor);
      Rectangle r = g.getClipBounds();

      int x, y;
      Insets theInsets = getInsets();
      int imWidth = wallPaperImage.getWidth(this);
      int imHeight = wallPaperImage.getHeight(this);

      if (imWidth <= 0 || imHeight <= 0)
      {
         g.fillRect(r.x, r.y, r.width, r.height);
         return true;
      }

      int winWidth = clientArea->getSize().width;
      int winHeight = clientArea->getSize().height;

      // Paint the background wallpaper
      switch (dlgWallPaper.flag)
      {
         case GWallPaperParams::DWP_TILEIMAGE:
              for (x = theInsets.left; x < winWidth; x += imWidth)
                 for (y = theInsets.top; y < winHeight; y += imHeight)
                    g.drawImage(wallPaperImage, x, y, imWidth, imHeight, this);
              break;

         case GWallPaperParams::DWP_STRETCHIMAGE:
              g.drawImage(wallPaperImage, 0, 0, winWidth-1, winHeight-1, 0, 0, imWidth-1, imHeight-1, bckColor, this);
              break;

         case GWallPaperParams::DWP_CENTERIMAGE:
              g.fillRect(r.x, r.y, r.width, r.height);
              g.drawImage(wallPaperImage, winWidth/2 - imWidth/2, winHeight/2 - imHeight/2, imWidth, imHeight, this);
              break;
      }
   }
   */

   return true;
}

/**
 * @author  Leif Erik Larsen
 * @since   2004.09.18
 */
void GWindow::onPaintBorder ( GGraphics& g )
{
   if (border == null)
      return;
   GInsets ins = getInsets();
   GRectangle winRect = getWindowRect();
   if (parentWin != null)
      g.setColor(parentWin->getBackgroundColor());
   else
      g.setColor(getBackgroundColor());
   border->onPaint(g, winRect, ins);
   g.excludeClipRectangle(border->leftBackgroundArea);
   g.excludeClipRectangle(border->rightBackgroundArea);
   g.excludeClipRectangle(border->topBackgroundArea);
   g.excludeClipRectangle(border->bottomBackgroundArea);
}

/**
 * Usually to be used by {@link #handleWindowMessage} only.
 */
GWindowMessage::Answer GWindow::callDefaultMsgProc ( GWindowMessage& msg )
{
   GWindowMessage::Answer ret = 0;

   if (hWnd != null)
   {
      int id = msg.getID();
      GWindowMessage::Param1 p1 = msg.getParam1();
      GWindowMessage::Param2 p2 = msg.getParam2();
      if (defaultMsgProc != null)
         ret = defaultMsgProc(hWnd, id, p1, p2);
      else
         ret = ::WinDefWindowProc(hWnd, id, p1, p2);
   }

   return ret;
}

/**
 * This is the "low level" message handler for any GLib-window,
 * from the underlying system point of view. It will take the 
 * message and dispatch it to the {@link #handleWindowMessage}
 * method of the correct window object.
 *
 * @see #handleWindowMessage
 */
GWindowMessage::Answer EXPENTRY
GWindow::GenericWindowProc ( HWND hWnd,
                             GWindowMessage::IDType ulMsg,
                             GWindowMessage::Param1 mp1,
                             GWindowMessage::Param2 mp2 )
{
   GWindowEntry* we = (GWindowEntry*) ::WinQueryWindowPtr(hWnd, QWL_USER);

   GWindowMessage::Answer ret;
   if (we != null && we->theWindow->isAlive())
   {
      GWindowMessage msg(ulMsg, mp1, mp2);
      ret = we->dispatchMessage(msg);
   }
   else
   {
      ret = ::WinDefWindowProc(hWnd, ulMsg, mp1, mp2);
   }

   return ret;
}

/**
 * Utility method that can be used as a platform independent wrapper
 * for subclassing a window message handler procedure.
 *
 * @author  Leif Erik Larsen
 * @since   2001.10.26
 */
GWindow::WindowProc GWindow::SubclassWindow ( HWND hwnd, WindowProc newProc )
{
   return WinSubclassWindow(hwnd, newProc);
}

/**
 * This method is automatically called whenever the window receives 
 * keyboard input focus, so that the window can update its keybar 
 * with respect to the currently focused window. The default 
 * implementation does nothing but call the same method recursively
 * on its parent window. It is intended to be overridden by 
 * {@link GFrameWindow}, which will update its keybar if it has one.
 *
 * @author  Leif Erik Larsen
 * @since   2004.03.10
 * @param   focusWin  The window that is about to receive keyboard 
 *                    input focus. This is the window on which the 
 *                    method is first called. It represent the window
 *                    for which the keybar (if any) should reflect 
 *                    the current keyboard shortcuts map.
 */
void GWindow::updateKeybarCommands ( GWindow* focusWin )
{
   GWindow* parent = getParentWindow();
   if (parent != null)
      parent->updateKeybarCommands(focusWin);
}

bool GWindow::fireMouseMovedEvent ( const GMouseEvent& ev )
{
   bool handled = false;
   if (mouseListeners == null)
      return handled;

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = mouseListeners->getCount();
   GVector<GMouseListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(mouseListeners->elementAt(i));

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GMouseListener* l = llist[i];
      if (l->mouseMoved(ev))
         handled = true;
   }

   return handled;
}

bool GWindow::fireMousePressedEvent ( const GMouseEvent& ev )
{
   bool handled = false;
   if (mouseListeners == null)
      return handled;

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = mouseListeners->getCount();
   GVector<GMouseListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(mouseListeners->elementAt(i));

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GMouseListener* l = llist[i];
      if (l->mousePressed(ev))
         handled = true;
   }

   return handled;
}

bool GWindow::fireMouseReleasedEvent ( const GMouseEvent& ev )
{
   bool handled = false;
   if (mouseListeners == null)
      return handled;

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = mouseListeners->getCount();
   GVector<GMouseListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(mouseListeners->elementAt(i));

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GMouseListener* l = llist[i];
      if (l->mouseReleased(ev))
         handled = true;
   }
   return handled;
}

bool GWindow::fireMouseClickedEvent ( const GMouseEvent& ev )
{
   bool handled = false;
   if (mouseListeners == null)
      return handled;

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = mouseListeners->getCount();
   GVector<GMouseListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(mouseListeners->elementAt(i));

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GMouseListener* l = llist[i];
      if (l->mouseClicked(ev))
         handled = true;
   }
   return handled;
}

bool GWindow::fireMouseDoubleClickedEvent ( const GMouseEvent& ev )
{
   bool handled = false;
   if (mouseListeners == null)
      return handled;

   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   const int count = mouseListeners->getCount();
   GVector<GMouseListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(mouseListeners->elementAt(i));

   // Invoke the listeners.
   for (int i=0; i<count; i++)
   {
      GMouseListener* l = llist[i];
      if (l->mouseDoubleClicked(ev))
         handled = true;
   }
   return handled;
}

bool GWindow::onActivate () 
{ 
   return false; 
}

bool GWindow::onBackgroundColorChanged ( const GColor& color ) 
{ 
   return false; 
}

bool GWindow::onButton1Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

bool GWindow::onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

bool GWindow::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   if (isStatic())
   {
      GWindow* parent = getParentWindow();
      if (parent != null)
         setActive(true);
      return true;
   }
   else
   {
      return false;
   }
}

bool GWindow::onButton1Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

bool GWindow::onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

bool GWindow::onButton2DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

bool GWindow::onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

bool GWindow::onButton2Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   return false; 
}

/**
 * @author  Leif Erik Larsen
 * @since   2004.08.04
 * @return  True if the keyboard message has been handled, 
 *          or else false.
 * @see     #onKey
 */
bool GWindow::onKeyDown ( const GKeyMessage& key ) 
{ 
   return false;
}

/**
 * @author  Leif Erik Larsen
 * @since   2004.08.04
 * @return  True if the keyboard message has been handled, 
 *          or else false.
 * @see     #onKey
 */
bool GWindow::onKeyUp ( const GKeyMessage& key ) 
{ 
   return false;
}

/** 
 * Call the default keyboard handling code directly.
 * This method is to be called from {@link #onKey} only.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.09
 * @param   key        The key event of which to do default handling.
 * @param   lowLevel   True if we should do "low level" default 
 *                     handling only. That is the system dependent 
 *                     keyboard handling, such as the default handling 
 *                     of the KEY_LEFT even for GTextEntry, which will
 *                     move the caret one character to the left. If 
 *                     false is specified to this parameter then we 
 *                     will do our "high level" default handling. That
 *                     is translating the key into an accelerator 
 *                     command (if any, and <i>transAccel</i> is true) 
 *                     or else call the {@link onKey} method on our 
 *                     parent window.
 * @param   transAccel True if we shall attempt translating the key 
 *                     into an accelerator command. If false is 
 *                     specified then we will call the {@link #onKey}
 *                     method on our parent window directly and without
 *                     any attempt to tyranslate it to an accelerator 
 *                     command for this window. This parameter is 
 *                     ignored if <i>lowLevel</i> is true.
 */
bool GWindow::callDefaultKeyHandler ( const GKeyMessage& key, 
                                      bool lowLevel,
                                      bool transAccel )
{
   if (lowLevel)
   {
      callDefaultMsgProc(key.originalMsg);
      return true;
   }

   // Translate into associated accelerator command, if any.
   if (transAccel && accelTable != null)
   {
      GString kname = key.getName();
      if (kname != "")
      {
         GAbstractCommand *cmd = accelTable->getAccelItemByKeyName(kname);
         if (cmd != null)
         {
            postCommand(cmd);
            return true;
         }
      }
   }

   // Don't call the parent if we are requested not to do it.
   if ((winStyle2 & WS2_NO_RECURSIVE_KEY_HANDLING) != 0)
      return false;

   // Key is still not handled, so pass it recursively to our parent window.
   GWindow* parent = getParentWindow();
   if (parent != null)
      return parent->onKey(key);

   // Noone seems to be interested in this key event!
   return false;
}

/**
 * This method is called automatically to let the window handle the 
 * specified keyboard message event.
 *
 * The default implementation does three things:
 *
 * 1) Calls {@link #onKeyDown} or {@link #onKeyUp} with respect to the
 *    specified key message. If any of these returns true then we will
 *    do nothing more but return true our self. If the key message 
 *    object is "consumed" (see {@link GKeyMessage#consume}) by one of 
 *    these methods then we will return immediately even if the key is 
 *    not actually "handled".
 *
 * 2) Translate the keyboard code to the corresponding accelerator 
 *    table command if the window has any associated accelerator table. 
 *    If this is the case then the default implementation will do 
 *    nothing more but return true.
 *
 * 3) If the key is not translated into any accelerator command then we
 *    will automatically call the parent window (if any) implementation
 *    of this method. This is done recursively until the top level 
 *    window is reached.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.03
 * @return  True if the keyboard message has been handled, 
 *          or else false. True is returned by the default 
 *          implementation only if the key was actually 
 *          translated into an accelerator command.
 * @see     #onKeyDown
 * @see     #onKeyUp
 * @see     #setAccelTable
 */
bool GWindow::onKey ( const GKeyMessage& key ) 
{ 
   if (key.isKeyDown())
   {
      if (onKeyDown(key))
         return true;
   }
   else
   {
      if (onKeyUp(key))
         return true;
   }

   return callDefaultKeyHandler(key, false, true);
}

/**
 * Called when the user has requested the window to be closed.
 * For instance by selecting the "Close" command from the system menu
 * of the window.
 */
bool GWindow::onClose () 
{ 
   return false; 
}

/**
 * Called when a control has sent a notification message to its
 * owner and/or parent window. This is WM_CONTROL on OS/2, and
 * WM_NOTIFY on Windows.
 *
 * @param  ctrlID    The ID of the control, as seen by the system.
 * @param  notifyID  The ID of the notification event.
 * @param  data      Control and notification specific additional
 *                   data. For most controls this parameter is
 *                   not used.
 * @param  sysAnswerToReturn The value of which to return directly to 
 *                   the underlying operating system. On OS/2 this is 
 *                   not used, since the documentation of WM_CONTROL
 *                   says that it should always return zero.
 */
bool GWindow::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn ) 
{ 
   return false; 
}

bool GWindow::onDeactivate () 
{ 
   return false; 
}

bool GWindow::onHide () 
{ 
   return false; 
}

bool GWindow::onInitMenu () 
{ 
   return false; 
}

/**
 * @return  True if and only if you have actually set the mouse cursor.
 */
bool GWindow::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   bool anyMouseButtonDown = flags.isAnyMouseButtonPressed();
   if (!anyMouseButtonDown && (winStyle2 & WS2_AUTO_SHOW_TOOLTIP) != 0)
   {
      GString tt = getTooltipText();
      if (tt != "")
      {
         if (isPointInWindowRect(xpos, ypos))
         {
            if (!isMouseCapture())
               captureMouse(true);
         }
         else
         {
            if (isMouseCapture())
               captureMouse(false);
         }
         return true;
      }
   }
   return false; 
}

/**
 * @see #onPaintBackground
 */
bool GWindow::onPaint ( GGraphics& g, const GRectangle& rect ) 
{ 
   return false; 
}

bool GWindow::onShow () 
{ 
   return false; 
}

/**
 * @author  Leif Erik Larsen
 * @since   2004.01.24
 * @param   timerID   The case sensitive name of the timer, as 
 *                    specified to {@link #startTimer}.
 * @param   userData  The optional user data object that is 
 *                    associated with the timer, as specified to
 *                    method {@link #startTimer}.
 */
bool GWindow::onTimer ( const GString& timerID, GObject* userData ) 
{ 
   return false; 
}

/**
 * Handle the {@link #GM_USERMESSAGE} message.
 *
 * We will check if the user
 * message is an instance of {@link GDialogMessage} and if it is then
 * we will call {@link #onDialogMessage} for further handling of the
 * dialog message. If it is not an instance of {@link GDialogMessage}
 * then will will do nothing but call the super implementation 
 * of this method.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.30
 * @see     #sendUserMessage
 * @see     GThread#sendGuiUserMessage
 */
bool GWindow::onUserMessage ( GUserMessage& msg ) 
{ 
   GDialogMessage* dm = dynamic_cast<GDialogMessage*>(&msg);
   if (dm == null)
      return false;
   return onDialogMessage(*dm);
}

/**
 * This method is automatically called by {@link #onUserMessage} when
 * the received user message is an instance of {@link GDialogMessage}.
 *
 * The default implementation of this method will call the user 
 * defined dialog message handler (if any). A user defined dialog
 * message handler is set with {@link #setDialogMessageHandler}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.30
 * @return  True if the message was handler, or else false.
 */
bool GWindow::onDialogMessage ( GDialogMessage& msg )
{
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "%s: msg=[%s]", GVArgs(__FUNCTION__).add(msg.toString()));
   // Pass the message to the user code only if the dialog 
   // is actually in execution mode.
   if (hasSent_GM_INITDIALOG && userMsgProc != null)
      return userMsgProc->handleDialogMessage(msg);
   else
      return false;
}

/**
 * This is the message handler method of the window object.
 *
 * It is probably to be called by <i>genericWindowProc()</i> and
 * <i>sendMessage()</i> only, but we have made it protected to make
 * it possible for sub-classes to call it directly as well. Just in case
 * it should be required.
 *
 * @see #sendMessage
 * @see #genericWindowProc
 */
GWindowMessage::Answer GWindow::handleWindowMessage ( GWindowMessage& msg )
{
   /*
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "%s: msg=%s", GVArgs(__FUNCTION__).add(msg.toString()));
   */
   switch (msg.getID())
   {
      // --------------------------------------------------------------------
      case WM_CREATE:
      {
         // This is the very first message that the window receives.
         // At this early stage the constructor of <i>GWindow</i> has not
         // yet finished. Initialization code should therefore go into the
         // constructor of the sub-class, rather in something
         // like <i>onCreate()</i>.
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_DESTROY:
      {
         // This message is actually never received by us, because the window
         // destructor has already "undocked" our self from the window
         // handle when the window handle is being destroyed. Any destruction
         // code should therefore go into the destructor of the sub-class,
         // rather in something like <i>onDestroy()</i>.
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_TIMER:
      {
         bool docall = true;
         if (msg.getParam1Int() == TID_CURSOR)
            docall = false; // The blinking caret is to be maintained by the default message handler only.
         bool handled = false;
         if (docall && activeTimersBySysID != null)
         {
            int sysTimerID = msg.getParam1Int();
            GString sysTimerIDStr = GInteger::ToString(sysTimerID);
            GWindowTimerData* timerData = activeTimersBySysID->get(sysTimerIDStr);
            if (timerData != null)
            {
               const GString& timerID = timerData->userTimerID;
               // GProgram::GetProgram().printF("WM_TIMER: timerID=%s", GVArgs(timerID));
               GObject* userData = timerData->userData;
               if (timerID == "_PostLayout")
               {
                  GObject::Synchronizer synch(this); // Synch. with {@link #postLayout}.
                  stopTimer("_PostLayout");
                  layoutPosted = false;
                  layout();
                  handled = true;
               }
               else
               {
                  handled = onTimer(timerID, userData);
               }
            }
         }
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_SIZE:
      {
         int width = msg.getParam2LoUShort();
         int height = msg.getParam2HiUShort();
         GWindowMessage::Answer ret = callDefaultMsgProc(msg);
         // TODO: disableAutoShowTooltip = true;
         onSize(width, height);
         return ret;
      }

      // --------------------------------------------------------------------
      case WM_TRANSLATEACCEL:
      {
         if (AboutToHandleKey)
         {
            // Ok, the message has been sent manually by us, during 
            // our handling of the WM_CHAR key. So give it to the 
            // default OS/2 message handler.
            return callDefaultMsgProc(msg);
         }
         // This message is automatically sent synchronously by OS/2 
         // upon ::WinGetMsg() and ::WinPeekMsg(). Due to the way we
         // handles keyboard events we must not give OS/2 a chance to
         // translate keyboard events into accelerator commands before 
         // we has finished handling the key. Only if we does not handle
         // the key at all we should do this translation. This is in order
         // to prevent OS/2 from handling e.g. system accelerators such as
         // Alt+F5 (Restore Window) except if we are not interested in the
         // same keyboard code.
         return GWindowMessage::Answer(FALSE);
      }

      case WM_CHAR:
      {
         if (AboutToHandleKey)
         {
            // OS/2 Warp automatically sends the keyboard event recursively
            // to all parent windows until it reaches the top level window.
            // Since "we are already here" we can assume that this keyboard 
            // event is due to OS/2 doing the recursive calls. In that case
            // do nothing but call the system default message handler.
            // This behaviour is needed in order to guarantee that the 
            // keyboard handling mechanisms works the same way in G-Lib 
            // on both OS/2 Warp and Windows.
            return callDefaultMsgProc(msg);
         }
         GIsHereTracker isHereTracker(AboutToHandleKey, true);
         GKeyMessage key(msg);
         if (key.getCharacter() != '\0')
            GTooltip::Hide();
         bool handled = onKey(key);
         if (!handled)
         {
            // Translate the keyboard message into a WM_COMMAND, 
            // WM_SYSCOMMAND or WM_HELP and send it to its window.
            GWindow& top = getTopLevelWindow();
            GFrameWindow* fw = dynamic_cast<GFrameWindow*>(&top);
            if (fw != null)
            {
               QMSG m;
               memset(&m, 0, sizeof(m));
               GWindow& frame = fw->getFrame();
               m.hwnd = frame.getHWND();
               m.msg = msg.getID();
               m.mp1 = msg.getParam1();
               m.mp2 = msg.getParam2();
               m.time = ::WinQueryMsgTime(GProgram::hAB);
               ::WinQueryMsgPos(GProgram::hAB, &m.ptl);
               bool translated = frame.sendMessage(WM_TRANSLATEACCEL, GWindowMessage::Param1(&m));
               if (!translated)
                  return callDefaultMsgProc(msg);
               ::WinSendMsg(m.hwnd, m.msg, m.mp1, m.mp2);
            }
         }
         return 0;
      }

      // --------------------------------------------------------------------
      case WM_PAINT:
      {
         if (winStyle2 & WS2_DEFAULTPAINT)
         {
            // The window is to paint it self, without calling the
            // program defined {@link #onPaint} method. This is typically
            // true for system defined window classes, such as title bars,
            // dialog boxes, push buttons, text entry fields, etc.
            return callDefaultMsgProc(msg);
         }
         RECTL r;
         HPS gh = ::WinBeginPaint(hWnd, null, &r);
         try {
            GRectangle rect = GWindow::MakeLibRect(r, *this);
            GGraphics g(gh, *this, false, false);
            onPaintBorder(g);
            onPaintBackground(g, rect);
            if ((winStyle2 & WS2_AUTO_PAINT_TEXT) != 0)
               onPaintText(g, rect);
            onPaint(g, rect);
            ::WinEndPaint(gh);
         } catch (...) {
            ::WinEndPaint(gh);
            throw;
         }
         return 0;
      }

      // --------------------------------------------------------------------
      case WM_MOUSEMOVE:
      {
         bool doFire = true;
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         bool handled = onMouseMove(x, y, flags);
         if (doFire)
         {
            // Use "BUTTON3" as a dummy buttonId (not relevant for 
            // this type of mouse event.
            GMouseEvent::ButtonId buttonId = GMouseEvent::BUTTON3;
            GMouseEvent ev(x, y, flags, buttonId);
            fireMouseMovedEvent(ev); // Fire every mouse listener, if any.
         }
         if (handled)
            return GWindowMessage::Answer(true);
         // Yes, do the default processing before return, so that the default mouse cursor can be set
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON1DOWN:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Left button down: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton1Down(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON1);
         if (fireMousePressedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON1UP:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Left button up: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton1Up(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON1);
         if (fireMouseReleasedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON1CLICK:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Left button click: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton1Click(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON1);
         if (fireMouseClickedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON1DBLCLK:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Left button double click: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton1DblClk(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON1);
         if (fireMouseDoubleClickedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON2DOWN:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Right button down: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton2Down(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON2);
         if (fireMousePressedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON2UP:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Right button up: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton2Up(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON2);
         if (fireMouseReleasedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON2CLICK:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Right button click: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton2Click(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON2);
         if (fireMouseClickedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_BUTTON2DBLCLK:
      {
         int x = msg.getParam1LoShort();
         int y = msg.getParam1HiShort();
         y = sysY(y);
         GWindowMessage::InputFlags flags = msg.getParam2HiUShort();
         // GProgram::GetProgram().printF("Right button double click: x=%d, y=%d", GVArgs(x).add(y));
         bool handled = onButton2DblClk(x, y, flags);
         GMouseEvent ev(x, y, flags, GMouseEvent::BUTTON2);
         if (fireMouseDoubleClickedEvent(ev)) // Fire every mouse listener, if any.
            handled = true;
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_COMMAND:
      {
         int cmdID = msg.getParam1Int();
         bool handled = onCommand(cmdID);
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_CONTROL:
      {
         int ctrlID = msg.getParam1LoUShort();
         int notifyID = msg.getParam1HiUShort();
         int data = msg.getParam2Int();
         int sysAnswerToReturn = 0; // Not actually used on OS/2. We will always return zero.
         if (!onNotify(ctrlID, notifyID, data, sysAnswerToReturn))
            return callDefaultMsgProc(msg);
         return 0;
      }

      // --------------------------------------------------------------------
      case WM_BEGINDRAG:
      {
         if (dragDropHandler != null)
         {
            int xpos = msg.getParam1LoShort();
            int ypos = msg.getParam1HiShort();
            dragDropHandler->handleBeginDrag(xpos, ypos);
            return 0;
         }
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case DM_DRAGOVER:
      {
         GWindowMessage::Answer mresult = 0;
         bool handled = false;
         DRAGINFO* di = (DRAGINFO*) msg.getParam1();
         if (::DrgAccessDraginfo(di))
         {
            GDragOverAnswer answ;
            memset(&answ, 0, sizeof(answ));
            HPS hps = DrgGetPS(getHWND());
            if (hps != null)
            {
               GGraphics g(hps, *this, false, false);
               GPoint pos = getWindowPosOnScreen();
               int xpos = msg.getParam2LoShort() - pos.x;
               int ypos = msg.getParam2HiShort() - pos.y;
               answ.canDrop = GDragOverAnswer::DOANSW_NEVERDROP;
               answ.defaultOp = GDragOverAnswer::DOANSW_UNKNOWN;
               GDragInfo dragInfo(*di);
               if (dragDropHandler != null)
               {
                  dragDropHandler->handleDragOver(dragInfo, g, xpos, ypos, answ);
                  handled = true;
               }
               ::DrgReleasePS(hps);
            }
            ::DrgFreeDraginfo(di);
            if (handled)
               mresult = MRFROM2SHORT(answ.canDrop, answ.defaultOp);
         }
         if (handled)
            return mresult;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case DM_DRAGLEAVE:
      {
         bool handled = false;
         DRAGINFO* di = (DRAGINFO*) msg.getParam1();
         if (::DrgAccessDraginfo(di))
         {
            HPS hps = ::DrgGetPS(getHWND());
            if (hps != null)
            {
               GGraphics g(hps, *this, false, false);
               GDragInfo dragInfo(*di);
               if (dragDropHandler != null)
               {
                  dragDropHandler->handleDragLeave(dragInfo, g);
                  handled = true;
               }
               ::DrgReleasePS(hps);
            }
            ::DrgFreeDraginfo(di);
         }
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case DM_DROP:
      {
         bool handled = false;
         DRAGINFO* di = (DRAGINFO*) msg.getParam1();
         if (::DrgAccessDraginfo(di))
         {
            HPS hps = ::DrgGetPS(getHWND());
            if (hps != null)
            {
               GGraphics g(hps, *this, false, false);
               GPoint pos = getWindowPosOnScreen();
               int xpos = di->xDrop - pos.x;
               int ypos = di->yDrop - pos.y;
               GDragInfo dragInfo(*di);
               if (dragDropHandler != null)
               {
                  dragDropHandler->handleDrop(dragInfo, g, xpos, ypos);
                  handled = true;
               }
               ::DrgReleasePS(hps);
            }
            // TODO: ::DrgFreeDraginfo(di);
         }
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_SETFOCUS:
      // case WM_FOCUSCHANGE:
      {
         bool set = msg.getParam2LoShort();
         if (set)
            updateKeybarCommands(this);
         onFocus(set);
         return GWindow::callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_ACTIVATE:
      {
         // Call the default window procedure first, because the Win32 API
         // ::GetActiveWindow() will not return true until this message has 
         // been handled by the default message handler of the window.
         callDefaultMsgProc(msg);
         bool activate = msg.getParam1LoShort();
         /*
         GWindow* modalDlg = getFirstOwnedModalDialog();
         if (activate && modalDlg != null)
         {
            modalDlg->setActive();
            return 0;
         }
         */

         if (activate)
            onActivate();
         else
            onDeactivate();
         return 0;
      }

      // --------------------------------------------------------------------
      case WM_SHOW:
      {
         if (msg.getParam1LoShort())
            onShow();
         else
            onHide();
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_MENUEND:
      {
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_PRESPARAMCHANGED:
      {
         if (getHWND() == null)
         {
            // This can happen if some presentation parameter (usually the
            // FontNameSize parameter) has been set before the window
            // constructor has finished. In that case, don't attempt
            // to call any event method on the instance object.
         }
         else
         {
            onPresParamChanged(msg.getParam1Int());
         }
         // Yes, do the default processing, even if the window did handle the message
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_CLOSE:
      {
         bool handled = onClose();
         if (handled)
            return 0;
         // Do nothing on window-close-request, by default.
         return 0;
      }

      // --------------------------------------------------------------------
      case WM_INITMENU:
      {
         // Make sure to close the tooltip (if visible).
         GTooltip::Hide();
         bool handled = onInitMenu();
         if (handled)
            return 0;
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_MENUSELECT:
      {
         int id = msg.getParam1LoShort();
         bool handled = onMenuSelect(id);
         if (handled)
         {
            bool postCmd = bool(msg.getParam1HiShort());
            return GWindowMessage::Answer(postCmd);
         }
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_DRAWITEM:
      {
         bool handled = false;
         bool isFromMenu = (msg.getParam1LoUShort() == FID_MENU);
         if (isFromMenu)
         {
            OWNERITEM* oi = (OWNERITEM*) msg.getParam2();
            GMenuPopupItem* item = (GMenuPopupItem*) oi->hItem;
            if (item != null && oi->idItem >= 1)
            {
               GMenuPopup::TheWindow& menwin = item->ownerMenu.getTheWindow();
               GGraphics g(oi->hps, menwin, false, false);
               bool itemDisabled = ((oi->fsAttribute & MIA_DISABLED) != 0);
               bool itemHilited = ((oi->fsAttribute & MIA_HILITED) != 0);
               bool itemChecked = ((oi->fsAttribute & MIA_CHECKED) != 0);
               GRectangle rect;
               rect.x = oi->rclItem.xLeft;
               rect.y = sysY(oi->rclItem.yBottom);
               rect.width = oi->rclItem.xRight - oi->rclItem.xLeft + 1;
               rect.height = oi->rclItem.yTop - oi->rclItem.yBottom + 1;
               rect.inflateRect(-2, 0);
               item->ownerMenu.drawMenuItem(g, *item, rect, itemDisabled, itemHilited, itemChecked);
            }

            // Set bits the same so that the menu window does not highlight
            // the item or remove its highlight. Also prevent the system
            // from drawing the check mark and the arrow used to visualize
            // popup menues.
            oi->fsAttributeOld = (oi->fsAttribute &= ~(MIA_HILITED | MIA_CHECKED | MIA_DISABLED | MIA_FRAMED));
            handled = true;
         }
         if (handled)
            return GWindowMessage::Answer(true); // Yes, we did draw it manually!
         // All other types of controls must be handled by the default
         // message handler.
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case WM_MEASUREITEM:
      {
         bool handled = false;
         bool isFromMenu = (msg.getParam1LoUShort() == FID_MENU);
         if (isFromMenu)
         {
            OWNERITEM* oi = (OWNERITEM*) msg.getParam2();
            GMenuPopupItem* item = (GMenuPopupItem*) oi->hItem;
            if (item != null && oi->idItem >= 1)
            {
               GMenu& topMen = item->ownerMenu.getTopLevelMenu();
               GMenuPopup::TheWindow& menwin = item->ownerMenu.getTheWindow();
               const GString& fontNS = topMen.getFontNameSize();
               menwin.setFontNameSize(fontNS);
               GGraphics g(menwin);
               GDimension dim = item->ownerMenu.measureMenuItem(g, *item);
               oi->rclItem.yTop = oi->rclItem.yBottom + dim.height - 1;
               oi->rclItem.xRight = oi->rclItem.xLeft + dim.width - 1;
            }
            handled = true;
         }
         if (handled)
            return 0;
         // All other types of controls must be handled by the default
         // message handler.
         return callDefaultMsgProc(msg);
      }

      // --------------------------------------------------------------------
      case GM_ABSTRACTCOMMAND:
      {
         GAbstractCommand* acmd = (GAbstractCommand*) msg.getParam1();
         executeAbstractCommand(acmd);
         return 0;
      }

      // --------------------------------------------------------------------
      case GM_USERMESSAGE:
      {
         bool handled = false;
         GUserMessage* um = (GUserMessage*) msg.getParam1();
         bool autoDelete = msg.getParam2Bool();
         if (um != null) // Should always be true, but in case.
         {
            handled = onUserMessage(*um);
            if (autoDelete)
               delete um;
         }
         return GWindowMessage::Answer(handled);
      }

      // --------------------------------------------------------------------
      case GM_USERMESSAGE_FROM_THREAD:
      {
         bool handled = false;
         GUserMessage* um = (GUserMessage*) msg.getParam1();
         GThread::UMParams* ump = (GThread::UMParams*) msg.getParam2();
         try {
            if (um != null) // Should always be true, but in case.
               handled = onUserMessage(*um);
         } catch (...) {
            // Notify the waiting thread automatically, if requested.
            if (ump != null && ump->autoNotify)
               ump->postingThread.guiUserMessageHandlerHasFinished();
            throw;
         }
         // Notify the waiting thread automatically, if requested.
         if (ump != null && ump->autoNotify) 
            ump->postingThread.guiUserMessageHandlerHasFinished();
         return GWindowMessage::Answer(handled);
      }

      // --------------------------------------------------------------------
      case GM_ASYNC_PAINT:
      {
         bool inclChildren = msg.getParam1Int();
         invalidateAll(inclChildren);
         return GWindowMessage::Answer(0);
      }

      // --------------------------------------------------------------------
      case GM_ASYNC_GRAB_FOCUS:
      {
         grabFocus();
         return GWindowMessage::Answer(0);
      }

      // --------------------------------------------------------------------
      default:
         return callDefaultMsgProc(msg);
   }
}

/**
 * Test if the component is empty or contains an illegal value.
 *
 * The default implementation will always return true.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @return  True if the component is empty or if it contains an 
 *          illegal value, or else false.
 * @see     GDialogPanel#isComponentEmpty
 */
bool GWindow::isEmpty () const
{
   return true;
}

/**
 * Test if the Component is Static or not. A Static Component is a 
 * Component that can never possibly receive the keyboard focus or 
 * being activated by any means, even if it isn't disabled or grayed. 
 * Examples of Static Components are STATICTEXT, STATICICON 
 * and GROUPBOX.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @see     #WS2_NOT_STATIC
 */
bool GWindow::isStatic () const 
{ 
   return (winStyle2 & WS2_NOT_STATIC) == 0;
}

/**
 * Test if the Component can be activated or not. This method is
 * specially important regarding the logic required to support so called
 * mandatory Components. A Component can only be activated if the owner
 * dialog panel doesn't have any empty mandatory Components with an index
 * that is less than the index of this Component. This strict mandatory
 * control can be disabled by using the
 * method {@link GDialogPanel#setMandatoryMode}, however.
 *
 * <b>Mark:</b> The Component can be none-activable even if it isn't
 * currently disabled. Whether or not the Component can be activated
 * depends on the mandatory state of the dialog panel in addition to
 * the enable status of the Component. Therefore, this test isn't the
 * same as is done when calling {@link #isEnabled}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @return  True if the control is able to receive keyboad
 *          focus, or else false.
 * @see     GDialogPanel#setMandatoryMode
 */
bool GWindow::isCtrlActivable ()
{
   if (isStatic())
      return false;
   if (!isVisible())
      return false;
   if (!isEnabled())
      return false;
   return true;
}

/**
 * This method will be called when our associated hot-key has been
 * pressed by the user. This default implementation will try to activate
 * the next editable component (that is, our closest neightbor to the
 * right, that is currently editable) upon this event.
 *
 * Any component class that need some other action upon this event
 * should override this method. This is true for instance for
 * PUSHBUTTON's, LOCKBUTTON's and TOGGLEBUTTON's.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */
void GWindow::hotKeyAction ()
{
   GDialogPanel* odlg = getOwnerDialogPanel();
   if (odlg == null)
      return;
   int max = odlg->getComponentCount() - 1;
   for (int i=odlg->getComponentIndex(name)+1; i<max; i++)
   {
      GWindow& comp = odlg->getComponentByIndex(i);
      if (!comp.isStatic())
      {
         comp.grabFocus(false);
         break;
      }
   }
}

/**
 * Get the current number of components that are contained in the
 * window, but only if the window is used as a dialog panel.
 *
 * The default implementation will always return zero.
 * {@link GDialogPanel} overrides this and returns the number of 
 * components contained in its contained client area.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */
int GWindow::getComponentCount () const
{
   return 0;
}

/**
 * Get a reference to the named Dialog Component Object.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.30
 * @throws  GNoSuchComponentException if the specified Component does not
 *                                    exist in the window.
 */
GWindow& GWindow::getComponentByID ( const GString& id )
{
   return getChildWindow(id);
}

/**
 * Get a reference to the indexed Dialog Component Object.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @throws  GNoSuchComponentException if the specified index is out of bounds.
 */
GWindow& GWindow::getComponentByIndex ( int index )
{
   return getChildWindow(index);
}

/**
 * Seach for the specified Component, and return the index of it.
 * <p>
 * In most cases the requested Component has an index that is the same to
 * what was the index of the previous requested Component (previous time
 * this method was called). Therefore it is very speed optimizing to
 * check if this is the case first, before eventually starting to search
 * for the requested Component. The chances are great that we will find
 * the requested Component then, without having to actually search at all.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @return  Index ofthe specified component, or else -1 if it doesn't exist.
 */
int GWindow::getComponentIndex ( const GString& id ) const
{
   int num = getComponentCount();
   for (int i=0; i<num; i++)
   {
      if (recentSearchIndex >= num)
         recentSearchIndex = 0;
      GWindow& child = const_cast<GWindow*>(this)->getComponentByIndex(recentSearchIndex);
      if (child.name == id)
         return recentSearchIndex;
      if (++recentSearchIndex >= num)
         recentSearchIndex = 0;
   }

   return -1; // Specified Component wasn't found!
}

/**
 * Activate the indexed Component. If we fails to activate the specified
 * Component then we will return <code>false</code>, or else we will
 * return <code>true</code>. It might fail e.g. if the Component is
 * currently disabled.
 *
 * The dialog messages GM_FOCUSSET and GM_FOCUSLOSS will be 
 * automatically sent to all affected windows.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @param   index    Index of which Component to activate.
 */
bool GWindow::setComponentFocus ( int index )
{
   try {
      // Make sure that the requested Component is activable.
      GWindow* ctrl = &getComponentByIndex(index);
      if (!ctrl->isCtrlActivable())
         return false;
      // ---
      ctrl->grabFocus();
      return true;
   } catch (GNoSuchComponentException& /*e*/) {
      return false;
   }
}

/**
 * Activate specified Component. If we fails to activate the specified
 * Component then we will return <code>false</code>, or else we will
 * return <code>true</code>. It might fail e.g. if the Component is
 * currently disabled.
 *
 * The dialog messages GM_FOCUSSET and GM_FOCUSLOSS will be 
 * automatically sent to all affected windows.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @param   id       ID of which Component to activate.
 */
bool GWindow::setComponentFocus ( const GString& id )
{
   int index = getComponentIndex(id);
   return setComponentFocus(index);
}

/**
 * Get the dialog panel of where we are contained, or null if we are 
 * not part of any parent dialog panel.
 *
 * Note that this method will search recursively upward in the parent 
 * window stack, and we will return a pointer to our first parent or 
 * grand parent that is an instance of {@link GDialogPanel}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */
GDialogPanel* GWindow::getOwnerDialogPanel ()
{
   GWindow* p = this;
   for (;;)
   {
      p = p->getParentWindow();
      if (p == null)
         return null;
      GDialogPanel* dlg = dynamic_cast<GDialogPanel*>(p);
      if (dlg != null)
         return dlg;
   }
}

/**
 * Se whether we should initially send a <i>GM_CTRLCHANGED</i> message for
 * each Component on the Dialog Panel or not. The default is that we will
 * send the GM_CTRLCHANGED message once for each Component opon
 * initialization. To disable this, just call this method with a true
 * parameter before the dialog is executed.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */
void GWindow::setNoInitCtrlChanged ( bool flag )
{
   noInitCtrlChanged = flag;
}

/**
 * Test if the Dialog Panel has been initialized or not. The Dialog Panel
 * is reported to be initialized as soon as the <i>GM_INITDIALOG</i>
 * message has been sent.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 */
bool GWindow::isInitialized () const
{
   return hasSent_GM_INITDIALOG;
}

/**
 * Return true if we are currently being executed as part 
 * of a dialog box.
 *
 * The default implementation does nothing but foreward the call to 
 * our top-level window (see {@link #getTopLevelWindow}). 
 * The top-level window returns true if it is an instance 
 * of {@link GDialogFrame} that is currently being executed.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.03
 * @see     #isAlive
 * @see     GDialogFrame#isExecutingAsDialog
 */
bool GWindow::isExecutingAsDialog () const
{
   GWindow& top = const_cast<GWindow*>(this)->getTopLevelWindow();
   return top.isExecutingAsDialog();
}

/**
 * Get the current value of the filter for GM_CTRLCHANGED messages.
 *
 * @author  Leif Erik Larsen
 * @since   2004.09.07
 * @see     #enterCtrlChangedFilter
 * @see     #exitCtrlChangedFilter
 */
int GWindow::getCtrlChangedFilter () const
{
   return ctrlChangedFilter;
}

/**
 * Use this method to temporarily prevent the DM_CTRLCHANGED message
 * from being sent upon {@link #sendCtrlChanged}.
 *
 * This method is typically used by components within the
 * {@link #changeValue} method to respect the received notify-flag.
 * Here you have a sample of a typical usage of this method.
 *
 * <pre>
 *
 * bool changeValue ( const GString& newValue, bool notify )
 * {
 *    enterCtrlChangedFilter();
 *    // ... Perform the logic needed to update the Component value.
 *    exitCtrlChangedFilter();
 *    if (notify)
 *       sendCtrlChanged();
 * }
 *
 * </pre>
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @see     #exitCtrlChangedFilter
 * @see     #getCtrlChangedFilter
 */
void GWindow::enterCtrlChangedFilter () 
{ 
   ctrlChangedFilter++; 
}

/**
 * Always call this method before exiting from a region that was
 * recently initiated by a call to {@link #exitCtrlChangedFilter}.
 * This reenables the {@link #sendCtrlChanged} method again, but
 * only if {@link #getCtrlChangedFilter} is back to zero.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @see     #enterCtrlChangedFilter
 * @see     #getCtrlChangedFilter
 */
void GWindow::exitCtrlChangedFilter () 
{ 
   ctrlChangedFilter--; 
}

/**
 * Send the <i>DM_CTRLCHANGED</i> message to the Owner Dialog Panel,
 * but only if the {@link #getCtrlChangedFilter} counter variable is
 * zero and the component has been initialized (that is if, and only 
 * if, <i>DM_INITDIALOG</i> has been sent).
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @see     #getCtrlChangedFilter
 * @see     #isInitialized
 */
void GWindow::sendCtrlChanged ()
{
   if ((winStyle2 & WS2_DONT_SEND_CTRL_CHANGE_MSG) != 0)
      return;
   if (ctrlChangedFilter || !isInitialized())
      return;
   GDialogPanel* dlg = getOwnerDialogPanel();
   if (dlg != null)
   {
      GString value;
      if ((winStyle2 & WS2_DONT_INCLUDE_VALUE_IN_CTRL_CHANGE_MSG) == 0)
         value = queryValue();
      dlg->sendDialogMessage(GDialogMessageHandler::GM_CTRLCHANGED, name, value);
   }
}

/**
 * Add the specified dialog message to the event queue so it will
 * be sent asynchronously to the user defined message handle of the 
 * dialog.
 * 
 * @author  Leif Erik Larsen
 * @since   2006.04.13
 * @param   msg  The dialog message of which to post.
 * @param   autoDeleteMsg True if we shall automatically destroy the 
 *               specified dialog message object after it has been 
 *               handled by the dialog message handler. This is 
 *               usually the case only if the dialog message is 
 *               created on the heap.
 * @see     #sendDialogMessage
 * @see     #postUserMessage
 */
void GWindow::postDialogMessage ( GDialogMessage* msg, bool autoDeleteMsg )
{
   postUserMessage(msg, autoDeleteMsg);
}

/**
 * Send the specified dialog message to the dialog, which in turn 
 * will be sent to the user defined message handler 
 * of the dialog (if any).
 *
 * @author  Leif Erik Larsen
 * @since   2004.02.01
 * @see     #postDialogMessage
 */
bool GWindow::sendDialogMessage ( GDialogMessage& msg )
{
   if (filterDlgMessages)
   {
      // Filter out the message in case {@link #filterDlgMessages} is true.
      // This is the case upon execution of the dialog, to ensure that
      // <i>GM_INITDIALOG</i> is the very first message that the user
      // defined message handler will receive.
      return false;
   }
   else
   if (isExecutingAsDialog())
   {
      return sendUserMessage(msg);
   }
   else
   {
      return false;
   }
}

bool GWindow::sendDialogMessage ( GDialogMessageHandler::MessageID id, GObject* param1, GObject* param2 )
{
   GDialogPanel& dlg = dynamic_cast<GDialogPanel&>(*this);
   GDialogMessage msg(dlg, id, param1, param2);
   return sendDialogMessage(msg);
}

bool GWindow::sendDialogMessage ( GDialogMessageHandler::MessageID id, const GString& str1, GObject* param2 )
{
   GDialogPanel& dlg = dynamic_cast<GDialogPanel&>(*this);
   GDialogMessage msg(dlg, id, str1, param2);
   return sendDialogMessage(msg);
}

bool GWindow::sendDialogMessage ( GDialogMessageHandler::MessageID id, const GString& str1, const GString& str2 )
{
   GDialogPanel& dlg = dynamic_cast<GDialogPanel&>(*this);
   GDialogMessage msg(dlg, id, str1, str2);
   return sendDialogMessage(msg);
}

/**
 * Make the dialog panel ready for execution.
 *
 * This method is public as an implementation side effect.
 * It is to be called internally by the G-library only.
 *
 * It will be called from {@link GDialogFrame#doExecute} to let us 
 * perform all the required initializations when we are being used 
 * as a dialog panel.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @param   initArg   Object to send with GM_INITDIALOG in parameter #1.
 *                    Typically, this is a null reference.
 * @param   breakSem  A pointer to a boolean variable that acts as 
 *                    a break semaphore. That is, if this variable 
 *                    is changed to true then the executiuon 
 *                    preparation will break it self and return false
 *                    immediately.
 * @return  True if everything performed OK, or else false on any error.
 *          If we return false then the dialog will not be executed.
 *          We will return false if {@link #dismiss} is called from any
 *          of the components of the dialog, but we will return true if
 *          {@link #dismiss} was called by the message handler of the
 *          top level dialog panel.
 */
bool GWindow::prepareForExecution ( GObject* initArg, bool* breakSem )
{
   // Prevent all but <i>WM_CREATE</i> from being sent to the user
   // registered message handler until the child components of ours has
   // been initialized with <i>GM_INITDIALOG</i>.
   filterDlgMessages = true;

   // Send the <i>GM_INITDIALOG</i> to the Components of the window.
   int num = getComponentCount();
   for (int i=0; i<num; i++)
   {
      GWindow& comp = getComponentByIndex(i);
      if (!comp.prepareForExecution(initArg, breakSem))
         return false; // The Component did call dismiss() during message.
   }

   filterDlgMessages = false;

   // Send <i>DM_INITDIALOG</i> to the user message handler of this
   // Dialog Panel.
   hasSent_GM_INITDIALOG = true;
   if (dynamic_cast<GDialogPanel*>(this) != null)
   {
      isIn_DmInitDialog = true;
      sendDialogMessage(GDialogMessageHandler::GM_INITDIALOG, initArg);
      isIn_DmInitDialog = false;
      if (*breakSem)
      {
         if (getOwnerDialogPanel() == null) // If this is a top level dialog.
            return true;
         return false; // Program did call dismiss() during GM_INITDIALOG.
      }

      // Send an initial <i>DM_CTRLCHANGED</i> for each none-static
      // component, to the user program registered message handler.
      if (!noInitCtrlChanged)
      {
         num = getComponentCount();
         for (int i=0; i<num; i++)
         {
            GWindow& comp = getComponentByIndex(i);
            if (!comp.isStatic())
               comp.sendCtrlChanged();
         }
      }
   }

   return true;
}

/**
 * Close and terminate the dialog. This will cause both
 * <i>GM_DISMISSDIALOG</i> and <i>WM_CLOSE</i> to be sent.
 *
 * The <i>GM_DISMISSDIALOG</i> message will be sent to the user defined
 * message handler when all parts of the Dialog Panel is still intact and
 * the dialog is still visible (if not already hidden by your code).
 * Then we will be send the <i>GM_DISMISSDIALOG</i> message to all Dialog
 * Controls that are contained in the Dialog Panel.
 *
 * When all Components has processed the <i>GM_DISMISSDIALOG</i> message
 * then the message <i>WM_CLOSE</i> will be sent to the program defined
 * message handler and the Frame Window (the owner Dialog Box of the Dialog
 * Panel) will finally be closed. Only the top-level Dialog Panel will
 * receive the <i>WM_CLOSE</i> message because this message is intended to
 * be associated with the event of closing a Frame Window only, which makes
 * sense only for top-level Dialog Panels.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.26
 * @param   arg  The object of which to be returned by <i>getDismissArgument()</i>.
 * @see     #getDismissArgument
 * @see     #executeModeless
 * @see     GDialogFrame#dismiss
 * @see     GDialogMessageHandler#GM_DISMISSDIALOG
 */
void GWindow::dismiss ( const GString& arg )
{
   GWindow& top = getTopLevelWindow();
   if (!top.isInDismiss)
   {
      // The dismiss operation must start in the Frame Dialog Box Window.
      // It will recursively call back the <i>dismiss()</i> method in
      // the Dialog Panel object of ours.
      isInDismiss = true;
      top.dismiss(arg);
      isInDismiss = false;
      return;
   }

   // Send the DM_DISMISSDIALOG message to the user program code when all
   // parts of the Dialog Panel are still intact.
   if (dynamic_cast<GDialogPanel*>(this) != null)
      sendDialogMessage(GDialogMessageHandler::GM_DISMISSDIALOG, arg);

   // All contained components must also be dismissed.
   const int num = getComponentCount();
   for (int i=0; i<num; i++)
   {
      try {
         GWindow& comp = getComponentByIndex(i);
         comp.dismiss(arg);
      } catch (GNoSuchComponentException& /*e*/) {
         continue; // Probably one of the "Low Level" wrapped components.
      }
   }

   // Now the Dialog Pane is to be treated as uninitialized again.
   // This is to make it ready to be eventually executed again.
   hasSent_GM_INITDIALOG = false;
}

/**
 * This method will be used by the owner Dialog Panel when it needs to know
 * which Hot Key character is associated with the Component.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.27
 * @return  The Hot Key character of the Component, or 0 if no Hot Key has
 *          been associated with the Component.
 * @see     #autoAssocAHotKey
 */
char GWindow::getHotKeyChar () const 
{ 
   return hotChar; 
}

/**
 * Auto assign a Hot Key to the Component. The Hot Key will be the first
 * character of the specified text that isn't already in use as a Hot Key
 * of any other Component in the same Dialog Panel.
 *
 * A Hot Key is a keyboard scancode that the user can press to directly
 * activate the associated Component at any time. The scancode will be
 * a character in combination with the <b><i>alt</i></b> key. Most Components
 * that has been associated to such a Hot Key will make the associated key
 * visible for instance by painting the character with an underlined style.
 * It is the Component it self that is responsible for doing this.
 *
 * If we find that the specified control already has a registered Hot Key
 * then we will automatically unregister the Hot Key before trying to
 * re-associate a Hot Key to the same control. This feature is nice when
 * the control need to be associated by another Hot Key, e.g. if the
 * control has been given a new text. Thus, this method can be called by
 * the control at any time, both upon initialization and upon update due
 * to text change.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.27
 * @param   text    The control text, of where to find an unused hotkey.
 * @return  Index of hotkey within <code>text</code>, or -1 if none.
 * @see     #getHotKeyChar
 */
int GWindow::autoAssocAHotKey ( const GString& text )
{
   GDialogPanel* odlg = getOwnerDialogPanel();
   if (odlg == null)
      return -1; // Not part of a dialog, so hotkeys are not supported here!

   // ---
   int idx = odlg->autoAssocCtrlToAHotKey(*this, text);
   if (idx >= 0)
      hotChar = GCharacter::ToUpperCase(text[idx]);
   else
      hotChar = 0;
   return idx;
}

/**
 * Change the user defined message handler of the dialog.
 *
 * @author  Leif Erik Larsen
 * @since   2004.08.30
 * @return  The previous message handler interface pointer.
 */
GDialogMessageHandler* GWindow::setDialogMessageHandler ( GDialogMessageHandler* msgProc )
{
   GDialogMessageHandler* old = userMsgProc;
   userMsgProc = msgProc;
   return old;
}
