/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_COMMANDMAP
#define __GLIB_COMMANDMAP

#include "glib/gui/GMenuPopupItem.h"
#include "glib/primitives/GInteger.h"
#include "glib/util/GLog.h"

/**
 * This class is only used to form the base of every instance
 * of the GMenuCommandBag<T> template.
 *
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
class GMenuCommandImpl : public GAbstractCommand
{
   protected:

      GMenuCommandImpl ( const GString& cmdID );
      virtual ~GMenuCommandImpl ();
};

/**
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
template <class T> class GMenuCommand : public GMenuCommandImpl
{
   public:

      typedef void (T::*GCommandMethod)(GAbstractCommand* cmd);
      GCommandMethod function;
      GMenuCommand ( const GString& cmdID, GCommandMethod function ) : GMenuCommandImpl(cmdID), function(function) {}
      virtual ~GMenuCommand () {}
};

/**
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
class GMenuCommandBagImpl : public GObject
{
   protected:

      GMenuCommandBagImpl ();
      virtual ~GMenuCommandBagImpl ();

      /** Close the tooltip (if visible). */
      void closeTooltip () const;
};

/**
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
template <class T> class GMenuCommandBag : public GMenuCommandBagImpl
{
   private:

      typedef void (T::*GCommandMethod)(GAbstractCommand* cmd);
      GKeyBag<GMenuCommand<T> > commands;

   public:

      GMenuCommandBag () : commands(8) {}
      virtual ~GMenuCommandBag () {}

   public:

      int getCommandCount () const
      { 
         return commands.getCount(); 
      }

      void addCommand ( const GString& cmdID, GCommandMethod func )
      {
         if (commands.containsKey(cmdID))
         {
            GLog::Log(this, "%s: Command already defined: \"%s\"", GVArgs(__FUNCTION__).add(cmdID));
         }
         else
         {
            GMenuCommand<T>* cmd = new GMenuCommand<T>(cmdID, func);
            commands.put(cmdID, cmd);
         }
      }

      bool executeCommand ( T* ownerObj, GAbstractCommand* cmd )
      {
         const GString& cmdID = cmd->getIDString();
         GLog::Log(this, "Executing command '%s' in command-map for Window '%s'", GVArgs(cmdID).add(ownerObj->getName()));
         GMenuCommand<T>* menuCommand = commands.get(cmdID);
         if (menuCommand == null)
            return false;
         closeTooltip();
         GCommandMethod function = menuCommand->function;
         (ownerObj->*function)(cmd);
         return true;
      }

      GMenuCommand<T>* getCommand ( const GString& cmdID )
      {
         return commands.get(cmdID);
      }
};

/**
 * @author  Leif Erik Larsen
 * @since   2001.03.07
 */
class GCommandMap : public GObject
{
   private:

      GKeyBag<GMenuPopupItem> mapBag;
      GKeyBag<GInteger> idMapByIDString;

   public:

      GCommandMap ();
      virtual ~GCommandMap ();

   public:

      void addCommand ( int id, GMenuPopupItem* item );
      GMenuPopupItem* getCommand ( int id );

      /**
       * Get the numerical ID that is used internally by the underlying OS
       * to identify the specified high level command ID string.
       */
      int getCommandMapID ( const char* commandID );
};

// ----
#define DECLARE_COMMAND_TABLE(className) \
        private: \
           GMenuCommandBag<className> __commandBag; \
           typedef GMenuCommand<className>::GCommandMethod GCommandMethod; \
           typedef className ThisClass; \
        public: \
           /** @return The specified command object, or null if it is unknown. */ \
           GAbstractCommand* getCommand ( const GString& cmdID ) \
           { \
              return __commandBag.getCommand(cmdID); \
           } \
        protected: \
            bool __executeAbstractCommand ( GAbstractCommand* cmd )
// ----
#define DEFINE_COMMAND_TABLE(className) \
        bool className::__executeAbstractCommand ( GAbstractCommand* cmd ) \
        { \
           if (__commandBag.getCommandCount() <= 0) \
           {
// ----
#define ADD_COMMAND(commandName, method) \
              __commandBag.addCommand(commandName, &ThisClass::method)
// ----
#define END_COMMAND_TABLE \
           } \
           return __commandBag.executeCommand(this, cmd); \
        }

#endif // #ifndef __GLIB_COMMANDMAP

