/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DIALOGMESSAGEHANDLER
#define __GLIB_DIALOGMESSAGEHANDLER

#include "glib/primitives/GObject.h"

/**
 * An interface class that declares the required message handler method
 * for all user defined message handlers that can be registered
 * on a {@link GDialogPanel}.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.26
 * @see     GDialogPanel#setDialogMessageHandler
 */
class GDialogMessageHandler
{
   public:

      enum MessageID
      {
         /**
          * This message is sent when the Dialog Panel is about to be executed,
          * but <i>before</i> the panel is actually visible on the screen.
          * The typical usage of this message is to perform some dialog
          * or application initializaton.
          *
          * From the application point of view, this is usually the very first
          * message to handle during the lifetime of a Dialog Panel. On the other
          * hand, {@link #GM_DISMISSDIALOG} is usually the very last.
          *
          * When this message is received you can perform almost all the supported
          * operations on the panel, such as setting its position, size and
          * other attributes. It is nice to perform such initializations before the
          * panel is visible, so that the user sees it only when it has been
          * properly set up by the application.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains a reference to the application defined object as was
          *     eventually given (as the <i>initArg</i> parameter) to
          *     {@link IDialogFrame#executeModal(no.cba.ica.gui.IWindowPanel, java.lang.Object)}.
          *     This parameter is typically not used, though.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see #GM_DISMISSDIALOG
          * @see #GM_QUERYPROFILE
          * @see GDialogFrame#executeModal
          * @see GProgram#executeDialog
          */
         GM_INITDIALOG = 1,

         /**
          * This message is sent to the Dialog Panel when it has been dismissed.
          * That is when it is about being closed. This message is sent when the
          * Dialog Panel is not visible on screen. The typical usage of this
          * message is to perform some cleanup operations.
          *
          * From the application point of view, this is usually the very last
          * message to handle during the lifetime of a Dialog Panel. On the other
          * hand, {@link #GM_INITDIALOG} is usually the very first.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains a pointer to the program defined dismiss-argument as was
          *     eventually given to {@link GDialogPanel#dismiss}. This is a
          *     pointer to an instance of <i>GObject</i>.
          *     This parameter is typically used as something like a "return value"
          *     from a modal dialog, such as for instance the ID of which command
          *     that caused the Dialog to be dismissed. The program is, however,
          *     free to use this parameter to whatever it want.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see #GM_INITDIALOG
          * @see #GM_WRITEPROFILE
          * @see GDialogPanel#dismiss
          */
         GM_DISMISSDIALOG,

         /**
          * This message is sent whenever the editable value of a Component within
          * the Dialog Panel has been changed. The message is sent synchronosuly,
          * and it can be due to a change that has been done by the user or
          * programatically by a call to {@link IDialogPanel#setComponentValue}
          * or any other method that caused the editable value of the Component
          * to change.
          *
          * It is the Component it self that actually sends the message. Thus,
          * the meaning of the message and whether it is sent or not is up to
          * the Component.
          *
          * Components send this message to its owner Dialog Panel only.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains a string that is the ID of the Component of which value
          *     has been changed. The program can assume that this string will
          *     never be null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Contains a reference to the "new value" of the Component.
          *     What this object means, and what type it is, is defined by the
          *     Component it self. See the documentation of the corresponding
          *     Component Class for more information. <b>Mark</b> that this 
          *     parameter will be empty if the component window was created 
          *     with the {@link GWindow#WS2_DONT_INCLUDE_VALUE_IN_CTRL_CHANGE_MSG}
          *     flag.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see GDialogPanel#setComponentValue
          * @see GWindow#changeValue
          * @see GWindow#WS2_DONT_INCLUDE_VALUE_IN_CTRL_CHANGE_MSG
          */
         GM_CTRLCHANGED,

         /**
          * This message is sent whenever a Component within the Dialog Panel is
          * about to loose the keyboard focus.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains a string that is the ID of the Component that is about
          *     to loose the focus.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Contains a string that is the ID of the Component that will soon
          *     receive the keyboard focus by default.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see #GM_FOCUSSET
          * @see GDialogPanel#setComponentFocus
          */
         GM_FOCUSLOSS,

         /**
          * This message is sent whenever a Component within the Dialog Panel
          * has received the keyboard focus.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains a string that is the ID of the Component that has
          *     received the keyboard focus.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Contains a string that is the ID of the Component that has
          *     recently lost the keyboard focus.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see #GM_FOCUSLOSS
          * @see GDialogPanel#setComponentFocus
          */
         GM_FOCUSSET,

         /**
          * This message can be used by various Component classes to represent a
          * notification to its owner Dialog Panel.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains a string that is the ID of the Component sending the
          *     notification.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Contains the Component defined notification ID as an instance
          *     of {@link java.lang#Integer}. The meaning, and the value, of
          *     this integer depends on the Component implementation. As an example,
          *     see {@link no.cba.ica.gui.dlg.DATABROWSER#DN_FILLERTHREADHASBEGUN}.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          */
         GM_CTRLNOTIFY,

         /**
          * When this message occurs is the best time for the program to call
          * {@link IDialogPanel#queryProfile}, if the program want the Dialog to
          * automatically restore it self to a state close to how it was when
          * {@link IDialogPanel#writeProfile} was last called on the same dialog.
          *
          * This message is sent to the top level Dialog Panel only. It will be one
          * of the first messages sent just after {@link #GM_INITDIALOG} has been
          * processed. It will be sent just before the dialog is made visible by
          * default (if the program didn't show the dialog manually upon
          * {@link #GM_INITDIALOG}).
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see     #GM_WRITEPROFILE
          * @see     GDialogPanel#queryProfile
          * @see     GDialogPanel#writeProfile
          */
         GM_QUERYPROFILE,

         /**
          * When this message occurs is the best time for the program to call
          * {@link IDialogPanel#writeProfile}, if the program want the Dialog to
          * automatically write it self to a profile so that it can be restored
          * by a future call to {@link IDialogPanel#queryProfile}.
          *
          * This message is sent to the top level Dialog Panel only. It will be the
          * last message sent just before {@link #GM_DISMISSDIALOG} is sent.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see     #GM_QUERYPROFILE
          * @see     GDialogPanel#queryProfile
          * @see     GDialogPanel#writeProfile
          */
         GM_WRITEPROFILE,

         /**
          * This message is sent when the user has requested a command by some
          * means. This can be because of several events: 1) User has selected
          * a menu command, 2) user has pressed an accelerator key on keyboard,
          * 3) user has pressed a toolbar button, 4) user has clicked on a
          * pushbutton in a dialog panel or 5) the program code has posted or
          * sent a GM_COMMAND message programatically.
          * <p>
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Contains an instance of a <i>GString</i> that is the ID of the
          *     command. The ID string is as defined by the <i>ID</i> parameter
          *     of the corresponding command in the Resource Script. This can
          *     be the ID of a menu item, push button, toolbar button,
          *     accelerator key, etc.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          */
         GM_COMMAND,

         /**
          * This message is sent to the top level Window Panel when the user
          * has requested to close the Owner Window Frame by selecting the
          * system dependent close command. On Windows and OS/2 this is the
          * window close button, the close command in the window system
          * menu or pressing ALT+F4.
          * <p>
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>If the dialog has an escape/cancel pushbutton then the
          *     default message handler of the dialog will post a
          *     GM_COMMAND message just as if the user has clicked on
          *     that button.
          *     </TD>
          * </TR>
          * </TABLE>
          */
         GM_CLOSEWIN,

         /**
          * This message is sent to the parent dialog panel when the user
          * has double clicked on a list box item.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>The ID of the listbox component.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>The index of the current selected item of which was double
          *     clicked.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          */
         GM_LISTBOX_DBLCLICK,

         /**
          * This message is the message used to represent a program defined
          * timer event. Take a look at {@link GWindow#startTimer} for
          * more information about program defined timers.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>A {@link GString} that contains the program defined
          *     timer ID as specified to {@link GWindow#startTimer}.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>A pointer that is the same as was given to the userData 
          *     parameter of {@link GWindow#startTimer}, which is null
          *     by default.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>Null.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          *
          * @see  GWindow#startTimer
          */
         GM_TIMER,

         /**
          * User defined message.
          *
          * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
          *
          * <TR>
          * <TD><B><I>Param1</I></B></TD>
          * <TD>The user program defined message ID string.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param2</I></B></TD>
          * <TD>User program defined.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Param3</I></B></TD>
          * <TD>User program defined.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Answer</I></B></TD>
          * <TD>Dummy.
          *     </TD>
          * </TR>
          *
          * <TR>
          * <TD><B><I>Default</I></B></TD>
          * <TD>Nothing.
          *     </TD>
          * </TR>
          * </TABLE>
          */
         GM_USER
      };

   public:

      virtual bool handleDialogMessage ( class GDialogMessage& msg ) = 0;
};

#endif


