/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/event/GUserMessage.h"
#include "glib/primitives/GBoolean.h"
#include "glib/primitives/GInteger.h"
#include "glib/util/GLog.h"

GUserMessage::GUserMessage ( GObject* param1, 
                             GObject* param2,
                             GObject* param3 )
               :param1(param1),
                param2(param2),
                param3(param3),
                str1(null),
                str2(null),
                autoDeleteParam1(false),
                autoDeleteParam2(false),
                autoDeleteParam3(false)
{
}

GUserMessage::GUserMessage ( const GString& str1, 
                             GObject* param2,
                             GObject* param3 )
               :param1(null),
                param2(param2),
                param3(param3),
                str1(null),
                str2(null),
                autoDeleteParam1(false),
                autoDeleteParam2(false),
                autoDeleteParam3(false)
{
   this->str1 = new GString(str1);
   this->param1 = this->str1;
}

GUserMessage::GUserMessage ( const GString& str1, 
                             const GString& str2,
                             GObject* param3 )
               :param1(null),
                param2(null),
                param3(param3),
                str1(null),
                str2(null),
                autoDeleteParam1(false),
                autoDeleteParam2(false),
                autoDeleteParam3(false)
{
   this->str1 = new GString(str1);
   this->str2 = new GString(str2);
   this->param1 = this->str1;
   this->param2 = this->str2;
}

GUserMessage::~GUserMessage ()
{
   delete str1;
   delete str2;
   if (autoDeleteParam1)
      delete param1;
   if (autoDeleteParam2)
      delete param2;
   if (autoDeleteParam3)
      delete param3;
}

void GUserMessage::setAutoDeleteParam1 ( bool autoDel )
{
   autoDeleteParam1 = autoDel;
}

void GUserMessage::setAutoDeleteParam2 ( bool autoDel )
{
   autoDeleteParam2 = autoDel;
}

void GUserMessage::setAutoDeleteParam3 ( bool autoDel )
{
   autoDeleteParam3 = autoDel;
}

GObject* GUserMessage::getParam1 ()
{
   return param1;
}

GObject* GUserMessage::getParam2 ()
{
   return param2;
}

GObject* GUserMessage::getParam3 ()
{
   return param3;
}

int GUserMessage::getParam1Int ( int def ) const
{
   try {
      GString defStr = GInteger::ToString(def);
      GString str = getParam1String(defStr);
      return GInteger::ParseInt(str);
   } catch (GNumberFormatException& /*e*/) {
      return def;
   }
}

int GUserMessage::getParam2Int ( int def ) const
{
   try {
      GString defStr = GInteger::ToString(def);
      GString str = getParam2String(defStr);
      return GInteger::ParseInt(str);
   } catch (GNumberFormatException& /*e*/) {
      return def;
   }
}

GString GUserMessage::getParam1String ( const GString& def ) const
{
   if (str1 != null)
      return *str1;
   else
   if (param1 == null)
      return def;
   else
      return param1->toString();
}

GString GUserMessage::getParam2String ( const GString& def ) const
{
   if (str2 != null)
      return *str2;
   else
   if (param2 == null)
      return def;
   else
      return param2->toString();
}

bool GUserMessage::getParam1Bool ( bool def ) const
{
   GString str = getParam1String();
   return GBoolean::ParseBoolean(str, def);
}

bool GUserMessage::getParam2Bool ( bool def ) const
{
   GString str = getParam2String();
   return GBoolean::ParseBoolean(str, def);
}

GString GUserMessage::toString () const
{
   const int maxSLen = 32;
   GString s1 = getParam1String();
   if (s1.length() > maxSLen)
   {
      s1.cutTailFrom(maxSLen);
      s1 += "...";
   }
   GString s2 = getParam2String();
   if (s2.length() > maxSLen)
   {
      s2.cutTailFrom(maxSLen);
      s2 += "...";
   }
   return GString("GWindow::GM_USERMESSAGE(param1='%s', param2='%s')", GVArgs(s1).add(s2));
}

