/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TREEPATH
#define __GLIB_TREEPATH

#include "glib/gui/GWindow.h"

/**
 * Represents a path to a node. A GTreePath is an array of {@link GTreeNode}s 
 * that are vended from a {@link GTreeModel}. The elements of the array are 
 * ordered such that the root is always the first element (index 0) of the 
 * array.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.04
 */
class GTreePath : public GObject
{

private:

   /** Path representing the parent, null if lastPathComponent represents the root. */
   const GTreePath* parentPath;

   /** Last path component, will never be null after successfull construction. */
   class GTreeNode* lastPathComponent;

public:

   /**
    * Constructs a GTreePath containing only a single element. This is
    * usually used to construct a GTreePath for the the root of the GTreeModel.
    * 
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   singlePath  A GTreeNode representing the path to a node.
    */
   GTreePath ( class GTreeNode& singlePath );

   /**
    * Constructs a path from an array of Objects, uniquely identifying 
    * the path from the root of the tree to a specific node, as returned
    * by the tree's data model.
    * 
    * The model is free to return an array of any Objects it needs to 
    * represent the path. The {@link GDefaultTreeModel} returns an array of 
    * {@link GTreeNode} objects. The first GTreeNode in the path is the root 
    * of the tree, the last GTreeNode is the node identified by the path.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   path A vector of GTreeNode references representing the 
    *               path to a node.
    */
   GTreePath ( const GArray<class GTreeNode>& path );

   /**
    * Copy constructor is fully supported by this class.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   GTreePath ( const GTreePath& src );

   /**
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   virtual ~GTreePath ();

   /**
    * Assignment operator is fully supported by this class.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   GTreePath& operator= ( const GTreePath& src );

public:

   /**
    * Returns the last component of this path. For a path returned 
    * by {@link GDefaultTreeModel} this will return an instance 
    * of {@link GTreeNode}.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  The Object at the end of the path.
    */
   class GTreeNode& getLastPathComponent ();

   /**
    * Returns the number of elements in the path.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  The number of node elements in the path.
    */
   int getPathCount () const;

   /**
    * Returns the path component at the specified index.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @param   idx  The index of which element component to get.
    * @return  A reference to the indexed path element.
    * @throws  GIllegalArgumentException if the index is beyond the length 
    *                                    of the path.
    */
   class GTreeNode& getPathComponent ( int idx );

   /**
    * Returns true if <code>aTreePath</code> is a descendant of this 
    * GTreePath. A GTreePath P1 is a descendent of a GTreePath P2
    * if P1 contains all of the components that make up P2's path.
    * For example, if this object has the path [a, b],
    * and <code>aTreePath</code> has the path [a, b, c], 
    * then <code>aTreePath</code> is a descendant of this object.
    * However, if <code>aTreePath</code> has the path [a],
    * then it is not a descendant of this object.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    * @return  True if <code>aTreePath</code> is a descendant of this path.
    */
   bool isDescendant ( const GTreePath& aTreePath ) const;

   /**
    * Returns a path containing all the elements of this object, 
    * except the last path component, or null if this is the root node.
    *
    * @author  Leif Erik Larsen
    * @since   2006.01.04
    */
   const GTreePath* getParentPath () const;
};

#endif
