/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_INTEGERNUMBER
#define __GLIB_INTEGERNUMBER

#include "glib/primitives/GNumber.h"
#include "glib/primitives/GCharacter.h"
#include "glib/primitives/GVArgs.h"
#include "glib/exceptions/GNumberFormatException.h"

/**
 * This is the abstract base class for every class that is
 * used to wrap a primitive integer number, such as e.g. GShort,
 * GInteger and GLong.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.23
 */
template <class T> class GIntegerNumber : public GNumber
{
   protected:

      GIntegerNumber () {}
      GIntegerNumber ( const GIntegerNumber<T>& src ) : GNumber(src) {}

      virtual ~GIntegerNumber () {}

   public:

      /**
       * Used by {@link #compareObj} to determine if comparization should 
       * be signed or unsigned. This default implementation returns false,
       * but unsigned number classes such as {@link GUInteger},
       * {@link GULong} and {@link GUShort} overrides this method 
       * in order to return true.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       * @see     #compareObj
       */
      virtual bool isUnsigned () const { return false; }

      /**
       * A default implementation of object comparization that 
       * works on all subclasses of {@link GIntegerNumber}. It uses the
       * values returned by {@link GNumber#ulongValue} or 
       * {@link GNumber#longValue}, if {@link #isUnsigned} returns 
       * true or false respectively.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       * @param   obj  The object of which to compare to.
       *               Must be an instance of {@link GNumber}.
       * @see     GArrayImpl#compare2Objects
       * @see     GComparator
       */
      virtual int compareObj ( const GObject& obj ) const
      {
         const GNumber* num = dynamic_cast<const GNumber*>(&obj);
         if (num == null)
            return 1; // Don't know how to compare, so return "this < obj".
         if (isUnsigned())
         {
            ulonglong v1 = ulongValue();
            ulonglong v2 = num->ulongValue();
            if (v1 < v2)
               return -1;
            if (v1 > v2)
               return +1;
            return 0;
         }
         else
         {
            longlong v1 = longValue();
            longlong v2 = num->longValue();
            if (v1 < v2)
               return -1;
            if (v1 > v2)
               return +1;
            return 0;
         }
      }

   protected:

      /**
       * Used by GShort, GInteger, GLong and possibly others
       * to parse a string into a signed short, int or longlong, respectively.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.23
       * @param   str       String of which to parse.
       * @param   minValue  Minimum value.
       * @param   maxValue  Maximum value.
       * @param   radix     0, 2, 8, 10 or 16. If 0 is specified then we
       *                    will use the default radix as with respect to
       *                    the format of the string; assuming that the
       *                    string is specifying an integer number with
       *                    radix 10, or radix 16 if the string starts
       *                    with the character sequence "0x".
       * @return  The parsed integer value.
       * @throws  GNumberFormatException if the specified string is not
       *                                 possible to parse into an integer
       *                                 of type T.
       */
      static T ParseSigned ( const GString& str, 
                             const T minValue, 
                             const T maxValue, 
                             int radix = 0 )
      {
         int i = 0;

         if (radix == 0)
         {
            if (str.length() >= 2 && 
                str[0] == '0' && 
                GCharacter::ToLowerCase(str[1]) == 'x')
            {
               radix = 16;
               i = 2;
            }
            else
            {
               radix = 10;
            }
         }

         if (radix < 2 || radix > 16)
            gthrow_(GNumberFormatException(GString("Unsupported radix: %d", GVArgs(radix))));

         // ---
         T result = 0;
         int max = str.length();
         bool negative = false;

         if (max > 0)
         {
            T limit;
            T multmin;

            if (str[0] == '-')
            {
               negative = true;
               limit = minValue;
               i++;
            }
            else
            {
               limit = T(-maxValue);
            }
            multmin = T(limit / radix);
            if (i < max)
            {
               int digit = GCharacter::Digit(str[i++]);
               if (digit < 0)
                  gthrow_(GNumberFormatException(str));
               result = T(-digit);
            }
            while (i < max)
            {
               // Accumulating negatively avoids surprises near MAX_VALUE.
               int digit = GCharacter::Digit(str[i++]);
               if (digit < 0 || digit >= radix || result < multmin)
                  gthrow_(GNumberFormatException(str));
               result = T(result * radix);
               if (result < limit + digit)
                  gthrow_(GNumberFormatException(str));
               result = T(result - digit);
            }
         }
         else
         {
            gthrow_(GNumberFormatException(str));
         }

         if (negative)
         {
            if (i > 1)
               return result;
            else
               gthrow_(GNumberFormatException(str));
         }
         else
         {
            return T(-result);
         }
      };

      /**
       * Used by GUShort, GUInteger, GULong and possibly others
       * to parse a string into an unsigned short, unsigned int or
       * ulonglong, respectively.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.23
       * @param   str       String of which to parse.
       * @param   maxValue  Maximum value.
       * @param   radix     0, 2, 8, 10 or 16. If 0 is specified then we
       *                    will use the default radix as with respect to
       *                    the format of the string; assuming that the
       *                    string is specifying an integer number with
       *                    radix 10, or radix 16 if the string starts
       *                    with the character sequence "0x".
       * @return  The parsed unsigned integer value.
       * @throws  GNumberFormatException if the specified string is not
       *                                 possible to parse into an unsigned
       *                                 integer of type T.
       */
      static T ParseUnsigned ( const GString& str, 
                               const T maxValue, 
                               int radix = 0 )
      {
         int i = 0;

         if (radix == 0)
         {
            if (str.length() >= 2 && 
                str[0] == '0' && 
                GCharacter::ToLowerCase(str[1]) == 'x')
            {
               radix = 16;
               i = 2;
            }
            else
            {
               radix = 10;
            }
         }

         if (radix < 2 || radix > 16)
            gthrow_(GNumberFormatException(GString("Unsupported radix: %d", GVArgs(radix))));

         T res = 0;
         ulonglong base = 1;
         const int len = str.length();
         for (; i<len; i++)
         {
            char chr = str[len-i-1];
            int digit;
            if (chr >= '0' && chr <= '9')
               digit = int(chr - '0');
            else
            {
               chr = GCharacter::ToUpperCase(chr);
               if (chr >= 'A' && chr <= 'F')
                  digit = 9 + int(chr - 'A');
               else
                  gthrow_(GNumberFormatException(str));
            }
            if (digit >= radix)
               gthrow_(GNumberFormatException(str));
            ulonglong tmp = res + digit * base;
            if (tmp > maxValue)
               gthrow_(GNumberFormatException(str));
            res = T(tmp);
            if (i == 0)
               base = ulonglong(radix);
            else
               base *= radix;
         }

         return res;
      };

      /**
       * Used by GShort, GInteger, GLong and possibly others
       * to convert a signed integer value into a string.
       *
       * @author  Leif Erik Larsen
       * @since   2000.11.01
       * @param   value   The value to convert into a string.
       * @param   radix   2, 8, 10 or 16.
       * @param   width   Width of the returned string. A zero value means 
       *                  that we will not pad the string with any additional
       *                  <i>padChar</i>'s at all.
       * @param   padChar If the value it self is not large enough to produce
       *                  a string that is at least <i>width</i> characters
       *                  long then we will insert a number of this character
       *                  to satisfy the requested width.
       */
      static GString ToSignedString ( T value, 
                                      int radix = 10, 
                                      int width = 0, 
                                      char padChar = '0' )
      {
         if (radix < 2 || radix > 16)
            gthrow_(GNumberFormatException(GString("Unsupported radix: %d", GVArgs(radix))));

         // ---
         GString dest(16);
         bool isnegative = false;

         if (value < 0)
         {
            isnegative = true;
            width -= 1; // Make room for the minus sign.
         }

         do
         {
            T digit = T(value % radix);
            char chr = char('0' + (digit < 0 ? -digit : digit));
            if (chr > '9')
               chr += char(7); // Convert to alfanumeric (hex, etc.).
            dest += chr;
         }
         while ((value = T(value / radix)) != 0);

         while (dest.length() < width)
            dest += padChar;

         if (isnegative)
            dest += '-';

         dest.reverse();
         return dest;
      };

      /**
       * Used by GUShort, GUInteger, GULong and possibly others
       * to convert an unsigned integer value into a string.
       *
       * @author  Leif Erik Larsen
       * @since   2000.11.01
       * @param   value   The value to convert into a string.
       * @param   radix   2, 8, 10 or 16.
       * @param   width   Width of the returned string. A zero value means 
       *                  that we will not pad the string with any additional
       *                  <i>padChar</i>'s at all.
       * @param   padChar If the value it self is not large enough to produce
       *                  a string that is at least <i>width</i> characters
       *                  long then we will insert a number of this character
       *                  to satisfy the requested width.
       */
      static GString ToUnsignedString ( T value, 
                                        unsigned short radix = 10, 
                                        int width = 0, 
                                        char padChar = '0' )
      {
         if (radix < 2 || radix > 16)
            gthrow_(GNumberFormatException(GString("Unsupported radix: %d", GVArgs(radix))));

         // ---
         GString dest(16);

         do
         {
            char chr = char('0' + (value % radix));
            if (chr > '9')
               chr += char(7); // Convert to alfanumeric (hex, etc.).
            dest += chr;
         }
         while ((value = T(value / radix)) > 0);

         while (dest.length() < width)
            dest += padChar;

         dest.reverse();
         return dest;
      };
};

#endif
