/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include "glib/primitives/GString.h"
#include "glib/primitives/GUInteger.h"
#include "glib/primitives/GInteger.h"
#include "glib/primitives/GULong.h"
#include "glib/primitives/GLong.h"
#include "glib/primitives/GBoolean.h"
#include "glib/primitives/GDouble.h"
#include "glib/primitives/GFloat.h"
#include "glib/primitives/GVArgs.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/GProgram.h"

GString::GString ()
        :theString(null),
         currentLength(0),
         allocatedSize(0),
         cachedHashCode(0)
{
}

GString::GString ( const GString& str )
        :theString(null),
         currentLength(str.currentLength),
         allocatedSize(currentLength),
         cachedHashCode(0)
{
   if (currentLength <= 0)
      return;
   theString = new char[allocatedSize + 1];
   memcpy(theString, str.theString, currentLength + 1);
}

GString::GString ( char chr )
        :theString(null),
         currentLength(1),
         allocatedSize(1),
         cachedHashCode(0)
{
   theString = new char[2];
   theString[0] = chr;
   theString[1] = '\0';
}

GString::GString ( const GString& str, int fromIndex, int count )
        :theString(null),
         currentLength(0),
         allocatedSize(0),
         cachedHashCode(0)
{
   int len = str.currentLength;
   if (len <= 0)
      return;
   if (fromIndex < 0)
      fromIndex = 0;
   else
   if (fromIndex >= len)
   {
      fromIndex = 0;
      count = 0;
   }
   if (count < 0)
      count = len - fromIndex;
   if (fromIndex + count > len)
      count = len - fromIndex;
   if (count <= 0)
      return;

   // ---
   currentLength = count;
   allocatedSize = count;
   theString = new char[allocatedSize + 1];
   memcpy(theString, str.theString + fromIndex, currentLength);
   theString[currentLength] = '\0';
}

GString::GString ( const char* str )
        :theString(null),
         currentLength(int(strlen(str))),
         allocatedSize(currentLength),
         cachedHashCode(0)
{
   if (currentLength <= 0)
      return;
   theString = new char[allocatedSize + 1];
   memcpy(theString, str, currentLength + 1);
}

GString::GString ( int allocs )
        :theString(null),
         currentLength(0),
         allocatedSize(allocs < 0 ? 0 : allocs),
         cachedHashCode(0)
{
   if (allocatedSize <= 0)
      return;
   theString = new char[allocatedSize + 1];
   theString[0] = '\0';
}

GString::GString ( const char* format, const GVArgs& args )
        :theString(null),
         currentLength(0),
         allocatedSize(0),
         cachedHashCode(0)
{
   int len = int(strlen(format));
   init(format, len, args);
}

GString::GString ( const GString& format, const GVArgs& args )
        :theString(null),
         currentLength(0),
         allocatedSize(0),
         cachedHashCode(0)
{
   int len = format.currentLength;
   init(format, len, args);
}

void GString::init ( const char* format, const int formatLen, const GVArgs& args )
{
   allocatedSize = formatLen + (16 * args.num());
   theString = new char[allocatedSize + 1];
   theString[0] = '\0';

   if (args.num() > 0)
   {
      GVArgs::FormatArgs(*this, format, formatLen, args);
   }
   else
   {
      // append(format);
      memcpy(theString, format, formatLen + 1);
      currentLength = formatLen;
   }
}

GStringl::GStringl ( const char* format, const GVArgs& args )
         :GString(GProgram::LoadText(format), args)
{
}

GStringl::GStringl ( const char* format )
         :GString(GProgram::LoadText(format))
{
}

GString::~GString ()
{
   delete [] theString;
}

int GString::hashCode () const
{
   int h = cachedHashCode;
   if (h == 0) 
   {
      for (int i=0, len=currentLength; i<len; i++) 
         h = 31*h + theString[i];
      cachedHashCode = h;
   }

   return cachedHashCode;
}

bool GString::equals ( const GObject& obj ) const
{
   if (&obj == this)
      return true;
   const GString* s = dynamic_cast<const GString*>(&obj);
   if (s == null)
      return false;
   return operator==(*s);
}

GString GString::toString () const
{
   return GString(*this);
}

GString& GString::clear ()
{
   if (theString != null)
      *theString = '\0';
   currentLength = 0;
   cachedHashCode = 0;
   return *this;
}

GString& GString::setCharAt ( int index, char chr )
{
   if (index < 0 || index >= currentLength)
      gthrow_(GArrayIndexOutOfBoundsException());
   theString[index] = chr;
   cachedHashCode = 0;
   return *this;
}

char GString::firstChar () const
{
   if (currentLength > 0)
      return theString[0];
   else
      return '\0';
}

char GString::lastChar () const
{
   if (currentLength > 0)
      return theString[currentLength-1];
   else
      return '\0';
}

GString& GString::operator= ( long strval )
{
   GString str = GLong::ToString(strval);
   return operator=(str);
}

GString& GString::operator= ( char chr )
{
   char str[2] = { chr, '\0' };
   return operator=(str);
}

GString& GString::operator= ( const char* str )
{
   cachedHashCode = 0;
   currentLength = int(strlen(str));
   if (currentLength > allocatedSize)
   {
      allocatedSize = currentLength;
      delete [] theString;
      theString = new char[allocatedSize + 1];
   }
   if (theString != null)
      memcpy(theString, str, currentLength + 1);
   return *this;
}

GString& GString::operator= ( const GString& str )
{
   if (&str == this)
      return *this;
   cachedHashCode = 0;
   currentLength = str.currentLength;
   if (currentLength > allocatedSize)
   {
      allocatedSize = currentLength;
      delete [] theString;
      theString = new char[allocatedSize + 1];
   }
   if (theString != null)
   {
      if (str.theString != null)
         memcpy(theString, str.theString, currentLength + 1);
      else
         theString[0] = '\0';
   }
   return *this;
}

char GString::operator[] ( int index ) const
{
   if (theString == null)
      return '\0';
   else
      return theString[index];
}

GString& GString::operator+= ( char chr ) 
{ 
   if (allocatedSize <= currentLength)
      enlargeTo(currentLength + 1);
   theString[currentLength++] = chr;
   theString[currentLength] = '\0';
   cachedHashCode = 0;
   return *this;
}

GString& GString::operator+= ( signed char chr ) 
{ 
   if (allocatedSize <= currentLength)
      enlargeTo(currentLength + 1);
   theString[currentLength++] = chr;
   theString[currentLength] = '\0';
   cachedHashCode = 0;
   return *this;
}

GString& GString::operator+= ( unsigned char chr ) 
{ 
   if (allocatedSize <= currentLength)
      enlargeTo(currentLength + 1);
   theString[currentLength++] = chr;
   theString[currentLength] = '\0';
   cachedHashCode = 0;
   return *this;
}

GString& GString::operator+= ( float val ) 
{ 
   GString str = GFloat::ToString(val);
   return operator+=(str);
}

GString& GString::operator+= ( double val ) 
{ 
   GString str = GDouble::ToString(val);
   return operator+=(str);
}

GString& GString::operator+= ( int val ) 
{ 
   GString str = GInteger::ToString(val);
   return operator+=(str);
}

GString& GString::operator+= ( const char* str ) 
{ 
   int len = int(strlen(str));
   if (len > 0)
   {
      ensureAllocatedSize(currentLength + len);
      char *ptr = theString + currentLength;
      memcpy(ptr, str, len + 1);
      currentLength += len;
   }
   cachedHashCode = 0;
   return *this;
}

GString& GString::operator+= ( const GString& str ) 
{ 
   if (&str == this) 
      return operator+=(GString(str)); 
   int len = str.currentLength;
   if (len > 0)
   {
      ensureAllocatedSize(currentLength + len);
      char *ptr = theString + currentLength;
      memcpy(ptr, str.theString, len + 1);
      currentLength += len;
   }
   cachedHashCode = 0;
   return *this;
}

GString& GString::operator+= ( bool val ) 
{ 
   if (val)
      return operator+=(GBoolean::TrueStr);
   else
      return operator+=(GBoolean::FalseStr);
}

GString& GString::operator+= ( unsigned val )
{
   GString str = GUInteger::ToString(val);
   return operator+=(str);
}

GString& GString::operator+= ( longlong val )
{
   GString str = GLong::ToString(val);
   return operator+=(str);
}

GString& GString::operator+= ( ulonglong val )
{
   GString str = GULong::ToString(val);
   return operator+=(str);
}

GString& GString::operator+= ( const GObject& obj ) 
{ 
   GString str = obj.toString();
   return operator+=(str);
}

GString GString::operator+ ( int val ) const 
{ 
   GString ret(10 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( unsigned val ) const
{
   GString ret(10 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( float val ) const 
{ 
   GString ret(12 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( double val ) const
{ 
   GString ret(12 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( longlong val ) const
{
   GString ret(10 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( ulonglong val ) const
{
   GString ret(10 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( char chr ) const 
{ 
   GString ret(1 + currentLength);
   ret += *this;
   ret += chr; 
   return ret;
}

GString GString::operator+ ( const char* str ) const 
{ 
   GString ret(32 + currentLength);
   ret += *this;
   ret += str; 
   return ret;
}

GString GString::operator+ ( const GString& str ) const 
{ 
   GString ret(str.currentLength + currentLength);
   ret += *this;
   ret += str; 
   return ret;
}

GString GString::operator+ ( bool val ) const 
{ 
   GString ret(5 + currentLength);
   ret += *this;
   ret += val; 
   return ret;
}

GString GString::operator+ ( const GObject& obj ) const 
{ 
   GString ret(32 + currentLength);
   ret += *this;
   ret += obj; 
   return ret;
}

bool GString::operator!= ( const char* str ) const 
{ 
   return !operator==(str); 
}

bool GString::operator!= ( const GString& str ) const 
{ 
   return !operator==(str); 
}

GString::operator const char* () const 
{ 
   return cstring(); 
}

bool GString::contains ( char chr ) const 
{ 
   return indexOf(chr) >= 0; 
}

bool GString::isEmpty () const 
{ 
   return currentLength <= 0; 
}

int GString::length () const 
{ 
   return currentLength; 
}

GString GString::substring ( int startIndex ) const 
{ 
   return substring(startIndex, currentLength); 
}

GString operator+ ( const char* str1, const GString& str2 ) 
{ 
   return GString(str1).operator+=(str2); 
}

GString operator+ ( const GObject& str1, const GString& str2 ) 
{ 
   return GString(str1.toString()).operator+=(str2); 
}

GString operator+ ( int str1, const GString& str2 ) 
{ 
   GString ret(10 + str2.length());
   ret += str1;
   ret += str2; 
   return ret;
}

GString operator+ ( float str1, const GString& str2 ) 
{ 
   GString ret(12 + str2.length());
   ret += str1;
   ret += str2; 
   return ret;
}

char GString::charAt ( int index ) const
{
   if (index < 0 || index > currentLength)
      gthrow_(GArrayIndexOutOfBoundsException());
   if (currentLength <= 0)
      return '\0';
   else
      return theString[index];
}

bool GString::operator== ( const char *str ) const
{
   if (theString != null)
      return strcmp(theString, str) == 0;
   else
      return str[0] == '\0';
}

bool GString::operator== ( const GString& str ) const
{
   if (this == &str)
      return true; // Same instance.
   if (currentLength != str.currentLength)
      return false;
   if (currentLength <= 0)
      return true;
   if (memcmp(theString, str.theString, currentLength) == 0)
      return true;
   return false;
}

bool GString::equalsString ( const char *str, bool ignoreCase ) const
{
   if (ignoreCase)
      return equalsIgnoreCase(str);
   else
      return operator==(str);
}

bool GString::equalsString ( const GString& str, bool ignoreCase ) const
{
   if (ignoreCase)
      return equalsIgnoreCase(str);
   else
      return operator==(str);
}

bool GString::equalsIgnoreCase ( const char *str ) const
{
   if (theString != null)
      return stricmp(theString, str) == 0;
   else
      return str[0] == '\0';
}

bool GString::equalsIgnoreCase ( const GString& str ) const
{
   if (currentLength != str.currentLength)
      return false;
   else
   if (currentLength <= 0)
      return true;
   else
   if (equalsIgnoreCase(str.theString))
      return true;
   else
      return false;
}

int GString::compare ( const char* str ) const
{
   if (theString != null)
      return strcmp(theString, str);
   else
   if (str[0] == '\0')
      return 0; // "this" equal "str"
   else
      return -1; // "this" is less than "str"
}

int GString::compare ( const GString& str ) const
{
   if (theString != null && str.theString != null)
      return strcmp(theString, str.theString);
   else
   if (theString == null && str.theString == null)
      return 0; // "this" equal "str" (both are empty)
   else
   if (theString == null)
      return -1; // "this" is less than "str"
   else
      return 1; // "this" is larger than "str"
}

int GString::compare ( const char* str, bool ignoreCase ) const
{
   if (ignoreCase)
      return compareIgnoreCase(str);
   else
      return compare(str);
}

int GString::compare ( const GString& str, bool ignoreCase ) const
{
   if (ignoreCase)
      return compareIgnoreCase(str);
   else
      return compare(str);
}

int GString::compareIgnoreCase ( const char* str ) const
{
   if (theString != null)
      return stricmp(theString, str);
   else
   if (str[0] == '\0')
      return 0; // "this" equal "str"
   else
      return -1; // "this" is less than "str"
}

int GString::compareIgnoreCase ( const GString& str ) const
{
   if (theString != null && str.theString != null)
      return stricmp(theString, str.theString);
   else
   if (theString == null && str.theString == null)
      return 0; // "this" equal "str" (both are empty)
   else
   if (theString == null)
      return -1; // "this" is less than "str"
   else
      return 1; // "this" is larger than "str"
}

int GString::compareObj ( const GObject& obj ) const
{
   const GString* str = dynamic_cast<const GString*>(&obj);
   if (str == null)
      return 1; // Don't know how to compare, so return "this < obj".
   else
      return compare(*str);
}

bool GString::equalsNum ( const char *str, int num ) const
{
   if (num <= 0)
      return true;
   else
   if (theString != null)
      return strncmp(theString, str, num) == 0;
   else
      return false;
}

bool GString::equalsNum ( const GString& str, int num ) const
{
   if (num <= 0)
      return true;
   else
   if (theString != null && str.theString != null)
      return equalsNum(str.theString, num);
   else
      return false;
}

bool GString::equalsIgnoreCaseNum ( const char *str, int num ) const
{
   if (num <= 0)
      return true;
   else
   if (theString != null)
      return strnicmp(theString, str, num) == 0;
   else
      return false;
}

bool GString::equalsIgnoreCaseNum ( const GString& str, int num ) const
{
   if (num <= 0)
      return true;
   else
   if (theString == null)
      return false;
   else
      return num <= str.currentLength && 
             equalsIgnoreCaseNum(str.theString, num);
}

const char* GString::cstring () const
{
   if (theString != null)
      return theString;
   else
      return "";
}

void GString::enlargeTo ( int length )
{
   char* oldPtr = theString;
   int prelloc = currentLength / 3; // Prealloc by a factor of 1/3.
   allocatedSize = length + prelloc;
   theString = new char[allocatedSize + 1];
   if (oldPtr != null)
      memcpy(theString, oldPtr, currentLength + 1);
   else
      theString[0] = '\0';
   delete [] oldPtr;
}

GString& GString::ensureAllocatedSize ( int length )
{
   if (allocatedSize < length)
      enlargeTo(length);
   return *this;
}

GString& GString::append ( char chr, int num )
{
   if (num <= 0)
      return *this;
   ensureAllocatedSize(currentLength + num);
   char *ptr = &theString[currentLength];
   currentLength += num;
   while (num--)
      *ptr++ = chr;
   *ptr = '\0';
   cachedHashCode = 0;
   return *this;
}

GString& GString::appendNum ( const char *str, int num )
{
   if (num > 0)
   {
      ensureAllocatedSize(currentLength + num);
      char *ptr = theString + currentLength;
      memcpy(ptr, str, num);
      currentLength += num;
      theString[currentLength] = '\0';
      cachedHashCode = 0;
   }
   return *this;
}

int GString::lastIndexOf ( char chr ) const
{
   if (currentLength <= 0)
      return -1;
   char *ptr = theString + currentLength;
   while (*ptr != chr)
      if (ptr-- == theString)
         return -1;
   return int(ptr - theString);
}

int GString::lastIndexOfAnyChar ( const GString& chrStr ) const
{
   if (currentLength <= 0)
      return -1;
   char *ptr = theString + currentLength;
   do {
      if (chrStr.indexOf(*ptr) >= 0)
         return int(ptr - theString);
   } while (ptr-- != theString);
   return -1;
}

int GString::indexOf ( const char *str, int startIndex ) const
{
   if (startIndex >= currentLength)
      return -1;

   if (startIndex < 0)
      startIndex = 0;

   if (theString == null)
   {
      if (str[0] == '\0' && startIndex == 0)
         return 0;
      else
         return -1;
   }

   char *ptr = theString + startIndex;
   const char *foundStr = strstr(ptr, str);
   if (foundStr == null)
      return -1;
   return int(foundStr - theString);
}

int GString::indexOf ( const GString& str, int startIndex ) const
{
   const char* ptr = str.theString;
   if (ptr == null)
      ptr = "";
   return indexOf(ptr, startIndex);
}

int GString::indexOf ( char chr, int startIndex ) const
{
   if (startIndex >= currentLength)
      return -1;

   if (theString == null)
      return -1;

   if (startIndex < 0)
      startIndex = 0;
   char *ptr = theString + startIndex;
   for (int index = startIndex; index < currentLength; ptr++, index++)
      if (*ptr == chr)
         return index;

   return -1;
}

int GString::indexOfAnyChar ( char chr1, char chr2, int startIndex ) const
{
   if (startIndex >= currentLength)
      return -1;

   if (theString == null)
      return -1;

   if (startIndex < 0)
      startIndex = 0;
   char *ptr = theString + startIndex;
   for (int index = startIndex; index < currentLength; ptr++, index++)
      if (*ptr == chr1 || *ptr == chr2)
         return index;

   return -1;
}

GString& GString::insert ( char chr, int index )
{
   if (index < 0 || index > currentLength)
      gthrow_(GArrayIndexOutOfBoundsException());
   ensureAllocatedSize(currentLength + 1);
   memmove(theString + index + 1, theString + index, currentLength - index + 1);
   theString[index] = chr;
   currentLength += 1;
   cachedHashCode = 0;
   return *this;
}

GString& GString::insert ( const char* str, int index )
{
   if (index < 0 || index > currentLength)
      gthrow_(GArrayIndexOutOfBoundsException());
   int len = int(strlen(str));
   if (len > 0)
   {
      ensureAllocatedSize(currentLength + len);
      memmove(theString + index + len, theString + index, currentLength - index + 1);
      memcpy(theString + index, str, len);
      currentLength += len;
   }
   cachedHashCode = 0;
   return *this;
}

GString& GString::insert ( const GString& str, int index )
{
   if (index < 0 || index > currentLength)
      gthrow_(GArrayIndexOutOfBoundsException());
   int len = str.currentLength;
   if (len > 0)
   {
      ensureAllocatedSize(currentLength + len);
      memmove(theString + index + len, theString + index, currentLength - index + 1);
      memcpy(theString + index, str.theString, len);
      currentLength += len;
   }
   cachedHashCode = 0;
   return *this;
}

bool GString::containsWhiteSpace () const
{
   char *ptr = theString;
   for (int i=0; i<currentLength; ptr++, i++)
      if (GCharacter::IsWhiteSpace(*ptr))
         return true;
   return false;
}

bool GString::ContainsWhiteSpace ( const char* str )
{
   while (*str)
      if (GCharacter::IsWhiteSpace(*str++))
         return true;
   return false;
}

bool GString::isMixedCase () const
{
   if (theString == null)
      return false;

   // Find first alphanumeric character in string
   const char *ptr = theString;
   for (; *ptr && !isalpha(*ptr); ptr++);

   // Test if any of the remaining alpanumeric characters has
   // another case than the first. In that case, return true.
   bool ret = false; // Until the opposite has been proven
   if (*ptr)
   {
      for (bool isupr1 = (isupper(*ptr) != 0); *ptr && !ret; ptr++)
      {
         if (isalpha(*ptr))
         {
            bool isupr2 = (isupper(*ptr) != 0);
            ret = (isupr2 != isupr1);
         }
      }
   }

   return ret;
}

GString& GString::cutTailFrom ( int index )
{
   if (index < 0 || index >= currentLength)
      return *this;

   theString[index] = '\0';
   currentLength = index;
   cachedHashCode = 0;
   return *this;
}

GString& GString::remove ( int index, int count )
{
   if (index < 0 || index >= currentLength || count <= 0)
      return *this;
   if (index + count >= currentLength)
      count = currentLength - index;
   int tailLen = currentLength - index - count;
   char* headPtr = &theString[index];
   const char* tailPtr = &theString[index + count];
   for (int i=0; i<tailLen; i++)
      *headPtr++ = *tailPtr++;
   *headPtr = '\0'; // Make sure there is a terminating null-character.
   currentLength -= count;
   cachedHashCode = 0;
   return *this;
}

GString& GString::removeCharAt ( int index )
{
   if (index < 0 || index >= currentLength)
      gthrow_(GArrayIndexOutOfBoundsException());

   memmove(theString + index, theString + index + 1, currentLength - index);
   currentLength -= 1;
   cachedHashCode = 0;
   return *this;
}

GString& GString::removeFirstChar ()
{
   if (currentLength <= 0)
      return *this;

   memmove(theString, theString + 1, currentLength);
   currentLength -= 1;
   cachedHashCode = 0;
   return *this;
}

GString& GString::removeLastChar ()
{
   if (currentLength >= 1)
   {
      currentLength -= 1;
      theString[currentLength] = '\0';
      cachedHashCode = 0;
   }
   return *this;
}

bool GString::beginsWith ( char chr, bool ignoreCase ) const
{
   if (currentLength <= 0)
      return chr == '\0';
   if (ignoreCase)
      return toupper(chr) == toupper(*theString);
   return chr == *theString;
}

bool GString::beginsWith ( const char* str, bool ignoreCase ) const
{
   if (str[0] == '\0')
      return true; // All strings "begins with an empty string".
   if (theString == null)
      return false;
   if (ignoreCase)
   {
      for (const char* ptr = theString; *str; str++, ptr++)
         if (toupper(*str) != toupper(*ptr))
            return false;
      return true;
   }
   else
   {
      for (const char* ptr = theString; *str; str++, ptr++)
         if (*str != *ptr)
            return false;
      return true;
   }
}

bool GString::beginsWith ( const GString& str, bool ignoreCase ) const
{
   if (str.theString == null)
      return true; // All strings "begins with an empty string".
   return beginsWith(str.theString, ignoreCase);
}

bool GString::endsWith ( char chr, bool ignoreCase ) const
{
   if (chr == '\0')
      return true; // All strings "ends with a null-character".
   if (theString == null)
      return false;
   char lastChar = theString[currentLength-1];
   if (ignoreCase)
      return toupper(lastChar) == toupper(chr);
   return lastChar == chr;
}

bool GString::endsWith ( const char* str, bool ignoreCase ) const
{
   if (str[0] == '\0')
      return true; // All strings "ends with an empty string".
   if (theString == null)
      return false;
   int len = int(strlen(str));
   if (len > currentLength)
      return false;
   const char *tail = theString + currentLength - len;
   if (ignoreCase)
      return stricmp(tail, str) == 0;
   return strcmp(tail, str) == 0;
}

bool GString::endsWith ( const GString& str, bool ignoreCase ) const
{
   int len = str.currentLength;
   if (len == 0)
      return true; // All strings "ends with an empty string".
   if (len > currentLength)
      return false;
   const char *tail = theString + currentLength - len;
   if (ignoreCase)
      return stricmp(tail, str.theString) == 0;
   return strcmp(tail, str.theString) == 0;
}

bool GString::endsWithEol () const
{
   char chr = lastChar();
   return chr == '\n' || chr == '\r';
}

GString GString::substring ( int startIndex, int endIndex ) const
{
   if (startIndex < 0)
      startIndex = 0;
   if (endIndex > currentLength)
      endIndex = currentLength;
   if (endIndex <= startIndex)
      return Empty;

   const int len = endIndex - startIndex;
   if (len <= 0)
      return Empty;

   GString ret(len);
   ret.currentLength = ret.allocatedSize;
   memcpy(ret.theString, theString + startIndex, ret.currentLength);
   ret.theString[ret.currentLength] = '\0';
   return ret;
}

GString GString::getIndexedPart ( int index, const char* sep ) const
{
   if (theString == null)
      return Empty;
   const char* src = theString;
   while (*src && index >= 0)
   {
      // Ignore separating characters.
      while (strchr(sep, *src))
         if (*src++ == 0)
             return Empty;

      if (index-- > 0)
      {
         // This is not the correct index, so just skip it.
         while (!strchr(sep, *src))
            if (*src++ == 0)
               return Empty;
      }
      else
      {
         // This is the requested index, so grab and return it.
         GString dest(16);
         while (*src && !strchr(sep, *src))
            dest += *src++;
         return dest;
      }
   }

   return Empty;
}

GString& GString::reverse ()
{
   if (theString != null)
   {
      strrev(theString);
      cachedHashCode = 0;
   }
   return *this;
}

GString& GString::toLowerCase ()
{
   if (theString != null)
   {
      strlwr(theString);
      cachedHashCode = 0;
   }
   return *this;
}

GString& GString::toUpperCase ()
{
   if (theString != null)
   {
      strupr(theString);
      cachedHashCode = 0;
   }
   return *this;
}

GString& GString::replaceAll ( char oldchr, char newchr )
{
   if (oldchr != newchr && theString != null)
   {
      char *ptr = theString;
      for (int i=0; i<currentLength; ptr++, i++)
         if (*ptr == oldchr)
            *ptr = newchr;
      cachedHashCode = 0;
   }
   return *this;
}

GString& GString::replaceAll ( const char* oldchars, char newchr )
{
   if (theString != null)
   {
      while (*oldchars)
         replaceAll(*oldchars++, newchr);
   }
   return *this;
}

GString& GString::replaceAll ( const GString& oldchars, char newchr )
{
   if (theString != null && oldchars.currentLength > 0)
      replaceAll(oldchars.theString, newchr);
   return *this;
}

int GString::ensureCompatibleEolFormat ()
{
   if (!endsWithEol())
      return 0;
   int countEOLs = stripTrailingEol();
   for (int i=0; i<countEOLs; i++)
      operator+=(GString::EOL);
   return countEOLs;
}

int GString::stripTrailingEol ()
{
   if (theString == null)
      return 0;
   int countEOLs = 0;
   char* lastCharPtr = theString + currentLength - 1;
   while (lastCharPtr >= theString)
   {
      if (*lastCharPtr == '\n')
      {
         if (--lastCharPtr > theString && *lastCharPtr == '\r')
         {
            // We have identifyed another "\r\n" sequence (Windows/OS2).
            lastCharPtr--;
            currentLength -= 2;
            theString[currentLength] = '\0';
            cachedHashCode = 0;
            countEOLs++;
         }
         else
         {
            // We have identifyed another "\n" sequence (Unix).
            currentLength--;
            theString[currentLength] = '\0';
            cachedHashCode = 0;
            countEOLs++;
         }
      }
      else
      if (*lastCharPtr == '\r')
      {
         // We have identifyed another "\n" sequence (Unix).
         lastCharPtr--;
         currentLength--;
         theString[currentLength] = '\0';
         cachedHashCode = 0;
         countEOLs++;
      }
      else
      {
         break;
      }
   }
   return countEOLs;
}

GString& GString::trim ( TrimType tt, const GString& chars )
{
   if (theString == null)
      return *this;
   const char* charsPtr = chars;
   switch (tt)
   {
      case TrimEndsOnly:
      {
         // Strip all from the start
         while (currentLength > 0 && strchr(charsPtr, theString[0]))
            removeFirstChar();

         // Strip all from the end
         while (currentLength > 0 && strchr(charsPtr, theString[currentLength-1]))
            removeLastChar();
         break;
      }

      case TrimEndsAndRepeating: 
      {
         // Strip all from the start
         while (currentLength > 0 && strchr(charsPtr, theString[0]))
            removeFirstChar();

         // Strip all from the end
         while (currentLength > 0 && strchr(charsPtr, theString[currentLength-1]))
            removeLastChar();

         // Strip from the middle
         int index = 1;
         bool previous = false;
         while (index < currentLength)
         {
            if (strchr(charsPtr, theString[index]))
            {
               if (previous)
               {
                  removeCharAt(index);
               }
               else
               {
                  previous = true;
                  index++;
               }
            }
            else
            {
               previous = false;
               index++;
            }
         }
         break; 
      }

      case TrimAll: 
      {
         int index = 0;
         while (index < currentLength)
         {
            if (strchr(charsPtr, theString[index]))
               removeCharAt(index);
            else
               index++;
         }
         break; 
      }

      default:
      {
         gthrow_(GIllegalArgumentException(GString("Unknown trim type: %d", GVArgs(tt))));
         break;
      }
   }

   return *this;
}

