/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_VARGS
#define __GLIB_VARGS

#include "glib/util/GArray.h"
#include "glib/primitives/GNumber.h"

/**
 * This class is implemented as a helper for easy emulation of
 * variable argument lists for usage in situations where native C++
 * variable argument lists are not suitable.
 *
 * See the implementation of
 * {@link GString#GString(const GString&, const GVArgs&)} for more
 * information and for an example of how objects of this type can
 * be used.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.23
 */
class GVArgs : public GObject
{
   private:

      /** */
      class Arg : public GObject
      {
         friend class GVArgs;

         private:

            GObject* obj;

            Arg ( GObject* obj );

         public:

            virtual ~Arg ();
      };

      /** The variable argument list, implemented as a dynamic vector. */
      GArray<GVArgs::Arg>* args;

   public:

      /** Create a variable argument list that is initially empty but with preallocated room for 16 arguments. */
      explicit GVArgs ();

      /** Create a variable argument list that is initially containing the specified string only, but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( const char* a );

      /** Create a variable argument list that is initially containing the specified string only, but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( const GString& a );

#if __HAS_KEYWORD_BOOL__
      /** Create a variable argument list that is initially containing the specified byte only (as an instance of {@link GBoolean}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( bool a );
#endif

      /** Create a variable argument list that is initially containing the specified char only (as an instance of {@link GCharacter}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( char a );

      /** Create a variable argument list that is initially containing the specified short only (as an instance of {@link GShort}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( short a );

      /** Create a variable argument list that is initially containing the specified unsigned short only (as an instance of {@link GUShort}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( unsigned short a );

      /** Create a variable argument list that is initially containing the specified int only (as an instance of {@link GInteger}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( int a );

      /** Create a variable argument list that is initially containing the specified unsigned int only (as an instance of {@link GUInteger}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( unsigned int a );

      /** Create a variable argument list that is initially containing the specified long only (as an instance of {@link GLong}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( long a );

      /** Create a variable argument list that is initially containing the specified unsigned long only (as an instance of {@link GULong}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( unsigned long a );

      /** Create a variable argument list that is initially containing the specified longlong only (as an instance of {@link GLong}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( longlong a );

      /** Create a variable argument list that is initially containing the specified unsigned longlong only (as an instance of {@link GULong}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( ulonglong a );

      /** Create a variable argument list that is initially containing the specified float only (as an instance of {@link GFloat}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( float a );

      /** Create a variable argument list that is initially containing the specified double only (as an instance of {@link GDouble}), but with preallocated room for additionally 15 objects. */
      explicit GVArgs ( double a );

      /** Destructor. */
      virtual ~GVArgs ();

   private:

      /** Disable the copy constructor. */
      GVArgs ( const GVArgs& src ) {}

      /** Disable the assignment operator for this class. */
      GVArgs& operator= ( const GVArgs& ) { return *this; }

   public:

      /** Add the specified object to the argument list, and return a reference to our self. */
      GVArgs& add ( const char* a );

      /** Add the specified object to the argument list, and return a reference to our self. */
      GVArgs& add ( const GString& a );

#if __HAS_KEYWORD_BOOL__
      /** Add the specified byte (as an instance of {@link GBoolean}) to the argument list, and return a reference to our self. */
      GVArgs& add ( bool a );
#endif

      /** Add the specified char (as an instance of {@link GCharacter}) to the argument list, and return a reference to our self. */
      GVArgs& add ( char a );

      /** Add the specified short (as an instance of {@link GShort}) to the argument list, and return a reference to our self. */
      GVArgs& add ( short a );

      /** Add the specified unsigned short (as an instance of {@link GUShort}) to the argument list, and return a reference to our self. */
      GVArgs& add ( unsigned short a );

      /** Add the specified int (as an instance of {@link GInteger}) to the argument list, and return a reference to our self. */
      GVArgs& add ( int a );

      /** Add the specified unsigned int (as an instance of {@link GUInteger}) to the argument list, and return a reference to our self. */
      GVArgs& add ( unsigned int a );

      /** Add the specified long (as an instance of {@link GLong}) to the argument list, and return a reference to our self. */
      GVArgs& add ( long a );

      /** Add the specified unsigned long (as an instance of {@link GLong}) to the argument list, and return a reference to our self. */
      GVArgs& add ( unsigned long a );

      /** Add the specified longlong (as an instance of {@link GLong}) to the argument list, and return a reference to our self. */
      GVArgs& add ( longlong a );

      /** Add the specified unsigned longlong (as an instance of {@link GLong}) to the argument list, and return a reference to our self. */
      GVArgs& add ( ulonglong a );

      /** Add the specified float (as an instance of {@link GFloat}) to the argument list, and return a reference to our self. */
      GVArgs& add ( float a );

      /** Add the specified double (as an instance of {@link GDouble}) to the argument list, and return a reference to our self. */
      GVArgs& add ( double a );

   public:

      /** Get the current number of arguments that are contained in the list. */
      int num () const;

      /** Get a reference to the indexed argument, or null if the index is out of bounds. */
      const GObject* get ( int idx ) const;

   private:

      /**
       * Exception class that can be thrown by {@link #get}.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      class UndefinedArgumentException : public GException
      {
         friend class GVArgs;

         UndefinedArgumentException ( int idx );
      };

      /**
       * Same as {@link #get} but throw UndefinedArgumentException if
       * index is out of bounds.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      const GObject* get_excp ( int idx ) const;

   private:

      /**
       * This inner class keeps the description of a single expression
       * code tag, as well as the code to format arguments of specific
       * types according to the tag.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      class Tag : public GObject
      {
         friend class GVArgs;

         /** Length of field (-1 if not specified). */
         int width;

         /** Number of decimals (-1 if not specified). */
         int precision;

         /** Pad with leading zeros. */
         bool leadingZeroes;

         /** Output plus sign for positive numbers. */
         bool showPlus;

         /** */
         bool alternate;

         /** */
         bool showSpace;

         /** */
         bool leftAlign;

         /** Conversion character. */
         char formatCode;

         /** Number of characters recognized as part of the expression tag. */
         int codeLength;

         /**
          * This constructor scans the expression for specifier:
          * [-+0123456789.dxXoufeEgGcs].
          * <p>
          * Example for expression to scan: "-+20.5f" will give this result:
          * <p>
          * <code>
          * width         = 20;
          * precision     = 5;
          * leadingZeroes = false;
          * showPlus      = true;
          * alternate     = false;
          * showSpace     = true;
          * leftAlign     = false;
          * formatCode    = 'f';
          * codeLength    = 7;
          * </code>
          *
          * @author  Leif Erik Larsen
          * @since   2001.02.08
          */
         Tag ( const char* exp, const int startPos, const int explen );
         virtual ~Tag ();

         GString fixedFormatted ( double d );
         GString expFormatted ( double d );
         GString packWith ( char c, int n );
         GString setIndication ( int parameter, const GString& formattedExp );
         GString formatString ( const GObject* arg );
         GString formatCharacter ( const GObject* arg );
         GString formatDouble ( const GNumber* arg );
         GString formatLong ( const GNumber* arg );
      };

      friend class Tag;

   private:

      static const GString STR_NULL;
      static const GString STR_ERROR;

   public:

      /**
       * Use this method to format a variable argument list with a
       * source string containing optional format tags almost like the
       * classic <i>fprintf()</i> set of functions in ANSI C.
       *
       * This static method can also be used as an example of how the
       * IVArgs class it self can be used.
       *
       * This method accepts a series of arguments, where each applies to
       * a format specifier contained in the format string pointed to by
       * <i>str</i>, and outputs the formatted data to a new string that is
       * returned. There must be the same number of format specifiers as
       * arguments or else the result is unpredictable.
       *
       * The following format tags are supported:
       *
       * <table border cellspacing=0 cellpadding=5 width="90%" bgcolor="#CCFFFF">
       * <tr><td><b>%d</b></td><td>
       *     Signed decimal integer number.
       *     Argument objects of type {@link java.lang.Number} are supported
       *     (e.g. Byte, Short, Integer, Long, Float and Double) as well
       *     as objects whose toString() can be converted to a Double with
       *     method {@link java.lang.Double#parseDouble(java.lang.String)}.
       *     </td></tr>
       * <tr><td><b>%i</b></td><td>
       *     Ditto.
       *     </td></tr>
       * <tr><td><b>%b</b></td><td>
       *     Ditto, but binary.
       *     </td></tr>
       * <tr><td><b>%o</b></td><td>
       *     Ditto, but octal.
       *     </td></tr>
       * <tr><td><b>%x</b></td><td>
       *     Ditto, but hexadecimal with lowercase a...f.
       *     </td></tr>
       * <tr><td><b>%X</b></td><td>
       *     Ditto, but uppercase A...F.
       *     </td></tr>
       * <tr><td><b>%f</b></td><td>
       *     Signed floating point number in fixed format (999.999999).
       *     Argument objects of type {@link GNumber} are supported
       *     (e.g. GShort, GInteger, GLong, GFloat and GDouble) as well
       *     as objects whose toString() can be converted to a Double with
       *     method {@link GDouble#parseDouble(GString)}.
       *     </td></tr>
       * <tr><td><b>%e</b></td><td>
       *     Ditto, but in exponential notation (scientific format) with
       *     lowercase e. For instance: 0.999999e+999.
       *     </td></tr>
       * <tr><td><b>%E</b></td><td>
       *     Ditto, but with uppercase E. For instance: 0.999999E+999.
       *     </td></tr>
       * <tr><td><b>%g</b></td><td>
       *     Floating point number in general format (fixed format for small
       *     numbers, exponential format for large numbers). Trailing zeroes
       *     are suppressed. <b>Note</b> that this is the same as "f" for
       *     absolute values not smaller than 0.001 and not greater than or
       *     equal to 1000. Otherwise, same as "e".
       *     </td></tr>
       * <tr><td><b>%G</b></td><td>
       *     Ditto, but with uppercase E..
       *     </td></tr>
       * <tr><td><b>%c</b></td><td>
       *     Single character (Character or "arg.toString().charAt(0)").
       *     </td></tr>
       * <tr><td><b>%s</b></td><td>
       *     Character string (String or "arg.toString()").
       *     </td></tr>
       * <tr><td><b>%%</b></td><td>
       *     The % character it self.
       *     </td></tr>
       * <tr><td><b>#n</b></td><td>
       *     Referencing the n'th argument, where n is the 1-based
       *     index (not 0-based). Note that this tag does not increment
       *     the argument index as is used by the %-tags to access the
       *     arguments one-by-one. <b>Note</b> that for backward compatibility
       *     reasons we will allow syntax forms like "%1" instead of the
       *     new "#1" to access numbered arguments, as long as the digit is
       *     only one character long (1..9, that is) and the character next to
       *     it is not equal to any supported format characters.
       *     </td></tr>
       * <tr><td><b>##</b></td><td>
       *     The # character it self.
       *     </td></tr>
       * </table>
       *
       * <b>Example</b>
       *
       * <pre>
       *    IVArgs.formatArgs("Variable %s equals %d, which is %d%% of %d",
       *                      new IVArgs("X").add(11).add(10).add(110));
       * </pre>
       *
       * The above code will produce the following result:
       *
       * <pre>
       *    "Variable X equals 11, which is 10% of 110"
       * </pre>
       *
       * You can get the same result by referencing the arguments
       * by their one-based index numbers, like this:
       * <pre>
       *    IVArgs.formatArgs("Variable #1 equals #2, which is #3%% of #4",
       *                      new IVArgs("X").add(11).add(10).add(110));
       * </pre>
       *
       * <b><i>More details</i></b>
       *
       * This method provides a number of formatted output of primitive data
       * types following the ANSI C-language <i>printf()</i> conventions.
       * Each formatting code/tag handles one parameter at a time. Thus, you
       * must use multiple format code tags to format more than one argument.
       *
       * The format code tag has a prefix, a format pattern and a suffix.
       * The prefix and suffix will be part of the formatted output.
       * The format code handle the formatting of the (single) parameter to
       * be formatted. The pattern has the following structure:
       *
       * <code>
       * %[-][+][0-9][.0-9][lL][diboxXfgGeEcs]
       * </code>
       *
       * <b><i>Format code pattern</i></b>
       *
       * <table border cellspacing=0 cellpadding=5 width="90%" bgcolor="#CCFFFF">
       * <tr><td><b>%</b></td><td>
       *     The percent sign is mandatory. It starts the format specifier.
       *     </td></tr>
       * <tr><td><b>[+]</b></td><td>
       *     Optional. Forces display of + for positive numbers.
       *     </td></tr>
       * <tr><td><b>[0]</b></td><td>
       *     Optional. Display leading zeroes.
       *     </td></tr>
       * <tr><td><b>[-]</b></td><td>
       *     Optional. Align left in the field (by default it is right-aligned).
       *     </td></tr>
       * <tr><td><b>[ ]</b></td><td>
       *     Optional. Space prepend a space in front of positive numbers.
       *     </td></tr>
       * <tr><td><b>[#]</b></td><td>
       *     Optional. Use "alternate" format. Add 0 or 0x for octal or
       *     hexadecimal numbers, respectively. Don't suppress trailing
       *     zeroes in general floating point format.
       *     </td></tr>
       * <tr><td><b>[0-9]</b></td><td>
       *     Optional. An integer denoting field width.
       *     </td></tr>
       * <tr><td><b>[.0-9]</b></td><td>
       *     Optional. A period followed by an integer denoting precision.
       *     </td></tr>
       * <tr><td><b>[diboxXfgGeEcs]</b></td><td>
       *     Mandatory. One and only one of these letters must be included
       *     in the format code in order to describe the expected type of the
       *     corresponding argument. See the above (first) table for
       *     details on each supported letter.
       *     </td></tr>
       * </table>
       *
       * The number of decimals specifies the length of the fractional
       * part for floating point numbers. If ommitted, the default is 6.
       *
       * For use of prefix [e] or [E]:<br>
       * The mantissa is normalized to its canonical form, i.e. it's
       * always smaller than 1.0 and greater than or equal to 0.1.
       * The exponent has always three digits.
       * The exponent always shows a sign, even for positive values.
       * The length of the frational part is subject to the number
       * of decimals specified in the format string.
       * If omitted, it defaults to 6.
       *
       * The fractional part is subject to rounding for the last visible
       * digit. Overflows could be propagated into the integer part or
       * even into the exponent.
       *
       * Other:<br>
       * The number of decimals is applied to all format chars except "c"
       * and "s". For integer values, the fractional part is always 0.
       *
       * <b><i>Example</i></b>
       *
       * Just as a technical sample, the below code will produce the
       * following output:
       *
       * <pre>
       * 01: "Simplest." ==> "Simplest."
       * 02: "Simple %s" ==> "Simple test."
       * 03: "Simple %s within quotes." ==> "Simple test within quotes."
       * 04: "%c%c%c%c. %%+##." ==> "Test. %+#."
       * 05: "%-10s." ==> "Test      ."
       * 06: "% 10s." ==> "      Test."
       * 07: "%+08d/%+08d." ==> "+0000005/-0000005."
       * 08: "%f." ==> "8.090000."
       * 09: "%0.6f." ==> "1.100000."
       * 10: "Null: %s." ==> "Null: [null]."
       * 11: "Error: %s." ==> "Error: [Error: Undefined argument #1]."
       * 12: "'%1' is ##1, '%2' is ##2." ==> "'1' is #1, '2' is #2."
       * 13: "'#1' is ##1, '#2' is ##2." ==> "'[Error: Undefined argument #11]' is #1, '[Error: Undefined argument #22]' is #2."
       * 14: "Dec=%d, Bin=%b, Oct=%o, AltOct=%#o, Hex=%x, AltHex=%#x." ==> "Dec=16, Bin=10000, Oct=20, AltOct=020, Hex=10, AltHex=0x10."
       * 15: "%d, %i, %f, %g, %G, %e, %E." ==> "25, 25, 25.010000, 25.01, 25.01, 2.501000e+001, 2.501000E+001."
       * 16: "%%" ==> "%"
       * 17: "##" ==> "#"
       * 18: "%" ==> "[error]"
       * 19: "#" ==> "[error]"
       * 20: "Test %" ==> "Test [error]"
       * 21: "Test % again." ==> "Test [error]gain."
       * 22: "Test %%" ==> "Test %"
       * 23: "Test #" ==> "Test [error]"
       * 24: "Test # again." ==> "Test [error] again."
       * 25: "Test ##" ==> "Test #"
       * 26: "%1" ==> "First"
       * 27: "#1" ==> "First"
       * 28: "%4d" ==> "   4"
       * </pre>
       *
       * <pre>
       * public void myMethod ()
       * {
       *    void* tests[56] =
       *    {
       *       "Simplest.", new GVArgs(),
       *       "Simple %s", new GVArgs("test."),
       *       "Simple %s within quotes.", new GVArgs("test"),
       *       "%c%c%c%c. %%+##.", new GVArgs('T').add('e').add('s').add('t'),
       *       "%-10s.", new GVArgs("Test"),
       *       "% 10s.", new GVArgs("Test"),
       *       "%+08d/%+08d.", new GVArgs(+5).add(-5),
       *       "%f.", new GVArgs(8.09f),
       *       "%0.6f.", new GVArgs(1.1f),
       *       "Null: %s.", new GVArgs((Object) null),
       *       "Error: %s.", new GVArgs(),
       *       "'%1' is ##1, '%2' is ##2.", new GVArgs(1).add(2),
       *       "'#1' is ##1, '#2' is ##2.", new GVArgs(1).add(2),
       *       "Dec=%d, Bin=%b, Oct=%o, AltOct=%#o, Hex=%x, AltHex=%#x.", new GVArgs(16).add(16).add(16).add(16).add(16).add(16),
       *       "%d, %i, %f, %g, %G, %e, %E.", new GVArgs(25.01).add(25.01).add(25.01).add(25.01).add(25.01).add(25.01).add(25.01),
       *       "%%", new GVArgs(),
       *       "##", new GVArgs(),
       *       "%", new GVArgs(),
       *       "#", new GVArgs(),
       *       "Test %", new GVArgs(),
       *       "Test % again.", new GVArgs(),
       *       "Test %%", new GVArgs(),
       *       "Test #", new GVArgs(),
       *       "Test # again.", new GVArgs(),
       *       "Test ##", new GVArgs(),
       *       "%1", new GVArgs("First"),
       *       "#1", new GVArgs("First"),
       *       "%4d", new GVArgs(4)
       *    };
       *
       *    for (int i=0; i<56; i+=2)
       *    {
       *       const char* expr = (const char*) tests[i+0];
       *       GVArgs* args = (GVArgs*) tests[i+1];
       *       GString frmt = GVArgs::formatArgs(expr, args);
       *       GString out = GVArgs::formatArgs("%02d: \"%s\" ==> \"%s\"", GVArgs(i/2+1).add(expr).add(frmt));
       *       System.out.println(out);
       *    }
       * }
       * </pre>
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       * @param   str   The string that will be used to format the
       *                variable argument list.
       * @param   args  The list of variable arguments that are to be
       *                joined into the format string in a formatted way.
       * @return  The formatted string.
       */
      static GString FormatArgs ( const GString& str, const GVArgs& args );

      /**
       * Specially optimized version for being used by the
       * constructor/initializer of GString's only.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      static void FormatArgs ( GString& dst, 
                               const char* str, 
                               const int len, 
                               const GVArgs& args );
};

#endif
