/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GAccelResource.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"
#include "glib/util/GLog.h"

GAccelResource::GAccelResource ( const GString& id )
               :GAbstractResource(id),
                keysByKeyName(32, -3, true),
                keysByCommandID(32, -3)
{
}

GAccelResource::~GAccelResource ()
{
}

bool GAccelResource::addAccelItem ( const GString& keyName, GAccelItem* item )
{
   // The same accelerator key can only be used on one menu command.
   // Therefore we must return false if the current accelerator key is
   // already in use by another menu command.
   if (!keysByKeyName.put(keyName, item, true))
      return false;

   // The <i>keysByCommandID</i> bag are only needed to be able to lookup
   // items by the command ID, to see if there are any key accelerator for the
   // given command ID string. If the current ID string is already contained
   // in the bag then just ignore it, because it is perfectly legal to have
   // more than one key accelerator for the same menu command.
   // The items in this bag must not be automatically destroyed, because they
   // will be automatically destroyed by the <i>keysByKeyName</i> bag.
   const GString& id = item->getIDString();
   keysByCommandID.put(id, item, false);
   return true;
}

GAccelItem* GAccelResource::getAccelItemByKeyName ( const GString& keyName ) const
{
   GAccelItem* ret = keysByKeyName.get(keyName);
   if (ret == null && keyName.endsWith("_N"))
   {
      // The key is one of the extended keys on the "num" keypad.
      // In most cases the accelerator table does not explicitly associate
      // a command directly to "gray key only". Therefore we must lookup
      // the accelerator item without the "gray tag" as a second attempt.
      // This way the user can press e.g. the PLUSS key on both the 
      // extended keypad and on the standard keyboard part transparently.
      GString tmp = keyName.substring(0, keyName.length() - 2);
      ret = keysByKeyName.get(tmp);
   }

   return ret;
}

GAccelItem* GAccelResource::getAccelItemByCommandID ( const GString& commandID ) const
{
   return keysByCommandID.get(commandID);
}

void GAccelResource::loadFromScript ( GRcTokenizer& tokenizer, GResourceTable* table )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lbracket)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("{").add(token->getString()))); // Expected '(' but found '%2' in statement!

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);
      if (*token == GRcTokenizer::Token_rbracket)
      {
         break;
      }
      else
      {
         const GString& id = token->getString();
         aptr<GAccelItem> newItem(new GAccelItem(id));
         newItem->loadFromScript(tokenizer, table);
         const GString& keyName = newItem->getKeyName();
         if (keyName.equalsIgnoreCase("VK_NULL"))
            ; // Yes, just ignore this null-item.
         else
         if (addAccelItem(keyName, newItem.get()))
            newItem.release();
         else
            gthrow_(GRcException(tokenizer, GRcException::ERRACCELARDEF)); // The specified key combination is already defined!
      }
   }
}

