/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_COMPONENTPARAMS
#define __GLIB_COMPONENTPARAMS

#include "glib/resource/GAbstractResource.h"
#include "glib/util/GParameterList.h"
#include "glib/gui/GComponentPos.h"
#include "glib/gui/GComponentSize.h"

/**
 * Each dialog COMPONENT is represented with an instance of this class.
 */
class GComponentParams : public GAbstractResource
{
   friend class GDialogResource;

   private:

      /** Literal name of control type, e.g. "PushButton" or "TextEntry". */
      GString typeName;

      /** Initial text on control (possibly a link to another TEXT). */
      GString textID;

      /** Text to show as control hint (possibly a link to another TEXT). */
      GString hintID;

      GString constraints;
      GComponentPos pos;
      GComponentSize size;
      bool disabled;
      bool toggleOn;
      bool hidden;
      bool oily;

   public:

      /** Additional arguments not recognized by the default component parser. */
      GParameterList extraArgs;

   public:

      GComponentParams ();

      /**
       * Create a new set of Dialog Component Parameters from the specified
       * resource script string.
       *
       * The script must be a single Dialog Component definition
       * statement. For example:
       *
       * <code>"TEXTENTRY(ID=TEST TEXT='TEXT' XPOS=4 YPOS=2 WIDTH=33)"</code>
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   script  The script of which to compile.
       * @throws  GRcException in case of any error compiling the script.
       */
      explicit GComponentParams ( const GString& script );

      GComponentParams ( const GString& typeName, const GString& id );
      GComponentParams ( const GComponentParams& src );

      virtual ~GComponentParams ();

   public:

      GComponentParams& operator= ( const GComponentParams& src );

   public:

      /**
       * Parse the header parameters of the specified dialogbox resource.
       *
       * Since we assume that the ID-part of the resource has already been
       * parsed the next token (which is the first, from our point of view)
       * returned from the tokenizer must be "(".
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.08
       * @param   tokenizer   The tokenizer of where to get tokens to parse.
       * @throws  GRcException in case of any error.
       */
      virtual void loadFromScript ( class GRcTokenizer& tokenizer, class GResourceTable* table );

      /**
       * Return a string representation of the parameters, for debugging
       * purposes only.
       */
      GString toString () const;

      const GString& getConstraints () const;
      const GString& getHintID () const;
      const GString& getTextID () const;
      const GString& getTypeName () const;
      const GComponentPos& getPos () const;
      const GComponentSize& getSize () const;
      bool isDisabled () const;
      bool isToggleOn () const;
      bool isHidden () const;
      bool isOily () const;
};

#endif
