/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GDialogResource.h"
#include "glib/resource/GResourceTable.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"

GDialogResource::ComponentTypes::ComponentTypes ()
{
   put("combobox", &dummyObj, false);
   put("droplist", &dummyObj, false);
   put("groupbox", &dummyObj, false);
   put("lineplotter", &dummyObj, false);
   put("listbox", &dummyObj, false);
   put("multilineedit", &dummyObj, false);
   put("progressbar", &dummyObj, false);
   put("pushbutton", &dummyObj, false);
   put("radiobutton", &dummyObj, false);
   put("spinner", &dummyObj, false);
   put("staticicon", &dummyObj, false);
   put("statictext", &dummyObj, false);
   put("tabbedpanel", &dummyObj, false);
   put("textentry", &dummyObj, false);
   put("togglebutton", &dummyObj, false);
   put("tree", &dummyObj, false);
}

GDialogResource::ComponentTypes GDialogResource::CompTypes;

GDialogResource::GDialogResource ( const GString& id )
                :GAbstractResource(id),
                 ctrls(16),
                 closeButton(false),
                 resizable(false)
{
}

GDialogResource::~GDialogResource ()
{
}

void GDialogResource::parseHeader ( GRcTokenizer& tokenizer, GResourceTable* table )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lpar)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);

      if (*token == GRcTokenizer::Token_rpar)
         break;
      else
      if (*token == GRcTokenizer::Token_comma)
         continue;
      else
      if (*token == GRcTokenizer::Token_text)
      {
         if (textID != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRTEXTARDEF)); // TEXT already defined!
         token = tokenizer.queryArgValue();
         textID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_acceltable)
      {
         if (accelID != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRACCELTABLEARDEF)); // ACCELTABLE already defined!
         token = tokenizer.queryArgValue();
         if (table != null)
            if (!table->getAcceleratorBag().containsKey(token->getString()))
               gthrow_(GRcException(tokenizer, GRcException::ERRUNDEFACCEL, GVArgs(token->getString()))); // Undefined ACCELTABLE: '%s'
         accelID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_menu)
      {
         if (menuID != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRMENUARDEF)); // MENU already defined!
         token = tokenizer.queryArgValue();
         if (table != null)
            if (!table->getMenuBag().containsKey(token->getString()))
               gthrow_(GRcException(tokenizer, GRcException::ERRUNDEFMENU, GVArgs(token->getString()))); // Undefined MENU: '%s'
         menuID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_toolbar)
      {
         if (toolbarID != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRTOOLBARARDEF)); // TOOLBAR already defined!
         token = tokenizer.queryArgValue();
         if (table != null)
            if (!table->getToolbarBag().containsKey(token->getString()))
               gthrow_(GRcException(tokenizer, GRcException::ERRUNDEFTOOLBAR, GVArgs(token->getString()))); // Undefined TOOLBAR: '%s'
         toolbarID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_xpos)
      {
         if (pos.x)
            gthrow_(GRcException(tokenizer, GRcException::ERRXPOSARDEF)); // XPOS already defined!
         pos.x = tokenizer.queryArgValueFloat();
      }
      else
      if (*token == GRcTokenizer::Token_ypos)
      {
         if (pos.y)
            gthrow_(GRcException(tokenizer, GRcException::ERRYPOSARDEF)); // YPOS already defined!
         pos.y = tokenizer.queryArgValueFloat();
      }
      else
      if (*token == GRcTokenizer::Token_width)
      {
         if (size.width)
            gthrow_(GRcException(tokenizer, GRcException::ERRWIDTHARDEF)); // WIDTH already defined!
         size.width = tokenizer.queryArgValueFloat();
         if (size.width <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRWIDTHLTHAN0)); // WIDTH must be larger than zero!
      }
      else
      if (*token == GRcTokenizer::Token_height)
      {
         if (size.height)
            gthrow_(GRcException(tokenizer, GRcException::ERRHEIGHTARDEF)); // HEIGHT already defined!
         size.height = tokenizer.queryArgValueFloat();
         if (size.height <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRHEIGHTLTHAN0)); // HEIGHT must be larger than zero!
      }
      else
      if (*token == GRcTokenizer::Token_icon)
      {
         if (iconID != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRICONARDEF)); // ICON already defined!
         token = tokenizer.queryArgValue();
         iconID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_layout)
      {
         if (layoutManagerID != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRLAYOUTARDEF)); // LAYOUT already defined!
         token = tokenizer.queryArgValue();
         layoutManagerID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_closebutton)
         closeButton = true;
      else
      if (*token == GRcTokenizer::Token_resizable)
         resizable = true;
      else
         gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(token->getString()))); // Unknown token: %s
   }
}

void GDialogResource::loadFromScript ( GRcTokenizer& tokenizer, 
                                       GResourceTable* table )
{
   parseHeader(tokenizer, table);

   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lbracket)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("{").add(token->getString()))); // Expected '(' but found '%2' in statement!

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);

      if (*token == GRcTokenizer::Token_rbracket)
      {
         // Make sure all ID-links are legal (NEXT, PREVIOUS, FORCENEXT and
         // FORCEPREVIOUS), and that the dialog doesn't contain two or more
         // controls that share the same ID.

         int num = ctrls.getCount();
         for (int i=0; i<num; i++)
         {
            GComponentParams& comp = ctrls.get(i); /* Make this a fast one */

            // Make sure that the two controls doesn't share the same ID.
            if (comp.getIDString() != "")
               for (int i2=0; i2<num; i2++)
                  if (ctrls.get(i2).getIDString() != "")
                     if ((i != i2) && (comp.getIDString() == ctrls.get(i2).getIDString()))
                        gthrow_(GRcException(tokenizer, GRcException::ERR2CTRLSSAMEID, GVArgs(comp.getIDString()).add(getIDString()))); // Control '%s' is defined twice in dialog '%s'!
         }

         break;
      }
      else
      if (*token == GRcTokenizer::Token_comma)
      {
         continue;
      }
      else
      {
         // Next token should be the name of some component type.
         const GString& typeName = token->getString();
         if (CompTypes.get(typeName) == null)
            gthrow_(GRcException(tokenizer, GRcException::ERRUNKNOWNCTRL, GVArgs(token->getString()))); // Unknown type of control: '%s'!

         aptr<GComponentParams> newComp(new GComponentParams());
         newComp->typeName = typeName;
         newComp->loadFromScript(tokenizer, table);
         ctrls.add(newComp.release());
      }
   }
}

const GString& GDialogResource::getAccelID () const 
{ 
   return accelID; 
}

const GArray<GComponentParams>& GDialogResource::getComponents () const 
{ 
   return ctrls; 
}

const GString& GDialogResource::getIconID () const 
{ 
   return iconID; 
}

const GString& GDialogResource::getLayoutManagerID () const 
{ 
   return layoutManagerID; 
}

const GString& GDialogResource::getMenuID () const 
{ 
   return menuID; 
}

const GComponentPos& GDialogResource::getPos () const 
{ 
   return pos; 
}

const GComponentSize& GDialogResource::getSize () const 
{ 
   return size; 
}

const GString& GDialogResource::getTextID () const 
{ 
   return textID; 
}

const GString& GDialogResource::getToolbarID () const 
{ 
   return toolbarID; 
}

bool GDialogResource::isResizable () const 
{ 
   return resizable; 
}

