/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/sys/GSystem.h"
#include "glib/sys/GSystemInfo.h"
#include "glib/GProgram.h"
#include "glib/io/GFileInputStream.h"
#include "glib/exceptions/GIllegalStateException.h"

GObject GSystem::LockGetCpuUsageInPercent;

GSystem::GSystem ()
{
}

GSystem::~GSystem ()
{
}

bool GSystem::BreakProcess ( ProcessID pid, BreakType bt, bool breakTree )
{
   switch (bt)
   {
      case BT_CtrlC:
      {
         if (::DosSendSignalException(pid, XCPT_SIGNAL_INTR) == NO_ERROR)
            return true;
         break;
      }

      case BT_CtrlBreak:
      {
         if (::DosSendSignalException(pid, XCPT_SIGNAL_BREAK) == NO_ERROR)
            return true;
         break;
      }

      case BT_Exit: // OS2 doesn't have any way to "exit" a remote process,
      case BT_Kill: // so just kill it anyway.
      {
         int action = (breakTree ? DKP_PROCESSTREE : DKP_PROCESS);
         if (::DosKillProcess(action, pid) == NO_ERROR)
            return true;
         break;
      }
   }
   return false;
}

void GSystem::Beep ( int freq, int ms )
{
   ::DosBeep(freq, ms);
   // Since most beeps are produced due to some error condition it might be 
   // usable to print the stack-trace to the log.
   if (GLog::Filter(GLog::TEST))
      GLog::PrintStackTrace("%s: freq=%d, ms=%d", GVArgs(__FUNCTION__).add(freq).add(ms));
}

GDimension GSystem::GetScreenSize ()
{
   int width = GetSystemMetrics(GSystem::SMID_CXSCREEN);
   int height = GetSystemMetrics(GSystem::SMID_CYSCREEN);
   return GDimension(width, height);
}

bool GSystem::IsShiftKeyDown ( ShiftKeyId keyId )
{
   int vk;
   switch (keyId)
   {
      case SK_SHIFT: vk = VK_SHIFT; break;
      case SK_ALT: vk = VK_ALT; break;
      case SK_CONTROL: vk = VK_CTRL; break;
      case SK_CAPS_LOCK: vk = VK_CAPSLOCK; break;
      case SK_NUM_LOCK: vk = VK_NUMLOCK; break;
      case SK_SCROLL_LOCK: vk = VK_SCRLLOCK; break;
      case SK_MOUSE_BUTTON1: vk = VK_BUTTON1; break;
      case SK_MOUSE_BUTTON2: vk = VK_BUTTON2; break;
      default: gthrow_(GIllegalArgumentException(GString("Unsupported key id: %d", GVArgs(keyId))));
   }
   LONG state = ::WinGetKeyState(HWND_DESKTOP, vk);
   return (state & 0x8000) != 0;
}

bool GSystem::IsShiftKeyToggledOn ( ShiftKeyId keyId )
{
   int vk;
   switch (keyId)
   {
      case SK_SHIFT: vk = VK_SHIFT; break;
      case SK_ALT: vk = VK_ALT; break;
      case SK_CONTROL: vk = VK_CTRL; break;
      case SK_CAPS_LOCK: vk = VK_CAPSLOCK; break;
      case SK_NUM_LOCK: vk = VK_NUMLOCK; break;
      case SK_SCROLL_LOCK: vk = VK_SCRLLOCK; break;
      default: gthrow_(GIllegalArgumentException(GString("Unsupported key id: %d", GVArgs(keyId))));
   }
   LONG state = ::WinGetKeyState(HWND_DESKTOP, vk);
   return (state & 0x0001) != 0;
}

bool GSystem::CopyTextToClipboard ( const GString& text )
{
   if (::WinOpenClipbrd(GProgram::hAB))
   {
      char* textBuff = null;
      int textLen = text.length();
      if (::DosAllocSharedMem((void**) &textBuff, null, textLen, OBJ_GIVEABLE | PAG_COMMIT | PAG_READ | PAG_WRITE) != NO_ERROR)
      {
         ::WinCloseClipbrd(GProgram::hAB);
         return false;
      }

      strcpy(textBuff, text.cstring());
      ::WinSetClipbrdData(GProgram::hAB, ULONG(textBuff), CF_TEXT, CFI_POINTER);
      ::WinCloseClipbrd(GProgram::hAB);
   }
   return true;
}

GString GSystem::GetClipboardText ()
{
   GString ret(256);

   if (WinOpenClipbrd(GProgram::hAB))
   {
      ULONG hdata = WinQueryClipbrdData(GProgram::hAB, CF_TEXT);
      if (hdata != 0)
      {
         const char* txt = (const char*) hdata;
         ret = txt;
      }
      WinCloseClipbrd(GProgram::hAB);
   }

   return ret;
}

bool GSystem::IsAnyTextOnClipboard ()
{
   bool ret = false;

   if (WinOpenClipbrd(GProgram::hAB))
   {
      ret = WinQueryClipbrdData(GProgram::hAB, CF_TEXT) != 0;
      WinCloseClipbrd(GProgram::hAB);
   }

   return ret;
}

GSystem::PlatformID GSystem::GetPlatformID ()
{
   static PlatformID Id = PlatformID(-1);
   if (Id != PlatformID(-1))
      return Id;

   ULONG majorVer = 0;
   ULONG minorVer = 0;
   APIRET rc;
   rc = ::DosQuerySysInfo(QSV_VERSION_MAJOR, QSV_VERSION_MAJOR, &majorVer, sizeof(majorVer));
   if (rc != NO_ERROR)
      gthrow_(GIllegalStateException(GString("Error getting major OS/2 version. Error message is: %s", GVArgs(GSystem::GetApiRetString(rc)))));
   rc = ::DosQuerySysInfo(QSV_VERSION_MINOR, QSV_VERSION_MINOR, &minorVer, sizeof(minorVer));
   if (rc != NO_ERROR)
      gthrow_(GIllegalStateException(GString("Error getting minor OS/2 version. Error message is: %s", GVArgs(GSystem::GetApiRetString(rc)))));
   if (majorVer == 20)
   {
      if (minorVer < 10)
         Id = Platform_OS2_2_0;
      else
      if (minorVer < 11)
         Id = Platform_OS2_2_1;
      else
      if (minorVer < 30)
         Id = Platform_OS2_2_11;
      else
      if (minorVer < 40)
         Id = Platform_OS2_3_0;
      else
      if (minorVer < 45)
         Id = Platform_OS2_4_0;
      else
         Id = Platform_OS2_4_5;
   }
   else
   {
      // Something's wrong, assume OS/2 Warp 3.
      Id = Platform_OS2_3_0;
   }
   return Id;
}

GString GSystem::GetPlatformName ()
{
   switch (GSystem::GetPlatformID())
   {
      case Platform_OS2_2_0: 
         return "OS/2 2.0";
	   case Platform_OS2_2_1: 
         return "OS/2 2.1";
	   case Platform_OS2_2_11: 
         return "OS/2 2.11";
	   case Platform_OS2_3_0: 
         return "OS/2 Warp 3.0";
      case Platform_OS2_4_0: 
         return "OS/2 Warp 4.0";
      case Platform_OS2_4_5: 
      default: 
         return "OS/2 Warp 4.5 (Warp 4 FP 13, WSeB, MCP/ACP, eComStation)";
   }
}

ulonglong GSystem::CurrentTimeMillis ()
{
   // One thread at a time here!
   static GObject* LockCurrentTimeMillis = null;
   if (LockCurrentTimeMillis == null)
      LockCurrentTimeMillis = new GObject();
   GObject::Synchronizer synch(*LockCurrentTimeMillis);
   ULONG ticks = 0;
   ::DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT, &ticks, sizeof(ticks));
   // Attempt to make it work longer than those ~49.7 days.
   static ULONG OldTicks = 0;
   static ulonglong TotalTicks = 0;
   if (ticks < OldTicks)
      OldTicks = 0; // The uptime wrapped! Slightly inaccurate.
   TotalTicks += ticks - OldTicks;
   OldTicks = ticks;
   return TotalTicks;
}

int GSystem::GetCpuUsageInPercent ()
{
   // One thread at a time here!
   GObject::Synchronizer synch(LockGetCpuUsageInPercent);

   static bool FirstTime = true;

   int ret = 0;

   // This code is partly taken from the public EDM/2 article
   // "Measuring CPU Usage" at http://www.edm2.com/0607/cpu.html

   static double ts_val_prev;
   static double idle_val_prev;
   static double busy_val_prev;
   static double intr_val_prev;

   CPUUTIL cpu;
   APIRET rc = ::DosPerfSysCall(CMD_KI_RDCNT, (ULONG) &cpu, 0, 0);
   if (rc != NO_ERROR)
   {
      if (GLog::Filter(GLog::DEBUG))
         GLog::Log(null, "CMD_KI_RDCNT failed, rc=%d (%s:%d)", GVArgs(rc).add(__FUNCTION__).add(__LINE__));
      return 0;
   }

   // Convert 8-byte (low, high) time value to double.
   #define LL2F(high, low) (4294967296.0*(high)+(low))
   double ts_val = LL2F(cpu.ulTimeHigh, cpu.ulTimeLow);
   double idle_val = LL2F(cpu.ulIdleHigh, cpu.ulIdleLow);
   double busy_val = LL2F(cpu.ulBusyHigh, cpu.ulBusyLow);
   double intr_val = LL2F(cpu.ulIntrHigh, cpu.ulIntrLow);
   #undef LL2F

   if (!FirstTime)
   {
      double  ts_delta = ts_val - ts_val_prev;
      double idlePercent = (idle_val - idle_val_prev) / ts_delta * 100.0;
      // double busyPercent = (busy_val - busy_val_prev) / ts_delta * 100.0;
      // double intrPercent = (intr_val - intr_val_prev) / ts_delta * 100.0;
      double cpuUsage = 100.0 - idlePercent;
      ret = (int) cpuUsage;
   }

   ts_val_prev = ts_val;
   idle_val_prev = idle_val;
   busy_val_prev = busy_val;
   intr_val_prev = intr_val;
	
   FirstTime = false;
	return ret;
}

GString GSystem::ScanSystemConfig ( const GString& keyWord )
{
   ULONG BootDrive;
   ::DosQuerySysInfo(QSV_BOOT_DRIVE, QSV_BOOT_DRIVE, &BootDrive, sizeof(BootDrive)) ;

   try {

      GVfsLocal vfs;
      GString buff(4096);
      GString path("%c:\\config.sys", GVArgs(BootDrive + 'A' - 1));
      GFileInputStream file(vfs, path, true);
      for (;;)
      {
         // Read the next line. The buffer will never exceed, because the
         // GString automatically expand it self if needed.
         if (file.readString(buff).length() <= 0)
            break;

         buff.stripTrailingEol();
         const char *p = buff.cstring();
         while (*p && ((*p == ' ') || (*p == '\t')))
            p++;

         if (strnicmp(p, keyWord.cstring(), keyWord.length()) != 0)
            continue;

         p += keyWord.length();
         while (*p && ((*p == ' ') || (*p == '\t')))
            p++;

         if (*p++ != '=')
            continue;

         while (*p && ((*p == ' ') || (*p == '\t')))
            p++;

         return GString(p);
      }

   } catch (GIOException& e) {
      // Failed on open the source file.
   }

   return GString::Empty;
}

GString GSystem::GetLibPath ()
{
   return GSystem::ScanSystemConfig("LIBPATH");
}

APIRET GSystem::SetSystemRegistryString ( const GString& sectionName,
                                          const GString& entryName,
                                          const GString& value )
{
   return ::PrfWriteProfileString(HINI_USERPROFILE, sectionName, entryName, value);
}

GString GSystem::GetSystemRegistryString ( const GString& sectionName,
                                           const GString& entryName,
                                           const GString& defaultValue )
{
   const BUFF_SIZE = 32001;
   char* buff = new char[BUFF_SIZE];
   buff[0] = '\0';
   PrfQueryProfileString(HINI_USERPROFILE, sectionName, entryName, defaultValue, buff, BUFF_SIZE);
   GString ret = buff;
   delete[] buff;
   return ret;
}

GColor GSystem::GetSystemColor ( SystemColorID id )
{
   int sysID;
   switch (id)
   {
      case SCID_DIALOGBCK: sysID = SYSCLR_DIALOGBACKGROUND; break;
      case SCID_DIALOGTXT: sysID = SYSCLR_WINDOWTEXT; break;
      case SCID_BUTTONBCK: sysID = SYSCLR_BUTTONMIDDLE; break;
      case SCID_BUTTONTXT: sysID = SYSCLR_BUTTONDEFAULT; break;
      case SCID_BUTTONDBOX: return GColor::BLACK;
      case SCID_3DLIGHT: sysID = SYSCLR_BUTTONLIGHT; break;
      case SCID_3DSHADOW: sysID = SYSCLR_BUTTONDARK; break;
      case SCID_MENUBCK: sysID = SYSCLR_MENU; break;
      case SCID_MENUTXT: sysID = SYSCLR_MENUTEXT; break;
      case SCID_MENUDISABLEDTXT: sysID = SYSCLR_MENUDISABLEDTEXT; break;
      case SCID_MENUHILITEBCK: sysID = SYSCLR_MENUHILITEBGND; break;
      case SCID_MENUHILITETXT: sysID = SYSCLR_MENUHILITE; break;
      case SCID_SCROLLBARBCK: sysID = SYSCLR_SCROLLBAR; break;
      case SCID_TEXTENTRYBCK: sysID = SYSCLR_ENTRYFIELD; break;
      case SCID_TEXTENTRYTXT: sysID = SYSCLR_WINDOWTEXT; break;
      default:
         gthrow_(GIllegalArgumentException(GStringl("Unknown system metric ID: %d", GVArgs(int(id)))));
   }
   return GColor(::WinQuerySysColor(HWND_DESKTOP, sysID, 0));
}

int GSystem::GetSystemMetrics ( GSystem::SystemMetricsID id )
{
   // Convert the ID to its system dependent value.
   int sysID;
   switch (id)
   {
      case SMID_CYTITLEBAR: sysID = SV_CYTITLEBAR; break;
      case SMID_CXDLGFRAME: sysID = SV_CXDLGFRAME; break;
      case SMID_CYDLGFRAME: sysID = SV_CYDLGFRAME; break;
      case SMID_INSERTMODE: sysID = SV_INSERTMODE; break;
      case SMID_CXSCREEN: sysID = SV_CXSCREEN; break;
      case SMID_CYSCREEN: sysID = SV_CYSCREEN; break;
      case SMID_CXVSCROLL: sysID = SV_CXVSCROLL; break;
      case SMID_CYHSCROLL: sysID = SV_CYHSCROLL; break;
      case SMID_CXICON: sysID = SV_CXICON; break;
      case SMID_CXICONSMALL: return GetSystemMetrics(SMID_CXICON) / 2;
      default:
         gthrow_(GIllegalArgumentException(GStringl("Unknown system metric ID: %d", GVArgs(int(id)))));
   }
   return ::WinQuerySysValue(HWND_DESKTOP, sysID);
}

void GSystem::SetMouseCursorShape ( GSystem::MouseCursorShapeID id )
{
   int sysID;
   switch (id)
   {
      case MCSID_NORMAL: sysID = SPTR_ARROW; break;
      case MCSID_TEXT: sysID = SPTR_TEXT; break;
      case MCSID_WAIT: sysID = SPTR_WAIT; break;
      case MCSID_MOVE: sysID = SPTR_MOVE; break;
      case MCSID_SIZENWSE: sysID = SPTR_SIZENWSE; break;
      case MCSID_SIZENESW: sysID = SPTR_SIZENESW; break;
      case MCSID_SIZEWE: sysID = SPTR_SIZEWE; break;
      case MCSID_SIZENS: sysID = SPTR_SIZENS; break;
      default:
         gthrow_(GIllegalArgumentException(GStringl("Unknown mouse cursor shape ID: %d", GVArgs(int(id)))));
   }
   HPOINTER hicon = ::WinQuerySysPointer(HWND_DESKTOP, sysID, false);
   if (hicon != null)
      ::WinSetPointer(HWND_DESKTOP, hicon);
}

bool GSystem::OpenRecycleBinObject ()
{
   bool ok = GSystem::OpenShellObject("<XWP_TRASHCAN>");
   if (!ok)
   {
      if (GLog::Filter(GLog::PROD))
         GLog::Log(null, "Could not open shell object: <XWP_TRASHCAN> (%s:%d)", GVArgs(__FUNCTION__).add(__LINE__));
      GSystem::Beep();
      // This probably means that XWorkplace is not installed on this 
      // OS/2 system. Standard OS/2 Warp has no "trashcan", but only 
      // a "shredder" (which is not a container).
   }
   return ok;
}

bool GSystem::OpenShellObject ( const GString& path )
{
   // Get rid of the trailing slash (if any) if the path is a directory only.
   GString p = path;
   int pathLen = path.length();
   if (p.endsWith(GFile::SlashChar) && pathLen > 1 && p[pathLen - 2] != ':')
      p.cutTailFrom(pathLen - 1);
   HOBJECT hobj = ::WinQueryObject(p);
   if (hobj == null)
      return false;
   if (!::WinOpenObject(hobj, OPEN_DEFAULT, true))
      return false;
   // Open it once more, to make sure that it is opened in the foreground.
   GThread::Sleep(250);
   if (!::WinOpenObject(hobj, OPEN_DEFAULT, true))
      return false;
   return true;
}

GString& GSystem::StrUpper ( GString& str )
{
   ::WinUpper(GProgram::hAB, 0, 0, (char *) str.cstring());
   return str;
}

int GSystem::StrCompare ( const GString& str1,
                          const GString& str2,
                          bool caseSen )
{
   if (caseSen)
   {
      switch (::WinCompareStrings(GProgram::hAB, 0, 0, str1, str2, 0))
      {
         case WCS_LT: return -1; break;
         case WCS_GT: return +1; break;
         case WCS_EQ: default: return 0; break;
      }
   }
   else
   {
      GString comp1(str1);
      GString comp2(str2);
      StrUpper(comp1);
      StrUpper(comp2);
      switch (::WinCompareStrings(GProgram::hAB, 0, 0, comp1, comp2, 0))
      {
         case WCS_LT: return -1; break;
         case WCS_GT: return +1; break;
         case WCS_EQ: default: return 0; break;
      }
   }
}

GString GSystem::GetApiRetString ( APIRET rc )
{
   if (rc == NO_ERROR)
      return GString::Empty;
   char msg[512];
   ULONG bc = 0;
   if (::DosGetMessage(null, 0, msg, sizeof(msg)-1, rc, "OSO001.MSG", &bc) != NO_ERROR)
      return GStringl("%ERROR_Unknown_Error", GVArgs(rc));
   msg[bc] = '\0';
   GString ret = msg;
   ret.stripTrailingEol();
   return ret;
}

GString GSystem::GetApiRetString ( GError& err )
{
   return GSystem::GetApiRetString(err.sysErrorCode);
}

GString GSystem::GetUserApplicationDataDirectory ( const GString& applName )
{
   // Use the %HOME% environment variable by default.
   GProgram& prg = GProgram::GetProgram();
   GString home = prg.getEnvironmentVar("home");
   if (home == "")
      home = prg.getRootDirectory();
   GFile::Slash(home);
   return home;
}

void GSystem::CreateSystemShellObject ( const GString& dstPath, 
                                        const GString& workDir,
                                        const GString& name )
{
   GVfsLocal localVfs;
   GFileItem fitem(localVfs, dstPath);
   if (fitem.isProgramFileName())
   {
      // Create a program object on the desktop.
      GString params = "EXENAME=" + dstPath + ";STARTUPDIR=" + workDir;
      ::WinCreateObject("WPProgram", name.cstring(), params.cstring(), "<WP_DESKTOP>", CO_FAILIFEXISTS);
   }
   else
   {
      // Create a shadow object on the desktop.
      HOBJECT hObjDesktop = ::WinQueryObject("<WP_DESKTOP>");
      HOBJECT hObjThisObj = ::WinQueryObject(dstPath.cstring());
      if (hObjDesktop != null && hObjThisObj != null)
      {
         HOBJECT hObjShadow = ::WinCreateShadow(hObjThisObj, hObjDesktop, 0);
         if (hObjShadow != null)
            return;
      }
   }
}

bool GSystem::IsPlatformOS2 ()
{
   return true;
}

bool GSystem::IsPlatformWindows ()
{
   return false;
}
