/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ARRAY
#define __GLIB_ARRAY

#include <malloc.h>
#include "glib/primitives/GObject.h"
#include "glib/util/GComparator.h"
#include "glib/util/GArrayIndexOutOfBoundsException.h"

/**
 * This is the base class of the <i>GArray</i> template.
 *
 * This class contains most of the actual implementation of the code
 * needed for a dynamic array (vector) of void pointers. By putting the
 * implementation into a non-template base class like this then we achieve
 * to minimize the size of the code generated for each template instance.
 *
 * @author  Leif Erik Larsen
 * @since   2000.02.29
 * @see     GArray
 */
class GArrayImpl : public GObject, public GComparator
{
   protected:

      /** Current number of items in array. */
      int num;

      /** Current allocated size of the array. */
      int size;

      /** Initial size of the array. */
      int initial;

      /** The array grow factor, by number (if positive) or by factor (if negative). A default grow size will be used if this is zero. */
      int incremental;

      /** Class used to wrap the actual data element and its "autoDelete" flag. */
      class Item : public GObject
      {
         public:

            bool autoDelete;
            GObject* data;

         public:

            Item ( GObject* data, bool autoDelete );
            virtual ~Item ();
      };

      /** The array it self, containing pointers to items. */
      Item** items;

   protected:

      /** @see GArray#GArray */
      GArrayImpl ( int initial, int incremental );

      /** @see GArray#~GArray */
      virtual ~GArrayImpl ();

   private:

      /** Disable the copy constructor, because it is implemented by our subclass. */
      GArrayImpl ( const GArrayImpl& src ) {}

      /** Disable the assignment operator, because it is implemented by our subclass. */
      virtual GArrayImpl& operator= ( const GArrayImpl& src ) { return *this; }

   private:

      /**
       * Swap the two specified items.
       */
      void swap ( int idx1, int idx2 );

   public:

      /**
       * A default implementation of {@link GComparator#compare2Objects} 
       * that is intended to be overridden. This default implementation 
       * will do a comparization of the two objects by calling 
       * {@link GObject#compareObj} on obj1, giving obj2 as an argument.
       *
       * This method is used by {@link #sortItems(bool).
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       */
      virtual int compare2Objects ( const GObject& obj1, const GObject& obj2 ) const;

      /**
       * Extend the Array with as many additional items as was set by the
       * 'incremental' parameter.
       */
      void enlarge ();

      /**
       * If the current capacity is less than the specified count then
       * reallocate space to make room for exactly the specified count of items.
       */
      void ensureCapacity ( int count );

      /**
       * Return the allocated size of the Array.
       */
      int getAllocatedSize () const;

      /**
       * Return the current number of items that are contained in the Array.
       */
      int getCount () const;

      /**
       * Get the index of the specified item, or -1 if it does not exist
       * in the array.
       */
      int indexOf ( const GObject& anItem ) const;

      /**
       * Return true if and only if the indexed item is to be automatically 
       * destroyed by us. That is if the <code>autoDelete</code> argument 
       * was true when adding the indexed item to the array.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.02
       */
      bool isAutoDeleteAt ( int index ) const;

      /**
       * Return  True if and only if the array contain no items at all,
       *         or else return false.
       */
      bool isEmpty () const;

      /**
       * Remove the specified number of items from the Array, starting with
       * the item at the indexed position.
       *
       * @param  index    The index of the first item of which to remove.
       * @param  count    The number of items of which to remove.
       * @param  destroy  True if we shall destroy the item(s) of which we
       *                  remove, or else false if we shall just remove the
       *                  item(s) from the array without destroying the
       *                  contained object(s).
       */
      void remove ( int index, int count = 1, bool destroy = true );

      /**
       * Remove all items from the array.
       */
      void removeAll ();

      /**
       * Sort the items in the array, using the specified comparator.
       */
      void sortItems ( const GComparator& comparator, bool ascending = true );

      /**
       * Sort the items in the array, using our built-in comparator,
       * which is {@link #compare2Objects}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       */
      void sortItems ( bool ascending = true );

   protected:

      /** @see GArray#add */
      void add ( GObject* anItem, bool autoDelete );

      /** @see GArray#insert */
      void insert ( GObject* anItem, int index, bool autoDelete );

      /**
       * This method is to be used by {@link GArray#sortItems} only.
       *
       * If you think of a one dimensional array as going from
       * the lowest index on the left to the highest index on the right
       * then the parameters to this function are lowest index at
       * left and highest index at right. The first time you call
       * this method it will be with the parameters lo0=0, hi0=list.length - 1.
       *
       * @param list The array to sort.
       * @param lo0  Left boundary of array partition.
       * @param hi0  Right boundary of array partition.
       * @see GArray#sortItems
       */
      void qsort ( int lo0, 
                   int hi0, 
                   const GComparator& comparator, 
                   bool ascending );

   private:

      /**
       * This method is to be called by {@link #qsort} only.
       *
       * Compare the two objects, respecting the ascending/descending flag
       * so that the calling sorting algorithm doesn't need to care
       * about that flag.
       */
      int qsort_compare ( const GComparator& comparator, 
                          const GObject& obj1, 
                          const GObject& obj2, 
                          bool ascending ) const
      {
         int res = comparator.compare2Objects(obj1, obj2);
         return ascending ? res : -res;
      }
};

/**
 * Dynamic array for objects that have been allocated from the heap.
 *
 * All the objects in the array will be automatically destroyed whenever
 * needed, such as upon desctruction or upon calling {@link #remove}
 * or {@link #removeAll}.
 *
 * The array is even sortable. Use <i>sortItems()</i> to perform sorting.
 * The sorting algorithm used is Quick-Sort.
 *
 * This container template can store {@link GObject}'s of any type, as 
 * long as they supports the following: A public copy constructor and 
 * a public destructor. Also, if sort operation will be done on the 
 * array, all contained objects must implement {@link GObject#compareObj}
 * correctly.
 *
 * @author  Leif Erik Larsen
 * @since   1998.07.02
 * @see     GVector
 */
template <class T> class GArray : public GArrayImpl
{
   public:

      /**
       * Construct a new Array Object with the specified 
       * initial and incremental size.
       *
       * @author  Leif Erik Larsen
       * @since   1998.07.02
       * @param   initial     The initial size of the array. If this 
       *                      is <= 0 then we will use a default.
       * @param   incremental Specifies the number of items for which the 
       *                      array will automatically extend it self 
       *                      whenever needed. This can be an absolute 
       *                      positive value, or a negative factor. If 
       *                      negative, -1 means a factor of 1/1, 
       *                      -2 means factor of 1/2, -3 means factor 
       *                      of 1/3, etc. If zero is specified then we 
       *                      will use a default grow value.
       */
      explicit GArray ( int initial = 64, int incremental = -3 ) 
                 :GArrayImpl(initial, incremental) 
      {
      }

      /**
       * Be careful using the copy constructor if the array contains 
       * objects of different types.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.06
       */
      GArray ( const GArray<T>& src )
         :GArrayImpl(src.num, src.incremental) 
      {
         operator=(src);
      }

      /**
       * Destruct the Array Object and all of its elements.
       */
      virtual ~GArray () 
      { 
      }

      /**
       * The assignment operator.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       */
      GArray<T>& operator= ( const GArray<T>& src )
      {
         if (this == &src)
            return *this; 
         removeAll();
         ensureCapacity(src.num);
         for (int i=0, len=src.num; i<len; i++)
         {
            T& obj = src.get(i);
            T* clone = new T(obj);
            add(clone, true);
         }
         return *this; 
      }

   public:

      /**
       * Add another item to the end of the Array.
       * Extends the Array if necessary. Null-items are not supported,
       * and if a null-item is specified then we will throw 
       * a {@link GIllegalArgumentException}.
       */
      void add ( T* anItem, bool autoDelete = true ) 
      { 
         GArrayImpl::add(anItem, autoDelete); 
      }

      /**
       * Add another item to the end of the Array.
       * The item will be added with the "autoDelete" flag set to false.
       * Extends the Array if necessary.
       */
      void add ( T& anItem ) 
      { 
         GArrayImpl::add(&anItem, false); 
      }

      /**
       * Insert another item just above the indexed item in the Array.
       * Extends the Array if necessary. Null-items are not supported,
       * and if a null-item is specified then we will throw 
       * a {@link GIllegalArgumentException}.
       */
      void insert ( T* anItem, int index, bool autoDelete = true ) 
      { 
         GArrayImpl::insert(anItem, index, autoDelete); 
      }

      /**
       * Insert another item just above the indexed item in the Array.
       * The item will be inserted with the "autoDelet" flag set to false.
       * Extends the Array buffer if necessary.
       */
      void insert ( const T& anItem, int index ) 
      { 
         GArrayImpl::insert((T*) &anItem, index, false); 
      }

      /**
       * Get a reference to the indexed item.
       *
       * @throws GArrayIndexOutOfBoundsException if the specified index is
       *                                         outside legal range.
       */
      T& get ( int index ) const
      {
         if (index < 0 || index >= num)
            gthrow_(GArrayIndexOutOfBoundsException());
         return dynamic_cast<T&>(*items[index]->data);
      }

      /**
       * Use this operator just as {@link #get}.
       *
       * @throws GArrayIndexOutOfBoundsException if the specified index is
       *                                         outside legal range.
       */
      T& operator[] ( int index ) const
      {
         if (index < 0 || index >= num)
            gthrow_(GArrayIndexOutOfBoundsException());
         return dynamic_cast<T&>(*items[index]->data);
      }

      /**
       * Get a pointer to the last element in the array, or null if there
       * currently is no element in the array.
       */
      T* getLast () const
      {
         if (num >= 1)
            return &get(num - 1);
         else
            return null;
      }

      /**
       * Get a pointer to the first element in the array, or null if there
       * currently is no element in the array.
       */
      T* getFirst () const
      {
         if (num >= 1)
            return &get(0);
         else
            return null;
      }

      /**
       * Change the indexed item.
       */
      void set ( int index, T* item, bool autoDelete = true )
      {
         if (index < 0 || index >= num)
            gthrow_(GArrayIndexOutOfBoundsException());
         Item* itm = items[index];
         if (itm->autoDelete)
            delete itm->data;
         itm->autoDelete = autoDelete;
         itm->data = item;
      }
};

#endif
