/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_BUFFER
#define __GLIB_BUFFER

#include "glib/util/GArrayIndexOutOfBoundsException.h"

/**
 * A template class that is typically used to allocate a local memory 
 * buffer of some type, that automatically destroys its allocated 
 * memory upon scope exit. For instance:
 * 
 * <pre>
 *    GSysFileHandle hfile = ...;
 *    int numberOfBytesRead = 0;
 *    const int buffSize = 1024;
 *    GBuffer<byte> in(buffSize);
 *    GFile::FromFile(hfile, in.theBuffer, buffSize, &numberOfBytesRead);
 * </pre>
 *
 * @author  Leif Erik Larsen
 * @since   2004.12.20
 */
template <class T> class GBuffer : public GObject
{
   public:

      T* theBuffer;

   private:

      unsigned allocatedCount;

   public:

      GBuffer ( unsigned buffSize )
      {
         allocatedCount = buffSize;
         theBuffer = new T[buffSize];
      }

      virtual ~GBuffer ()
      {
         delete [] theBuffer;
      }

   public:

      void clearBufferContent ( int val = 0 )
      {
         memset(theBuffer, val, allocatedCount * sizeof(T));
      }

      /**
       * Get a reference to the indexed element.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.21
       * @throws  GArrayIndexOutOfBoundsException if the specified index is
       *                    out of bounds.
       */
      T& operator[] ( int index ) const
      {
         if (index < 0 || index >= num)
            gthrow_(GArrayIndexOutOfBoundsException());
         return items[index];
      }

      /**
       * Alternative method that can be used instead of {@link #operator[]}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.21
       * @throws  GArrayIndexOutOfBoundsException if the specified index is
       *                    out of bounds.
       */
      T& elementAt ( int index ) const throw(GArrayIndexOutOfBoundsException)
      {
         return operator[](index);
      }

      /**
       * Get the allocated size of the buffer, measured in number of 
       * elements (not bytes).
       *
       * @author  Leif Erik Larsen
       * @since   2005.04.20
       */
      int getSize () const
      {
         return allocatedCount;
      }
};

#endif
