/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_EXPRESSIONPARSER
#define __GLIB_EXPRESSIONPARSER

#include "glib/primitives/aptr.h"
#include "glib/primitives/GInteger.h"
#include "glib/primitives/GDouble.h"
#include "glib/util/GAbstractTokenizer.h"
#include "glib/util/GArray.h"

/**
 * This is a general expression parser class that can be used to analyze
 * and calculate mathematical expressions. It supports all the standard
 * operators (+, -, * and /) as well as parenthesis and correct operator
 * precedence.
 *
 * Below is a list of all the operators that are recognized by the parser.
 * Be aware, however, that these are the tokens recognized by the parser it
 * self only. It is up to the tokenizer to produce the supported tokens, or
 * else the parser will produce a {@link ISyntaxErrorException}. The
 * operators are listed in their respective precedence order.
 *
 * <table border cellspacing=0 cellpadding=5 width="100%" bgcolor="#ccffff">
 * <tr><td>"="</td><td>Assign.</td></tr>
 * <tr><td>"%"</td><td>Percent.</td></tr>
 * <tr><td>"!"</td><td>Boolean inverse.</td></tr>
 * <tr><td>"~"</td><td>Bitwise inverse.</td></tr>
 * <tr><td>"^"</td><td>Power.</td></tr>
 * <tr><td>"*"</td><td>Multiply.</td></tr>
 * <tr><td>"/"</td><td>Divide.</td></tr>
 * <tr><td>"&"</td><td>Bitwise and.</td></tr>
 * <tr><td>"+"</td><td>Add.</td></tr>
 * <tr><td>"-"</td><td>Subtract.</td></tr>
 * <tr><td>"|"</td><td>Bitwise or.</td></tr>
 * <tr><td>"=="</td><td>Equals.</td></tr>
 * <tr><td>"!="</td><td>Unlike.</td></tr>
 * <tr><td>"<="</td><td>Less or like.</td></tr>
 * <tr><td>">="</td><td>Larger or like.</td></tr>
 * <tr><td>"<"</td><td>Less.</td></tr>
 * <tr><td>">"</td><td>Larger.</td></tr>
 * <tr><td>"&&"</td><td>Boolean and.</td></tr>
 * <tr><td>"||"</td><td>Boolean or.</td></tr>
 * </table>
 *
 * The parser also has built in support for the below listed
 * set of functions.
 *
 * <table border cellspacing=0 cellpadding=5 width="100%" bgcolor="#ccffff">
 * <tr><td>abs</td><td>See {@link GMath#abs}.</td></tr>
 * <tr><td>acos</td><td>See {@link GMath#acos}.</td></tr>
 * <tr><td>asin</td><td>See {@link GMath#asin}.</td></tr>
 * <tr><td>atan</td><td>See {@link GMath#atan}.</td></tr>
 * <tr><td>ceil</td><td>See {@link GMath#ceil}.</td></tr>
 * <tr><td>cos</td><td>See {@link GMath#cos}.</td></tr>
 * <tr><td>exp</td><td>See {@link GMath#exp}.</td></tr>
 * <tr><td>floor</td><td>See {@link GMath#floor}.</td></tr>
 * <tr><td>log</td><td>See {@link GMath#log}.</td></tr>
 * <tr><td>random</td><td>See {@link GMath#random}.</td></tr>
 * <tr><td>sin</td><td>See {@link GMath#sin}.</td></tr>
 * <tr><td>sqrt</td><td>See {@link GMath#sqrt}.</td></tr>
 * <tr><td>tan</td><td>See {@link GMath#tan}.</td></tr>
 * </table>
 *
 * In addition the parser has built in support for the below listed
 * set of constants.
 *
 * <table border cellspacing=0 cellpadding=5 width="100%" bgcolor="#ccffff">
 * <tr><td>pi</td><td>Standard pi (3.14...).</td></tr>
 * <tr><td>true</td><td>Boolean true.</td></tr>
 * <tr><td>false</td><td>Boolean false.</td></tr>
 * </table>
 *
 * When it comes to numerical literals, the following formats are
 * supported: (<b>Mark</b> that all non-digit characters are
 * case-insensitive.)
 *
 * <table border cellspacing=0 cellpadding=5 width="100%" bgcolor="#ccffff">
 * <tr><td>11</td><td>Decimal integer 11</td></tr>
 * <tr><td>11.0</td><td>Decimal float 11.0</td></tr>
 * <tr><td>11.</td><td>Decimal float 11.0</td></tr>
 * <tr><td>.11</td><td>Decimal float 0.11</td></tr>
 * <tr><td>0x11</td><td>Hex 11 (decimal 17)</td></tr>
 * <tr><td>0h11</td><td>Hex 11 (decimal 17)</td></tr>
 * <tr><td>0o11</td><td>Octal 11 (decimal 9)</td></tr>
 * <tr><td>0b11</td><td>Binary 11 (decimal 3)</td></tr>
 * <tr><td>11t</td><td>Decimal integer 11*1000</td></tr>
 * <tr><td>11m</td><td>Decimal integer 11*1000*1000</td></tr>
 * <tr><td>11b</td><td>Decimal integer 11*1000*1000*1000</td></tr>
 * <tr><td>11.0t</td><td>Decimal float 11.0*1000.0</td></tr>
 * <tr><td>11.0m</td><td>Decimal float 11.0*1000.0*1000.0</td></tr>
 * <tr><td>11.0b</td><td>Decimal float 11.0*1000.0*1000.0*1000.0</td></tr>
 * <tr><td>11e10</td><td>Decimal float 11.0*10^10</td></tr>
 * </table>
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.29
 */
class GExpressionParser : public GObject
{
   private:

      static const GString Token_lpar;
      static const GString Token_rpar;
      static const GString Token_plus;
      static const GString Token_minus;
      static const GString Token_percent;
      static const GString Token_pi;
      static const GString Token_true;
      static const GString Token_false;

   private:

      GAbstractTokenizer& tokenizer;

   public:

      /**
       * Create a new expression parser that can be used to analyze
       * expression(s) in the specified tokenizer.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      explicit GExpressionParser ( GAbstractTokenizer& tokenizer );

      virtual ~GExpressionParser ();

   public:

      /**
       * Class used to represent a single operator.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      class Operator : public GObject
      {
         friend class GExpressionParser;

         public:

            /**
             * These are all the supported operators.
             *
             * It is very important that each operator is defined with a value
             * that corresponds with their precedence, so that the operator with
             * the greatest precedence is defined first and has the largest value.
             */
            enum OPERATOR_ID
            {
               OP_ASSIGN       = 99, // "="
               OP_BOOLINVERSE  = 98, // "!"
               OP_BITINVERSE   = 97, // "~"
               OP_POWER        = 96, // "^"
               OP_MULTIPLY     = 95, // "*"
               OP_DIVIDE       = 94, // "/"
               OP_BITAND       = 93, // "&"
               OP_ADD          = 92, // "+"
               OP_SUBTRACT     = 91, // "-"
               OP_BITOR        = 90, // "|"
               OP_EQUALS       = 89, // "=="
               OP_UNLIKE       = 88, // "!="
               OP_LESSORLIKE   = 87, // "<="
               OP_LARGERORLIKE = 86, // ">="
               OP_LESS         = 85, // "<"
               OP_LARGER       = 84, // ">"
               OP_BOOLAND      = 83, // "&&"
               OP_BOOLOR       = 82, // "||"
               OP_NOOP         = 0   // No operation
            };

         private:

            OPERATOR_ID idPrecedence;
            int numArgs;

            /** Line number this operator. */
            int lineNr;

            /** Column position of the last character of this operator. */
            int clmnPos;

         private:

            explicit Operator ( const GAbstractTokenizer& tokenizer );
            Operator ( const GAbstractTokenizer& tokenizer, OPERATOR_ID idPrecedence, int numArgs );
            Operator ( const Operator& src );

            virtual ~Operator ();

         public:

            int getLineNr () const;
            int getColumnPos () const;

            OPERATOR_ID getIDPrecedence () const;
            int getNumArgs () const;
            void set ( OPERATOR_ID idPrecedence, int numArgs );

            virtual GString toString () const;
      };

   public:

      /**
       * Represents a single precompiled R-Value.
       *
       * These are values that are not logically bound to a memory buffer
       * and therefore cannot reside on the "left side" of an ASSIGN
       * operator. R-Values can only resign on the "right side" of it.
       *
       * The various methods of this class and its sub-classes will
       * not throw any other types of exceptions than
       * {@link #GSyntaxErrorException}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       */
      class RValue : public GObject
      {
         private:

            /** Line number this token. */
            int lineNr;

            /** Column position of the last character of this token. */
            int clmnPos;

         protected:

            const GAbstractTokenizer& tokenizer;

         protected:

            RValue ( const GAbstractTokenizer& tokenizer );

         public:

            /**
             * This destructor does nothing but some debug logging.
             */
            virtual ~RValue ();

         public:

            int getLineNr () const;
            int getColumnPos () const;

            // Below methods are overrideable by subclasses.
            // Remember that only GSyntaxErrorException's are allowed to
            // be thrown.

            virtual RValue* clone () const = 0;
            virtual aptr<RValue> operate ( const Operator& op ) const;
            virtual aptr<RValue> operate ( const Operator& op, const RValue& arg ) const;
            virtual aptr<RValue> allegate () const;
            virtual aptr<RValue> negate () const;
            virtual GString getTypeName () const = 0;

         public:

            virtual GString toString () const;
            virtual bool isDouble () const = 0;
            virtual bool isTrue () const = 0;
            virtual GString getString () const = 0;
            virtual int getInt () const = 0;
            virtual double getDouble () const = 0;
      };

   public:

      /**
       */
      class RValueInteger : public RValue
      {
         private:

            int val;

         public:

            RValueInteger ( const GAbstractTokenizer& tokenizer, int val );
            RValueInteger ( const GAbstractTokenizer& tokenizer, double val );

            virtual ~RValueInteger ();

         public:

            RValue* clone () const;
            bool isDouble () const;
            aptr<RValue> allegate () const;
            aptr<RValue> negate () const;
            GString getTypeName () const;
            bool isTrue () const;
            GString getString () const;
            int getInt () const;
            double getDouble () const;
            aptr<RValue> operate ( const Operator& op ) const;
            aptr<RValue> operate ( const Operator& op, const RValue& arg ) const;
      };

   public:

      /**
       */
      class RValueDouble : public RValue
      {
         private:

            double val;

         public:

            RValueDouble ( const GAbstractTokenizer& tokenizer, double val );

            virtual ~RValueDouble ();

         public:

            RValue* clone () const;
            bool isDouble () const;
            aptr<RValue> allegate () const;
            aptr<RValue> negate () const;
            GString getTypeName () const;
            bool isTrue () const;
            GString getString () const;
            int getInt () const;
            double getDouble () const;
            aptr<RValue> operate ( const Operator& op ) const;
            aptr<RValue> operate ( const Operator& op, const RValue& arg ) const;
      };

   public:

      /**
       */
      class RValueBoolean : public RValue
      {
         private:

            bool val;

         public:

            RValueBoolean ( const GAbstractTokenizer& tokenizer, bool val );

            virtual ~RValueBoolean ();

         public:

            RValue* clone () const;
            bool isDouble () const;
            GString getTypeName () const;
            bool isTrue () const;
            GString getString () const;
            int getInt () const;
            double getDouble () const;
            aptr<RValue> operate ( const Operator& op ) const;
            aptr<RValue> operate ( const Operator& op, const RValue& arg ) const;
      };

   public:

      /**
       */
      class RValueString : public RValue
      {
         private:

            GString val;

         public:

            RValueString ( const GAbstractTokenizer& tokenizer, const GString& val );

            virtual ~RValueString ();

         public:

            RValue* clone () const;
            bool isDouble () const;
            GString getTypeName () const;
            bool isTrue () const;
            GString getString () const;
            int getInt () const;
            double getDouble () const;
            aptr<RValue> operate ( const Operator& op ) const;
            aptr<RValue> operate ( const Operator& op, const RValue& arg ) const;
      };

   friend class Operator;
   friend class RValue;
   friend class RValueBoolean;
   friend class RValueDouble;
   friend class RValueInteger;
   friend class RValueString;

   public:

      /**
       * Parse and calculate the result of the specified expression.
       *
       * We will automatically create a temporary {@link GStringTokenizer}
       * object to tokenize the specified string.
       *
       * Since the string tokenizer will always use an in-memory
       * input stream it will not happen that any GIOException is
       * thrown when this static method is used. Thus, the caller
       * does not need to catch any other types of exceptions
       * than {@link GSyntaxErrorException}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The resulting R-Value of the expression.
       * @throws  GSyntaxErrorException in case of any syntax error in
       *                                the specified string.
       */
      static aptr<RValue> GetParsedExpression ( const GString& expr );

      /**
       * Get the value as is the result of the next expression.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The resulting R-Value of the next expression.
       * @throws  GSyntaxErrorException in case of any syntax error in
       *                                the tokenized expression.
       * @throws  GIOException in case of any I/O error when reading
       *                       from the input stream of the tokenizer.
       */
      aptr<RValue> getParsedExpression ();

   private:

      /**
       * Get the operator code of the specified token.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   token   The token of which to parse.
       * @param   op      The operator object of which to update.
       * @return  True on success, or else false if the specified token is
       *          not a known operator. If we return false then the caller
       *          can assume that we has not touched the content of the
       *          <i>op</i> parameter.
       */
      bool getOperator ( const GAbstractToken& token, Operator& op ) const;

      /**
       * Get the next R-value.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GSyntaxErrorException in case of any syntax error in
       *                                the tokenized expression.
       * @throws  GIOException in case of any I/O error when reading
       *                       from the input stream of the tokenizer.
       */
      aptr<RValue> getRValue ();

      /**
       * Get the value as is the result of the next sub-expression.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The resulting R-Value of the next expression.
       * @throws  GSyntaxErrorException in case of any syntax error in
       *                                the tokenized expression.
       * @throws  GIOException in case of any I/O error when reading
       *                       from the input stream of the tokenizer.
       */
      aptr<RValue> parseSubExpression ();
};

#endif


