/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FIXEDVECTOR
#define __GLIB_FIXEDVECTOR

#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/util/GArrayIndexOutOfBoundsException.h"
#include "glib/util/GComparator.h"

/**
 * Array for objects that are not allocated on the heap.
 *
 * The array is fixed in that its number of elements is constant.
 * In other words: The vector is not extendible, its size must be set
 * upon construction.
 *
 * All elements of the whole array will always contain a valid object.
 * Elements not explicitly set by the client is created automatically
 * using its default constructor.
 *
 * Thus, this template can be used this way:
 * <pre>
 *   GFixedVector<int> v(3);
 *   v[0] = 123;
 *   v[1] = 234;
 *   int i = v[2]; // OK: i will be 0.
 * </pre>
 *
 * Doing the same with {@link GVector} requires another code:
 * <pre>
 *   GVector<int> v(3);
 *   v.add(123);
 *   v.add(234);
 *   int i = v[2]; // Error: Will throw GArrayIndexOutOfBoundsException
 * </pre>
 *
 * @author  Leif Erik Larsen
 * @since   2001.05.07
 * @see     GVector
 * @see     GArray
 */
template <class T> class GFixedVector : public GObject
{
   private:

      /** Current number of items in the vector. */
      int num;

      /** The array it self, containing items by value. */
      T* items;

   public:

      /**
       * Construct a new vector with the specified initial and incremental
       * size.
       */
      explicit GFixedVector ( int num )
         :num(num),
          items(null)
      {
         if (num < 0)
            gthrow_(GIllegalArgumentException("num < 0"));
         items = new T[num];
      }

      /**
       * Copy constructor.
       */
      explicit GFixedVector ( const GFixedVector<T>& src )
      {
         num = src.num;
         items = new T[num];

         // Copy the new content.
         for (int i=0; i<num; i++)
            items[i] = src.items[i];
      }

      /**
       * Destruct the vector and all of its elements.
       */
      virtual ~GFixedVector ()
      {
         destruct();
      }

   private:

      /**
       * Destruct all elements, and delete the element array it self.
       */
      void destruct ()
      {
         delete [] items;
         items = null;
         num = 0;
      }

   public:

      GFixedVector<T>& operator= ( const GFixedVector<T>& src )
      {
         // Remove the old content.
         destruct();

         // Prepare for the new content.
         num = src.num;
         items = new T[num];

         // Copy the new content.
         for (int i=0; i<num; i++)
            items[i] = src.items[i];

         return *this;
      }

      bool operator== ( const GFixedVector<T>& src ) const
      {
         if (num != src.num)
            return false;
         for (int i=0; i<num; i++)
            if (items[i] != src.items[i])
               return false;
         return true;
      }

      bool operator!= ( const GFixedVector<T>& src ) const
      {
         return !operator==(src);
      }

      T& operator[] ( int index ) const throw(GArrayIndexOutOfBoundsException)
      {
         if (index < 0 || index >= num)
            gthrow_(GArrayIndexOutOfBoundsException());
         return items[index];
      }

   public:

      /**
       * Get the index of the first item in the vector that is equal to the
       * specified item, or -1 if it does not exist in the vector.
       */
      int indexOf ( const T& anItem ) const
      {
         for (int i=0; i<num; i++)
            if (items[i] == anItem)
               return i;
         return -1;
      }

      /**
       * Return the current number of items that are contained in the vector.
       */
      int length () const
      {
         return num;
      }

      /**
       * Swap the two indexed items.
       */
      void swap ( int idx1, int idx2 )
      {
         T tmp = items[idx1];
         items[idx1] = items[idx2];
         items[idx2] = tmp;
      }

   private:

      /**
       * This method is to be called by {@link #qsort} only.
       *
       * Compare the two objects, respecting the ascending/descending flag
       * so that the calling sorting algorithm doesn't need to care
       * about that flag.
       */
      int qsort_compare ( const GComparator& comparator, 
                          const GObject& obj1, 
                          const GObject& obj2, 
                          bool ascending ) const
      {
         int res = comparator.compare2Objects(obj1, obj2);
         return ascending ? res : -res;
      }

      /**
       * This method is to be used by {@link #sortItems} only.
       *
       * If you think of a one dimensional array as going from
       * the lowest index on the left to the highest index on the right
       * then the parameters to this function are lowest index at
       * left and highest index at right. The first time you call
       * this method it will be with the parameters lo0=0, hi0=list.length - 1.
       *
       * @param list The array to sort.
       * @param lo0  Left boundary of array partition.
       * @param hi0  Right boundary of array partition.
       * @see   #sortItems
       */
      void qsort ( int lo0, int hi0, const GComparator& comparator, bool ascending )
      {
         int low = lo0;
         int high = hi0;
         if (hi0 > lo0)
         {
            // Arbitrarily establishing partition element as the midpoint of
            // the array.
            const T& mid = items[(lo0 + hi0) / 2];

            // loop through the array until indices cross
            while (low <= high)
            {
               // find the first element that is greater than or equal to
               // the partition element starting from the left Index.
               while ((low < hi0) && (qsort_compare(comparator, &items[low], &mid, ascending) < 0))
                  ++low;

               // find an element that is smaller than or equal to
               // the partition element starting from the right Index.
               while ((high > lo0) && (qsort_compare(comparator, &items[high], &mid, ascending) > 0))
                  --high;

               // if the indexes have not crossed, swap
               if (low <= high)
                  swap(low++, high--);
            }

            // If the right index has not reached the left side of array
            // must now sort the left partition.
            if (lo0 < high)
               qsort(lo0, high, comparator, ascending);

            // If the left index has not reached the right side of array
            // must now sort the right partition.
            if (low < hi0)
               qsort(low, hi0, comparator, ascending);
         }
      }

   public:

      /**
       * Sort the items in the array, using the specified comparator.
       */
      void sortItems ( const GComparator& comparator, bool ascending = true )
      {
         qsort(0, num - 1, comparator, ascending);
      }
};

#endif


