/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_KEYBAG
#define __GLIB_KEYBAG

#include "glib/util/GArray.h"

/**
 * This is the base class of the <i>GKeyBag</i> template and should
 * not be used by application code directly.
 *
 * This class contains some parts of the actual implementation of the
 * keybag code. By putting as much as possible of the implementation into
 * a non-template base class like this we achieve to minimize the size
 * of the code generated for each template instance.
 *
 * @author  Leif Erik Larsen
 * @since   2000.02.29
 * @see     GKeyBag
 */
class GKeyBagImpl : public GObject
{
   protected:

      /**
       * Class used as the storage container for objects in a {@link GKeyBag}.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.16
       */
      class Entry : public GObject
      {
         public:

            /**
             * The key name string of the bag entry.
             */
            GString key;

            /**
             * A reference to the actual object of the bag entry.
             */
            GObject* object;

            /**
             * True if the contained object should be destroyed when the
             * entry is destroyed and removed from the key bag.
             */
            bool autoDestroyObj;

         public:

            /**
             * Create a new GKeyBag entry.
             */
            Entry ( const GString& key, 
                    GObject* object, 
                    bool autoDestroyObj );

            virtual ~Entry ();

         private:

            /** Disable the copy constructor. */
            Entry ( const Entry& );

            /** Disable the assignment operator. */
            void operator= ( const Entry& );
      };

   protected:

      /** Table of reference to actual objects and keys. */
      GArray<Entry> objTable;

      /** True if the keys are not case significant, or else false. */
      bool ignoreCase;

   protected:

      /** @see GKeyBag#GKeyBag */
      explicit GKeyBagImpl ( int initial, int incremental, bool ignoreCase );
      virtual ~GKeyBagImpl ();

   private:

      /** Disable the copy constructor. */
      GKeyBagImpl ( const GKeyBagImpl& );

      /** Disable the assignment operator. */
      void operator= ( const GKeyBagImpl& );

   protected:

      /** @see GKeyBag#get */
      GObject* get ( const GString& key ) const;

      /** @see GKeyBag#put */
      bool put ( const GString& key, GObject* obj, bool autoDestroyObj );

      /** @see GKeyBag#update */
      void update ( int idx, GObject* obj, bool autoDestroyObj );
      bool update ( const GString& key, GObject* obj, bool autoDestroyObj );

   public:

      /**
       * Check for existense of the specified <code>key</code> in the bag.
       * Returns true if the bag contains an object associated to the specified
       * key, or else it will return false.
       *
       * @author  Leif Erik Larsen
       * @since   1997.07.05
       */
      bool containsKey ( const GString& key ) const;

      /**
       * Enlarge the bag with room for as many additional objects as was given
       * in the <code>iIncremental</code> parameter upon creation of the GKeyBag.
       * This method is probably to be used internally by this class only, but
       * it can also be called by program to manually extend the bag should it
       * be required.
       *
       * @author  Leif Erik Larsen
       * @since   1997.07.05
       * @return  The new size of the GKeyBag after the enlargement.
       */
      void enlarge ();

      /**
       * Find the object as is associated to the specified key.
       * This method will use the binary search algorithm to look up the key
       * as quick as possible.
       *
       * @author     Leif Erik Larsen
       * @since      1997.07.07
       * @param      key        Key of which object to find.
       * @param      closestIdx The index of the closest mach will be returned
       *                        in this int-wrapper. Regardless if the
       *                        specified key does actually exist or not.
       *                        By storing that index now we don't need to
       *                        perform another search of where to place it.
       *                        When we shall store it in the sorted table later
       *                        we simply can use this index value rather than
       *                        perform the search again. This solution is very
       *                        speed optimizing. If this argument is
       *                        <code>NULL</code> then we will not touch it at
       *                        all.
       * @return     Index of specified key, or else -1 if the key isn't
       *             contained in the bag.
       */
      int findEntry ( const GString& key, int* closestIdx = null ) const;

      /**
       * Return the number of objects currently contained in this GKeyBag.
       *
       * @author  Leif Erik Larsen
       * @since   1997.07.11
       */
      int getCount () const;

      /**
       * Get a reference to the indexed key string.
       *
       * @author  Leif Erik Larsen
       * @since   1998.08.19
       */
      const GString& getKey ( int idx ) const;

      /**
       * Return the "ignore case" flag of the bag.
       *
       * @author     Leif Erik Larsen
       * @since      1999.05.09
       */
      bool isIgnoreCase () const;

      /**
       * Remove the specified item from the bag.
       *
       * @param   key  Key of which object to remove.
       * @return  True on success, or else false on any error (e.g. the
       *          specified item didn't exist in the bag).
       */
      bool remove ( const GString& key );

      /**
       * Remove all items from the bag.
       *
       * @author     Leif Erik Larsen
       * @since      1998.07.13
       */
      void removeAll ();

      /**
       * Remove the indexed item from the bag.
       *
       * @author  Leif Erik Larsen
       * @since   2000.01.26
       * @param   idx     Index of which object to remove.
       * @param   destroy True if we shall automatically destroy the removed
       *                  item, but of course only if it was added with the
       *                  parameter "autoDestroyObj" as true.
       * @return  True on success, or else false on
       *          any error (e.g. the specified item didn't
       *          exist in the bag).
       */
      void removeIndexedItem ( int idx, bool destroy = true );
};

/**
 * General bag of where to store and access an unlimited number of
 * objects of any type, where each object is stored and accessed using a
 * string key.
 *
 * All objects in the bag must have an unique key.
 * The keys are treated to be case sensitive by default, but case
 * insensitive keys are also supported.
 *
 * Internally, this class will keep the table of objects sorted by key.
 * This is to be able to lookup a requested object by key as fast as
 * possible, using the binary search algorithm.
 *
 * Null-objects are not allowed to be contained in the bag, and if 
 * an attempt is done to add or change an object in the bag with a 
 * null-value then we will throw a {@link GIllegalArgumentException}.
 *
 * Here you have an example of how to use this template:
 *
 * <pre>
 *
 *    GKeyBag<GString> tst(3);
 *    tst.put("Key1", new GString("Test Nummer 1"));
 *    tst.put("Key2", new GString("Test Nummer 2"));
 *    tst.put("Key3", new GString("Test Nummer 3"));
 *    GString* str = tst.get("Key1");
 *
 * </pre>
 *
 * This container class can only store objects that are inherited
 * from {@link GObject}.
 *
 * @author  Leif Erik Larsen
 * @since   1997.07.05
 */
template <class T> class GKeyBag : public GKeyBagImpl
{
   public:

      /**
       * Constructs a new bag with initial room for <code>iInitial</code> number
       * of objects of any type. When the bag is full and there is an attempt
       * to put another object into it then the bag will be automatically
       * enlarged to be able to store <code>iIncremental</code> number of
       * additional objects.
       *
       * @param   iInitial     Initial size of the new bag. If this is a
       *                       value less than or equal to zero we will use
       *                       a default initial size.
       * @param   iIncremental Incremental size of the new bag. Whenever the
       *                       bag is enlarged it will be enlarged with room
       *                       for as many additional objects as was specified
       *                       by this parameter. If this is a negative value
       *                       we will use it as an incremental factor. For
       *                       example: If it is -2 we will enlarge the bag
       *                       with double the current size whenever it is
       *                       enlarged. If it is -4 we will enlarge it with a
       *                       factor of four, etc. -1 will be treated the
       *                       same way as 0, which is a default linear
       *                       incremental value.
       * @param   ignoreCase   True if the GKeyBag is to treate the Keys in a
       *                       case insignificant way.
       */
      explicit GKeyBag ( int initial = 100, 
                         int incremental = -3, 
                         bool ignoreCase = false ) 
                  : GKeyBagImpl(initial, 
                                incremental, 
                                ignoreCase) 
      {
      }

      /** Copy constructor. */
      GKeyBag ( const GKeyBag<T>& src ) 
         :GKeyBagImpl(src.getCount(), -3, src.isIgnoreCase())
      {
         operator=(src);
      }

      virtual ~GKeyBag () 
      { 
      }

   public:

      /** Assignment operator. */
      GKeyBag<T>& operator= ( const GKeyBag<T>& src ) 
      { 
         if (this == &src)
            return *this;
         if (isIgnoreCase() != src.isIgnoreCase())
            gthrow_(GIllegalArgumentException("Conflicting 'Ignore Case'!"));
         removeAll();
         int num = src.getCount();
         objTable.ensureCapacity(num);
         for (int i=0; i<num; i++)
         {
            Entry& next = src.objTable.get(i);
            const GString& key = next.key;
            T* obj = new T(dynamic_cast<T&>(*next.object));
            Entry* e = new Entry(key, obj, true);
            objTable.add(e, true);
         }
         return *this;
      }

      /**
       * Get a pointer to the object in the bag that is associated to the
       * specified key, or null if the bag doesn't contain any object
       * with the specified key.
       *
       * @author  Leif Erik Larsen
       * @since   1997.07.05
       */
      T* get ( const GString& key ) const 
      { 
         return dynamic_cast<T*>(GKeyBagImpl::get(key)); 
      }

      /**
       * Get a reference to the object in the bag at the specified index.
       * This method is to be used in special situations only, for instance for
       * speed optimizing reasons.
       *
       * @author  Leif Erik Larsen
       * @since   1998.09.06
       * @throws  GArrayIndexOutOfBoundsException if the specified index
       *                                          is out of bounds.
       */
      T& getIndexedItem ( int index ) const 
      { 
         return dynamic_cast<T&>(*objTable.get(index).object); 
      }

      /**
       * Put another object into the bag, associated to the specified key.
       * If an object in the bag is already using the specified key then
       * the new object will not be added and this method returns false,
       * or else it will return true to signal that the object was successfully
       * added.
       *
       * @author  Leif Erik Larsen
       * @since   1997.07.05
       * @return  True on success or else false of the key is already in use.
       */
      bool put ( const GString& key, T* obj, bool autoDestroyObj = true ) 
      { 
         return GKeyBagImpl::put(key, obj, autoDestroyObj); 
      }

      /**
       * Update an existing object in the bag, associated to the specified key.
       * If the specified object doesn't exist in the bag then
       * this method returns false, or else it will return true to signal that
       * the bag entry was successfully updated.
       *
       * @author  Leif Erik Larsen
       * @since   1997.11.19
       * @param   obj  The new object of which to put into the bag.
       * @param   key  The key of where to put the new object into the bag.
       * @param   autoDestroyObj  True if the new object should be 
       *               automatically destroyed when it is removed from 
       *               the bag.
       */
      bool update ( const GString& key, T* obj, bool autoDestroyObj = true ) 
      { 
         return GKeyBagImpl::update(key, obj, autoDestroyObj); 
      }

      bool update ( int idx, T* obj, bool autoDestroyObj = true ) 
      { 
         return GKeyBagImpl::update(idx, obj, autoDestroyObj); 
      }
};

#endif
