/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/util/GMath.h"
#include "glib/util/GRandom.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/sys/GSystem.h"

const longlong GRandom::Multiplier = (longlong(0x5DEECE66)<<4) | longlong(0xD); // Ends up as a "0x5DEECE66D" (36 bits).
const longlong GRandom::Addend = 0xBL;
const longlong GRandom::Mask = (longlong(1) << 48) - 1;

GRandom::GRandom ()
        :seed(0),
         nextNextGaussian(0),
         haveNextNextGaussian(false)
{
   longlong seed = GSystem::CurrentTimeMillis();
   setSeed(seed);
}

GRandom::GRandom ( longlong seed )
        :seed(0),
         nextNextGaussian(0),
         haveNextNextGaussian(false)
{
   setSeed(seed);
}

void GRandom::setSeed ( longlong s )
{
   seed = (s ^ Multiplier) & Mask;
   haveNextNextGaussian = false;
}

int GRandom::next ( int bits )
{
   longlong nextseed = (seed * Multiplier + Addend) & Mask;
   seed = nextseed;
   return (int) (nextseed >> (48 - bits));
}

void GRandom::nextBytes ( GFixedVector<byte>& bytes )
{
   int numRequested = bytes.length();
   int numGot = 0;
   int rnd = 0;

   const int BITS_PER_BYTE = 8;
   const int BYTES_PER_INT = sizeof(int);

   for (;;)
   {
      for (int i=0; i<BYTES_PER_INT; i++)
      {
         if (numGot == numRequested)
            return;

         if (i == 0)
            rnd = next(BITS_PER_BYTE * BYTES_PER_INT);
         else
            rnd >>= BITS_PER_BYTE;
         bytes[numGot++] = byte(rnd);
      }
   }
}

int GRandom::nextInt ()
{
   return next(32);
}

int GRandom::nextInt ( int n )
{
   if (n <= 0)
      gthrow_(GIllegalArgumentException("n must be positive"));

   if ((n & -n) == n)  // i.e., n is a power of 2
      return (int) ((n * (longlong) next(31)) >> 31);

   int bits, val;
   do {
      bits = next(31);
      val = bits % n;
   } while(bits - val + (n-1) < 0);
   return val;
}

longlong GRandom::nextLong ()
{
   // it's okay that the bottom word remains signed.
   return (longlong(next(32)) << 32) + next(32);
}

bool GRandom::nextBool ()
{
   return next(1) != 0;
}

float GRandom::nextFloat ()
{
   int i = next(24);
   return i / float(1 << 24);
}

double GRandom::nextDouble ()
{
   longlong l = (longlong(next(26)) << 27) + next(27);
   return l / double(longlong(1) << 53);
}

double GRandom::nextGaussian ()
{
   // See Knuth, ACP, Section 3.4.1 Algorithm C.
   if (haveNextNextGaussian)
   {
      haveNextNextGaussian = false;
      return nextNextGaussian;
   }
   else
   {
      double v1, v2, s;
      do {
         v1 = 2 * nextDouble() - 1; // between -1 and 1
         v2 = 2 * nextDouble() - 1; // between -1 and 1
         s = v1 * v1 + v2 * v2;
      } while (s >= 1 || s == 0);
      double multiplier = GMath::Sqrt(-2 * GMath::Log(s)/s);
      nextNextGaussian = v2 * multiplier;
      haveNextNextGaussian = true;
      return v1 * multiplier;
    }
}

